
/*
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

// $Id: Date.i,v 1.21 2002/03/11 14:08:03 lballabio Exp $

#ifndef quantlib_date_i
#define quantlib_date_i

%include Types.i
%include String.i
%include Vectors.i
%include Null.i

%{
#include <cstdlib>
#include <string>
using QuantLib::Date;

using QuantLib::Day;
using QuantLib::Year;
%}

typedef int Day;
typedef int Year;

// typemap weekdays to corresponding strings

%{
using QuantLib::Weekday;
using QuantLib::Sunday;
using QuantLib::Monday;
using QuantLib::Tuesday;
using QuantLib::Wednesday;
using QuantLib::Thursday;
using QuantLib::Friday;
using QuantLib::Saturday;

using QuantLib::StringFormatter;
using QuantLib::DateFormatter;
%}

%typemap(python,in) Weekday (Weekday temp), const Weekday & (Weekday temp) {
    if (PyString_Check($source)) {
        std::string s(PyString_AsString($source));
        s = StringFormatter::toLowercase(s);
        if (s == "sun" || s == "sunday")
            temp = Sunday;
        else if (s == "mon" || s == "monday")
            temp = Monday;
        else if (s == "tue" || s == "tuesday")
            temp = Tuesday;
        else if (s == "wed" || s == "wednesday")
            temp = Wednesday;
        else if (s == "thu" || s == "thursday")
            temp = Thursday;
        else if (s == "fri" || s == "friday")
            temp = Friday;
        else if (s == "sat" || s == "saturday")
            temp = Saturday;
        else {
            PyErr_SetString(PyExc_TypeError,"not a weekday");
            return NULL;
        }
    } else {
        PyErr_SetString(PyExc_TypeError,"not a weekday");
        return NULL;
    }
    $target = &temp;
};

%typemap(python,out) Weekday, const Weekday & {
    switch (*$source) {
      case Sunday:      $target = PyString_FromString("Sunday");     break;
      case Monday:      $target = PyString_FromString("Monday");     break;
      case Tuesday:     $target = PyString_FromString("Tuesday");    break;
      case Wednesday:   $target = PyString_FromString("Wednesday");  break;
      case Thursday:    $target = PyString_FromString("Thursday");   break;
      case Friday:      $target = PyString_FromString("Friday");     break;
      case Saturday:    $target = PyString_FromString("Saturday");   break;
      default:          throw Error("unknown weekday");
    }
};

%typemap(python,ret) Weekday {
    delete $source;
}


// typemap months to corresponding strings or numbers

%{
using QuantLib::Month;
using QuantLib::January;
using QuantLib::February;
using QuantLib::March;
using QuantLib::April;
using QuantLib::May;
using QuantLib::June;
using QuantLib::July;
using QuantLib::August;
using QuantLib::September;
using QuantLib::October;
using QuantLib::November;
using QuantLib::December;
%}

%typemap(python,in) Month (Month temp), const Month & (Month temp) {
    if (PyString_Check($source)) {
        std::string s(PyString_AsString($source));
        s = StringFormatter::toLowercase(s);
        if (s == "jan" || s == "january")
            temp = January;
        else if (s == "feb" || s == "february")
            temp = February;
        else if (s == "mar" || s == "march")
            temp = March;
        else if (s == "apr" || s == "april")
            temp = April;
        else if (s == "may")
            temp = May;
        else if (s == "jun" || s == "june")
            temp = June;
        else if (s == "jul" || s == "july")
            temp = July;
        else if (s == "aug" || s == "august")
            temp = August;
        else if (s == "sep" || s == "september")
            temp = September;
        else if (s == "oct" || s == "october")
            temp = October;
        else if (s == "nov" || s == "november")
            temp = November;
        else if (s == "dec" || s == "december")
            temp = December;
        else {
            PyErr_SetString(PyExc_TypeError,"not a month");
            return NULL;
        }
    } else if (PyInt_Check($source)) {
        int i = int(PyInt_AsLong($source));
        if (i>=1 && i<=12)
            temp = Month(i);
        else {
            PyErr_SetString(PyExc_TypeError,"not a month");
            return NULL;
        }
    } else {
        PyErr_SetString(PyExc_TypeError,"not a weekday");
        return NULL;
    }
    $target = &temp;
};

%typemap(python,out) Month, const Month & {
    switch (*$source) {
      case January:     $target = PyString_FromString("January");    break;
      case February:    $target = PyString_FromString("February");   break;
      case March:       $target = PyString_FromString("March");      break;
      case April:       $target = PyString_FromString("April");      break;
      case May:         $target = PyString_FromString("May");        break;
      case June:        $target = PyString_FromString("June");       break;
      case July:        $target = PyString_FromString("July");       break;
      case August:      $target = PyString_FromString("August");     break;
      case September:   $target = PyString_FromString("September");  break;
      case October:     $target = PyString_FromString("October");    break;
      case November:    $target = PyString_FromString("November");   break;
      case December:    $target = PyString_FromString("December");   break;
      default:          throw Error("unknown month");
    }
};

%typemap(python,ret) Month {
    delete $source;
}


// typemap time units to corresponding strings

%{
using QuantLib::TimeUnit;
using QuantLib::Days;
using QuantLib::Weeks;
using QuantLib::Months;
using QuantLib::Years;
%}

%typemap(python,in) TimeUnit (TimeUnit temp), const TimeUnit& (TimeUnit temp) {
    if (PyString_Check($source)) {
        std::string s(PyString_AsString($source));
        s = StringFormatter::toLowercase(s);
        if (s == "d" || s == "day" || s == "days")
            temp = Days;
        else if (s == "w" || s == "week" || s == "weeks")
            temp = Weeks;
        else if (s == "m" || s == "month" || s == "months")
            temp = Months;
        else if (s == "y" || s == "year" || s == "years")
            temp = Years;
        else {
            PyErr_SetString(PyExc_TypeError,"not a time unit");
            return NULL;
        }
    } else {
        PyErr_SetString(PyExc_TypeError,"not a time unit");
        return NULL;
    }
    $target = &temp;
};

%typemap(python,out) TimeUnit, const TimeUnit & {
    switch (*$source) {
      case Days:    $target = PyString_FromString("days");   break;
      case Weeks:   $target = PyString_FromString("weeks");  break;
      case Months:  $target = PyString_FromString("months"); break;
      case Years:   $target = PyString_FromString("years");  break;
      default:      throw Error("unknown time unit");
    }
};

%typemap(python,ret) TimeUnit {
    delete $source;
}


// time period

%{
using QuantLib::Period;
%}

class Period {
  public:
    Period(int n, TimeUnit units);
    ~Period();
    int length() const;
    TimeUnit units() const;
};

%addmethods Period {
    String __str__() {
        String s = IntegerFormatter::toString(self->length());
        switch (self->units()) {
          case Days:
            return s + " day(s)";
          case Weeks:
            return s + " week(s)";
          case Months:
            return s + " month(s)";
          case Years:
            return s + " year(s)";
          default:
            return "Unknown period";
        }
        QL_DUMMY_RETURN(String());
    }
}

TypemapVector(Period,Period,PeriodVector,PeriodVector);


// and finally, the Date class

class Date {
  public:
    Date(Day d, Month m, Year y);
    ~Date();
    // access functions
    Weekday weekday() const;
    Day dayOfMonth() const;
    Day dayOfYear() const;        // one-based
    Month month() const;
    Year year() const;
    int serialNumber() const;
    // increment/decrement dates
    Date plusDays(int days) const;
    Date plusWeeks(int weeks) const;
    Date plusMonths(int months) const;
    Date plusYears(int years) const;
    Date plus(int units, TimeUnit) const;
    // leap years
    static bool isLeap(Year y);
    // earliest and latest allowed date
    static Date minDate();
    static Date maxDate();
};

%addmethods Date {
    int monthNumber() {
        return int(self->month());
    }
    int weekdayNumber() {
        return int(self->weekday());
    }
    Date __add__(int days) {
        return self->plusDays(days);
    }
    Date __sub__(int days) {
        return self->plusDays(-days);
    }
    int __cmp__(const Date& other) {
        if (*self < other)
            return -1;
        if (*self == other)
            return 0;
        return 1;
    }
    String __str__() {
        return DateFormatter::toString(*self);
    }
    String __repr__() {
        return "Date(" +
               IntegerFormatter::toString(self->dayOfMonth()) +
               "," +
               IntegerFormatter::toString(int(self->month())) +
               "," +
               IntegerFormatter::toString(self->year()) +
               ")";
    }
    Date __iadd__(int days) {
        return self->plusDays(days);
    }
    Date __isub__(int days) {
        return self->plusDays(-days);
    }
    bool __nonzero__() {
        return (*self != Date());
    }
}

// typemap None to null Date

AllowNoneAsInput(Date,Date);

// build date from serial number

%inline %{
    Date DateFromSerialNumber(int serialNumber) {
        return Date(serialNumber);
    }
%}


// Date vector

TypemapVectorAllowingNone(Date,Date,DateVector,DateVector);
ExportVector(Date,DateVector,DateVector);

%addmethods DateVector {
    String __str__() {
        String s = "(";
        /* Size */ int size_ = self->size();
        if (size_>0) {
            for (/* Size */ int i=0; i<size_-1; i++) {
                s += DateFormatter::toString((*self)[i]);
                s += ", ";
            }
            s += DateFormatter::toString((*self)[size_-1]);
        }
        s += ")";
        return s;
    }
};


#endif
