/* Copyright (C) 2007 Alejandro Álvarez
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA. 
 */
 
#include <QContextMenuEvent>
#include "codeedit.h"

CodeEdit::CodeEdit(QWidget *parent, QString syntaxF): QTextEdit(parent), contextMenu(this)
{
  syntax=new Syntax(this);
  if(!syntaxF.isEmpty() && get_config("syntaxHighlighting")=="true")
    syntax->load(syntaxF);

  setUndoRedoEnabled(true);
  setTabStopWidth(32);
  setFrameStyle(QFrame::NoFrame);
  
  autocompletion_ok=(get_config("autoCompletion")!="false");
  brakets_match_ok=(get_config("bracketsMatch")!="false");

  // ContextMenu
  connect(contextMenu.addAction(tr("Undo")), SIGNAL(triggered()),
	  this, SLOT(undo()));
  connect(contextMenu.addAction(tr("Redo")), SIGNAL(triggered()),
	  this, SLOT(redo()));

  contextMenu.addSeparator();

  connect(contextMenu.addAction(tr("Cut")), SIGNAL(triggered()),
	  this, SLOT(cut()));
  connect(contextMenu.addAction(tr("Copy")), SIGNAL(triggered()),
	  this, SLOT(copy()));
  connect(contextMenu.addAction(tr("Paste")), SIGNAL(triggered()),
	  this, SLOT(paste()));
  connect(contextMenu.addAction(tr("Delete")), SIGNAL(triggered()),
	  this, SLOT(deleteSelection()));

  contextMenu.addSeparator();

  connect(contextMenu.addAction(tr("Select all")), SIGNAL(triggered()),
	  this, SLOT(selectAll()));

  contextMenu.addSeparator();

  connect(contextMenu.addAction(tr("Toggle breakpoint")), SIGNAL(triggered()),
	  this, SLOT(toggleBreakpoint()));
 
  if(autocompletion_ok)
  {
	connect(this->document(), SIGNAL( contentsChange ( int , int , int  )), this, SLOT(buildAutoCompletionList(int , int , int)) );
	connect(&completion, SIGNAL(activated ( const QModelIndex &)), this, SLOT(doCompletion(const QModelIndex &)) );
  }
  
  if(brakets_match_ok) connect(&braketsTimer, SIGNAL(timeout ()), this, SLOT(braketsMatch()));
  braketsTimer.setSingleShot(true);
  braketsTimer.setInterval(100);
  
   connect(this, SIGNAL( cursorPositionChanged() ), this, SLOT( cursorChanged_cb() ) );
  
  auto_indent=true;
  setAcceptDrops(false);
  if(autocompletion_ok)
  {
	completion.setWidget(this);
	completion_model=new QStringListModel(&completion);
	completion.setModel(completion_model);
	completion.setCompletionMode(QCompleter::UnfilteredPopupCompletion);
  }
  
  context_changed_ok=false;
  
  auto_indent=("false"!=get_config("autoindent"));
  automatic_indention_statement_ok=(get_config("autoindent_statements")=="true");
}

CodeEdit::~CodeEdit()
{
	//printf("Borrando la sintaxis\n");
	delete syntax;
	//printf("CodeEdit destruido\n");
}

void CodeEdit::contextMenuEvent(QContextMenuEvent *e)
{
  //setTextCursor(cursorForPosition(e->pos()));
  contextMenu.exec(e->globalPos());
}

void CodeEdit::undo()
{
  document()->undo();
}

void CodeEdit::redo()
{
  document()->redo();
}

void CodeEdit::deleteSelection()
{
  textCursor().removeSelectedText();
}

void CodeEdit::toggleBreakpoint()
{
  int line = 1;
  for(QTextBlock tb = document()->begin();
      tb.isValid();
      line++, tb = tb.next())
  {
    if(tb == textCursor().block())
    {
      emit toggleBreakpoint(line);
      return;
    }
  }
}

bool CodeEdit::event( QEvent * e )
{
	if(QEvent::KeyPress==e->type() )
	{
		QKeyEvent *k=(QKeyEvent *)e;
		if( autocompletion_ok && ( Qt::Key_Left==k->key() || Qt::Key_Right==k->key() ) )
		{
			completion.popup()->hide();
		}
		else if( Qt::Key_Return==k->key() ||  Qt::Key_Enter==k->key())
		{
			if(autocompletion_ok && !completion.popup()->isHidden())
			{
				doCompletion(completion.popup()->currentIndex());
			}
			else if( auto_indent )
			{
				QTextCursor cursor=textCursor();
				int pos=cursor.position();
				cursor.movePosition(QTextCursor::StartOfLine,QTextCursor::KeepAnchor);
				QString line=cursor.selectedText();
				QString start_blank;
				start_blank.append('\n');
				for(int i=0;i<line.length() && (line.at(i)==' ' || line.at(i)=='\t');i++)
					start_blank.append(line.at(i));
				if( automatic_indention_statement_ok )
				{
					QRegExp re("^while[ |(].*|^if[ |(].*|^for .*|^switch[ |(].*|^do$|^try$");
					if(re.exactMatch( line.trimmed() ) )
						start_blank.append("\t");
				}
				
				cursor.setPosition(pos);
				cursor.insertText(start_blank);
				setTextCursor( cursor );
			}
			else
			{
				return QTextEdit::event(e);
			}
			return true;
		}
	}
	return QTextEdit::event(e);
}

void CodeEdit::setAutoindent(bool ai_ok)
{
	auto_indent=ai_ok;
}

bool CodeEdit::getAutoindent() {return auto_indent;}


void CodeEdit::cursorChanged_cb()
{
	if(brakets_match_ok)
	{
		if(!context_changed_ok) braketsTimer.start();
		else
		{
			context_changed_ok=false;
			//braketsTimer.stop();
		}
	}
}


void CodeEdit::braketsMatch()
{
	if(autocompletion_ok) disconnect(this->document(), SIGNAL( contentsChange ( int , int , int  )), this, SLOT(buildAutoCompletionList(int , int , int)) );
	int pos=textCursor().position();
	if(pos>=0) syntax->braketsMacth(pos);
	if(autocompletion_ok) connect(this->document(), SIGNAL( contentsChange ( int , int , int  )), this, SLOT(buildAutoCompletionList(int , int , int)) );
}

void CodeEdit::buildAutoCompletionList(int pos, int /*charsRemoved*/, int /*charsAdded*/ )
{
	if(brakets_match_ok) braketsMatch();
	document()->setModified(true);
	context_changed_ok=true;
	//Cleans brakets match information
	QTextBlock block=document()->findBlock(0);
	//while( block.isValid() )
	//{
		//block.setUserData(NULL);
		//block=block.next();
	//}
	//Buscando palabra a completar
	block=document()->findBlock(pos);
	//int _pos=pos;
	pos-=block.position();
	int i=pos;
	QString text=block.text();
	QRegExp re("([^a-zA-Z_0-9]+)");
	i=re.lastIndexIn(text, i);
	//printf("pos=%d i=%d len=%d\n", pos, i, re.matchedLength());
	if( i==pos ) {completion.popup()->hide();return;}
	QString word_to_complete=text.mid(i+1,pos-i);
	//printf("i=%d word=>%s<\n",i, word_to_complete.toLocal8Bit().data());
	
	if(word_to_complete.length()<3) {completion.popup()->hide();return;}
	
	//Searchs help for command
	//printf("%s\n", word_to_complete.toLocal8Bit().data());
	emit dynamic_help_required(word_to_complete);
	
	//printf("Buscando lista\n");
	block=document()->findBlock(0);
	
	QStringList list;
	//QString match;
	QRegExp rx("([a-zA-Z_0-9]+)");
	
	while( block.isValid() )
	{
		QString text=block.text();
		int i = 0;
		
		//printf("textblock=>%s<\n",text.toLocal8Bit().data());

		while ((i = rx.indexIn(text, i)) != -1) {
			QString word=rx.cap(1);
			//printf("i=%d word=>%s<\n",i, word.toLocal8Bit().data());
			if( word_to_complete!=word && word.startsWith(word_to_complete) && !list.contains(word) ) list << word;
			
			i += rx.matchedLength();
		}
		
		block=block.next();
	}
	
	if(list.isEmpty()) {completion.popup()->hide();return;}
	
	//disconnect(this->document(), SIGNAL( contentsChange ( int , int , int  )), this, SLOT(buildAutoCompletionList(int , int , int)) );
	
	//match=match.mid(word_to_complete.length());
	//QTextCursor cursor=textCursor();
	//cursor.setPosition(_pos+1);
	//cursor.insertText(match);
	//cursor.setPosition(_pos+1,QTextCursor::KeepAnchor);
	//setTextCursor(cursor);
	
	//connect(this->document(), SIGNAL( contentsChange ( int , int , int  )), this, SLOT(buildAutoCompletionList(int , int , int)) );
	
	//printf("Se ordena la lista\n");
	//list.sort();
	
	completion_model->setStringList(list);
	
	QRect _position=cursorRect();
	
	//printf("x=%d y=%d width=%d height=%d\n", _position.x(), _position.y(), _position.width(), _position.height() );
	
	//_position.moveTo(_position.bottomRight() );
	////_position.setWidth(100);
	_position.setWidth(width());
	
	completion.setCompletionPrefix(word_to_complete);
	completion.complete(_position);
	completion.popup()->show();
	completion.popup()->setFocus(Qt::TabFocusReason);
}

void CodeEdit::doCompletion(const QModelIndex &index)
{
	QString word=index.data().toString();
	QString prefix=completion.completionPrefix();
	
	QString suffix=word.mid(prefix.length());
	
	QTextCursor cursor=textCursor();
	cursor.insertText(suffix);
	
	completion.popup()->hide();
}

bool CodeEdit::getbraketsMatchOk()
{
	return syntax->getIsActiveBraketsMacth();
}
