/*
 * CP2D.h
 * $Id: CP2D.h,v 1.4 2003/06/24 14:50:02 anxo Exp $
 *
 * Copyright (C) 1999, 2000 Michael Meissner
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Class CP2D
//  Purpose     : Provides  funcionality


#ifndef __CP2D_H
#define __CP2D_H


// System
///////////
#include <math.h>
#ifdef _MSC_VER
#if _MSC_VER >= 1300
#include <iostream>
#endif
#else
#include <iostream.h>
#endif

// Own
///////////
#include "CV2D.h"
#include "CP3D.h"


// forward declarations
/////////////////////////
class CP3D;




/** This class provides a interface to 2D point 
  *
  * @author Michael Meissner
  */
class CP2D 
{   
public:
  static double epsilon;

  /** Default constructor.
   * The default value of the instantiated point will be (0.0,0.0). */
  CP2D() { m_ard[0] = 0.0;
           m_ard[1] = 0.0; };

  /** Construct new point.
   * The value of the point will be (rdX, rdY). */
  CP2D(double rdX, double rdY) { m_ard[0] = rdX;
                                 m_ard[1] = rdY; }

  /** Copy constructor.
   * The parameters will simply be copied. */
  CP2D(const CP2D& Point) { m_ard[0] = Point[0];
                            m_ard[1] = Point[1]; }


  //////////////////////////
  // OVERLOADED OPERATORS //
  //////////////////////////
 
  /** Cast operator to convert CP2D points to CP3D points.
   * The third component is set zero. */
  operator CP3D() const;

  /** Assign one point to another.*/
  const CP2D& operator=(const CP2D&);

  /** Compares to points for being equal.
   * The result will be 'true'(1) if the two point are indentically
   * up to <= CP2D::epsilon for each component. Otherwise 'false'(0) 
   * will be returned. */
  int operator==(const CP2D&) const;

  /** Compares to points for not being equal.
   * Same as operator== but inverted.
   * @see operator==() */
  int operator!=(const CP2D&) const;

  /** Adds a vector to this point. */
  CP2D& operator+=(const CV2D&);

  /** Subtracts a vector from this point. */
  CP2D& operator-=(const CV2D&);
  
  /** Multiplies a point by a vector. */
  CP2D& operator*=(const CV2D&);

  /** Multiplies a point by a scalar. */
  CP2D& operator*=(double);

  /** Divides a point by a scalar. */
  CP2D& operator/=(double);

  /** Adds a vector to a point. */
  CP2D operator+(const CV2D&) const;

  /** Adds a point to a point. */
  CP2D operator+(const CP2D&) const;  // eigentlich nur fuer affine Punkte
  
  /** Substracts a vector from a point. */
  CP2D operator-(const CV2D&) const;

  /** Substracts a point from a point. */
  CV2D operator-(const CP2D&) const ;

  /** Multiplies a point by a vector. */
  CP2D operator*(const CV2D&) const;  // scaling

  /** Multiplies a point by a scalar. */
  CP2D operator*(double) const;

  /** Divides a point by a vector. */
  CP2D operator/(const CV2D&) const;

  /** Divides a point by a scalar. */
  CP2D operator/(double) const;

  /** Returns the i-th component of the point.
   * The index goes from 0 to 2, 0 stands for the
   * x-coordinate, 1 for the y-coordinate and so on. */
  double& operator [] (int i) { return m_ard[i]; };

  /** Same as above but does not alter anything. */
  double operator[](int i) const { return m_ard[i]; };


  /////////////
  // METHODS //
  /////////////
  
  /** Returns the value of the minimal point component. */
  double getMinComponent(void) const    { return m_ard[getMinComponentCoord()]; };
 
  /** Returns the value of the minimal point component. */
  double getAbsMinComponent(void) const { return m_ard[getAbsMinComponentCoord()]; };
 
  /** Returns the value of the maximal point component. */
  double getMaxComponent(void) const    { return m_ard[getMaxComponentCoord()]; };
 
  /** Returns the value of the maximal point component. */
  double getAbsMaxComponent(void) const { return m_ard[getAbsMaxComponentCoord()]; };
 
  /** Returns the coordinate index of the minial point component. */
  int getMinComponentCoord(void) const;

  /** Returns the coordinate index of the minial point component (using fabs). */
  int getAbsMinComponentCoord(void) const;

  /** Returns the coordinate index of the maximum point component. */
  int getMaxComponentCoord(void) const;

  /** Returns the coordinate index of the maximum point component (using fabs). */
  int getAbsMaxComponentCoord(void) const;

  /** Converts a point to a vector.
   * It's implemented as 'get'-method to prevent
   * implicit casting by the compiler. */
  CV2D getCV2D() const;

  /** Returns the x-coordinate of the point. */
  double getX(void) const  { return m_ard[0]; };

  /** Returns the y-coordinate of the point. */
  double getY(void) const  { return m_ard[1]; };

  /** Sets the x-coordinate of the point to 'rdX'. */
  void setX(double rdX)    { m_ard[0] = rdX; return; };

  /** Sets the y-coordinate of the point to 'rdX'. */
  void setY(double rdY)    { m_ard[1] = rdY; return; };

  /** Set the values of the point.
   * The value of the point will be (rdX, rdY, rdZ). */
  void setCoord(double rdX, double rdY) { m_ard[0] = rdX; 
                                          m_ard[1] = rdY; 
                                          return; };

  /////////////
  // FRIENDS //
  /////////////
  
  /** Returns the affine combination of the points and vectors. */
  friend CP2D AffinComb(const CP2D&, double, const CP2D&);

  /** Returns the affine combination of the points and vectors. */
  friend CP2D AffinComb2(double r, const CP2D& R, 
                         double s, const CP2D& S) {
                                  return CP2D(r*R[0] + s*S[0],
                                              r*R[1] + s*S[1]); };

  /** Returns the distance between two points. */
  friend double dist(const CP2D&, const CP2D&);

  /** Returns the square of the distance between two points. */
  friend double quaddist(const CP2D&, const CP2D&);

  /** Returns the minimum of all components of two points. */
  friend CP2D Min(const CP2D&, const CP2D&);

  /** Returns the maximum of all components of two points. */
  friend CP2D Max(const CP2D&, const CP2D&);

  /** Returns a point being the result of multiplying a scalar and a point. */
  friend CP2D operator*(double, const CP2D&);

  /** Returns the point in the middle between two points. */
  friend CP2D MidPoint(const CP2D&, const CP2D&);


  // output to stderr
  /////////////////////
  /** Prints a point to the standard output. */
  void print() const;

  /** Same as above. But more useful for streams. */
  friend inline ostream& operator<<(ostream&, const CP2D&);

  /** Reads a point from the given stream. */
  friend inline istream& operator>>(istream&, CP2D&);


protected:
  double m_ard[2];
};



// Function   : operator=
// Parameters : const CP2D& p1
// Purpose    : assign another point to this point
// Comments   : 
inline const CP2D& CP2D::operator=(const CP2D& p1)
/*******************************************************************/
{
  m_ard[0] = p1.m_ard[0];
  m_ard[1] = p1.m_ard[1];
  return *this;
}



// Function   : getCV2D
// Parameters : 
// Purpose    : Convert CP2D to CV2D
// Comments   : 
inline CV2D CP2D::getCV2D() const
/*******************************************************************/
{
  return CV2D(m_ard[0], m_ard[1]);
}



// Function   : operator<<
// Parameters : ostream& s, const CP2D& pnt
// Purpose    : 
// Comments   : 
inline ostream& operator<<(ostream& s, const CP2D& pnt)
/*******************************************************************/
{   
  return s << "(" << pnt.m_ard[0] << "," << pnt.m_ard[1] << ")"; 
}



// Function   : operator>>
// Parameters : istream& s, CP2D& pnt
// Purpose    : 
// Comments   : 
inline istream& operator>>(istream& s, CP2D& pnt)
/*******************************************************************/
{   
  char c;
  return s >> c >> pnt.m_ard[0] >> c >> pnt.m_ard[1] >> c;
}

#endif
