/*
 * CBoundingBox3D.h
 * $Id: CBoundingBox3D.h,v 1.5 2003/06/24 14:50:02 anxo Exp $
 *
 * Copyright (C) 1999, 2000 Rainer Jaeger, Michael Meissner
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Class CBoundingBox3D
//  Purpose     : Provides  funcionality


#ifndef __CBOUNDING_BOX3D_H
#define __CBOUNDING_BOX3D_H


// Qt
///////

// System
///////////

// Own
///////////
#include "CP3D.h"
#include "CP4D.h"


/** This class provides a interface to a 
  * 3D bounding box.
  *
  * @author Rainer Jaeger and Michael Meissner
  */
class CBoundingBox3D {
 public:

  //////////////////
  // CONSTRUCTORS //
  //////////////////

  /** default constructor (Bounding box will be set to 0,0,0 to 0,0,0. */
  CBoundingBox3D(void) { m_LowerLeft = CP3D(0.0, 0.0, 0.0);
                       m_UpperRight = CP3D(0.0, 0.0, 0.0); };

  /** Constructor using the two given points. The minimum and the
   * maximum of all components of these points will be used to set
   * the bounding box. */
  CBoundingBox3D(double, double, double, double, double, double);

  /** Constructor using the two given points. The minimum and the
   * maximum of all components of these points will be used to set
   * the bounding box. */
  CBoundingBox3D(const CP3D &minVec, const CP3D &maxVec);

  /** Constructor using the two given points. The minimum and the
   * maximum of all components of these points will be used to set
   * the bounding box. */
  CBoundingBox3D(const CP4D &minVec, const CP4D &maxVec);

  /** copy constructor. */
  CBoundingBox3D(const CBoundingBox3D &bbox);

  /** Initializes common stuff, called by all constructors. */
  void CommonConstructor(CP3D, CP3D); 

  /** default destructor. */
  ~CBoundingBox3D();


  static double epsilon;

  //////////////////////
  // MEMBER FUNCTIONS //
  //////////////////////

   /** Set the bounding box to the specified bounding box. */
   void setBBox(CBoundingBox3D &bbox);

   /** operator= assigns specified bounding box to this. */
   const CBoundingBox3D& operator=(const CBoundingBox3D& bbox);

   /** operator+ returns sum of both bounding boxes. */
   CBoundingBox3D operator+(const CBoundingBox3D& bbox) const;

   /** operator+= adds specified bounding box to this. */
   CBoundingBox3D& operator+=(const CBoundingBox3D& bbox);

   /** Adds the specified point to the bounding box, extending it,
    *  to include the point. */
   void addPoint(const CP3D&);

   /** Adds the specified point to the bounding box, extending it,
    *  to include the point. */
   void addPoint(const CP4D&);

   /** Move the bounding box by the specified vector. */
   void move(const CV3D&);

   /** Scale the bounding box by the specified vector. CV3D(1.0, 1.0, 1.0)
    *  would let the bounding box unmodified, negative components of
    *  the vector are illegal but are not checked against. */
   void scale(const CV3D&);

  /** Returns the lower left corner point of the bounding box. */
  CP3D getLowerLeft() const  { return m_LowerLeft; }

  /** Returns the upper right corner point of the bounding box. */
  CP3D getUpperRight() const { return m_UpperRight; }

  /** Returns the corner vertex.
   * 0=000 (xyz), 1=100, 2=010, 3=110, 4=001, 5=101, 6=011, 7=111. */
  CP3D getCornerVertex(int) const;

  /** Returns the range in the given dimension. */
  double getSize(int) const;

  /** Returns the minimum of the range in all dimensions. */
  double getMinSize(void) const;

  /** Returns the maximum of the range in all dimensions. */
  double getMaxSize(void) const;

  /** Returns the center of the bounding box. */
  const CP3D getCenter(void) const;

  /** Returns the center of the bounding box in the specified dimension. */
  double getCenter(int nDimension) { return 0.5 * (m_UpperRight[nDimension] 
                                                   + m_LowerLeft[nDimension]); };

  /** Returns the radius of the minimal sphere enclosing the bounding box. */
  double getOuterRadius(void) const { return 0.5 * sqrt(3*getMaxSize()*getMaxSize()); };  

  /** Returns the radius of the maximum sphere being enclosing in the bounding box. */
  double getInnerRadius(void) const { return 0.5 * sqrt(3*getMinSize()*getMinSize()); };

  /** Returns the radius of the maximum sphere being enclosing in the bounding box. */
  double getDiagonal(void) const;

  /** Returns the radius of the maximum sphere being enclosing in the bounding box. */
  double getVolume(void) const;

  /** Returns whether a point/vector is within the bounding box. */
  bool isInside(const CP3D&) const;

  /** Returns whether a point/vector is within the bounding box. */
  bool isInside(const CP4D&) const;

  /** Compares two bounding boxes. */
  bool operator==(const CBoundingBox3D &cSource) const {
    return (m_LowerLeft == cSource.m_LowerLeft) &&
           (m_UpperRight == cSource.m_UpperRight);
  };


  //////////////////////
  // HELPER FUNCTIONS //
  //////////////////////

  /** Prints the parameters of CBoundingBox3D to standard out. */
  void print(void) const;

  /** Same as above. But more useful for streams. */
  friend ::ostream& operator<<(::ostream&, const CBoundingBox3D&);

  /** Reads a vector from the given stream. */
  friend ::istream& operator>>(::istream&, CBoundingBox3D&);

protected:
  CP3D m_LowerLeft;
  CP3D m_UpperRight;

}; // class CBoundingBox3D

#endif // __CBOUNDING_BOX_H
