/***************************************************************************
  graindelegate.cpp
  -------------------
  Grain delegate editor
  -------------------
  Copyright 2006, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <QAbstractItemModel>
#include <QComboBox>
#include <QDoubleSpinBox>

#include "data.h"
#include "grain.h"
#include "graindelegate.h"

const int extra = 1; // always display one extra blank row

GrainDelegate::GrainDelegate(QObject *parent) : QItemDelegate(parent) {}

GrainDelegate::~GrainDelegate() {}

QWidget *GrainDelegate::createEditor(QWidget *parent,
                                     const QStyleOptionViewItem &/*option*/,
                                     const QModelIndex &index) const
{
    QComboBox *combo;
    QDoubleSpinBox *spin;
    QString suffix;

    // can only edit name on blank row
    bool blank = index.row() >= index.model()->rowCount() - extra;

    // different kind of editor for each column
    switch (index.column()) {
      case Grain::NAME:
          combo = new QComboBox(parent);
          combo->setEditable(true);
          combo->addItem(QString());
          combo->addItems(Data::instance()->grainsList());
          combo->installEventFilter(const_cast<GrainDelegate*>(this));
          return combo;

      case Grain::QUANTITY:
          if (blank) return 0;
          spin = new QDoubleSpinBox(parent);
          spin->setDecimals(2);
          spin->setRange(0.00, 100.00);
          spin->setSingleStep(0.25);
          suffix = " " + Data::instance()->defaultGrainUnit().symbol();
          spin->setSuffix(suffix);
          spin->installEventFilter(const_cast<GrainDelegate*>(this));
          return spin;

      case Grain::EXTRACT:
          if (blank) return 0;
          spin = new QDoubleSpinBox(parent);
          spin->setDecimals(3);
          spin->setRange(1.000, 1.100);
          spin->setSingleStep(0.001);
          spin->installEventFilter(const_cast<GrainDelegate*>(this));
          return spin;

      case Grain::COLOR:
          if (blank) return 0;
          spin = new QDoubleSpinBox(parent);
          spin->setDecimals(1);
          spin->setRange(0.0, 500.0);
          spin->setSingleStep(1.0);
          spin->setSuffix(Resource::DEGREE);
          spin->installEventFilter(const_cast<GrainDelegate*>(this));
          return spin;

      case Grain::USE:
          if (blank) return 0;
          combo = new QComboBox(parent);
          combo->setEditable(false);
          combo->addItems(Grain::useStringList());
          combo->installEventFilter(const_cast<GrainDelegate*>(this));
          return combo;

      default:
          return 0;
    }
}

void GrainDelegate::setEditorData(QWidget *editor,
                                  const QModelIndex &index) const
{
    QComboBox *combo;
    QDoubleSpinBox *spin;
    int comboindex;

    QVariant value = index.model()->data(index, Qt::EditRole);

    // different kind of editor for each column
    switch (index.column()) {
      case Grain::NAME:
          combo = static_cast<QComboBox*>(editor);
          if (!combo) return;
          comboindex = combo->findText(value.toString());
          if (comboindex > 0) {
              combo->setCurrentIndex(comboindex);
          } else {
              combo->setEditText(value.toString());
          }
          break;

      case Grain::QUANTITY:
      case Grain::EXTRACT:
      case Grain::COLOR:
          spin = static_cast<QDoubleSpinBox*>(editor);
          if (!spin) return;
          spin->setValue(value.toDouble());
          break;

      case Grain::USE:
          combo = static_cast<QComboBox*>(editor);
          if (!combo) return;
          combo->setCurrentIndex(combo->findText(value.toString()));
          break;

      default:
          QItemDelegate::setEditorData(editor, index);
          break;
    }
}

void GrainDelegate::setModelData(QWidget *editor, QAbstractItemModel *model,
                      const QModelIndex &index) const
{
    QComboBox *combo;
    QDoubleSpinBox *spin;
    QVariant value;

    // different kind of editor for each column
    switch (index.column()) {
      case Grain::NAME:
      case Grain::USE:
          combo = static_cast<QComboBox*>(editor);
          if (!combo) return;
          value = combo->currentText();
          model->setData(index, value);
          break;

      case Grain::QUANTITY:
      case Grain::EXTRACT:
      case Grain::COLOR:
          spin = static_cast<QDoubleSpinBox*>(editor);
          if (!spin) return;
          value = spin->value();
          model->setData(index, value);
          break;

      default:
          QItemDelegate::setModelData(editor, model,index);
          break;
    }
}

void GrainDelegate::updateEditorGeometry(QWidget *editor,
                              const QStyleOptionViewItem &option,
                              const QModelIndex &/*index*/) const
{
    editor->setGeometry(option.rect);
}
