/***************************************************************************
  qbrew.cpp
  -------------------
  A brewing recipe calculator for Unix/X11
  -------------------
  begin         September 20th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include "qbrew.h"

#include <qapplication.h>
#include <qdir.h>
#include <qfiledialog.h>
#include <qlabel.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qpaintdevicemetrics.h>
#include <qprinter.h>
#include <qstatusbar.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qwhatsthis.h>

#include <qwindowsstyle.h>
#include <qplatinumstyle.h>
#include <qmotifstyle.h>
#include <qcdestyle.h>
#include <qmotifplusstyle.h>
#include <qsgistyle.h>
#include <stepstyle.h>

#include "qbrewdoc.h"
#include "qbrewview.h"
#include "qbrewcalc.h"
#include "helpwindow.h"
#include "hydrometertool.h"
#include "setupdialog.h"
#include "preferences.h"
#include "oldsettings.h"

#include "filesave.xpm"
#include "fileopen.xpm"
#include "filenew.xpm"
#include "fileprint.xpm"
#include "qbrew.xpm"

using namespace AppResource;

static QMap<QString, int> stylemap; // TODO: I don't like this as a static...

//////////////////////////////////////////////////////////////////////////////
// Construction and Initialization                                          //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// QBrew()
// -------
// Constructor

QBrew::QBrew(const QString &filename) : view_(0), doc_(0), calc_(0), preferences_(0), filename_(filename),
    filemenu_(0), optionsmenu_(0), toolsmenu_(0), helpmenu_(0), toolbar_(0)

{
    setCaption(ID_TITLE + VERSION);
    setIcon(QPixmap(qbrewIcon));

    // setup themable styles
    stylemap.insert(STYLES[STYLE_MOTIF], STYLE_MOTIF);
    stylemap.insert(STYLES[STYLE_PLATINUM], STYLE_PLATINUM);
    stylemap.insert(STYLES[STYLE_WINDOWS], STYLE_WINDOWS);
    stylemap.insert(STYLES[STYLE_CDE], STYLE_CDE);
    stylemap.insert(STYLES[STYLE_STEP], STYLE_STEP);
    stylemap.insert(STYLES[STYLE_DEFAULT], STYLE_DEFAULT);
    stylemap.insert(STYLES[STYLE_MOTIFPLUS], STYLE_MOTIFPLUS);
    stylemap.insert(STYLES[STYLE_SGI], STYLE_SGI);

    // other initializations
    initPreferences();
    initMenuBar();
    initToolBar();
    initStatusBar();

    doc_ = new QBrewDoc(this);
    if ((filename_ != ID_DEFAULT_FILE) && (filename_ != ""))
        doc_->load(filename);
    else
        doc_->setStyle(preferences_->getString(ID_PREF_RECIPE_STYLE, ID_PREF_RECIPE_STYLE_DEFAULT));
    calc_ = new QBrewCalc(preferences_, doc_);
    view_ = new QBrewView(this, "view_", doc_, calc_);
    setCentralWidget(view_);
    // new documents have not been modified...
    doc_->unsetModified();

    // setup connections
    connect(doc_,SIGNAL(documentModified()),this,SLOT(slotDocumentModified()));
    connect(calc_,SIGNAL(calcDone()),doc_,SIGNAL(styleChanged())); // piggybacked signals
}

//////////////////////////////////////////////////////////////////////////////
// ~Qbrew
// ------
// Destructor

QBrew::~QBrew()
{
    if (preferences_) delete preferences_;
    if (doc_) delete doc_;
    if (view_) delete view_;
    if (calc_) delete calc_;
    // the menus and toolbars get destroyed by Qt
}

//////////////////////////////////////////////////////////////////////////////
// initPreferences()
// -----------------
// Initialize the application preferences

void QBrew::initPreferences()
{
    // load preferences from rc file
    preferences_ = new Preferences(QDir::homeDirPath() + "/" + ID_PREFERENCES_FILE, PACKAGE, VERSION);

    // do any stuff that needs to be done
    switch (stylemap[preferences_->getString(ID_PREF_WIDGET_STYLE, ID_PREF_WIDGET_STYLE_DEFAULT)]) {
        case STYLE_MOTIF:
            QApplication::setStyle(new QMotifStyle);
            break;
        case STYLE_PLATINUM:
            QApplication::setStyle(new QPlatinumStyle);
            break;
        case STYLE_WINDOWS:
            QApplication::setStyle(new QWindowsStyle);
            break;
        case STYLE_CDE:
            QApplication::setStyle(new QCDEStyle);
            break;
        case STYLE_STEP:
            QApplication::setStyle(new StepStyle);
            break;
        case STYLE_MOTIFPLUS:
            QApplication::setStyle(new QMotifPlusStyle);
            break;
        case STYLE_SGI:
            QApplication::setStyle(new QSGIStyle);
            break;
        default:
            break;
    }
}

//////////////////////////////////////////////////////////////////////////////
// initMenuBar()
// -------------
// Initialize the menu bar

void QBrew::initMenuBar()
{
    // TODO: don't hardcode the accelerator keys
    // file menu
    filemenu_ = new QPopupMenu();
    filemenu_->insertItem(tr("&New"), this, SLOT(slotFileNew()), CTRL+Key_N, ID_FILE_NEW);
    filemenu_->insertItem(tr("&Open..."), this, SLOT(slotFileOpen()), CTRL+Key_O, ID_FILE_OPEN);
    // TODO: Open Recent...
    filemenu_->insertSeparator();
    filemenu_->insertItem(tr("&Save"), this, SLOT(slotFileSave()), CTRL+Key_S, ID_FILE_SAVE);
    filemenu_->insertItem(tr("Save &as..."), this, SLOT(slotFileSaveAs()), 0, ID_FILE_SAVE_AS);
    filemenu_->insertSeparator();
    filemenu_->insertItem(tr("&Print..."), this, SLOT(slotFilePrint()), CTRL+Key_P, ID_FILE_PRINT);
    filemenu_->insertSeparator();
    filemenu_->insertItem(tr("&Quit"), this, SLOT(slotFileQuit()), CTRL+Key_Q, ID_FILE_QUIT);
    // options menu
    optionsmenu_ = new QPopupMenu();
    optionsmenu_->setCheckable(true);
    optionsmenu_->insertItem(tr("Tool&bar"), this, SLOT(slotOptionsToolbar()), 0, ID_OPTIONS_TOOLBAR);
    optionsmenu_->insertItem(tr("&Statusbar"), this, SLOT(slotOptionsStatusbar()), 0, ID_OPTIONS_STATUSBAR);
    optionsmenu_->insertSeparator();
    // TODO: Configure Key Bindings...
    optionsmenu_->insertItem(tr("&Configure qbrew..."), this, SLOT(slotOptionsSetup()), 0, ID_OPTIONS_SETUP);
    optionsmenu_->insertSeparator();
    optionsmenu_->insertItem(tr("Sa&ve Options"), this, SLOT(slotOptionsSaveSettings()), 0, ID_OPTIONS_SAVESETTINGS);

    optionsmenu_->setItemChecked(ID_OPTIONS_TOOLBAR, true);
    optionsmenu_->setItemChecked(ID_OPTIONS_STATUSBAR, true);
    // tools menu
    toolsmenu_ = new QPopupMenu();
    toolsmenu_->insertItem(tr("&Hydrometer Correction..."), this, SLOT(slotToolsHydrometer()), 0, ID_TOOLS_HYDROMETER);
    // help menu
    helpmenu_ = new QPopupMenu();
    helpmenu_->insertItem(tr("&Contents..."), this, SLOT(slotHelpContents()), Key_F1,ID_HELP_CONTENTS);
    helpmenu_->insertItem(tr("&Primer..."), this, SLOT(slotHelpPrimer()), 0,ID_HELP_PRIMER);
    helpmenu_->insertSeparator();
    helpmenu_->insertItem(tr("About Qt..."), this, SLOT(slotHelpAboutQt()), 0,ID_HELP_ABOUT_QT);
    helpmenu_->insertItem(tr("About ") + ID_TITLE + "...", this, SLOT(slotHelpAbout()), 0,ID_HELP_ABOUT);
    // insert submenus into main menu
    menuBar()->insertItem(tr("&File"), filemenu_);
    menuBar()->insertItem(tr("&Options"), optionsmenu_);
    menuBar()->insertItem(tr("&Tools"), toolsmenu_);
    menuBar()->insertSeparator();
    menuBar()->insertItem(tr("&Help"), helpmenu_);

    // enable/disable appropriate items in menu
    filemenu_->setItemEnabled(ID_FILE_SAVE, false);
    // check if preferences for toolbar is set
    if (preferences_->getBool(ID_PREF_TOOLBAR, ID_PREF_TOOLBAR_DEFAULT)) {
        optionsmenu_->setItemChecked(ID_OPTIONS_TOOLBAR, true);
    } else {
        optionsmenu_->setItemChecked(ID_OPTIONS_TOOLBAR, false);
    }
    // now check preferences for statusbar
    if (preferences_->getBool(ID_PREF_STATUSBAR, ID_PREF_STATUSBAR_DEFAULT)) {
        optionsmenu_->setItemChecked(ID_OPTIONS_STATUSBAR, true);
    } else {
        optionsmenu_->setItemChecked(ID_OPTIONS_STATUSBAR, false);
    }

    // setup menu connections
    connect(filemenu_, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
    connect(optionsmenu_, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
    connect(toolsmenu_, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
    connect(helpmenu_, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
}

//////////////////////////////////////////////////////////////////////////////
// initToolBar()
// -------------
// Initialize the toolbar

void QBrew::initToolBar()
{
    setRightJustification(true);
    toolbar_ = new QToolBar("Main Toolbar", this);
    QToolButton* fileNew = new QToolButton(QPixmap(filenewxpm), tr("New File"), 0, this,
        SLOT(slotFileNew()), toolbar_);
    QToolButton* fileOpen = new QToolButton(QPixmap(fileopenxpm), tr("Open File"), 0, this,
        SLOT(slotFileOpen()), toolbar_);
    QToolButton* fileSave = new QToolButton(QPixmap(filesavexpm), tr("Save File"), 0, this,
        SLOT(slotFileSave()), toolbar_);
    QToolButton* filePrint = new QToolButton(QPixmap(fileprintxpm), tr("Print File"), 0, this,
        SLOT(slotFilePrint()),  toolbar_);
    // TODO: | print | cut copy paste | help
    toolbar_->addSeparator();

    QWhatsThis::whatsThisButton(toolbar_);
    QWhatsThis::add(fileNew, tr("Click this button to create a new file.\n\n"
            "You can also select the New command from the File menu."));
    QWhatsThis::add(fileOpen, tr("Click this button to open a new file.\n\n"
            "You can also select the Open command from the File menu."));
    QWhatsThis::add(fileSave, tr("Click this button to save the file you are "
            "editing.  You will be prompted for a file name.\n\n"
            "You can also select the Save command from the File menu."));
    QWhatsThis::add(filePrint, tr("Click this button to print the current "
            "recipe.\n\n"
            "You can also select the Print command from the File menu."));

    // set a filler widget so that right justified toolbars look okay
    QLabel* mtfiller =  new QLabel(toolbar_, "mtfiller");
    mtfiller->setBackgroundMode(PaletteButton);
    toolbar_->setStretchableWidget(mtfiller);

    // now show or hide toolbar depending on initial setting
    if (preferences_->getBool(ID_PREF_TOOLBAR, true)) toolbar_->show();
    else toolbar_->hide();
}

//////////////////////////////////////////////////////////////////////////////
// initStatusBar()
// ---------------
// Initialize the status bar

void QBrew::initStatusBar()
{
    statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
    // now show or hide statusbar depending on initial setting
    if (preferences_->getBool(ID_PREF_STATUSBAR, true)) statusBar()->show();
    else statusBar()->hide();
}

//////////////////////////////////////////////////////////////////////////////
// File Menu Implementation                                                 //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// slotFileNew()
// -------------
// Received to create a new recipe

void QBrew::slotFileNew()
{
    if (doc_->isModified()) {
        // file needs to be saved, what do we do
        switch (querySave()) {
            case 0: // yes, save the file
                slotFileSave();
                break;
            case 1: // no, don't save the file
                break;
            case 2: // cancel creating new file
                statusBar()->message(tr("Canceled..."), 2000);
                // exit function
                return;
        }
    }
    // create a new file
    statusBar()->message(tr("Creating new recipe..."));
    doc_->newDoc(preferences_->getString(ID_PREF_RECIPE_STYLE, ID_PREF_RECIPE_STYLE_DEFAULT));
    filemenu_->setItemEnabled(ID_FILE_SAVE, false);
    // no file name yet, so set it as junk
    filename_ = ID_DEFAULT_FILE;
    setCaption( ID_TITLE + (QString)" - " + filename_);
    statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// slotFileOpen()
// --------------
// Received to open a recipe

void QBrew::slotFileOpen()
{
    if (doc_->isModified()) {
        // file needs to be saved, what do we do
        switch (querySave()) {
            case 0: // yes, save the file
                slotFileSave();
                break;
            case 1: // no, don't save the file
                break;
            case 2: // cancel creating new file
                statusBar()->message(tr("Canceled..."), 2000);
                return;
        }
    }
    // open the file
    statusBar()->message(tr("Opening file..."));
    QString fname = QFileDialog::getOpenFileName(0, ID_FILE_FILTER, this);
    // keep information about the file
    QFileInfo *finfo = new QFileInfo(fname);
    if (!fname.isEmpty()) {
        // file dialog returned a file name
        if (doc_->load(fname)) {
            // load was successful
            filemenu_->setItemEnabled(ID_FILE_SAVE, false);
            setCaption(ID_TITLE + (QString)" - " + finfo->fileName());
            statusBar()->message(tr("Loaded document: ") + finfo->fileName() , 2000);
            // save name of file
            filename_ = fname;
        } else {
            // load was unsuccessful
            QMessageBox::warning(this, ID_TITLE, ID_TITLE + tr(" was unable to load the file ") + finfo->fileName());
            statusBar()->message(tr("Error in loading ") + finfo->fileName() , 2000);
        }
    } else {
        // file dialog didn't return a file name
        statusBar()->message(tr("Loading aborted") , 2000);
    }
    delete finfo;
}

//////////////////////////////////////////////////////////////////////////////
// slotFileSave()
// --------------
// Received to save a recipe

void QBrew::slotFileSave()
{
    if (doc_->isNew())
    {
        slotFileSaveAs();
    } else {
        // file exists so save it
        statusBar()->message(tr("Saving file..."));
        if (doc_->save(filename_)) {
            // successful in saving file
            filemenu_->setItemEnabled(ID_FILE_SAVE, false);
            statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
        } else {
            // error in saving file
            QMessageBox::warning(this, ID_TITLE, ID_TITLE + tr(" was unable to save the file " + filename_));
            statusBar()->message(tr("Error in saving file"), 2000);
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// slotFileSaveAs()
// --------------
// Received to save a recipe under a new name

void QBrew::slotFileSaveAs()
{
    statusBar()->message(tr("Saving recipe under new filename..."));
    QString fname = QFileDialog::getSaveFileName(0, ID_FILE_FILTER, this);
    QFileInfo *finfo = new QFileInfo(fname);
    if (!fname.isEmpty()) {
        // we got a valid filename
        if (doc_->save(fname)) {
            // successfully saved
            filemenu_->setItemEnabled(ID_FILE_SAVE, false);
            setCaption(ID_TITLE + (QString)" - " + finfo->fileName());
            statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
            // save name of file
            filename_ = fname;
        } else {
            // error in saving
            QMessageBox::warning(this, ID_TITLE, ID_TITLE + tr(" was unable to save the file " + finfo->fileName()));
            statusBar()->message(tr("Error in saving file"), 2000);
        }
    } else {
        // no file name chosen
        statusBar()->message(tr("Saving aborted"), 2000);
    }
    delete finfo;
}

//////////////////////////////////////////////////////////////////////////////
// slotFilePrint()
// ---------------
// Received to print the current recipe. Much of this method derived from
// Qt example programs, including "helpviewer" copyright 1992-2000 Troll Tech AS.

void QBrew::slotFilePrint()
{
    // TODO: eventually will have an export function to export to html. At that time
    // switch this to print html/richtext.
    statusBar()->message(tr("Printing..."));

    QPrinter* printer = new QPrinter();
    printer->setFullPage(true);

    if (printer->setup())
    {
        QPainter painter(printer);
        QPaintDeviceMetrics metrics(painter.device());

        QString line;
        int tpos;
        int ypos = 12;              // y position for each line
        const int margin = 18;      // eighteen point margin

        // painter.begin(printer);
        QFontMetrics fm = painter.fontMetrics();
        QFont font("times", 10);    // serif font best for printouts
        font.setPointSize(10);
        painter.setFont(font);      // use fixed width font
        QString text = getRecipeText();
        int pageno = 1;             // keep track of pages
        // for each line of text...
        while (text.length() > 0) {
            // get line of text
            tpos = text.find('\n');
            if (tpos > 0) {
                line = text.left(tpos);
                text.remove(0, tpos+1);
            } else {
                // get last line if text doesn't end in newline
                line = text; text = "";
            }
            // is there space for this line on page?
            if ((margin + ypos) > (metrics.height() - margin)) {
                statusBar()->message("Printing (page " + QString::number(++pageno) + ")...");
                printer->newPage();         // no more room on this page
                ypos = 12;                  // back to top of page
            }
            // print the line
            painter.drawText(margin, margin + ypos, metrics.width(), fm.lineSpacing(),
                    ExpandTabs | DontClip, line );
            // update paper position
            ypos += fm.lineSpacing();
        }
        painter.end();        // send job to printer
        statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
    } else {
        // user chose not to print
        statusBar()->message("Printing cancelled", 2000);
    }
    delete printer;
}

//////////////////////////////////////////////////////////////////////////////
// slotFileQuit()
// --------------
// Received to quit the application

void QBrew::slotFileQuit()
{
    if (doc_->isModified()) {
        // file needs to be saved, what do we do
        switch (querySave()) {
            case 0: // yes, save the file
                slotFileSave();
                break;
            case 1: // no, go ahead and exit
                break;
            case 2: // cancel exit
                statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
                return;
        }
    }
    // TODO: do we need to do anything else before we quit?
    qApp->quit();
}

//////////////////////////////////////////////////////////////////////////////
// Options Menu Implementation                                              //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// slotOptionsToolbar()
// --------------------
// Received to toggle toolbar status

void QBrew::slotOptionsToolbar()
{
    if (toolbar_->isVisible()) {
        toolbar_->hide();
        optionsmenu_->setItemChecked(ID_OPTIONS_TOOLBAR, false);
        preferences_->setBool(ID_PREF_TOOLBAR, false);
    } else {
        toolbar_->show();
        optionsmenu_->setItemChecked(ID_OPTIONS_TOOLBAR, true);
        preferences_->setBool(ID_PREF_TOOLBAR, true);
    }
    statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// slotOptionsSatusbar()
// --------------------
// Received to toggle statusbar status

void QBrew::slotOptionsStatusbar()
{
    if (statusBar()->isVisible()) {
        statusBar()->hide();
        optionsmenu_->setItemChecked(ID_OPTIONS_STATUSBAR, false);
        preferences_->setBool(ID_PREF_STATUSBAR, false);
    } else {
        statusBar()->show();
        optionsmenu_->setItemChecked(ID_OPTIONS_STATUSBAR, true);
        preferences_->setBool(ID_PREF_STATUSBAR, true);
    }
    statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// slotOptionsSetup()
// ------------------
// Received to display the setup dialog

void QBrew::slotOptionsSetup()
{
    statusBar()->message(tr("Configuring ") + ID_TITLE + "...");
    SetupDialog* dialog = new SetupDialog(this, "SetupDialog");

    // setup widget style combobox
    QMap<QString, int>::Iterator it;
    for(it=stylemap.begin(); it!=stylemap.end(); ++it) dialog->addWidgetStyle(it.key());

    // setup widget style combobox
    dialog->setRecipeStyleBox(calc_->stylesList());

    // load current preferences into dialog
    dialog->setCaption(ID_TITLE + tr(" - Configure"));
    dialog->setQBrewDir(preferences_->getString(ID_PREF_QBREWDIR, ID_PREF_QBREWDIR_DEFAULT));
    dialog->setWidgetStyle(preferences_->getString(ID_PREF_WIDGET_STYLE, ID_PREF_WIDGET_STYLE_DEFAULT));
    dialog->setBatch(preferences_->getDouble(ID_PREF_BATCH, ID_PREF_BATCH_DEFAULT));
    dialog->setEfficiency(preferences_->getDouble(ID_PREF_EFFICIENCY, ID_PREF_EFFICIENCY_DEFAULT));
    dialog->setRecipeStyle(preferences_->getString(ID_PREF_RECIPE_STYLE, ID_PREF_RECIPE_STYLE_DEFAULT));
    dialog->setTinseth(preferences_->getBool(ID_PREF_TINSETH, ID_PREF_TINSETH_DEFAULT));

    // save some stuff we are interested in...
    double efficiency = preferences_->getDouble(ID_PREF_EFFICIENCY, ID_PREF_EFFICIENCY_DEFAULT);
    bool tinseth = preferences_->getBool(ID_PREF_TINSETH, ID_PREF_TINSETH_DEFAULT);
    unsigned oldwidgetstyle = stylemap[preferences_->getString(ID_PREF_WIDGET_STYLE, ID_PREF_WIDGET_STYLE_DEFAULT)];

    // execute the dialog
    int ret = dialog->exec();
    if (ret == QDialog::Accepted) {
        preferences_->setString(ID_PREF_QBREWDIR, dialog->qbrewDir());
        preferences_->setString(ID_PREF_WIDGET_STYLE, dialog->widgetStyle());
        preferences_->setDouble(ID_PREF_BATCH, dialog->batch());
        preferences_->setDouble(ID_PREF_EFFICIENCY, dialog->efficiency());
        preferences_->setString(ID_PREF_RECIPE_STYLE, dialog->recipeStyle());
        preferences_->setBool(ID_PREF_TINSETH, dialog->tinseth());
    }

    // activate new style
    unsigned newwidgetstyle = stylemap[preferences_->getString(ID_PREF_WIDGET_STYLE, ID_PREF_WIDGET_STYLE_DEFAULT)];
    if (oldwidgetstyle != newwidgetstyle) switch (newwidgetstyle) {
        case STYLE_MOTIF:
            QApplication::setStyle(new QMotifStyle);
            break;
        case STYLE_PLATINUM:
            QApplication::setStyle(new QPlatinumStyle);
            break;
        case STYLE_WINDOWS:
            QApplication::setStyle(new QWindowsStyle);
            break;
        case STYLE_CDE:
            QApplication::setStyle(new QCDEStyle);
            break;
        case STYLE_STEP:
            QApplication::setStyle(new StepStyle);
            break;
        case STYLE_MOTIFPLUS:
            QApplication::setStyle(new QMotifPlusStyle(true));
            break;
        case STYLE_SGI:
            QApplication::setStyle(new QSGIStyle);
            break;
        default:
            QApplication::setStyle(new QMotifStyle);
    }

    // do we need to recalc?
    if ((efficiency != preferences_->getDouble(ID_PREF_EFFICIENCY, ID_PREF_EFFICIENCY_DEFAULT)) ||
    	(tinseth != preferences_->getBool(ID_PREF_TINSETH, ID_PREF_TINSETH_DEFAULT))) {	
    	calc_->slotRecalc();
    }
    statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// slotOptionsSaveSettings()
// -------------------------
// Received to save the application settings

void QBrew::slotOptionsSaveSettings()
{
    statusBar()->message(tr("Saving settings..."));
    preferences_->flush();
    statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// Tools Menu Implementation                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// slotToolsHydrometer()
// ---------------------
// A utility dialog for hydrometer correction

void QBrew::slotToolsHydrometer()
{
    (new HydrometerTool(this))->show();
    statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// Help Menu Implementation                                                 //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// slotHelpContents()
// ------------------
// Received to display the application manual

void QBrew::slotHelpContents()
{
    QString home = QString(preferences_->getString(ID_PREF_QBREWDIR, ID_PREF_QBREWDIR_DEFAULT))
        + tr(ID_HELP_FILE);
    (new HelpWindow(home, ".", 0, "QBrewHelp"))->show();
    statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// slotHelpPrimer()
// ------------------
// Received to display the brewing primer

void QBrew::slotHelpPrimer()
{
    QString home = QString(preferences_->getString(ID_PREF_QBREWDIR, ID_PREF_QBREWDIR_DEFAULT))
        + tr(ID_PRIMER_FILE);
    (new HelpWindow(home, ".", 0, "QBrewPrimer"))->show();
    statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// slotHelpAboutQt()
// ------------------
// Received to display About Qt dialog

void QBrew::slotHelpAboutQt()
{
    QMessageBox::aboutQt(this, ID_TITLE);
    statusBar()->message(ID_DEFAULT_MESSAGE, 2000);
}

//////////////////////////////////////////////////////////////////////////////
// slotHelpAbout()
// ------------------
// Received to display the About dialog

void QBrew::slotHelpAbout()
{
    QMessageBox::about(this, ID_TITLE, ID_APP_ABOUT);
}

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous Slots                                                      //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// slotDocumentModified()
// ----------------------
// Received when document is modified

void QBrew::slotDocumentModified()
{
    filemenu_->setItemEnabled(ID_FILE_SAVE, true);
}

//////////////////////////////////////////////////////////////////////////////
// statusCallback()
// ----------------
// Received when pointer is over a menu item, displays help line

void QBrew::statusCallback(int id)
{
    switch (id) {
    case ID_FILE_NEW        :  statusHelpMessage(tr("Creates a new document")); break;
    case ID_FILE_OPEN       :  statusHelpMessage(tr("Opens an existing document")); break;
    case ID_FILE_SAVE       :  statusHelpMessage(tr("Save the document")); break;
    case ID_FILE_SAVE_AS    :  statusHelpMessage(tr("Save the document under a new name")); break;
    case ID_FILE_PRINT      :  statusHelpMessage(tr("Prints the file")); break;
    case ID_FILE_QUIT       :  statusHelpMessage(tr("Quits the program")); break;
    case ID_OPTIONS_TOOLBAR :  statusHelpMessage(tr("Enables / disables the main Toolbar")); break;
    case ID_OPTIONS_STATUSBAR   :  statusHelpMessage(tr("Enables / disables the Statusbar")); break;
    case ID_OPTIONS_SETUP   :  statusHelpMessage(tr("Configures general settings for QBrew")); break;
    case ID_OPTIONS_SAVESETTINGS    :  statusHelpMessage(tr("Save the options")); break;
    case ID_TOOLS_HYDROMETER    :  statusHelpMessage(tr("A hydrometer correction utility")); break;
    case ID_HELP_CONTENTS   :  statusHelpMessage(tr("Displays the main help contents")); break;
    case ID_HELP_PRIMER     :  statusHelpMessage(tr("Displays a brewing primer")); break;
    case ID_HELP_ABOUT_QT   :  statusHelpMessage(tr("All about the Qt library")); break;
    case ID_HELP_ABOUT      :  statusHelpMessage(tr("All about QBrew")); break;
    }
}

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// querySave()
// -----------
// Ask the user if they want to save their work before going on

int QBrew::querySave()
{
    return QMessageBox::information(this, ID_TITLE + " - Save?",
        tr("Do you wish to save your work first?"), tr("Yes"), tr("No"), tr("Cancel"), 0, 2);
}

//////////////////////////////////////////////////////////////////////////////
// getRecipeText()
// ---------------
// Get the ascii text of the recipe, for printing and exporting

QString QBrew::getRecipeText()
{
    // TODO: need to implement metric as well as english
    QString buffer = "\n";

    // title stuff
    buffer += tr("Recipe: ") + doc_->title() + '\n';
    buffer += tr("Brewer: ") + doc_->brewer() + '\n';
    buffer += tr("Style: ") + doc_->style() + '\n';
    buffer += tr("Batch: ") + QString::number(double(doc_->size()), 'f', 2);
    if (doc_->mash()) buffer += tr(", Mashed");
    buffer += "\n\n";

    // style stuff
    buffer += tr("Recipe Gravity: ") + QString::number(calc_->OG(), 'f', 3) + '\n';
    buffer += tr("Recipe Bitterness: ") + QString::number(calc_->IBU(), 'f', 0) + '\n';
    buffer += tr("Recipe Color: ") + QString::number(calc_->SRM(), 'f', 0) + CHAR_LATIN_DEGREE + '\n';
    buffer += tr("Estimated FG: ") + QString::number(calc_->FGEstimate(), 'f', 3) + '\n';
    buffer += tr("Alcohol by Volume: ") + QString::number(calc_->ABV() * 100.0, 'f', 1) + "%\n";
    buffer += tr("Alcohol by Weight: ") + QString::number(calc_->ABW() * 100.0, 'f', 1) + "%\n\n";

    // grains
    GrainList *grainlist = doc_->grainList();
    GrainList::Iterator itg;
    for (itg=grainlist->begin(); itg != grainlist->end(); ++itg) {
        buffer += (*itg).name().leftJustify(30, ' ');
        buffer += (*itg).quantityString() + tr(" lbs, ");
        buffer += (*itg).useString() + '\n';
    }
    buffer += '\n';

    // hops
    HopsList *hopslist = doc_->hopsList();
    HopsList::Iterator ith;
    for (ith=hopslist->begin(); ith != hopslist->end(); ++ith) {
        buffer += (*ith).name().leftJustify(30, ' ');
        buffer += (*ith).quantityString() + tr(" oz, ");
        buffer += (*ith).form() + ", ";
        buffer += (*ith).timeString() + tr(" minutes\n");
    }
    buffer += '\n';

    // misc ingredients
    MiscIngredientList *misclist = doc_->miscIngredientList();
    MiscIngredientList::Iterator itm;
    for (itm=misclist->begin(); itm != misclist->end(); ++itm) {
        buffer += (*itm).name().leftJustify(30, ' ');
        buffer += (*itm).quantityString() + ", ";
        buffer += (*itm).notes() + '\n';
    }

    buffer += ID_TITLE + " " + VERSION;
    return buffer;
}

//////////////////////////////////////////////////////////////////////////////
// getBatchSettings()
// ------------------
// Retrieve the batch size

double QBrew::getBatchSetting()
{
    return preferences_->getDouble(ID_PREF_BATCH, ID_PREF_BATCH_DEFAULT);
}

//////////////////////////////////////////////////////////////////////////////
// statusHelpMessage()
// -------------------
// Briefly display a help message on the status bar

void QBrew::statusHelpMessage(const char *text) { statusBar()->message(text, 2000); }
