/***************************************************************************
 $RCSfile$
                             -------------------
    cvs         : $Id$
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "transferlist.h"
#include <assert.h>
#include <qstring.h>
#include <qpainter.h>
#include <qdatetime.h>
#include <qheader.h>

#include <gwenhywfar/debug.h>


TransferListViewItem::TransferListViewItem(TransferListView *parent,
                                           RefPointer<Transfer> t)
:QListViewItem(parent)
,_transfer(t){
  assert(parent);
  _populate();
}



TransferListViewItem::TransferListViewItem(const TransferListViewItem &item)
:QListViewItem(item) {
  _transfer=item._transfer;
}



TransferListViewItem::TransferListViewItem(TransferListView *parent,
                                           QListViewItem *after,
                                           RefPointer<Transfer> t)
:QListViewItem(parent, after)
,_transfer(t){
  _populate();
}



TransferListViewItem::~TransferListViewItem(){
}



void TransferListViewItem::setColour(const QString &c){
  _colour=c.latin1();
  repaint();
}



RefPointer<Transfer> TransferListViewItem::getTransfer(){
  return _transfer;
}



void TransferListViewItem::redrawItem() {
  int i;

  for (i=0; i<TRANSFER_LISTVIEW_COLUMNS; i++) {
    delete _cell[i];
    _cell[i]=0;
  }

  _populate();
  setup();
}



void TransferListViewItem::_populate() {
  QString tmp;
  int i;
  const std::list<std::string> sl;
  std::list<std::string>::const_iterator it;
  int ht;
  const GWEN_TIME *ti;
  const AB_VALUE *v;

  i=0;

  switch(_transfer.ref().getStatus()) {
  case Transfer::StatusEnqueued:
    setColour(QString("yellow"));
    break;
  case Transfer::StatusPending:
    setColour(QString("light green"));
    break;
  case Transfer::StatusOk:
  case Transfer::StatusManualOk:
    setColour(QString("light green"));
    break;
  case Transfer::StatusError:
  case Transfer::StatusManualError:
    setColour(QString("red"));
    break;
  case Transfer::StatusAborted:
    setColour(QString("orange"));
    break;
  default:
    break;
  }

  for (i=0; i<TRANSFER_LISTVIEW_COLUMNS; i++)
    _cell[i]=0;

  i=0;
  ht=0;

  // date
  tmp="<qt>";
  ti=_transfer.ref().getDate();
  if (!ti)
    ti=_transfer.ref().getValutaDate();
  if (ti) {
    int year, month, day;

    if (!GWEN_Time_GetBrokenDownDate(ti, &day, &month, &year)) {
      QDate t;

      t=QDate(year, month+1, day);
      tmp+=t.toString(Qt::LocalDate);
    }
  }
  tmp+="</qt>";
  _cell[i]=new QSimpleRichText(tmp, listView()->font());
  _cell[i]->setWidth(listView()->columnWidth(i));
  ht=(ht<_cell[i]->height())?_cell[i]->height():ht;
  i++;

  // institute
  tmp="<qt>";
  tmp+=QString::fromUtf8(_transfer.ref().getRemoteBankCode().c_str());
  tmp+="</qt>";
  _cell[i]=new QSimpleRichText(tmp, listView()->font());
  _cell[i]->setWidth(listView()->columnWidth(i));
  ht=(ht<_cell[i]->height())?_cell[i]->height():ht;
  i++;

  // account
  tmp="<qt>";
  tmp+=QString::fromUtf8(_transfer.ref().getRemoteAccountNumber().c_str());
  tmp+="</qt>";
  _cell[i]=new QSimpleRichText(tmp, listView()->font());
  _cell[i]->setWidth(listView()->columnWidth(i));
  ht=(ht<_cell[i]->height())?_cell[i]->height():ht;
  i++;

  // other name
  tmp="<qt>";
  for (it=_transfer.ref().getRemoteName().begin();
       it!=_transfer.ref().getRemoteName().end();
       it++) {
    tmp+=QString::fromUtf8((*it).c_str());
    tmp+="<br>";
  } // for

  // description
  for (it=_transfer.ref().getPurpose().begin();
       it!=_transfer.ref().getPurpose().end();
       it++) {
    tmp+="<br>";
    tmp+=QString::fromUtf8((*it).c_str());
  } // for

  tmp+="</qt>";
  _cell[i]=new QSimpleRichText(tmp, listView()->font());
  _cell[i]->setWidth(listView()->columnWidth(i));
  ht=(ht<_cell[i]->height())?_cell[i]->height():ht;
  i++;

  // type
  tmp="<qt>";
  switch(_transfer.ref().getTransferType()) {
  case Transfer::TransferType_Unknown:
    tmp+=QWidget::tr("unknown");
    break;
  case Transfer::TransferType_Simple:
    tmp+=QWidget::tr("transfer");
    break;
  case Transfer::TransferType_DebitNote:
    tmp+=QWidget::tr("debit note");
    break;
  case Transfer::TransferType_EuTransfer:
    tmp+=QWidget::tr("EU transfer");
    break;
  }
  tmp+="</qt>";
  _cell[i]=new QSimpleRichText(tmp, listView()->font());
  _cell[i]->setWidth(listView()->columnWidth(i));
  ht=(ht<_cell[i]->height())?_cell[i]->height():ht;
  i++;

  // value
  tmp="<qt>";
  v=_transfer.ref().getValue();
  if (v) {
    tmp+=QString::number(AB_Value_GetValueAsDouble(v), 'f', 2);
    tmp+=" ";
    tmp+=QString::fromUtf8(AB_Value_GetCurrency(v));
  }
  tmp+="</qt>";
  _cell[i]=new QSimpleRichText(tmp, listView()->font());
  _cell[i]->setWidth(listView()->columnWidth(i));
  ht=(ht<_cell[i]->height())?_cell[i]->height():ht;
  i++;

  // status
  tmp="<qt>";
  switch(_transfer.ref().getStatus()) {
  case Transfer::StatusEnqueued:
    tmp+=QWidget::tr("enqueued");
    break;
  case Transfer::StatusPending:
    tmp+=QWidget::tr("sent to the bank");
    break;
  case Transfer::StatusOk:
    tmp+=QWidget::tr("ok");
    break;
  case Transfer::StatusError:
    tmp+=QWidget::tr("error");
    break;
  case Transfer::StatusManualOk:
    tmp+=QWidget::tr("ok (override)");
    break;
  case Transfer::StatusManualError:
    tmp+=QWidget::tr("error (override)");
    break;
  case Transfer::StatusAborted:
    tmp+=QWidget::tr("aborted");
    break;
  default:
    tmp+=QWidget::tr("unknown");
    break;
  }
  tmp+="</qt>";
  _cell[i]=new QSimpleRichText(tmp, listView()->font());
  _cell[i]->setWidth(listView()->columnWidth(i));
  ht=(ht<_cell[i]->height())?_cell[i]->height():ht;
  i++;


  _ht=ht;
  setHeight(ht);

  //_colour="light green";
}



// return string for sort
QString TransferListViewItem::key(int column, bool ascending) const{
  QString result("");

  switch(column) {
  case 0: { // date
    const GWEN_TIME *ti;

    ti=_transfer.ref().getDate();
    if (!ti)
      ti=_transfer.ref().getValutaDate();
    if (ti) {
      result=QString::number(GWEN_Time_Seconds(ti));
    }
    break;
  }

  case 1: // institute
    result=_transfer.ref().getRemoteBankCode().c_str();
    break;

  case 2: // account
    result=_transfer.ref().getRemoteAccountNumber().c_str();
    break;

  case 3: { // othername
    std::list<std::string>::const_iterator it;

    for (it=_transfer.ref().getRemoteName().begin();
         it!=_transfer.ref().getRemoteName().end();
         it++) {
      result+=QString::fromUtf8((*it).c_str()) + " ";
    }
    for (it=_transfer.ref().getPurpose().begin();
         it!=_transfer.ref().getPurpose().end();
         it++) {
      result+=QString::fromUtf8((*it).c_str()) + " ";
    }
    break;
  }

  case 4: // type
    result=text(column);
    break;

  case 5: { // value
    const AB_VALUE *v;

    v=_transfer.ref().getValue();
    if (v) {
      result=QString::number(AB_Value_GetValueAsDouble(v), 'f', 2);
      result+=" ";
      result+=QString::fromUtf8(AB_Value_GetCurrency(v));
    }
    break;
  }
  case 6:
    break;
  default:
    result="";
    break;
  } // switch

  return result;
}



void TransferListViewItem::setup(){
  widthChanged();
  setHeight(_ht);
}



void TransferListViewItem::paintCell(QPainter * pnt,
                                     const QColorGroup & cg,
                                     int column,
                                     int width,
                                     int align){
  QSimpleRichText *p;
  QBrush br;

  if (column>=TRANSFER_LISTVIEW_COLUMNS) {
    DBG_WARN(0, "WARNING: Column out of range (%d)",column);
    return;
  }
  p=_cell[column];
  if (!p) {
    DBG_DEBUG(0, "TransferListEntry::paintCell():Invalid pointer !!");
    return;
  }
  //pnt->eraseRect(pnt->viewport());
  if (isSelected())
    br=QBrush(QColor("light blue"));
  //br=cg.brush(QColorGroup::Highlight);
  else {
    if (!_colour.empty()) {
      br=QBrush(QColor(_colour.c_str()));
    }
    else {
      br=cg.brush(QColorGroup::Light);
    }
  }

  p->draw(pnt,
          listView()->itemMargin(),
          listView()->itemMargin(),
          QRect(0, 0, width, height()),
          cg, &br);
}



int TransferListViewItem::width(const QFontMetrics &,
                                const QListView *lv,
                                int column) const {
  int w;

  if (column>=TRANSFER_LISTVIEW_COLUMNS) {
    DBG_DEBUG(0, "WARNING: Column out of range (%d)",column);
    return 0;
  }
  if (!_cell[column])
    return 0;

  w=_cell[column]->widthUsed()+lv->itemMargin()*2;
  return w;
}













TransferListView::TransferListView(QWidget *parent, const char *name)
:QListView(parent, name){
  setAllColumnsShowFocus(true);
  setShowSortIndicator(true);
  addColumn(QWidget::tr("Date"),-1);
  addColumn(QWidget::tr("Institute"),-1);
  addColumn(QWidget::tr("Account"),-1);
  addColumn(QWidget::tr("Name/Purpose"),-1);
  addColumn(QWidget::tr("Transfertype"),-1);
  addColumn(QWidget::tr("Value"),-1);
  addColumn(QWidget::tr("Status"),-1);
  setColumnAlignment(5, Qt::AlignRight);
  header()->setTracking(false);
  QObject::connect(header(), SIGNAL(sizeChange(int, int, int)),
                   this, SLOT(slotHeaderChanged(int, int, int)));

}



TransferListView::~TransferListView(){
}



void TransferListView::addTransfer(RefPointer<Transfer> t){
  TransferListViewItem *entry;

  entry=new TransferListViewItem(this, t);
}



void
TransferListView::addTransfers(const std::list<RefPointer<Transfer> > &ts){
  std::list<RefPointer<Transfer> >::const_iterator it;

  for (it=ts.begin(); it!=ts.end(); it++) {
    addTransfer(*it);
  } /* for */
}



RefPointer<Transfer> TransferListView::getCurrentTransfer() {
  TransferListViewItem *entry;

  entry=dynamic_cast<TransferListViewItem*>(currentItem());
  if (!entry) {
    DBG_DEBUG(0, "No item selected in list.");
    return 0;
  }
  return entry->getTransfer();
}



std::list<RefPointer<Transfer> > TransferListView::getSelectedTransfers(){
  std::list<RefPointer<Transfer> > ts;

  getSelectedTransfersToList(ts);
  return ts;
}



void TransferListView::getSelectedTransfersToList(std::list<RefPointer<Transfer> > &ts){
  TransferListViewItem *entry;

  // Create an iterator and give the listview as argument
  QListViewItemIterator it(this);
  // iterate through all items of the listview
  for (;it.current();++it) {
    if (it.current()->isSelected()) {
      entry=dynamic_cast<TransferListViewItem*>(it.current());
      if (entry)
        ts.push_back(entry->getTransfer());
    }
  } // for
}



void TransferListView::slotHeaderChanged(int section,
                                         int oldSize,
                                         int newSize){
  TransferListViewItem *entry;

  // Create an iterator and give the listview as argument
  QListViewItemIterator it(this);
  // iterate through all items of the listview
  for (;it.current();++it) {
    entry=dynamic_cast<TransferListViewItem*>(it.current());
    if (entry) {
      entry->redrawItem();
    }
  } // for
}
































