#!/usr/bin/env python
# -*- coding: utf-8 -*-
# setup.py
# This file is part of python-otr
# 
# Copyright (C) 2008 - Kjell Braden <fnord@pentabarf.de>
# 
# python-otr is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# python-otr is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with python-otr.  If not, see <http://www.gnu.org/licenses/>.
 

import os
import sys
from glob import glob
from copy import copy
from unittest import TextTestRunner, TestLoader, TestSuite
import distutils
from distutils.core import setup, Extension, Command
from distutils.command.build_py import build_py as BuildPyCommand

def recursive_set_path(obj, path):
    if isinstance(obj, TestSuite):
        for test in obj:
            recursive_set_path(test, path)
    else:
        obj.path = path

class MyBuildPyCommand(BuildPyCommand):
    def run(self):
        self.run_command("build_ext")
        BuildPyCommand.run(self)

class TestCommand(Command):
    user_options = [
        ('build-lib=', 'b',
        "directory for compiled extension modules")
        ]

    def initialize_options(self):
        self.build_lib = None
        self.build_temp = None

    def finalize_options(self):
        self.set_undefined_options('build',
            ('build_lib','build_lib'),
            ('build_temp','build_temp'))

    def run(self):
        # the OTR module needs to be built before we can test it
        self.run_command('build')

        # find the tests
        testfiles = [ ]
        for t in glob(os.path.join('tests', '*.py')):
            if not t.endswith(r'_init__.py'):
                testfiles.append('.'.join([
                        'tests',
                        os.path.splitext(os.path.basename(t))[0]
                    ]))

        # we need to be able to find the module
        old_path = copy(sys.path)
        sys.path.insert(0, self.build_lib)

        # load the tests
        tests = TestLoader().loadTestsFromNames(testfiles)

        # pass the temporary path to the tests
        recursive_set_path(tests, self.build_temp)

        # run the tests
        t = TextTestRunner(verbosity=1)
        t.run(tests)
    
        # reset the search path
        sys.path = old_path

setup(
    cmdclass = { 'test':TestCommand, 'build_py':MyBuildPyCommand},
    name = "python-otr",
    version = "0.2.1",
    author = "Kjell Braden",
    author_email = "fnord@pentabarf.de",
    description = "python bindings for libOTR, for more information on OTR "
        "see http://www.cypherpunks.ca/otr/",
    url = "http://python-otr.pentabarf.de",
    download_url = "https://launchpad.net/python-otr/+download",
    license = "GPL-3",
    classifiers = ["License :: OSI Approved :: GNU General Public License "
        "(GPL)", "Development Status :: 2 - Pre-Alpha",
        "Intended Audience :: Developers",
        "Topic :: Communications :: Chat",
        "Operating System :: OS Independent",
        "Programming Language :: C",
        "Programming Language :: Python",
        "Topic :: Software Development :: Libraries :: Python Modules"],
    ext_modules = [
        Extension("_otr", ["otr.i"],
            libraries=["otr"],
            define_macros=[("SWIG_PYTHON_OUTPUT_TUPLE",None),
                    ("SWIG_PYTHON_SAFE_CSTRINGS", None)]
        )
    ],
    py_modules = ["otr"],
)
