# arch-tag: 4ccb7253-910d-41a0-8653-d1197af1601a
# Copyright (C) 2004 David Allouche <david@allouche.net>
#               2005 Canonical Limited.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""
Internal module providing output parsing functionality.

This module implements some of public interface for the
pybaz_ package. But for convenience reasons the author prefers
to store this code in a file separate from ``__init__.py``.

.. _pybaz: arch-module.html

This module is strictly internal and should never be used.
"""

__all__ = []

from pathname import *
from _escaping import *


__all__.extend((
    'Chatter',
    'classify_chatter',
    ))

class Chatter(object):
    """Chatter lines in ``tla`` output.

    :ivar text: chatter text without the ``"* "`` prefix.
    :type text: str
    """

    def __init__(self, text):
        """Create a chatter item.

        :param text: tla-style chatter line, starting with ``"* "``
        :type text: str
        """
        assert text[:2] == '* '
        self.text = text[2:]

    def __str__(self):
        """tla-style chatter line with ``"* "`` prefix."""
        return "* " + self.text


def classify_chatter(iter):
    """Classify chatter in a sequence of strings.

    Generator that yields Chatter objects for chatter lines, and
    yields other lines verbatim.

    :param iter: iterable of str
    :rtype: iterable of `Chatter` or str
    """
    for line in iter:
        if line.startswith("* "):
            yield Chatter(line)
        else:
            yield line


__all__.extend((
    'MergeOutcome',
    'TreeChange',
    ))

class MergeOutcome(object):
    """Abstract base class for changeset application summary output lines.

    :ivar name: name of the corresponding file
    :type name: `FileName` or `DirName`
    :ivar directory: does the change applies to a directory?
    :type directory: bool
    """

    def _parse(self, text, pad, filepre, dirpre=None):
        self._pad = pad
        assert 2 == len(filepre)
        assert dirpre is None or 2 == len(dirpre)
        prefix = text[0:2+len(pad)]
        if prefix == filepre+pad:
            self._directory = False
        elif dirpre is not None and prefix == dirpre+pad:
            self._directory = True
        else:
            raise AssertionError('bad prefix: %r' % prefix)
        name = name_unescape(text[len(prefix):])
        self._name = (FileName, DirName)[self._directory](name)

    def __str__(self):
        """tla-style changeset application line."""
        raise NotImplementedError

    def _to_str(self, filepre, dirpre=None):
        if self._directory:
            return ''.join((dirpre, self._pad, name_escape(self.name)))
        else:
            return ''.join((filepre, self._pad, name_escape(self.name)))

    def _get_directory(self):
        return self._directory
    directory = property(_get_directory)

    def _get_name(self):
        return self._name
    name = property(_get_name)


class TreeChange(MergeOutcome):
    """Abstract base class for ``changes`` summary output lines."""

    def __str__(self):
        """tla-style change line,"""
        raise NotImplementedError


__all__.extend((
    'FileAddition',
    'FileDeletion',
    'FileModification',
    'FilePermissionsChange',
    'FileRename',
    'SymlinkModification',
    ))

class FileAddition(TreeChange):
    """Changeset summary line for a new file.

    :ivar name: name of the added file or directory.
    :ivar directory: is ``name`` a directory name?
    """

    def __init__(self, text, pad):
        self._parse(text, pad, 'A ', 'A/')

    def __str__(self):
        return self._to_str('A ', 'A/')


class FileDeletion(TreeChange):
    """Changeset  summary line for a deleted file.

    :ivar name: name of the deleted file or directory.
    :ivar directory: is ``name`` a directory name?
    """

    def __init__(self, text, pad):
        self._parse(text, pad, 'D ', 'D/')

    def __str__(self):
        return self._to_str('D ', 'D/')


class FileModification(TreeChange):
    """Changeset summary line for file whose contents were modified.

    :ivar name: name of the modified file.
    :ivar binary: is ``name`` a  binary file?
    :type binary: bool
    :ivar directory: always False
    """

    def __init__(self, text, pad):
        self._directory = False
        self._pad = pad
        prefix = text[0:2+len(pad)]
        if prefix == 'M '+pad: self.binary = False
        elif prefix == 'Mb'+pad: self.binary = True
        else: raise AssertionError('bad prefix: %r' % prefix)
        name = name_unescape(text[len(prefix):])
        self._name = FileName(name)

    def __str__(self):
        if self.binary:
            return 'Mb%s%s' % (self._pad, name_escape(self.name))
        else:
            return 'M %s%s' % (self._pad, name_escape(self.name))


class FilePermissionsChange(TreeChange):
    """Changeset summary line for a change in permissions.

    :ivar name: name of the file or directory whose permissions changed.
    :ivar directory: is ``name`` a directory name?
    """

    def __init__(self, text, pad):
        if text.startswith('--/ '):
            self._patching = True
            self._directory = True
            name = name_unescape(text[len('--/ '):])
            self._name = DirName(name)
        else:
            self._patching = False
            self._parse(text, pad, '--', '-/')

    def __str__(self):
        if self._patching:
            assert self._directory
            return '--/ ' + name_escape(self.name)
        else:
            return self._to_str('--', '-/')


class FileRename(TreeChange):
    """Changeset summary line for a renaming.

    :ivar name: new name of the moved file or directory.
    :ivar oldname: old name of the moved file or directory.
    :type oldname: `FileName`, `DirName`
    :ivar directory: are ``name`` and ``oldname`` directory names?
    """

    def __init__(self, text, pad):
        self._pad = pad
        prefix = text[0:2+len(pad)]
        if prefix == '=>'+pad:
            self._directory = False
        elif prefix == '/>'+pad:
            self._directory = True
        else:
            raise AssertionError('bad prefix: %r' % prefix)
        oldnew = text[len(prefix):].split('\t')
        assert len(oldnew) == 2
        ctor = (FileName, DirName)[self.directory]
        self.oldname, self._name = map(ctor, map(name_unescape, oldnew))

    def __str__(self):
        if self.directory:
            format = '/>%s%s\t%s'
        else:
            format = '=>%s%s\t%s'
        names = map(name_escape, (self.oldname, self.name))
        return format % ((self._pad,) + tuple(names))


class SymlinkModification(TreeChange):
    """Changeset summary line for a symlink modification.

    :ivar name: name of the modified symbolic link.
    :ivar directory: always False
    """

    def __init__(self, text, pad):
        self._parse(text, pad, '->')

    def __str__(self):
        return self._to_str('->')


__all__.extend((
    'classify_changeset_creation',
    'PatchConflict',
    'classify_changeset_application',
    ))

def classify_changeset_creation(lines, pad=' '):
    """Classify the output of a changeset creation command.

    :param lines: incremental output from a changeset creation command.
    :type lines: iterator of str
    :param pad: padding text between prefix and file name.
    :type pad: str
    :rtype: Iterable of `Chatter`, `TreeChange`, str

    :note: diff output (*e.g.* ``changes --diffs``) is not supported.
    """
    for line in classify_chatter(lines):
        if isinstance(line, Chatter) or len(line) < 2:
            yield line
        elif line[:3] in ('A  ', 'A/ '):
            yield FileAddition(line, pad)
        elif line[:3] in ('D  ', 'D/ '):
            yield FileDeletion(line, pad)
        elif line[:3] in ('M  ', 'Mb '):
            yield FileModification(line, pad)
        elif line[:3] in ('-- ', '-/ '):
            yield FilePermissionsChange(line, pad)
        elif line[:3] in ('=> ', '/> '):
            yield FileRename(line, pad)
        elif line[:3] == '-> ':
            yield SymlinkModification(line, pad)
        else:
            yield line


class PatchConflict(MergeOutcome):
    """Changeset application summary line for a patch conflict.

    :ivar name: name of the file where the patch conflict occurred.
    :ivar directory: always False
    """
    def __init__(self, text):
        self._parse(text, '  ', 'C ')

    def __str__(self):
        return self._to_str('C ')


def classify_changeset_application(lines):
    """Classify the output of a change-producing command.

    :param lines: incremental output from a changeset application command.
    :type lines: iterable of str
    :rtype: iterable of `Chatter`, `MergeOutcome`, str

    :note: diff output (*e.g.* ``changes --diffs``) is not supported.
    """
    for line in classify_changeset_creation(lines, pad='  '):
        if isinstance(line, (Chatter, TreeChange)) or len(line) < 2:
            yield line
        elif line[:4] == '--/ ':
            yield FilePermissionsChange(line, pad=' ')
        elif line[:4] == 'C   ':
            yield PatchConflict(line)
        else:
            yield line
