# -*- perl -*-
# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released "AS IS WITH ALL FAULTS"
# and WITHOUT ANY WARRANTIES under the terms of the GNU Lesser
# General Public License, Version 2.1, a copy of which can be
# found in the "COPYING" file of this distribution.

# $Id: data.t,v 1.1 2000/11/23 23:36:17 muaddib Exp $

=head1 NAME

data.t - script to test AtomicData datatypes

=head1 SYNOPSIS

 perl data.t [-v] [-n] [-t] [-q] [-f F<file name>]

=head1 DESCRIPTION

Will load all AtomicData types and test standard and edge cases for
cannocilazation, verification and formating. Checks function outputs
against expected outputs and reports on the results of the comparison.
All cases are coded in the file and one must edit the file to add
cases. Future versions may add support for reading in test files.

=head2 -v

Tells the program to run in verbose mode. Will report successes,
errors and timing information instead of just errors. This option will
overide a -q.

=head2 -q

Tells the program to run in quit mode. Will only tell if there were
errors or not, and not detail the errors encountered.

=head2 -t

Will toggle timeing information. That is, will supress it if -v is
active and report it if -q is active. This option can then be used
to get timing information alone.

=head2 -f F<file name>

Will output the report to F<file name>. This will overwrite F<file
name> if the file is already present.

=head1 SEE ALSO

Summary.doc (with AtomicData datatypes)
AtomicData.pm and children

=head1 AUTHOR

Zane Rockenbaugh

=cut

use Test;
use Benchmark;
use Getopt::Std;
use AtomicData::Types;
use strict;

use vars qw($opt_v $opt_q $opt_t $opt_f);

#used in the report func, see doc
my ($VERBOSE, $SUCCESS, $TIMING, $ERROR, $FINAL) = (4,3,2,1,0);
my $report = ''; #this will hold all the reporting information
my $error = 0; #will set to true if there are any problems

getopt('f');
getopt('q');
getopt('v');
getopt('t');

#######
# BEGIN DATA
#######

use vars qw($test_data);

#need to get the current YY so that we have good edge cases
my $year    = (localtime())[5] + 1900;
my $year1   = $year - 49;
my $year2   = $year + 50;
my $year1_2 = $year1 % 100;
my $year2_2 = $year2 % 100;

$test_data = 
  {
   'AtomicData::AnyThing' =>
   [
    ['',''],
    ['0','0'],
    [' 0',' 0'],
    [' 0  ',' 0  '],
    ['aoeuaoeu',
     'aoeuaoeu'],
    ['3s43s()*SNH eah au\' \'sn{snth})',
     '3s43s()*SNH eah au\' \'sn{snth})'],
    ['23131', 
     '23131'],
    ["  \n \n\t\n ",
     "  \n \n\t\n "],
    ['-',
     '-'],
    ['12345678',   '12345678', {min_length => 8, max_length => 8}],
    ['1234567',     '1234567', {min_length => 8, max_length => 8}, 1],
    ['123456789', '123456789', {min_length => 8, max_length => 8}, 1],
    ['1234567',     '1234567', {min_length => 7, max_length => 8}],
    ['12345678',   '12345678', {min_length => 7, max_length => 8}],
    ['123456789', '123456789', {min_length => 7, max_length => 8}, 1]
   ],

   'AtomicData::Text' =>
   [
    [ '',     ''],
    [ '0',   '0'],
    [ ' 0',  '0'],
    [ '0 ',  '0'],
    [ ' 0 ', '0'],
    [ "   nsthsnthoaeu",       'nsthsnthoaeu'],
    [ "  \nnsthsnthoaeu",      'nsthsnthoaeu'],
    [ "\n  nsthsnthoaeu\n  ",  'nsthsnthoaeu'],
    [ "\n \n\nnsthsnthoaeu\n  nsthsnthoaeu \n\n  ",
             "nsthsnthoaeu\n  nsthsnthoaeu"]
   ],

   'AtomicData::Integer' =>
   [
    ['',''],
    [   '0', '0' ],
    [  ' 0', '0' ],
    [  '0 ', '0' ],
    [  ' 1', '1' ],
    [  '2 ', '2' ],
    [ ' 3 ', '3' ]
   ],

   'AtomicData::Decimal' =>
   [
    ['',''],
    [           '1',       '1' ], #trivial test
    [        '1.01',    '1.01' ],
    [          ' 2',       '2' ], #strip leading test
    [          '3 ',       '3' ], #strip trailing test
    [         ' 4 ',       '4' ], #strip both
    [        ' 2.1',     '2.1' ], #strip leading test
    [        '3.1 ',     '3.1' ], #strip trailing test
    [       ' 4.1 ',     '4.1' ], #strip both
    [      ' 2,222',    '2222' ], #strip leading test
    [      '3,333 ',    '3333' ], #strip trailing test
    [     ' 4,444 ',    '4444' ], #strip both
    [    ' 2,222.1',  '2222.1' ], #strip leading test
    [    '3,333.1 ',  '3333.1' ], #strip trailing test
    [   ' 4,444.1 ',  '4444.1' ], #strip both
    [  ' 2,222,222', '2222222' ], #strip leading test
    [  '   2222222', '2222222' ], #strip without commas
    [  '3,333,333 ', '3333333' ], #strip trailing test
    [ ' 4,444,444 ', '4444444' ]  #strip both
   ],


   'AtomicData::Email' =>
   [
    ['',''],
    ['zane@mesas.com',         'zane@mesas.com'],
    ['zane.@mesas.com',        '~'],
    ['zane@mail.mesas.com',    'zane@mail.mesas.com'],
    ['   zane@mesas.com   ',   'zane@mesas.com'],
    ['zane&co@nowhere.org',    'zane&co@nowhere.org'],
    ['you@-here.com',          '~'],
    ['you@here-.com',          '~'],
    ['you@here-there.com',     'you@here-there.com'],
    ['bad@suck.com.',          '~'],
    ['bad@1suck.com',          'bad@1suck.com'],
    ['bad@suck..com',          '~'],
    ['bad@.suck.com',          '~'],
    ['bad@.suck.com-',         '~'],
    ['bad()@suck.com',         '~'],
    ['wh12/-+*\'&%$#!.@w.edu', '~'],
    ['wh12/-+*\'&%$#.!@w.edu', 
     'wh12/-+*\'&%$#.!@w.edu'],
    ['"Zane & Co" <zane&co@nowhere.org>', 
     '"Zane & Co" <zane&co@nowhere.org>'],
    ['Zane and Co <zaneco@nowhere.org>', 
     'Zane and Co <zaneco@nowhere.org>'],
    ['Zane & Co <zane&co@nowhere.org>', 
     'Zane & Co <zane&co@nowhere.org>'],
    ['Zane and Co. <zane&co@nowhere.org>', '~'],
    ['Zane & Co zane&co@nowhere.org>',     '~'],
    ['Zane.Co <zane&co@nowhere.org>',      '~'],
    ['Zane , Co <zane&co@nowhere.org>',    '~'],
    ['Zane & Co <zaneconowhere.org>',      '~'],
    ['zane@mesas.com',                     'zane@mesas.com', 
     {us => 'true', top_level => 'true'}],
    ['zane@com',                           'zane@com', 
     {us => 'true', top_level => 'true'}, 1],
    ['zane@mail.mesas',                    'zane@mail.mesas', 
     {top_level => 'true'}],
    ['zane@mail.mesas',                    'zane@mail.mesas', 
     {us => 'true', top_level => 'true'}, 1],
    ['"Zane & Co" <zane&co@nowhere.org>',  '"Zane & Co" <zane&co@nowhere.org>',
     {us => 'true', top_level => 'true'}],
    ['Zane and Co <zaneco@nowhere.org>',   'Zane and Co <zaneco@nowhere.org>',
     {us => 'true', top_level => 'true'}],
    ['Zane and Co. <ze&co@nowhere.org>',  #'Zane and Co. <ze&co@nowhere.org>', 
     '~', {us => 'true', top_level => 'true'}],
    ['Zane & Co <zane&co@nowhere.org>',    'Zane & Co <zane&co@nowhere.org>', 
     {us => 'true', top_level => 'true'}]
   ],

   'AtomicData::Y4Date' =>
   [
    ['',''],
    ['12/27/1998',  '12/27/1998'],
    ['2/7/1998',    '02/07/1998'],
    ['2-7-1998',    '02/07/1998'],
    ['Jan 1 1999',  '01/01/1999'],
    ['Jan-1-1999',  '01/01/1999'],
    ['Jan 1, 1999', '01/01/1999'],
    ['Jan 1,1999',  '01/01/1999'],
    ['Jan 1, 0000', '~'],
    ['09/20/0000',  '~'],
    ['Jan 1, 1999', '01/01/1999', 
     {min_length => 9, max_length => 9}] 
   ],

   'AtomicData::Date' =>
   [
    ['',''],
    ['12/27/1998',    '12/27/1998'],
    ['2/7/1998',      '02/07/1998'],
    ['2-7-1998',      '02/07/1998'],
    ['Jan 1 1999',    '01/01/1999'],
    ['Jan-1-1999',    '01/01/1999'],
    ['Jan 1, 1999',   '01/01/1999'],
    ['Jan 1 99',      '01/01/1999'],
    ['Jan-1-99',      '01/01/1999'],
    ['Jan 1, 99',     '01/01/1999'],
    ['Feb 14, 2000',  '02/14/2000'],
    ['Feb 14 00',     '02/14/2000'],
    ['Feb-14-00',     '02/14/2000'],
    ['Feb 14, 00',    '02/14/2000'],
    ['Feb 14, 2001',  '02/14/2001'],
    ['Feb 14 01',     '02/14/2001'],
    ['Feb-14-01',     '02/14/2001'],
    ['Feb 14, 01',    '02/14/2001'],
    ['2-7-'.$year1,   '02/07/'.$year1],
    ['2-7-'.$year2,   '02/07/'.$year2],
    ['2-7-'.$year1_2, '02/07/'.$year1],
    ['2-7-'.$year2_2, '02/07/'.$year2],
   ],

   'AtomicData::USPhone' =>
   [
    ['',''],
    [ '(512) 447-5816', '512-447-5816' ],
    [ '(512)-447-5816', '512-447-5816' ],
    [ '512 447 5816',   '512-447-5816' ],
    [ '512.447.5816',   '512-447-5816' ]
   ],

   'AtomicData::USZip' =>
   [
    ['',''],
    [ '78745',      '78745'      ],
    [ '78745-9873', '78745-9873' ],
    [ '787459873',  '78745-9873' ],
    [ '123',                 '~' ],
    [ '1234',                '~' ],
    [ '12 34',               '~' ],
    [ '123 45',              '~' ],
    [ '123456',              '~' ]
   ],

   'AtomicData::TaxID' =>
   [
    ['',''],
    [   '11-1111111', '111111111'],
    [ ' 11-1111111 ', '111111111'],
    [  '111-11-1111', '111111111'],
    [ ' 111-11-1111', '111111111'],
    [    '111111111', '111111111'],
    [  '000-19-0000', '000190000'],
    [     '11111111',         '~'],
    [   '1111111111',         '~']
   ],

   'AtomicData::USCurrency' =>
   [
    ['',''],
    [ '  $1.1  ',    '1.10' ],
    [        '1',    '1.00' ],
    [     ' $1 ',    '1.00' ],
    [     '1.01',    '1.01' ],
    [ '10,101.01',   '10101.01'   ],
    [  '10101.01',   '10101.01'   ],
    [ '1,000,101.1', '1000101.10' ],
    [      '1 1',    '~' ],
    [     '1 .1',    '~' ],
    [   '1.10.1',    '~' ],
    [  '1.10.10',    '~' ],
    [    '1.10.',    '~' ],
    [  '10101,01',   '~' ],
    [ '1.000,101.1', '~' ],
    [    '1,1.1',    '~' ]
   ],

   'AtomicData::RTNumber' =>
   [
    ['',''],
    [  '12312312', '123123123' ],
    [ '123123123', '123123123' ]
   ],

   'AtomicData::PAN' =>
   [
    ['',''],
    [ '', '', {blank_ok => 0}, 1]
   ]
  };

#######
# END DATA
#######

my $n = 0; map { $n += @{$test_data->{$_}} } keys %$test_data;
plan tests => $n * 2;

report("Beginning Canonicalization Test...\n", $VERBOSE);

my (%times,$interval);
my $errors = 0;
my $count = 0;

$times{canon_start} = new Benchmark();

for my $data_type (keys %$test_data) {
  $times{group_start} = new Benchmark();
  my $group_errors = 0;
  my $group_count = 0;
  my $test_group = $test_data->{$data_type};
  my $ERR = "ERROR($data_type)";

  # iterate through all the tests.
  for my $test (@$test_group) {

    # interpret the contents of each test.
    my ($init,$expected,$parameters,$verify_not) = @$test;

    # construct; report time.
    $times{start} = new Benchmark();
    my $data = new $data_type($init);
    $interval = timestr(timediff($times{start}, new Benchmark()));
    report("$data_type: new($init): $interval", $TIMING);

    # set parameters.
    $parameters and $data->set_parameters($parameters);

    # canonicalize; report time.
    $times{start} = new Benchmark();
    $group_count++;
    my $value = $data->canonicalize();
    $interval = timestr(timediff($times{start}, new Benchmark()));
    report("$data_type: canonicalize($init): $interval", $TIMING);

    if (! defined $value) {
      my @message = @{$data->{_can_failed}};
      if ($expected ne '~') {
	report("$ERR: '$init': unexpected canonicalization error: @message", $ERROR);
	$errors++;
      } else {
	report("$data_type: '$init' was correctly rejected with: @message", $SUCCESS);
      }
    } else {
      # test the expected canonical value.
      if ($value ne $expected) {
	report("$ERR: canon('$init') = '$value' != '$expected'",$ERROR);
	$errors++;
      } else {
	report("$data_type: '$init' canonicalized correctly to '$expected'", $SUCCESS);
      }
    }

    # do we wish to start from scratch here?
    $data = new $data_type($init);
    $parameters and $data->set_parameters($parameters);

    #verify this data; report time.
    $times{start} = new Benchmark();
    $group_count++;
    my ($success,$problems) = $data->verify();
    $interval = timestr(timediff($times{start}, new Benchmark()));
    report("$data_type: verify($init): $interval", $TIMING);

    if ($success) {
      # test if verification was not expected.
      if ($verify_not) {
	report("$ERR: '$init' was expected to be invalid.", $ERROR);
	$errors++;
      } elsif ($expected eq '~') {
	report("$ERR: '$init' validation not expected.", $ERROR);
	$errors++;
      } else {
	report("'$init' correctly validated", $SUCCESS);
      }
    } else {
      my @message = @$problems;
      if (!$verify_not and $expected ne '~') {
	report("$ERR: '$init' was expected to be valid:  @message", $ERROR);
	$errors++;
      } else {
	report("'$init' correctly invalidated: @message", $SUCCESS);
      }
    }
  }
  $interval = timestr(timediff($times{group_start}, new Benchmark()));
  report(($group_count - $group_errors)." (of $group_count)".
	 " of $data_type tests for completed: $interval", $TIMING);
  $count += $group_count;
  $errors += $group_errors;
}

$interval = timestr(timediff($times{canon_start}, new Benchmark()));
print $report;
report(($count - $errors)." (of $count) tests completed: $interval", $TIMING);

report($error?"Errors were encountered.":"Test successful.", $FINAL);


=head2 report

 [private] package
 () report (string $message, int $type)

DESCRIPTION:

This function handles all the reporting for the file. It builds up the
global C<$report> variable as it is called with messages of various
types based on the options that the program was started with. They
C<$type> may be C<$SUCCESS>, which is for success messages and
padding. These are only added if the -v option was
specified. C<$ERROR> messages are specific error reports and are added
unless supressed by -q. C<$FINAL> is always appended.

=cut

sub report {
  my ($message, $type) = @_;

  $message =~ s/\s*$//;
  $message =~ s/\n/\\n/;
  $message .= "\n";

  if ($type eq $SUCCESS) {
    ok 1;
  } elsif ($type eq $ERROR) {
    ok 0;
  }

  ($type eq $SUCCESS and !$opt_v) and return;
  ($type eq $VERBOSE and !$opt_v) and return;
  ($type eq $TIMING  and !$opt_t) and return;
  ($type eq $ERROR   and  $opt_q) and return;

  print "#".$message;
}
