/*
 * actionlist.cpp - the customizeable action list
 * Copyright (C) 2004  Michail Pishchagin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "actionlist.h"

#include <qobject.h>
#include <qdict.h>

#include "iconaction.h"

//----------------------------------------------------------------------------
// ActionList
//----------------------------------------------------------------------------

class ActionList::Private : public QObject
{
	Q_OBJECT
public:
	Private() { }
	Private( const Private & );

	QString name;
	int id;
	QStringList sortedActions;
	QDict<IconAction> actions;

public slots:
	void actionDestroyed(QObject *);
};

ActionList::ActionList( QString name, int id, bool autoDelete )
{
	d = new Private();
	d->actions.setAutoDelete( autoDelete );

	d->name = name;
	d->id   = id;
}

ActionList::ActionList( const ActionList &from )
{
	d = new Private( *from.d );
}

ActionList::~ActionList()
{
	delete d;
}

QString ActionList::name() const
{
	return d->name;
}

int ActionList::id() const
{
	return d->id;
}

IconAction *ActionList::action( QString name ) const
{
	return d->actions[name];
}

QStringList ActionList::actions() const
{
	return d->sortedActions;
}

void ActionList::addAction( QString name, IconAction *action )
{
	d->sortedActions << name;

	if ( action ) {
		action->setName( name.latin1() );
		d->actions.insert( name, action );
		d->connect( action, SIGNAL( destroyed(QObject *) ), d, SLOT( actionDestroyed(QObject *) ) );
	}
}

void ActionList::clear()
{
	d->actions.clear();
}

ActionList::Private::Private( const Private &from )
	: QObject()
{
	name = from.name;
	id   = from.id;
	
	actions = from.actions;
	actions.setAutoDelete( from.actions.autoDelete() );
	
	sortedActions = from.sortedActions;
}

void ActionList::Private::actionDestroyed(QObject *obj)
{
	bool autoDelete = actions.autoDelete();
	actions.setAutoDelete( false );

	actions.remove( obj->name() );

	actions.setAutoDelete( autoDelete );
}

//----------------------------------------------------------------------------
// MetaActionList
//----------------------------------------------------------------------------

class MetaActionList::Private
{
public:
	Private() { }

	QPtrList<ActionList> lists;
};

MetaActionList::MetaActionList()
{
	d = new Private();
	d->lists.setAutoDelete( true );
}

MetaActionList::~MetaActionList()
{
	delete d;
}

ActionList *MetaActionList::actionList( QString name ) const
{
	QPtrListIterator<ActionList> it(d->lists);
	for ( ; it.current() != 0; ++it ) {
		if ( it.current()->name() == name )
			return it.current();
	}

	return 0;
}

QPtrList<ActionList> MetaActionList::actionLists( int id ) const
{
	QPtrList<ActionList> list;

	for ( int i = 0; i < 32; i++ ) {
		if ( !(id & ( 1 << i )) )
			continue;

		QPtrListIterator<ActionList> it(d->lists);
		for ( ; it.current() != 0; ++it ) {
			ActionList *l = it.current();

			if ( l->id() & ( 1 << i ) )
				list.append( l );
		}
	}

	return list;
}

ActionList MetaActionList::suitableActions( int id ) const
{
	QPtrList<ActionList> lists = actionLists( id );
	ActionList actions("", 0, false), *list;

	QPtrListIterator<ActionList> it( lists );
	while ( (list = it.current()) ) {
		++it;
		QStringList actionList = list->actions();
		QStringList::Iterator it2 = actionList.begin();
		for ( ; it2 != actionList.end(); ++it2 )
			actions.addAction( *it2, list->action( *it2 ) );
	}

	return actions;
}

QStringList MetaActionList::actionLists() const
{
	QStringList names;

	QPtrListIterator<ActionList> it(d->lists);
	for ( ; it.current() != 0; ++it )
		names << it.current()->name();

	return names;
}

void MetaActionList::addList( ActionList *list )
{
	if ( list )
		d->lists.append( list );
}

void MetaActionList::clear()
{
	QPtrListIterator<ActionList> it(d->lists);
	for ( ; it.current() != 0; ++it )
		it.current()->clear();
}

#include "actionlist.moc"
