/***************************************************************************
                          gui_taglisting.cpp  -  description
                             -------------------
    begin                : Thu May 3 2001
    copyright            : (C) 2001 by Holger Sattel
    email                : hsattel@rumms.uni-mannheim.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "gui_taglisting.h"

#include "configuration.h"

#include "jobmanager.h"
#include "job_modifytracksindatabase.h"

#include "gui.h"
#include "lvi_track.h"
#include "tagger.h"

#include <qlayout.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qlistview.h>
#include <qpopupmenu.h>
#include <qaccel.h>
#include <qradiobutton.h>
#include <qtooltip.h>
#include <qmessagebox.h>

#include <libintl.h>
#define _(String) gettext (String)

#define EDIT_MODE    1
#define VIEWING_MODE 2

#define MAXGENRE 147

// *******************
// *** PUBLIC PART ***
// *******************

// ##################################################
// # constructor
// ##################################################
GUI_TagListing::GUI_TagListing(QWidget *parent, QListView *_tracklist, const char *name)
  : QFrame(parent, name), tracklist(_tracklist)
{
  int lineSpace = fontMetrics().lineSpacing();

  QGridLayout *lay0 = new QGridLayout(this, 3, 5);
  
  lay0->addRowSpacing(2, lineSpace / 2);
  lay0->addColSpacing(0, lineSpace / 2);
  lay0->addColSpacing(2, lineSpace / 2);
  lay0->addColSpacing(4, lineSpace / 2);

  QButtonGroup *infobox = new QButtonGroup(_("File Information"), this);
  QGridLayout *lay1 = new QGridLayout(infobox, 5, 10);
  
  lay1->addRowSpacing(0, lineSpace);
  lay1->addRowSpacing(2, lineSpace / 2);
  lay1->addRowSpacing(4, lineSpace / 2);
  lay1->addColSpacing(0, lineSpace / 2);
  lay1->addColSpacing(5, 16);
  lay1->addColSpacing(9, lineSpace / 2);

  lay1->setColStretch(7, 1);

  QLabel *filelabel = new QLabel("<b>" + QString(_("File")) + ": </b>", infobox);
  lay1->addWidget(filelabel, 1, 1, 2);

  labelFile = new QLabel( " ", infobox);
  lay1->addWidget(labelFile, 1, 2);

  QLabel *labelOf = new QLabel(" " + QString(_("of")) + " ", infobox);
  lay1->addWidget(labelOf, 1, 3);
  labelSum = new QLabel("0", infobox);
  lay1->addWidget(labelSum, 1, 4);
  QLabel *namelabel = new QLabel("<b>" + QString(_("Name")) + ": </b>", infobox);
  lay1->addWidget(namelabel, 1, 6);

  stackFilename = new QWidgetStack(infobox);
  labelFilename = new QLabel("", infobox);
  labelFilename->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  stackFilename->addWidget(labelFilename, 1);
  editFilename = new QLineEdit("", infobox);
  editFilename->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  stackFilename->addWidget(editFilename, 2);
  stackFilename->raiseWidget(1);
  lay1->addWidget(stackFilename, 1, 7);
  buttonFilename = new ExtendedPushButton("", infobox);
  QToolTip::add( buttonFilename, _( "Left click to apply change to all selected tracks.\n Right click for special options." ) );
  buttonFilename->setFixedSize(14, 14);
  lay1->addWidget(buttonFilename, 1, 8);

  connect(editFilename, SIGNAL(textChanged(const QString&)),
 	  this, SLOT(slot_filenameEdit(const QString&)));
  connect(buttonFilename, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
 	  this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

  QLabel *infolabel = new QLabel("<b>" + QString(_("Info")) + ": </b>", infobox);
  lay1->addWidget(infolabel, 3, 1, 2);
  labelInfo = new QLabel("", infobox);
  lay1->addMultiCellWidget(labelInfo, 3, 3, 2, 7);

  lay0->addWidget(infobox, 0, 1);

  QButtonGroup *tagbox = new QButtonGroup(_("Tag Information"), this);
  QGridLayout *lay3 = new QGridLayout(tagbox, 7, 12);

  lay3->addRowSpacing(0, lineSpace);
  lay3->addRowSpacing(6, lineSpace / 2);
  lay3->addColSpacing(0, lineSpace / 2);
  lay3->addColSpacing(6, 10);
  lay3->addColSpacing(11, lineSpace / 2);

  lay3->setColStretch(4, 1);

  QLabel *labtitle = new QLabel("<b>" + QString(_("Title")) + ": </b>", tagbox);
  lay3->addWidget(labtitle, 1, 1, 2);
  stackTitle = new QWidgetStack(tagbox);
  labelTitle = new QLabel("", tagbox);
  labelTitle->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  stackTitle->addWidget(labelTitle, 1);
  editTitle = new QLineEdit("", tagbox);
  editTitle->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  //    editTitle->setFrame(false);
  stackTitle->addWidget(editTitle, 2);
  stackTitle->raiseWidget(1);
  lay3->addMultiCellWidget(stackTitle, 1, 1, 2, 9);
  buttonTitle = new ExtendedPushButton("", tagbox);
  QToolTip::add( buttonTitle, _( "Left click to apply change to all selected tracks.\n Right click for special options." ) );
  buttonTitle->setFixedSize(14, 14);
  lay3->addWidget(buttonTitle, 1, 10);

  connect(editTitle, SIGNAL(textChanged(const QString&)),
	  this, SLOT(slot_titleEdit(const QString&)));
  connect(buttonTitle, SIGNAL(clicked()),
	  this, SLOT(slot_titleButton()));
  connect(buttonTitle, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
	  this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

  QLabel *labartist = new QLabel("<b>" + QString(_("Artist")) + ": </b>", tagbox);
  lay3->addWidget(labartist, 2, 1, 2);
  stackArtist = new QWidgetStack(tagbox);
  labelArtist = new QLabel("", tagbox);
  labelArtist->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  stackArtist->addWidget(labelArtist, 1);
  editArtist = new QLineEdit("", tagbox);
  editArtist->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  //    editArtist->setFrame(false);
  stackArtist->addWidget(editArtist, 2);
  stackArtist->raiseWidget(1);
  lay3->addMultiCellWidget(stackArtist, 2, 2, 2, 9);
  buttonArtist = new ExtendedPushButton("", tagbox);
  QToolTip::add( buttonArtist, _( "Left click to apply change to all selected tracks.\n Right click for special options." ) );
  buttonArtist->setFixedSize(14, 14);
  lay3->addWidget(buttonArtist, 2, 10);

  connect(editArtist, SIGNAL(textChanged(const QString&)),
	  this, SLOT(slot_artistEdit(const QString&)));
  connect(buttonArtist, SIGNAL(clicked()),
	  this, SLOT(slot_artistButton()));
  connect(buttonArtist, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
	  this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

  QLabel *labalbum = new QLabel("<b>" + QString(_("Album")) + ": </b>", tagbox);
  lay3->addWidget(labalbum, 3, 1, 2);
  stackAlbum = new QWidgetStack(tagbox);
  labelAlbum = new QLabel("", tagbox);
  labelAlbum->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  stackAlbum->addWidget(labelAlbum, 1);
  editAlbum = new QLineEdit("", tagbox);
  editAlbum->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  //    editAlbum->setFrame(false);
  stackAlbum->addWidget(editAlbum, 2);
  stackAlbum->raiseWidget(1);
  lay3->addMultiCellWidget(stackAlbum, 3, 3, 2, 4);
  buttonAlbum = new ExtendedPushButton("", tagbox);
  QToolTip::add( buttonAlbum, _( "Left click to apply change to all selected tracks.\n Right click for special options." ) );
  buttonAlbum->setFixedSize(14, 14);
  lay3->addWidget(buttonAlbum, 3, 5);

  connect(editAlbum, SIGNAL(textChanged(const QString&)),
	  this, SLOT(slot_albumEdit(const QString&)));
  connect(buttonAlbum, SIGNAL(clicked()),
	  this, SLOT(slot_albumButton()));
  connect(buttonAlbum, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
	  this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

  QLabel *labyear = new QLabel("<b>" + QString(_("Year")) + ": </b>", tagbox);
  lay3->addWidget(labyear, 3, 8, 2);
  stackYear = new QWidgetStack(tagbox);
  labelYear = new QLabel("", tagbox);
  stackYear->addWidget(labelYear, 1);
  editYear = new QLineEdit("", tagbox);
  //    editYear->setFrame(false);
  stackYear->addWidget(editYear, 2);
  stackYear->raiseWidget(1);
  stackYear->setFixedWidth(60);
  lay3->addWidget(stackYear, 3, 9);
  buttonYear = new ExtendedPushButton("", tagbox);
  QToolTip::add( buttonYear, _( "Left click to apply change to all selected tracks.\n Right click for special options." ) );

  buttonYear->setFixedSize(14, 14);
  lay3->addWidget(buttonYear, 3, 10);

  connect(editYear, SIGNAL(textChanged(const QString&)),
	  this, SLOT(slot_yearEdit(const QString&)));
  connect(buttonYear, SIGNAL(clicked()),
	  this, SLOT(slot_yearButton()));
  connect(buttonYear, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
	  this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

  QLabel *labcomment = new QLabel("<b>" + QString(_("Comment")) + ": </b>", tagbox);
  lay3->addWidget(labcomment, 4, 1, 2);
  stackComment = new QWidgetStack(tagbox);
  labelComment = new QLabel("", tagbox);
  labelComment->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  stackComment->addWidget(labelComment, 1);
  editComment = new QLineEdit("", tagbox);
  editComment->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  //    editComment->setFrame(false);
  stackComment->addWidget(editComment, 2);
  stackComment->raiseWidget(1);
  lay3->addMultiCellWidget(stackComment, 4, 4, 2, 9);
  buttonComment = new ExtendedPushButton("", tagbox);
  QToolTip::add( buttonComment, _( "Left click to apply change to all selected tracks.\n Right click for special options."));
  buttonComment->setFixedSize(14, 14);
  lay3->addWidget(buttonComment, 4, 10);

  connect(editComment, SIGNAL(textChanged(const QString&)),
	  this, SLOT(slot_commentEdit(const QString&)));
  connect(buttonComment, SIGNAL(clicked()),
	  this, SLOT(slot_commentButton()));
  connect(buttonComment, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
	  this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

  QLabel *labtrack = new QLabel("<b>" + QString(_("Track")) + ": </b>", tagbox);
  lay3->addWidget(labtrack, 5, 8, 2);
  stackTrack = new QWidgetStack(tagbox);
  labelTrack = new QLabel("", tagbox);
  stackTrack->addWidget(labelTrack, 1);
  spinTrack = new QSpinBox(0, 255, 1, tagbox);
  spinTrack->setSpecialValueText(" ");
  spinTrack->setButtonSymbols(QSpinBox::PlusMinus);
  stackTrack->addWidget(spinTrack, 2);
  stackTrack->raiseWidget(1);
  stackTrack->setFixedWidth(60);
  lay3->addWidget(stackTrack, 5, 9, 1);
  buttonTrack = new ExtendedPushButton("", tagbox);
  QToolTip::add( buttonTrack, _( "Left click to apply change to all selected tracks.\n Right click for special options.") );
  buttonTrack->setFixedSize(14, 14);
  lay3->addWidget(buttonTrack, 5, 10);

  connect(spinTrack, SIGNAL(valueChanged(int)), this, SLOT(slot_tracknumberChanged(int)));
  connect(buttonTrack, SIGNAL(clicked()),
 	  this, SLOT(slot_trackButton()));
  connect(buttonTrack, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
 	  this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

  QLabel *labgenre = new QLabel("<b>" + QString(_("Genre")) + ": </b>", tagbox);
  lay3->addWidget(labgenre, 5, 1, 2);
  stackGenre = new QWidgetStack(tagbox);
  labelGenre = new QLabel("", tagbox);
  labelGenre->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  stackGenre->addWidget(labelGenre, 1);
  comboGenre = new QComboBox(tagbox);
  comboGenre->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
  stackGenre->addWidget(comboGenre, 2);
  stackGenre->raiseWidget(1);

  for(int i=0; i <= MAXGENRE+1; i++) comboGenre->insertItem(Tagger::getGenreAlpha(i), i);
  comboGenre->setCurrentItem(0);

  stackGenre->setFixedWidth(160);
  lay3->addWidget(stackGenre, 5, 2, 1);
  buttonGenre = new ExtendedPushButton("", tagbox);
  QToolTip::add( buttonGenre, _( "Left click to apply change to all selected tracks.\n Right click for special options." ) );
  buttonGenre->setFixedSize(14, 14);
  lay3->addWidget(buttonGenre, 5, 3);

  connect(comboGenre, SIGNAL(activated(int)),
	  this, SLOT(slot_genreCombo(int)));
  connect(buttonGenre, SIGNAL(clicked()),
	  this, SLOT(slot_genreButton()));
  connect(buttonGenre, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
	  this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

  lay0->addWidget(tagbox, 1, 1);

  QGroupBox *commandbox = new QGroupBox(1, Qt::Horizontal, _("Commands"), this);
  QFrame *commandframe = new QFrame(commandbox);
  QVBoxLayout *lay4 = new QVBoxLayout(commandframe);

  buttonEdit = new QPushButton(_("Edit"), commandframe);
  lay4->addWidget(buttonEdit);
  buttonApply = new QPushButton(_("Save"), commandframe);
  lay4->addWidget(buttonApply);
  buttonDiscard = new QPushButton(_("Discard"), commandframe);
  lay4->addWidget(buttonDiscard);

  lay0->addWidget(commandbox, 1, 3);

  connect(buttonEdit, SIGNAL(clicked()),
	  this, SLOT(slot_editbutton()));
  connect(buttonApply, SIGNAL(clicked()),
	  this, SLOT(slot_applybutton()));
  connect(buttonDiscard, SIGNAL(clicked()),
	  this, SLOT(slot_discardbutton()));

  // MK begin

  QButtonGroup *albumbox = new QButtonGroup(_("Album Information"), this);
  QToolTip::add( albumbox, _( "View if track is part of sampler\n or soundtrack view" ));
  QGridLayout *lay2 = new QGridLayout(albumbox, 5, 4);

  lay2->addRowSpacing(0, lineSpace);
  lay2->addRowSpacing(4, lineSpace / 2);
  lay2->addColSpacing(0, lineSpace / 2);
  lay2->addColSpacing(3, lineSpace / 2);
  lay2->setRowStretch(4, 1);

  standardAlbum   = new QRadioButton(_("Normal"), albumbox);
  samplerAlbum    = new QRadioButton(_("Sampler"), albumbox);
  soundtrackAlbum = new QRadioButton(_("Soundtrack"), albumbox);
  lay2->addWidget(standardAlbum, 1, 1);
  lay2->addWidget(samplerAlbum, 2, 1);
  lay2->addWidget(soundtrackAlbum, 3, 1);

  buttonAlbumInfo = new ExtendedPushButton("", albumbox);
  QToolTip::add( buttonAlbumInfo, _( "Left click to apply change to all selected tracks." ) );
  buttonAlbumInfo->setFixedSize(14, 14);
  lay2->addWidget(buttonAlbumInfo, 2, 2);

  lay0->addWidget(albumbox, 0, 3);

  connect(albumbox, SIGNAL(clicked(int)), this, SLOT(slot_albumbox(int)));
  connect(buttonAlbumInfo, SIGNAL(clicked()), this, SLOT(slot_albumInfoButton()));

  // MK end

  QAccel *accel = new QAccel(this);
  accel->connectItem(accel->insertItem(CTRL+Key_P),this,
		     SLOT(slot_selPrevTrack()));
  accel->connectItem(accel->insertItem(CTRL+Key_N),this,
		     SLOT(slot_selNextTrack()));

  buttonEdit->setEnabled(false);
  buttonApply->setEnabled(false);
  buttonDiscard->setEnabled(false);

  buttonFilename->hide();
  buttonTitle->hide();
  buttonArtist->hide();
  buttonAlbum->hide();
  buttonYear->hide();
  buttonTrack->hide();
  buttonGenre->hide();
  buttonComment->hide();
  buttonAlbumInfo->hide();

  editLock = 0;
  editMode = false;
  doNothing = false;

}

void GUI_TagListing::setConnectionState(bool state) {
  if(state == false && editMode) {
    switchMode(VIEWING_MODE, false);
    notifySelectionChanged();
  }
}

// ##################################################
// # set switch for the ability to edit tags
// ##################################################
void GUI_TagListing::notifyAddEditLock() {
  editLock++;
  buttonEdit->setEnabled((currentTrack >= 0) && editLock == 0 && editEnabled);
}

void GUI_TagListing::notifyRemoveEditLock() {
  editLock--;
  buttonEdit->setEnabled((currentTrack >= 0) && editLock == 0 && editEnabled);
}

void GUI_TagListing::notifyEnableEdit(bool b) {
  buttonEdit->setEnabled(b);
  editEnabled = b;
}

void GUI_TagListing::getSelectedItems()
{
    selectedItems.clear();
    QListViewItem *curr = tracklist->firstChild();
    while(curr) {
	if(curr->isSelected()) selectedItems.append(curr);
	curr = curr->nextSibling();
    }
}

// ##################################################
// # selection has changed
// ##################################################
void GUI_TagListing::notifySelectionChanged()
{
/*  if(doNothing) return;
  if(!editMode) {
    getSelectedItems();
    currentTrack = (selectedItems.count() == 0) ? 0 : 1;
    buttonEdit->setEnabled((currentTrack != 0) && editLock == 0);
    buttonApply->setEnabled(false);
    buttonDiscard->setEnabled(false);
  } else {
        if(tracklist->selectedItem() == 0) tracklist->setSelected(tracklist->currentItem(), true);
//        currentTrack = selectedItems.find(tracklist->selectedItem()) +1;
   }
//  applyCurrentTrack(); */
    if(doNothing) return;
    if(!editMode) getSelectedItems();
    if(selectedItems.count() == 1 && selectedItems.at(0) != tracklist->currentItem()) {
      tracklist->setCurrentItem(selectedItems.at(0));
      tracklist->setSelected(selectedItems.at(0), true);
#if (QT_VERSION >= 0x030201)
      tracklist->setSelectionAnchor(selectedItems.at(0)); 
#endif
    };
    notifyCurrentChanged(0);
/*    if(doNothing) return;
    getSelectedItems();
    int selectcount = (selectedItems.count());
    if(selectcount == 0 && tracklist->currentItem() != 0) {
        tracklist->setSelected(tracklist->currentItem(), true);
        getSelectedItems();
        selectcount = 1;
    }

    if(!editMode) {
        buttonEdit->setEnabled((selectcount != 0) && editLock == 0);
        buttonApply->setEnabled(false);
        buttonDiscard->setEnabled(false);
    }

    currentTrack = selectedItems.find(tracklist->currentItem()) + 1;
    if(currentTrack == 0) currentTrack = 1;
    applyCurrentTrack(); */
}

void GUI_TagListing::notifyCurrentChanged(QListViewItem *lvi)
{
    if(doNothing) return;
    if(!editMode) getSelectedItems();
    int selectcount = (selectedItems.count());

    if(selectcount == 0 && tracklist->currentItem() != 0) {
        tracklist->setSelected(tracklist->currentItem(), true);
        if(!editMode) getSelectedItems();
        selectcount = 1;
    }

    if(!editMode) {
        buttonEdit->setEnabled((selectcount != 0) && editLock == 0 && editEnabled);
        buttonApply->setEnabled(false);
        buttonDiscard->setEnabled(false);
    }

//    currentTrack = selectedItems.find(tracklist->currentItem());
//    if(currentTrack == 0) currentTrack = 1;
    applyCurrentTrack();
}

// ##################################################
// # sorting has changed (editmode only)
// ##################################################
void GUI_TagListing::notifySortingInEditModeChanged()
{
  selectedItems.clear();
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    selectedItems.append(item);
    item = item->nextSibling();
  }

  applyCurrentTrack();
}

// ##################################################
// # destructor
// ##################################################
GUI_TagListing::~GUI_TagListing() {}

// ********************
// *** PRIVATE PART ***
// ********************

// ##################################################
// # draw information for current track
// ##################################################
void GUI_TagListing::applyCurrentTrack()
{

  if ( verbose == 4 ) {
    qWarning( "GUI_TagListing::applyCurrentTrack() starts");
        dumpSelectedItems( &selectedItems );
  }

  currentTrack = selectedItems.find(tracklist->currentItem());

  QString num;
  num.sprintf("%d", selectedItems.count());
  labelSum->setText(num);
  labelFile->setText(QString::number(currentTrack+1));

  standardAlbum->setEnabled(FALSE);
  samplerAlbum->setEnabled(FALSE);
  soundtrackAlbum->setEnabled(FALSE);

  if(currentTrack >= 0) {
    LVI_Track *lvi = dynamic_cast<LVI_Track*>(selectedItems.at(currentTrack));
    TRACK *track = lvi->getOriginalTrack();

    labelInfo->setText(Tagger::getTypeAsString(track) + ", " +
		       Tagger::getModeAsString(track) + ", " +
		       Tagger::getSizeAsString(track));

    int xgenre = Tagger::genreToAlphaIndex(lvi->getGenre());

    setAlbumInformationFromComment(lvi->getComment());

    if(!editMode) {
      QString s;
      labelFilename->setText(lvi->getFilename());
      labelTitle->setText(lvi->getTitle());
      labelArtist->setText(lvi->getArtist());
      labelAlbum->setText(lvi->getAlbum());
      if(lvi->getTracknumber() != 0) labelTrack->setText(s.sprintf("%02d", lvi->getTracknumber()));
      else labelTrack->setText("");
      labelYear->setText(lvi->getYear());
      labelComment->setText(lvi->getComment());
      labelGenre->setText(Tagger::getGenre(lvi->getGenre()));
    }  else {
      if(lvi->getType() == MEDIUM_CDROM || !lvi->getIsAvailable()) {
	stackFilename->raiseWidget(1);
	labelFilename->setText(lvi->getFilename());
	buttonFilename->hide();
      } else {
	stackFilename->raiseWidget(2);
	editFilename->setText(lvi->getFilename());
	buttonFilename->show();
      }
      editTitle->setText(lvi->getTitle());
      editArtist->setText(lvi->getArtist());
      editAlbum->setText(lvi->getAlbum());
      spinTrack->setValue(lvi->getTracknumber());
      editYear->setText(lvi->getYear());
      editComment->setText(lvi->getComment());
      comboGenre->setCurrentItem(xgenre);
      standardAlbum->setEnabled(TRUE);
      samplerAlbum->setEnabled(TRUE);
      soundtrackAlbum->setEnabled(TRUE);
    }
    tracklist->ensureItemVisible(lvi);
  } else {
    labelFilename->setText("");
    labelInfo->setText("");
    labelTitle->setText("");
    labelArtist->setText("");
    labelAlbum->setText("");
    labelTrack->setText("");
    labelYear->setText("");
    labelComment->setText("");
    labelGenre->setText("");
    standardAlbum->setChecked(FALSE);
    samplerAlbum->setChecked(FALSE);
    soundtrackAlbum->setChecked(FALSE);  }
  if ( verbose == 4 ) qWarning( "GUI_TagListing::applyCurrentTrack() stops");
}

// ##################################################
// # set album information from comment
// ##################################################
void GUI_TagListing::setAlbumInformationFromComment(QString comment)
{
	int widget = 1;

    if(comment == SAMPLER)
    {
    	samplerAlbum->setChecked(TRUE);
    }
    else if(comment == SOUNDTRACK)
    {
    	soundtrackAlbum->setChecked(TRUE);
    }
    else
    {
    	standardAlbum->setChecked(TRUE);
        widget = 2;
    }

    if(editMode)
    {
       	labelComment->setText(comment);
     	if(widget != stackComment->id(stackComment->visibleWidget()))
     	{
        	editComment->setText(comment);
     		stackComment->raiseWidget(widget);
     	}
    }
}

// ##################################################
// # switch between view- and editmode
// ##################################################
void GUI_TagListing::switchMode(int mode, bool saving)
{

  int widgetlayer;

  if ( verbose == 4 ) qWarning( "GUI_TagListing::switchMode( mode=%s, saving=%s ) starts",
				( mode == EDIT_MODE ) ? "EDIT_MODE" : "VIEWING_MODE",
				saving ? "true" : "false" );

  if ( mode == VIEWING_MODE ) { // restore previous tooltip
    QToolTip::remove( this );
    QToolTip::add( this, oldtip);
  };

  if ( mode == EDIT_MODE ) {  // save previous tooltip
    oldtip = QToolTip::textFor( this, QPoint(0,0));
    QToolTip::remove( this );
  };

  if(!saving) {
    buttonLastClicked = buttonFilename; slot_menuDiscardChangesALL();
    buttonLastClicked = buttonTitle; slot_menuDiscardChangesALL();
    buttonLastClicked = buttonArtist; slot_menuDiscardChangesALL();
    buttonLastClicked = buttonAlbum; slot_menuDiscardChangesALL();
    buttonLastClicked = buttonTrack; slot_menuDiscardChangesALL();
    buttonLastClicked = buttonYear; slot_menuDiscardChangesALL();
    buttonLastClicked = buttonGenre; slot_menuDiscardChangesALL();
    buttonLastClicked = buttonComment; slot_menuDiscardChangesALL();
  }

  if(mode == EDIT_MODE) widgetlayer = 2; else widgetlayer = 1;

  if(mode == VIEWING_MODE) stackFilename->raiseWidget(widgetlayer);
  stackTitle->raiseWidget(widgetlayer);
  stackArtist->raiseWidget(widgetlayer);
  stackAlbum->raiseWidget(widgetlayer);
  stackYear->raiseWidget(widgetlayer);
  stackTrack->raiseWidget(widgetlayer);
  stackGenre->raiseWidget(widgetlayer);
  stackComment->raiseWidget(widgetlayer);

  if(mode == EDIT_MODE) {
    buttonFilename->show();
    buttonTitle->show();
    buttonArtist->show();
    buttonAlbum->show();
    buttonYear->show();
    buttonTrack->show();
    buttonGenre->show();
    buttonComment->show();
    buttonAlbumInfo->show();

    buttonEdit->setEnabled(false);
    buttonApply->setEnabled(true);
    buttonDiscard->setEnabled(true);

    doNothing = true;
    gui->getListing()->setEditMode(true);
    tracklist->setSelected(selectedItems.at(currentTrack), true);
    doNothing = false;

    gui->setStatusInfo(" " + QString(_("Mode")) + ": " + _("tagedit"), 5);

    statusleft = gui->getSelector()->isVisible();
    statusright = gui->getPlaylisting()->isVisible();
    autohide = (config->getAutoHide() == 1);

    editMode = true;
	if(autohide) {
		gui->getSelector()->hide();
		gui->getSourcer()->hide();
		gui->getPlaylisting()->hide();
		gui->getSwitchBar()->setEnabled(false);
	}
  } else {
    buttonFilename->hide();
    buttonTitle->hide();
    buttonArtist->hide();
    buttonAlbum->hide();
    buttonYear->hide();
    buttonTrack->hide();
    buttonGenre->hide();
    buttonComment->hide();
    buttonAlbumInfo->hide();

    if(!saving) buttonEdit->setEnabled(true);
    buttonApply->setEnabled(false);
    buttonDiscard->setEnabled(false);

    gui->setStatusInfo(" " + QString(_("Mode")) + ": " + _("normal"), 5);
    editMode = false;
    if(autohide) {
	if(statusleft) gui->getSelector()->show();
	if(statusleft) gui->getSourcer()->show();
	if(statusright) gui->getPlaylisting()->show();
	gui->getSwitchBar()->setEnabled(true);
    }

//    LVI_Track *track = dynamic_cast<LVI_Track*>(selectedItems.at(currentTrack))));
    LVI_Track *track = dynamic_cast<LVI_Track*>(tracklist->currentItem());
    gui->getListing()->setEditMode(false);
    notifySelectionChanged();
    if(track->getIsShowed()) {
        tracklist->setCurrentItem(track);
//      currentTrack = selectedItems.find(track);
    }
  }

  applyCurrentTrack();

  if ( verbose == 4 ) qWarning( "GUI_TagListing::switchMode ends");
}

// ##################################################
// # callbacks for the 5 normal buttons
// ##################################################

void GUI_TagListing::slot_editbutton() {
  switchMode(EDIT_MODE, true);
}

void GUI_TagListing::slot_applybutton()
{
  bool updateSamplerTab = false;
  bool updateFavouritesTab = false;

  QList<TRACK> *trackList = new QList<TRACK>;
  QList<QString> *origFilenameList = new QList<QString>;

  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);
    if(!lvi->getIsAvailable()) lvi->setFilename(lvi->getOriginalTrack()->filename);
    if(lvi->hasFilenameChanged() ||
       lvi->hasTitleChanged() || lvi->hasArtistChanged() || lvi->hasAlbumChanged() ||
       lvi->hasYearChanged() || lvi->hasGenreChanged() || lvi->hasCommentChanged() || lvi->hasTracknumberChanged())
      {

	if(lvi->hasCommentChanged() || (lvi->hasAlbumChanged() && (lvi->getComment() == SAMPLER || lvi->getComment() == SOUNDTRACK)))
		updateSamplerTab = true;

	if(lvi->hasArtistChanged())
		updateFavouritesTab = true;

	TRACK *origTrack = lvi->getOriginalTrack();
	TRACK *changedTrack = new TRACK;
	changedTrack->id           = origTrack->id;
	changedTrack->path         = origTrack->path;
	changedTrack->filename     = lvi->getFilename();
	changedTrack->medium       = origTrack->medium;
	changedTrack->lastModified = origTrack->lastModified;
	changedTrack->hasChanged   = (lvi->hasTitleChanged() || lvi->hasArtistChanged() || lvi->hasAlbumChanged() ||
				      lvi->hasYearChanged() || lvi->hasGenreChanged() || lvi->hasCommentChanged() ||
				      lvi->hasTracknumberChanged() || origTrack->hasChanged);
	changedTrack->mimetype     = origTrack->mimetype;
	changedTrack->version      = origTrack->version;
	changedTrack->layer        = origTrack->layer;
	changedTrack->mode         = origTrack->mode;
	changedTrack->bitrate      = origTrack->bitrate;
	changedTrack->samplerate   = origTrack->samplerate;
	changedTrack->length       = origTrack->length;
	changedTrack->size         = origTrack->size;
	changedTrack->artist       = lvi->getArtist();
	changedTrack->title        = lvi->getTitle();
	changedTrack->album        = lvi->getAlbum();
	changedTrack->tracknumber  = lvi->getTracknumber();
	changedTrack->year         = lvi->getYear();
	changedTrack->genre        = lvi->getGenre();
	changedTrack->comment      = lvi->getComment();
	trackList->append(changedTrack);

	QString *fname = new QString(origTrack->filename);
	origFilenameList->append(fname);
      }
    lvi->applyChanges();
    item = item->nextSibling();
  }

  buttonLastClicked = buttonFilename; slot_menuDiscardChangesALL();
  jobman->addJob(new Job_ModifyTracksInDatabase(trackList, origFilenameList, updateSamplerTab, updateFavouritesTab));

  switchMode(VIEWING_MODE, true);
}

void GUI_TagListing::slot_discardbutton() {
  switchMode(VIEWING_MODE, false);
}

// ##################################################
// # callbacks for the editfields in editmode
// ##################################################
void GUI_TagListing::slot_artistEdit(const QString& artist) {
  dynamic_cast<LVI_Track*>(tracklist->selectedItem())->setArtist(artist);
}

void GUI_TagListing::slot_filenameEdit(const QString& filename) {
  dynamic_cast<LVI_Track*>(tracklist->selectedItem())->setFilename(filename);
}

void GUI_TagListing::slot_titleEdit(const QString& title) {
  dynamic_cast<LVI_Track*>(tracklist->selectedItem())->setTitle(title);
}

void GUI_TagListing::slot_albumEdit(const QString& album) {
  dynamic_cast<LVI_Track*>(tracklist->selectedItem())->setAlbum(album);
}

void GUI_TagListing::slot_tracknumberChanged(int tracknumber) {
  dynamic_cast<LVI_Track*>(tracklist->selectedItem())->setTracknumber(tracknumber);
}

void GUI_TagListing::slot_yearEdit(const QString& year) {
  dynamic_cast<LVI_Track*>(tracklist->selectedItem())->setYear(year);
}

void GUI_TagListing::slot_commentEdit(const QString& comment) {
  dynamic_cast<LVI_Track*>(tracklist->selectedItem())->setComment(comment);
  setAlbumInformationFromComment(comment);
}

void GUI_TagListing::slot_genreCombo(int genre) {
  dynamic_cast<LVI_Track*>(tracklist->selectedItem())->setGenre(Tagger::alphaToGenreIndex(genre));
}

void GUI_TagListing::slot_albumbox(int id) {
  if(id < 3) // 3 = PushButton
  {
  	QString comment = NORMAL;
    if(id == 1) comment = SAMPLER;
    if(id == 2) comment = SOUNDTRACK;

    dynamic_cast<LVI_Track*>(tracklist->selectedItem())->setComment(comment);
    setAlbumInformationFromComment(comment);
  }
}

// ##################################################
// # callbacks for the extended edit buttons
// ##################################################
void GUI_TagListing::slot_artistButton() {
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    dynamic_cast<LVI_Track*>(item)->setArtist(editArtist->text());
    item = item->nextSibling();
  }
}

void GUI_TagListing::slot_titleButton() {
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    dynamic_cast<LVI_Track*>(item)->setTitle(editTitle->text());
    item = item->nextSibling();
  }
}

void GUI_TagListing::slot_albumButton() {
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    dynamic_cast<LVI_Track*>(item)->setAlbum(editAlbum->text());
    item = item->nextSibling();
  }
}

void GUI_TagListing::slot_trackButton() {
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    dynamic_cast<LVI_Track*>(item)->setTracknumber(spinTrack->value());
    item = item->nextSibling();
  }
}

void GUI_TagListing::slot_autotrackButton() {
  int cur_tracknum;
  QListViewItem *item = tracklist->firstChild();
//  cur_tracknum = dynamic_cast<LVI_Track*>(item)->getTracknumber();
//  if (cur_tracknum <= 0)
      cur_tracknum = 1;
  while(item != 0) {
    dynamic_cast<LVI_Track*>(item)->setTracknumber(cur_tracknum++);
    item = item->nextSibling();
  }
}

void GUI_TagListing::slot_yearButton() {
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    dynamic_cast<LVI_Track*>(item)->setYear(editYear->text());
    item = item->nextSibling();
  }
}

void GUI_TagListing::slot_commentButton() {
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    dynamic_cast<LVI_Track*>(item)->setComment(editComment->text());
    item = item->nextSibling();
  }
}

void GUI_TagListing::slot_genreButton() {
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    dynamic_cast<LVI_Track*>(item)->setGenre(Tagger::alphaToGenreIndex(comboGenre->currentItem()));
    item = item->nextSibling();
  }
}

void GUI_TagListing::slot_albumInfoButton() {
  QString comment = NORMAL;
  if( samplerAlbum->isChecked() ) comment = SAMPLER;
  if( soundtrackAlbum->isChecked() ) comment = SOUNDTRACK;
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    dynamic_cast<LVI_Track*>(item)->setComment(comment);
    item = item->nextSibling();
  }
  setAlbumInformationFromComment(comment);
}

// ##################################################
// # context menu in editmode
// ##################################################
void GUI_TagListing::slot_buttonContext(ExtendedPushButton *but, const QPoint& p)
{
  buttonLastClicked = but;

  QPopupMenu *menu = new QPopupMenu(this);

  if(but == buttonFilename) {
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_writetags.png"), _("Standard Filename"), this, SLOT(slot_menuStandardFilename()), 0, 1);
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_writetags.png"), _("Andromeda Filename"), this, SLOT(slot_menuAndromedaFilename()), 0, 2);
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_discard.png"), _("Discard Changes"), this, SLOT(slot_menuDiscardChanges()), 0, 3);
    menu->insertSeparator();
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_writetags.png"), _("Standard Filename to ALL"), this, SLOT(slot_menuStandardFilenameALL()), 0, 4);
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_writetags.png"), _("Andromeda Filename to ALL"), this, SLOT(slot_menuAndromedaFilenameALL()), 0, 5);
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_discard.png"), _("Discard Changes to ALL"), this, SLOT(slot_menuDiscardChangesALL()), 0, 6);
    menu->insertSeparator();
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/help.png"), _("More Info"), this, SLOT(slot_menuHelpAndromeda()), 0, 7);
  }
  if(but == buttonTitle || but == buttonArtist || but == buttonAlbum || but == buttonComment) {
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_writetags.png"), _("First-Letter Upcase"), this, SLOT(slot_menuFLUpcase()), 0, 1);
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_writetags.png"), _("Downcase"), this, SLOT(slot_menuDowncase()), 0, 2);
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_discard.png"), _("Discard Changes"), this, SLOT(slot_menuDiscardChanges()), 0, 3);
    menu->insertSeparator();
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_writetags.png"), _("First-Letter Upcase to ALL"), this, SLOT(slot_menuFLUpcaseALL()), 0, 4);
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_writetags.png"), _("Downcase to ALL"), this, SLOT(slot_menuDowncaseALL()), 0, 5);
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_discard.png"), _("Discard Changes to ALL"), this, SLOT(slot_menuDiscardChangesALL()), 0, 6);
  }
  if(but == buttonTrack || but == buttonYear || but == buttonGenre) {
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_discard.png"), _("Discard Changes"), this, SLOT(slot_menuDiscardChanges()), 0, 1);
    menu->insertSeparator();
    if(but == buttonTrack) menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_writetags.png"), _("Auto Numbering By Listing Position"), this, SLOT(slot_autotrackButton()), 0, 3);
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_discard.png"), _("Discard Changes to ALL"), this, SLOT(slot_menuDiscardChangesALL()), 0, 2);
  }

  menu->exec(p, 0);
}

// ##################################################
// # callbacks for the menu-entries in context menu
// ##################################################
void GUI_TagListing::slot_menuDiscardChanges()
{
  LVI_Track *lvi = dynamic_cast<LVI_Track*>(selectedItems.at(currentTrack));

  if(buttonLastClicked == buttonFilename) editFilename->setText(lvi->getOriginalTrack()->filename);
  else if(buttonLastClicked == buttonTitle) editTitle->setText(lvi->getOriginalTrack()->title);
  else if(buttonLastClicked == buttonArtist) editArtist->setText(lvi->getOriginalTrack()->artist);
  else if(buttonLastClicked == buttonAlbum) editAlbum->setText(lvi->getOriginalTrack()->album);
  else if(buttonLastClicked == buttonTrack) spinTrack->setValue(lvi->getOriginalTrack()->tracknumber);
  else if(buttonLastClicked == buttonYear) editYear->setText(lvi->getOriginalTrack()->year);
  else if(buttonLastClicked == buttonGenre)
    {
      lvi->setGenre(lvi->getOriginalTrack()->genre);
      int xgenre = Tagger::genreToAlphaIndex(lvi->getGenre());
      comboGenre->setCurrentItem(xgenre);
    }
  else if(buttonLastClicked == buttonComment) editComment->setText(lvi->getOriginalTrack()->comment);
}

void GUI_TagListing::slot_menuDiscardChangesALL()
{
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);

    if(buttonLastClicked == buttonFilename) lvi->setFilename(lvi->getOriginalTrack()->filename);
    else if(buttonLastClicked == buttonTitle) lvi->setTitle(lvi->getOriginalTrack()->title);
    else if(buttonLastClicked == buttonArtist) lvi->setArtist(lvi->getOriginalTrack()->artist);
    else if(buttonLastClicked == buttonAlbum) lvi->setAlbum(lvi->getOriginalTrack()->album);
    else if(buttonLastClicked == buttonYear) lvi->setYear(lvi->getOriginalTrack()->year);
    else if(buttonLastClicked == buttonTrack) lvi->setTracknumber(lvi->getOriginalTrack()->tracknumber);
    else if(buttonLastClicked == buttonGenre) lvi->setGenre(lvi->getOriginalTrack()->genre);
    else if(buttonLastClicked == buttonComment) lvi->setComment(lvi->getOriginalTrack()->comment);

    item = item->nextSibling();
  }
  slot_menuDiscardChanges();
}

void GUI_TagListing::slot_menuFLUpcase()
{
  if(buttonLastClicked == buttonTitle) {
    QString s = editTitle->text();
    s = s.lower();
    for(uint i=0; i<s.length(); i++) if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace()))) s[i] = s[i].upper();
    editTitle->setText(s);
  } else if(buttonLastClicked == buttonArtist) {
    QString s = editArtist->text();
    s = s.lower();
    for(uint i=0; i<s.length(); i++) if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace()))) s[i] = s[i].upper();
    editArtist->setText(s);
  } else if(buttonLastClicked == buttonAlbum) {
    QString s = editAlbum->text();
    s = s.lower();
    for(uint i=0; i<s.length(); i++) if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace()))) s[i] = s[i].upper();
    editAlbum->setText(s);
  } else if(buttonLastClicked == buttonComment) {
    QString s = editComment->text();
    s = s.lower();
    for(uint i=0; i<s.length(); i++) if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace()))) s[i] = s[i].upper();
    editComment->setText(s);
  }
}

void GUI_TagListing::slot_menuFLUpcaseALL()
{
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);

    if(buttonLastClicked == buttonTitle) {
      QString s = lvi->getTitle();
      s = s.lower();
      for(uint i=0; i<s.length(); i++) if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace()))) s[i] = s[i].upper();
      lvi->setTitle(s);
    } else if(buttonLastClicked == buttonArtist) {
      QString s = lvi->getArtist();
      s = s.lower();
      for(uint i=0; i<s.length(); i++) if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace()))) s[i] = s[i].upper();
      lvi->setArtist(s);
    } else if(buttonLastClicked == buttonAlbum) {
      QString s = lvi->getAlbum();
      s = s.lower();
      for(uint i=0; i<s.length(); i++) if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace()))) s[i] = s[i].upper();
      lvi->setAlbum(s);
    } else if(buttonLastClicked == buttonComment) {
      QString s = lvi->getComment();
      s = s.lower();
      for(uint i=0; i<s.length(); i++) if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace()))) s[i] = s[i].upper();
      lvi->setComment(s);
    }

    item = item->nextSibling();
  }
  slot_menuFLUpcase();
}

void GUI_TagListing::slot_menuDowncase()
{
  if(buttonLastClicked == buttonTitle) {
    QString s = editTitle->text();
    s = s.lower();
    editTitle->setText(s);
  } else if(buttonLastClicked == buttonArtist) {
    QString s = editArtist->text();
    s = s.lower();
    editArtist->setText(s);
  } else if(buttonLastClicked == buttonAlbum) {
    QString s = editAlbum->text();
    s = s.lower();
    editAlbum->setText(s);
  } else if(buttonLastClicked == buttonComment) {
    QString s = editComment->text();
    s = s.lower();
    editComment->setText(s);
  }
}

void GUI_TagListing::slot_menuDowncaseALL()
{
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);

    if(buttonLastClicked == buttonTitle) {
      QString s = lvi->getTitle();
      s = s.lower();
      lvi->setTitle(s);
    } else if(buttonLastClicked == buttonArtist) {
      QString s = lvi->getArtist();
      s = s.lower();
      lvi->setArtist(s);
    } else if(buttonLastClicked == buttonAlbum) {
      QString s = lvi->getAlbum();
      s = s.lower();
      lvi->setAlbum(s);
    } else if(buttonLastClicked == buttonComment) {
      QString s = lvi->getComment();
      s = s.lower();
      lvi->setComment(s);
    }

    item = item->nextSibling();
  }
  slot_menuDowncase();
}

void GUI_TagListing::slot_menuStandardFilename()
{
  if(editArtist->text() == "" || editTitle->text() == "") return;
  else {
    QString ending = editFilename->text();
    ending = ending.remove(0, ending.findRev('.')).lower();
    editFilename->setText(filter_ext2(editArtist->text().lower() + " - " + editTitle->text().lower() + ending));
  }
}

void GUI_TagListing::slot_menuStandardFilenameALL()
{
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);

    if(lvi->getArtist() != "" && lvi->getTitle() != "") 
      {
	QString ending = lvi->getFilename();
	ending = ending.remove(0, ending.findRev('.')).lower();
	lvi->setFilename(filter_ext2(lvi->getArtist().lower() + " - " + lvi->getTitle().lower() + ending));
      }

    item = item->nextSibling();
  }
  slot_menuStandardFilename();
}

void GUI_TagListing::slot_menuAndromedaFilename()
{
  if(editArtist->text() == "" || editTitle->text() == "") return;
  else {
    QString ending = editFilename->text();
    ending = ending.remove(0, ending.findRev('.')).lower();

    QString s = editAlbum->text();
    s = s.lower();
    QString acro = ""; int acrocount = 0;
    for(uint i=0; i<s.length(); i++) if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace()))) acro[acrocount++] = s[i].upper();

    LVI_Track *lvi = dynamic_cast<LVI_Track*>(selectedItems.at(currentTrack));
    if(lvi->getTracknumber() != 0) {
      QString s = "";
      s.sprintf("%02d", lvi->getTracknumber());
      acro+=s;
    }

    editFilename->setText(filter_ext2(editArtist->text().lower() + " - " + acro + " - " + editTitle->text().lower() + ending));
  }
}

void GUI_TagListing::slot_menuHelpAndromeda() { 
  QMessageBox::information(this, _("About Andromeda Filename"),
_("The Andromeda option assigns a filename to a track, so it is lexically in "
"the right order including the order of tracknumbers. It was implemented from the " 
"great PHP-Script <b>Andromeda</b>.<br>"
"An Andromeda filename consists of the artist name, followed by the album abbreviation, a track number and "
" finally title name.<br>"
"By consistently using Andromeda file name, a simple alphanumeric sort will classify your tracks "
			   "by artist, album and track number.") );
}

void GUI_TagListing::slot_menuAndromedaFilenameALL()
{
  QListViewItem *item = tracklist->firstChild();
  while(item != 0) {
    LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);

    if (lvi->getArtist() != "" && lvi->getTitle() != "") {
      QString ending = lvi->getFilename();
      ending = ending.remove(0, ending.findRev('.')).lower();
      QString s = lvi->getAlbum();
      s = s.lower();
      QString acro = ""; int acrocount = 0;
      for(uint i=0; i<s.length(); i++) if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace()))) acro[acrocount++] = s[i].upper();

      if(lvi->getTracknumber() != 0) {
	QString s = "";
	s.sprintf("%02d", lvi->getTracknumber());
	acro+=s;
      }

      lvi->setFilename(filter_ext2(lvi->getArtist().lower() + " - " + acro + " - " + lvi->getTitle().lower() + ending));
    }

    item = item->nextSibling();
  }
  slot_menuAndromedaFilename();
}

void GUI_TagListing::slot_selPrevTrack()
{
  QListViewItem *item;
  if (!selectedItems.isEmpty() && selectedItems.count() == 1)
    item = selectedItems.getFirst();
  else if (tracklist->currentItem())
    item = tracklist->currentItem();
  else
    item = tracklist->firstChild();

  if (item) {
    item = item->itemAbove();
    if (item) {
      tracklist->clearSelection();
      tracklist->setSelected(item,true);
      tracklist->setCurrentItem(item);
      tracklist->ensureItemVisible(item);
    }
  }
}

void GUI_TagListing::slot_selNextTrack()
{
  QListViewItem *item;
  if (!selectedItems.isEmpty() && selectedItems.count() == 1)
    item = selectedItems.getFirst();
  else if (tracklist->currentItem())
    item = tracklist->currentItem();
  else
    item = tracklist->firstChild();

  if (item) {
    item = item->itemBelow();
    if (item) {
      tracklist->clearSelection();
      tracklist->setSelected(item,true);
      tracklist->setCurrentItem(item);
      tracklist->ensureItemVisible(item);
    }
  }
}

QString GUI_TagListing::filter_ext2(QString filename)
{  

// avoid filename using "/" and non printable character..

  for (int i=0; i<filename.length(); i++)
    if ( !filename[i].isPrint() || filename[i] == QChar('/') )
      filename[i]=QChar('_');
  return filename;
}
