/*
 * $id:$
 * 
 * Originally a kernel patch by Drew Hess <dhess@CS.Stanford.EDU>
 * ported to userspace by the Powertweak team.
 *
 */

#include <powertweak.h>
#include "pci.h"
#include "PCILIB/pciutils.h"

#define PCI_DEVICE_ID_INTEL_82850_MC1	0x2530

static struct pci_access *pacc;

 /*
 * The BIOS on some Intel i850 motherboards (notably the D850GB) 
 * may configure pool C RDRAM devices to run in nap mode due to thermal 
 * concerns. We set up pool C to run in standby mode for better
 * performance and set pools A and B to their max sizes.  Then we
 * write (but don't commit) a "safe" value in case an over-temperature
 * condition occurs.
 */

static void quirk_tehama (struct pci_dev *dev)
{
	u8 rdps;

	/*
	printk(KERN_INFO "i850: Configuring pool C RDRAM devices to run in standby mode\n");
	pci_read_config_byte(dev, 0x88, &rdps);
	printk(KERN_INFO "i850: BIOS set RDPS to 0x%02x\n", rdps);
	*/

	/* 
	 * i850 docs are unclear about how to unlock/set RDPS and
	 * then initialize the pool logic, so do it one step at a time.
	 * Wait for poolinit bit to clear for indication that pools
	 * are running in new mode.
	 */
	pci_write_byte(dev, 0x88, 0x0);
	pci_write_byte(dev, 0x88, 0x0f);
	pci_write_byte(dev, 0x88, 0x2f);
	do {
		pci_read_byte(dev, 0x88, &rdps);
	} while (rdps & 0x20);

	/* prep for nap mode in case of over-temperature condition */
	pci_write_byte(dev, 0x88, 0x19);
}

static int Closei850Backend(void)
{
	pci_cleanup (pacc);
	pacc = NULL;
	return TRUE;
}

int InitPlugin (void)
{
	struct pci_dev *dev;

	if (!fileexists("/proc/bus/pci"))
		return FALSE;

	pacc = pci_alloc();
	pacc->error = die;
	pci_init(pacc);
	pci_scan_bus(pacc);

	dev = pacc->devices;

	while (dev) {
		if ((dev->vendor_id == 0x8086) &&
			(dev->device_id == PCI_DEVICE_ID_INTEL_82850_MC1)) {
			quirk_tehama(dev);
			RegisterShutdownCallback(&Closei850Backend);
			return TRUE;
		}
		dev = dev->next;
	}

	Closei850Backend();
	return FALSE;
}

