/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef POWERSAVE_PM_INTERFACE_H
#define POWERSAVE_PM_INTERFACE_H

#include "throttle_interface.h"

#include "battery.h"
#include "powerlib.h"
#include "dbus_server.h"

using namespace std;

class MainLoop;
class EventManagement;
class Brightness;
class ClientConnection;
class CpufreqManagement;

typedef struct PM_STATE {
	// THERM_MODE comes from library libpower.h
	THERM_MODE TEMP_STATE;
	AC_ADAPTER_STATE AC_STATE;
	 PM_STATE():TEMP_STATE(OK),AC_STATE(AC_UNKNOWN) {}
} PM_STATE;


/** @brief global object for abstraction of acpi and apm
 *
 * mainly an abstraction of apm/acpi but contains other things like
 * ac adapter  handling which maybe moves to own class in future.
 */
class PM_Interface {
public:
	/** @brief constructor setting up the object */
	PM_Interface();
	
	/** @brief destructor freeing all members */
	virtual ~ PM_Interface();
	

	int sendSleepRequest(const char *method, int first_arg_type, ...);

	/** @brief triggers suspend to ram via HAL */
	bool suspendToDisk();

	/** @brief triggers suspend to ram via HAL */
	bool suspendToRam();

	/** @brief get the current state the battery is in
	 *
	 * @return the current battery state
	 */
	BATTERY_STATES batteryState();

	/** @brief used to execute scripts in INT_SCRIPT_PATH */
	int executeScript(const string &script, const string &param);

	/** @brief triggers a screenlock either from a client or script
	 *
	 * emits a screenlock signal over the bus and checks if a client
	 * with capability screenlock is connected. If not, execute script
	 * to lock the screen
	*/
	int screenlock(const string &param);

	/** @brief triggers a notification for the user
	 *
	 * emits a notification signal over the bus and checks if a client
	 * with capability notifications is connected. If not, execute
	 * script to display the notification
	 */
	int x_notification(const string &param);

	/** @brief starts the pm_object
	 *
	 * throws appropriate ac adapter/battery events and initializes
	 * the active scheme
	 */
	virtual int start();

	/** @brief checks for temperature events and executes appropriate
	 * events */
	virtual void checkTemperatureStateChanges();

	/** @brief Checks for Battery State changes (when AC adapter is
	 *         not plugged in) and throws appropriate event
	 *
	 * Be careful in ACPI you have to check whether the battery alarm also needs
	 * to be set
	 */
	virtual void checkBatteryStateChanges();

	/** @brief Checks for AC state changes and throws appropraite events
	 *
	 * if AC changes to offline, also battery state changes will be checked
	 * @return integer with state of AC 
	 * @retval 0 if AC state has not changed
	 * @retval 1 if AC state has changed
	 * @retval -1 if an error occur
	 */
	virtual int checkACStateChanges();

	/** @brief update temperature state
	 *
	 * @return integer with state of temperature
	 * @retval -1 on failure
	 * @retval 0 when state did not change
	 * @retval 1 when a more critical temperature level is reached
	 * @retval 2 when a less critical battery level is reached
	 */
	virtual int updateTemperatureState();

	/** @brief updates ACState
	 *
	 * @return integer with result of check for changes
	 * @retval 0 if AC state did not change
	 * @retval 1 else 
	 */
	virtual int updateACState();

	/** @brief updates the cpu state and throws appropriate events */
	virtual bool checkThrottling();

	/** @brief switches settings to a possibly new scheme
	 *         and activates them
	 * @return integer with result whether the scheme could (not) be set
	 *         or is already set.
	 * @retval -1 if scheme does not exist
	 * @retval  0 if scheme is already set
	 * @retval  1 if scheme has been successfully set
	 */
	virtual int activateScheme(const string &scheme_name);

	/** @brief switches scheme according to ac_state
	 *
	 * @param ac_state any ac adapter status
	 * @return integer with result whether the scheme could (not) be set
	 *         or is already set.
	 * @retval -1 if scheme does not exist
	 * @retval  0 if scheme is already set
	 * @retval  1 if scheme has been successfully set
	 */
	virtual int activateScheme(AC_ADAPTER_STATE ac_state);

	/** @brief Activate scheme settings 
	 *
	 * Inheriting functions in APM_Interface and ACPI_Interface
	 * set them specifically for APM/ACPI
	 *	    
	 * Add additional code that is needed after activating a new scheme
	 * or after re-reading configurations here (e.g. brightness?)
	 *
	 * Be aware that first the inhereting functions in the ACPI/APM_Interface
	 * layer are called
	 */
	virtual void activateSettings();

	/** @brief reread config files. */
	virtual int rereadConfig();

	/** @brief throttles the cpu
	 * 
	 * Only throttle if temperature is in unknown, ok or active state.
	 * Otherwise kernel may have throttled ...
	 */
	virtual void throttle(int percent = -1);

	/** @brief dethrottles the cpu
	 *
	 * Only dethrottle if temperature is in unknown, ok or active state.
	 * Otherwise kernel may have throttled ...
	 */
	virtual void dethrottle();

	/** @brief opens the event fd
	 *
	 * has to be implemented in derived classes (apm/acpi)
	 */
	virtual int openHWEventFD() = 0;

	/** @brief opens the acpid socket
	 *
	 * has to be implemented in derived classes (only acpi)
	 */
	virtual int openAcpidSocket() = 0;

	/** @brief handles a request on the event socket/fd 
	 *
	 * has to be implemented in derived classes (only acpi)
	 */
	virtual int handleHWEventRequest(int fd = -1) = 0;

	/** @brief handles a message on the events interface
	 * @param msg the dbus message received
	 * @return DBUS_ERROR
	 */
	DBUS_ERROR handleScriptsMessage(DBusMessage *msg, const string &msg_member);

	/** @brief handles a message on the request interface
	 * @param msg the dbus message received
	 * @param reply the dbus reply to send back
	 * @return DBUS_ERROR
	 */
	DBUS_ERROR handleDBusMessage(DBusMessage *msg, DBusMessage *reply,
				     const string &msg_member);

	/** @brief called when coming back from suspend
	 *
	 * updates ac state, battery state and adjust speeds
	 *
	 * @param cpufreq_mode cpufreq policy which was active before suspend
	 */
	void resume();

	/** @brief object to manage running events */
	EventManagement *_eM;

	/** @brief object implementing brightness functionality on some
	 * laptops */
	Brightness *_brightness;

protected:
	/** @brief defines whether cooling mode is supported */
	bool _cooling_mode_supported;

	/** @brief defines whether thermal trip points are supported */
	bool _thermal_trip_points_supported;

	/** @brief the current pm state contianing e.g. battery, thermal
	 * and cpu info */
	PM_STATE _cur_state;

	bool _can_suspend_to_disk;
	bool _can_suspend_to_ram;

	/** @brief object caring about throttling */
	ThrottleInterface _throttleInterface;

	/** @brief battery object represention one single battery */
	Battery _battery;

private:
};

#endif // POWERSAVE_PM_INTERFACE_H
