/*
 * main.c  Main part of the portslave.
 *
 * Version:  @(#)main.c  1.35  08-Nov-1997  MvS.
 *
 */
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <termios.h>
#include <ctype.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "radius.h"
#include "server.h"

static void usage()
{
    nsyslog(LOG_ERR, "Usage: portslave port|-\n");
    fprintf(stderr,  "Portslave RADIUS client v%s\tGPL2\n"
  "(c) 1997-1999 Miquel van Smoorenburg, "
  "Dave Cinege, Vesselin Atanasov, et al.\n"
  "(c) 2000-2001 Russell Coker et al.\n\n"
  "Usage: portslave port|-\n", PORTSLAVE_VERSION);
    exit(1);
}

/*
 *  Main loop for 1 login channel.
 */
int main(int argc, char **argv)
{
/* ai: structure that holds basically all data on the session, user-name,
       password, port number, IP address, MTU/MRU, etc

   tty: the device that is standard input

   i: usual loop counter

   rc: return code from system calls */
  struct auth ai;
  char *tty = NULL;
  int i, rc = 0;
/* do_acct: if we should do accounting in the main portslave program.
   printed: Whether we have displayed some text, in which case we wait 2 seconds
   is_ppp:  true if the protocol to use is PPP
*/
  bool do_acct = false;
  bool printed = false;
  bool is_ppp = false;
  char *param = argv[1];
  const char *config_file = CONFFILE;

  /*
   *    Check the syntax.
   */
  if(argc < 2)
    usage();
  if(argv[1][0] == '+')
  {
    if(argv[1][1])
    {
      config_file = &argv[1][1];
      param = argv[2];
    }
    else
    {
      config_file = argv[2];
      param = argv[3];
    }
  }
  if(param[0] != '-')
  {
    int port_num = atoi(param);
    if(port_num == 0 && param[0] != '0')
      usage();
    SetPortNo(port_num);
  }
  else
  {
    SetPortNo(RAD_INIT_PPPD);
    tty = ttyname(0);
    if (tty == NULL)
    {
      fprintf(stderr, "portslave: standard input is not a tty\n");
      exit(1);
    }
  }

  /* set the environment variable if using non-default file */
  if(config_file != CONFFILE)
    setenv("PORTSLAVE_CONF", config_file, 1);
  /*
   *    Read the config file.
   */
  if(rad_init(config_file, GetPortNo(), &ai, tty) < 0)
    return 1;

  if(lineconf.tty == NULL)
  {
    nsyslog(LOG_ERR, "port %d has no tty defined", GetPortNo());
    exit(1);
  }

  nsyslog(LOG_INFO, "portslave started on port %d (%s)", GetPortNo(),
                lineconf.tty);
  if(ai.proto == P_SOCKET_SERVER)
    goto socket_srv;

  /*
   *    Now put a getty on the port.
   */
  if(lineconf.emumodem)
    rc = emumodem(&ai, eLocModem);
  else
    rc = getty(&ai, eLocModem);

  if(rc < 0)
    exit(1);

  if(lineconf.protocol == P_PPP_ONLY)
    goto socket_srv;

  nsyslog(LOG_INFO, "Connected - waiting for login");

  /*
   *  Do login/password if needed.
   */
  if(lineconf.authtype)
  {
    for(i = 0; i < 3; i++)
    {
      rc = 0;
      
      if(lineconf.fixedlogin)
      {
        strncpy(ai.login, lineconf.fixedlogin, sizeof(ai.login));
        ai.login[sizeof(ai.login) - 1] = '\0';
      }
      else
      {
        rc = do_login(&ai);
      }
      
      if (rc == 2)  continue;
      if (rc < 0)  exit(1);

      do_acct = false;

      /*
       *  Do authentication if needed.
       */
      if(lineconf.locallogins && ai.login[0] == '!')
      {
        memmove(ai.login, ai.login+1, strlen(ai.login));
        ai.passwd[0] = '\0';
        ai.proto = P_LOCAL;
        break;
      }
      if(!strcmp(ai.login, "AutoPPP"))
      {
        ai.proto = P_AUTOPPP;
        break;
      }

      if(ai.proto == P_PPP)
        is_ppp = true;
      else
        do_acct = true;
      
      if(lineconf.fixedlogin)
        break;
      if(!do_local_or_server_authentication(&ai, is_ppp))
        break;
    }

    if(i == 3)
    {
      /*
       *  3 login failures - exit!
       */
      printf("Authentication failure - exiting\r\n"); 
      fflush(stdout);
      xusleep(250000);
      maketermsane();
      exit(1);
    }

    setenv("LOGNAME", ai.login, 1);

    if(do_acct)
      rad_acct(&ai, 1);

  }
  /*
   *  Okay, we are logged in here (or authentication comes later).
   */
  if(ai.message[0])
  {
    unsigned int x;
    for(x = 0; x < ai.msn; x++)
      printf("%s\r\n", ai.message[x]);

    printed = true;
  }

socket_srv:
  switch(ai.proto)
  {
    case P_AUTOPPP:
      nsyslog(LOG_INFO, "PPP frames detected - switching to PPP mode");
    break;
    case P_SLIP:
      nsyslog(LOG_INFO, "%s/SLIP session (%s)", ai.login, dotted(ai.address));
    break;
    case P_CSLIP:
      nsyslog(LOG_INFO, "%s/CSLIP session (%s)", ai.login, dotted(ai.address));
    break;
    case P_PPP:
    case P_PPP_ONLY:
      if(ai.login && strcmp(ai.login, "NONE"))
        nsyslog(LOG_INFO, "%s/PPP session (%s)", ai.login, dotted(ai.address));
      else
        nsyslog(LOG_INFO, "PPP session (%s)", dotted(ai.address));
    break;
    case P_SSH:
      if(!ai.host)
        ai.host = lineconf.host;
      nsyslog(LOG_INFO, "%s/ssh session (%s)", ai.login, dotted(ai.host));
    break;
    case P_SSH2:
      if(!ai.host)
        ai.host = lineconf.host;
      nsyslog(LOG_INFO, "%s/ssh2 session (%s)", ai.login, dotted(ai.host));
    break;
    case P_TELNET:
      if(!ai.host)
        ai.host = lineconf.host;
      nsyslog(LOG_INFO, "%s/telnet to %s", ai.login, dotted(ai.host));
    break;
    case P_RLOGIN:
      if(!ai.host)
        ai.host = lineconf.host;
      nsyslog(LOG_INFO, "%s/rlogin to %s", ai.login, dotted(ai.host));
    break;
    case P_SOCKET_SERVER:
      nsyslog(LOG_INFO, "socket server %d", GetPortNo());
    break;
    case P_SOCKET_CLIENT:
      nsyslog(LOG_INFO, "socket client %d", GetPortNo());
    break;
    case P_LOCAL:
      nsyslog(LOG_INFO, "%s/locallogin", ai.login);
    break;
    default:
      nsyslog(LOG_INFO, "%s/type %c starting", ai.login, ai.proto);
    break;
  }

  /*
   *  Print a portmaster compatible SLIP/PPP banner.
   */
  if (strchr("PCS", ai.proto)) {
    printf("%s session from (%s) ",
      ai.proto == 'P' ? "PPP" : "SLIP", dotted(lineconf.loc_host));
    printf("to %s beginning....\n", dotted(ai.address));
    printed = true;
  }

  /*
   *  Wait for output to drain.
   */
  fflush(stdout);
  if(printed)
    xsleep(1);
  tcdrain(1);

  /*
   *  Start the protocol.
   */
  rc = spawnit(&ai);

  /*
   *  If the protocol returned, tell the accounting server we're done.
   */
  if(do_acct)
    rad_acct(&ai, 0);

  if (ai.proto != P_PPP)
    nsyslog(LOG_INFO, "Session done.");

  return (rc == 0 ? 0 : 1);
}
