/*
 * libpsr.c  LibPortSlaveRadius.
 *    Shared library to be linked with pppd
 *    Takes care of the portslave side of things.
 *
 * Version  @(#)libpsr.c 1.03  09-Jan-1998  miquels@cistron.nl
 *
 */

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <pppd.h>
#include <fsm.h>
#include <lcp.h>
#include <ipcp.h>
#include <cbcp.h>

#include "server.h"

/* Version string for pppd so it knows that we were compiled with the right
   header files */
char pppd_version[] = VERSION;

/* thisauth is the struct that has all details of the user account in use */
static struct auth thisauth;
/* rad_using_pap is a boolean determining if user does PAP */
static int rad_using_pap;

int ul_login(char *, char *, char **, struct wordlist **, struct wordlist **);
void ul_ppp_ipup(void);
void ul_ppp_ipdown(void);
void rad_make_wordlist(u_int32_t, struct wordlist **);
#ifdef PORTSLAVE_CLIENT_IP_RULES
void rad_make_wordlist(u_int32_t, struct wordlist **);
void rad_make_wordlist2 (char * const *, int, struct wordlist **);
#endif
#ifdef PORTSLAVE_CALLBACK
void ul_cbcp_init(cbcp_state *);
#endif

int plugin_init(void)
{
  const char *env_data;
  const char *config = getenv("PORTSLAVE_CONF");
  if(!config) config = CONFFILE;
  if(rad_init(config, RAD_INIT_PPPD, &thisauth, ttyname(0)) < 0)
  {
    nsyslog(LOG_ERR, "libpsr plugin init failed!");
    exit(1);
  }
  env_data = getenv("PORTSLAVELOGNAME");
  if(!env_data)
  {
    nsyslog(LOG_ERR, "Bad environment, exit");
    exit(1);
  }
  if(strlen(env_data) == 4 && strncmp(env_data,"NONE",4) == 0){
    /* we start session for PPP only without auth */
    if(ipparam != NULL) {
      /* try get login name from ipparam */
      snprintf(thisauth.login, sizeof (thisauth.login), "%s", ipparam);
    } else {
      snprintf(thisauth.login, sizeof (thisauth.login), "%s", env_data);
    }
  } else {
    snprintf(thisauth.login, sizeof (thisauth.login), "%s", env_data);
  }
  update_utmp(lineconf.stripnames ? "%L" : "%l", lineconf.utmpfrom, &thisauth, lineconf.syswtmp);
  nopenlog(NULL, LOG_CONS|LOG_NDELAY, 0);
  pap_auth_hook = ul_login;
  ip_up_hook = ul_ppp_ipup;
  ip_down_hook = ul_ppp_ipdown;
#ifdef PORTSLAVE_CALLBACK
  cbcp_init_hook = ul_cbcp_init;
#endif
#ifdef PORTSLAVE_CLIENT_IP_RULES
  thisauth.valid_ip_rules = (char **) NULL;
  thisauth.num_ip_rules      = 0;
#endif
  rad_using_pap = 0;
  env_data = getenv("PORTSLAVE_SESSION");
  if(!env_data)
  {
    nsyslog(LOG_ERR, "Bad environment, exit");
    exit(1);
  }
  thisauth.acct_session_id = xstrdup(env_data);
  env_data = getenv("PORTSLAVE_START_TIME");
  if(!env_data)
  {
    nsyslog(LOG_ERR, "Bad environment, exit");
    exit(1);
  }
  thisauth.start = atoi(env_data);
  env_data = getenv("PORTSLAVE_CLI_SRC");
  if(env_data)
  {
    strncpy(thisauth.cli_src, env_data, sizeof(thisauth.cli_src));
    thisauth.cli_src[sizeof(thisauth.cli_src) - 1] = '\0';
  }
  env_data = getenv("PORTSLAVE_CLI_DST");
  if(env_data)
  {
    strncpy(thisauth.cli_dst, env_data, sizeof(thisauth.cli_dst));
    thisauth.cli_dst[sizeof(thisauth.cli_dst) - 1] = '\0';
  }
#ifdef PORTSLAVE_CALLBACK
  thisauth.cb_allowed = 0;
  thisauth.cb_number = NULL;
#endif
  return 0;
}

int ul_login(char *User, char *Passwd, char **unused, struct wordlist **addrs,
    struct wordlist **opts)
{
  char tmp[255];
  char *s;
#ifdef ALLOW_NO_LOCAL_IP
  bool set_ip_f;
#endif

  if((s = getenv ("CONNECT_INFO")) != NULL)
  {
    strncpy(thisauth.conn_info, s, sizeof(thisauth.conn_info));
    thisauth.conn_info[sizeof(thisauth.conn_info) - 1] = '\0';
  }

  snprintf(thisauth.login, sizeof (thisauth.login), "%s", User);
  snprintf(thisauth.passwd, sizeof (thisauth.passwd), "%s", Passwd);
  if(do_local_or_server_authentication(&thisauth, 1))
    return 0;
  if(thisauth.proto == P_PPP_ONLY)
    return 0;
  if(thisauth.netmask && thisauth.netmask != 0xFFFFFFFF)
    netmask = thisauth.netmask;
  if(thisauth.mtu)
    lcp_allowoptions[0].mru = thisauth.mtu;
  if(thisauth.mru)
  {
    lcp_wantoptions[0].mru = thisauth.mru;
    lcp_wantoptions[0].neg_mru = 1;
  }
#ifdef ALLOW_NO_LOCAL_IP
  if(thisauth.localip != 0xFFFFFFFF)
  {
    set_ip_f = true;
    strcpy(tmp, dotted(thisauth.localip));
    strcat(tmp, ":");
  }
  else
  {
    set_ip_f = false;
    tmp[0] = ':';
  }
#else
  strcpy(tmp, dotted(thisauth.localip));
  strcat(tmp, ":");
#endif

  /* If an IP address was set, use it as the valid address. */
  if(thisauth.address != 0xFFFFFFFF)
  {
#ifdef ALLOW_NO_LOCAL_IP
    set_ip_f = true;
#endif
    strcat(tmp, dotted (thisauth.address));
    rad_make_wordlist (thisauth.address, addrs);
  }
  else
  {
#ifdef PORTSLAVE_CLIENT_IP_RULES
    /* Return the list of IP rules, if any were */
    /*  specified, in a word list.  Otherwise,  */
    /*  just return NULL which means the auth   */
    /*  of the address will fail (pppd 2.4.0).  */

    if(thisauth.num_ip_rules > 0)
      rad_make_wordlist2(thisauth.valid_ip_rules, thisauth.num_ip_rules, addrs);
    else
      *addrs = NULL;
#else
    *addrs = NULL;
#endif
  }
#ifdef ALLOW_NO_LOCAL_IP
  if(set_ip_f)
  {
#endif
    if(setipaddr(tmp, NULL, 1) < 0)
    {
      nsyslog(LOG_ERR, "bad IP address %s", tmp);
      return 0;
    }
#ifdef ALLOW_NO_LOCAL_IP
  }
#endif
  *opts = NULL;
  if(thisauth.idletime > 0)
    idle_time_limit = thisauth.idletime;
  if(thisauth.sessiontime > 0)
    maxconnect = thisauth.sessiontime;
  setenv("LOGNAME", User, 1);
  update_utmp(lineconf.stripnames ? "%L" : "%l", lineconf.utmpfrom, &thisauth, lineconf.syswtmp);
  rad_using_pap = 1;
  nsyslog(LOG_NOTICE, "user %s logged in", User);
  return 1;
}

void ul_ppp_ipdown()
{
  thisauth.traffic.sent_bytes = link_stats.bytes_out;
  thisauth.traffic.recv_bytes = link_stats.bytes_in;
  thisauth.traffic.sent_pkts = link_stats.pkts_out;
  thisauth.traffic.recv_pkts = link_stats.pkts_in;
  rad_acct(&thisauth, 0);
  nsyslog(LOG_NOTICE, "user %s logged out", thisauth.login);
}

void ul_ppp_ipup()
{
  char *s;

  if(!rad_using_pap)
  {
    if((s = getenv("CONNECT_INFO")) != NULL)
    {
      thisauth.conn_info[0]='\0';
      strncat (thisauth.conn_info, s,sizeof(thisauth.conn_info)-1);
    }
    if(ipparam == NULL)
      ipparam = getenv("PORTSLAVELOGNAME");
    snprintf(thisauth.login, sizeof (thisauth.login), "%s", ipparam);
    getenv_from_rad("PORTSLAVE_FILTER", thisauth.filterid, MAX_FILTERS, &thisauth.fln);
    getenv_from_rad("PORTSLAVE_FRAMED_ROUTE", thisauth.framed_route, MAX_FRAMED_ROUTES, &thisauth.frn);
    thisauth.proto = P_PPP;
    thisauth.address = ipcp_gotoptions[0].hisaddr;
    thisauth.localip = ipcp_gotoptions[0].ouraddr; 
  }
  if(getenv("PORTSLAVE_DO_ACCT"))
    thisauth.do_acct = 1;
  rad_acct(&thisauth, 1);
}

void rad_make_wordlist (u_int32_t addr, struct wordlist **addrs)
{
  const char *ipaddr;
  struct wordlist *ap;

  ipaddr = dotted(addr);
  ap = (struct wordlist *) xmalloc(sizeof(struct wordlist) + strlen(ipaddr) + 1);
  ap->word = (char *) (ap + 1);   
  ap->next = NULL;
  strcpy(ap->word, ipaddr);
  *addrs = ap;
}

#ifdef PORTSLAVE_CLIENT_IP_RULES
/*
 * Create a word list for PPPD from an array of strings.
 */
void rad_make_wordlist2(char * const *strings, int num, struct wordlist **addrs)
{
  struct wordlist *head;
  struct wordlist *ap;
  struct wordlist *prev;
  int             len;
  int             cur;

  if(num == 0 || strings == NULL)
    return;

  head = NULL;
  prev = NULL;
  cur = 0;

  /* Loop through all of the strings given. */

  while(cur < num)
  {
    /* Allocate the structure; the "word" must live in */
    /*  the same allocated chunk of memory.            */
    len = sizeof ap[0] + strlen(strings[cur]) + 1;
    ap = (struct wordlist *) xmalloc(len);

    ap->word = (char *) (ap + 1);
    ap->next = NULL;
    strcpy(ap->word, strings[cur]);

    /* Link this entry into the list if it's not the */
    /*  head of the list.                            */
    if(prev != NULL)
      prev->next = ap;

    /* Remember the head of the list if it was not */
    /*  already set.                               */
    if(head == NULL)
      head = ap;

    prev = ap;
    cur++;
  }

  *addrs = head;
}
#endif

#ifdef PORTSLAVE_CALLBACK
void ul_cbcp_init(cbcp_state *us)
{
  if(thisauth.cb_allowed != 0)
  {
    us->us_allowed = thisauth.cb_allowed;
    if(thisauth.cb_number != NULL)
    {
      us->us_number = xstrdup(thisauth.cb_number);
      free(thisauth.cb_number); /* we don't need them anymore */
    }
  }
  return;
}
#endif
