/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxdef-globals.h"

#include "polyxdef-ui-atomdef.h"
#include "polyxdef-ui-syntaxcheck.h"


GtkWidget *
polyxdef_ui_atomdef_wnd_setup_window (PxmDefCtxt *defctxt)
{
  GtkWidget *widget = NULL;
  GtkWidget *entry = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;

  GtkWidget *vbox = NULL;
  GtkWidget *sw = NULL;

  GladeXML *xml = NULL;
  
  gchar *gui_file = NULL;
  
  
  /* We are setting up the atom definition window for the defctxt. The
     definition window is referenced in the defctxt with the
     definition_wnd member...
  */
  g_assert (defctxt != NULL);

  /* Basically, we don't want to call this function twice for the same
     definition context.
  */
  g_assert (defctxt->definition_wnd == NULL);

  /* Sanity check, the array of atoms is allocated on newing the
     PxmDefCtxt.
   */
  g_assert (defctxt->atomGPA != NULL);
  

  gui_file = 
    g_strdup_printf ("%s/polyxdef.glade", userspec->gladedir);

  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "polyxdef_atomdef_wnd", 
		       PACKAGE);
  g_free (gui_file);
  
  if (xml == NULL)
    {
      g_critical (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }

  defctxt->definition_wnd =
    glade_xml_get_widget (xml, "polyxdef_atomdef_wnd");
  
  if (defctxt->definition_wnd == NULL)
    {
      g_critical (_("%s@%d: failed to create the atom definition window\n"),
	     __FILE__, __LINE__);

      return NULL;
    }

  /* If the defctxt contains valid data, they certainly have been
     loaded from a file. If this is true, then the member
     defctxt->filename should be non-NULL. If so we display the value
     of that variable in a GtkEntry.
  */
  entry = glade_xml_get_widget (xml, "atomdef_filename_entry");
  g_assert (entry != NULL);

  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atomdef_filename_entry", entry);

  if (defctxt->filename != NULL)
    {
      /* Let's just make a real sanity check !
       */
      g_assert (TRUE == 
		g_file_test (defctxt->filename, G_FILE_TEST_EXISTS));
      
      /* Now we can set the file name to its corresponding 
	 GtkEntry widget.
      */
      gtk_entry_set_text (GTK_ENTRY (entry), defctxt->filename);
    }
  
  /* Now we can continue with the setup of the window. Note that if no
     atom definition file was read from disk, then defctxt->atomGPA is
     non-NULL but empty, which is OK; we should be able to deal with
     that eventuality.
  */


  /* Store the pointers of all the widgets that we may need to access
   * later...
   */
  widget = glade_xml_get_widget (xml, "atomdef_status_label");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atomdef_status_label", widget);


  /* Start preparing the treeview displaying the available atoms.
   */
  widget = glade_xml_get_widget (xml, "atoms_available_frame");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atoms_available_frame", widget);

  vbox = glade_xml_get_widget (xml, "atoms_available_vbox");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atoms_available_vbox", vbox);

  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atoms_available_sw", sw);
  
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);
  
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  

  
  /* Create the treeview model.
   */
  model = 
    polyxdef_ui_atomdef_wnd_create_atoms_treeview_model (defctxt->atomGPA);

  /* Set to the model a datum with a pointer to defctxt, so that the
     any data is easily accessible later. See the _cell_edited ()
     functions. And vice-versa !
  */
  g_object_set_data (G_OBJECT (model), "defctxt", defctxt);

  g_object_set_data (G_OBJECT (defctxt->definition_wnd), 
		     "atom_treeview_model", model);

  /* Create the treeview proper.
   */
  treeview = polyxdef_ui_atomdef_wnd_create_atoms_treeview (defctxt);
  g_assert (treeview != NULL);
  
  /* Now that the treeview was created, we can unref the model.
   */
  g_object_unref (G_OBJECT (model));



  /* Continue with other more conventional widgets ... :-)
   */

  widget = glade_xml_get_widget (xml, 
				 "atomdef_actions_frame");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atomdef_actions_frame", widget);

  widget = glade_xml_get_widget (xml, 
				 "atomdef_add_atom_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atomdef_add_atom_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxdef_ui_atomdef_wnd_add_atom_button),
		    defctxt);

  widget = glade_xml_get_widget (xml, 
				 "atomdef_add_isotope_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atomdef_add_isotope_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxdef_ui_atomdef_wnd_add_isotope_button),
		    defctxt);

  widget = glade_xml_get_widget (xml, 
				 "atomdef_remove_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atomdef_remove_button", widget);
  
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxdef_ui_atomdef_wnd_remove_button),
		    defctxt);

  widget = glade_xml_get_widget (xml, 
				 "atomdef_check_syntax_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atomdef_check_syntax_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxdef_ui_atomdef_wnd_check_syntax_button),
		    defctxt);

  widget = glade_xml_get_widget (xml, 
				 "atomdef_save_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atomdef_save_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxdef_ui_atomdef_wnd_save_button),
		    defctxt);

  widget = glade_xml_get_widget (xml, 
				 "atomdef_saveas_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atomdef_saveas_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxdef_ui_atomdef_wnd_saveas_button),
		    defctxt);

  /* There is a GtkEntry in the window that is commonly used to 
   * display messages.
   */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "messages_entry", widget);

  

  /* We don't need the GladeXML object any more, so unref it
   * to save some memory 
   */
  g_object_unref (G_OBJECT (xml));


  /* Connect some important signals for the window proper.
   */
  g_signal_connect (G_OBJECT (defctxt->definition_wnd),
		    "delete_event",
		    G_CALLBACK (polyxdef_ui_atomdef_wnd_delete_event),
		    defctxt);
  
  g_signal_connect (G_OBJECT (defctxt->definition_wnd),
		    "destroy_event",
		    G_CALLBACK (polyxdef_ui_atomdef_wnd_destroy_event),
		    defctxt);

  gtk_widget_show_all (GTK_WIDGET (defctxt->definition_wnd));

  return defctxt->definition_wnd;
}




/* SYNTAX ERROR CHECKING FUNCTIONS
 */
gboolean
polyxdef_ui_atomdef_wnd_check_syntax_button (GtkWidget *widget, 
					     gpointer data)
{
  PxmDefCtxt *defctxt = data;

  gboolean valid_result = TRUE;

  gchar *valid = NULL;
  


  g_assert (defctxt != NULL);
  g_assert (defctxt->atomGPA != NULL);


  /* Now validate all the atoms in the array.
   */
  valid_result = pxmchem_atom_array_validate_all (defctxt->atomGPA, &valid);
  
  if (valid_result == FALSE)
    {
      /* There was at least one error, which means that the string in
	 valid has been allocated. We should show that string and
	 later free it.
      */
      g_assert (valid != NULL);

      if (defctxt->syntax_check_wnd != NULL)
	{
	  polyxdef_ui_syntaxcheck_wnd_reset (defctxt);
	}
      else
	{
	  /* We have to setup the window first.
	   */
	  defctxt->syntax_check_wnd = 
	    polyxdef_ui_syntaxcheck_wnd_setup_window (defctxt, NULL);
	  
	  if (defctxt->syntax_check_wnd == NULL)
	    {
	      g_message (_("%s@%d: failed to set up the syntax-checking "
		       "window for the following errors: '%s'\n\n"),
		     __FILE__, __LINE__, valid);
	      
	      g_free (valid);
	      
	      return FALSE;
	    }
	}

      /* We now have an empty textview where to set the errors' string.
       */
      polyxdef_ui_syntaxcheck_wnd_set_errors (defctxt, valid);

      g_free (valid);
      
      return FALSE;
    }
  else if (valid_result == TRUE)
    {
      /* Great, the validation succeeded, just inform the user.
       */
      g_message (_("%s@%d: the validation succeeded\n"),
	     __FILE__, __LINE__);
      
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 _("The validation succeeded"), 
	 POLYXMASS_NORM_MSG_TIMEOUT);

      return TRUE;
    }
  else
    g_assert_not_reached ();

  return FALSE;
}





/* ATOMS TREEVIEW CREATION STUFF
 */
GtkTreeModel *
polyxdef_ui_atomdef_wnd_create_atoms_treeview_model (GPtrArray *GPA)
{
  GtkTreeStore *model;
  GtkTreeIter tree_iter;

  gint iter = 0;
  gint jter = 0;

  gdouble *mass = NULL;
  gdouble *abund = NULL;

  gchar *mono_mass = NULL;
  gchar *avg_mass = NULL;
  gchar *abundance = NULL;

  PxmAtom *atom = NULL;
  
  model = gtk_tree_store_new (COLUMN_ATOM_COL_COUNT,

			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,

			      G_TYPE_POINTER,

			      G_TYPE_BOOLEAN,
			      G_TYPE_BOOLEAN);


  /* Add data for each atom to the tree store.
   */
  for (iter = 0 ; iter < GPA->len ; iter++)
    {
      atom = g_ptr_array_index (GPA, iter);
      g_assert (atom != NULL);

      /* First we have to convert gdouble data to string.
       */
      avg_mass = 
	libpolyxmass_globals_dtoa (atom->avg, 10);
      
      
      gtk_tree_store_append (model, &tree_iter, NULL);

      gtk_tree_store_set 
	(model, &tree_iter,

	 COLUMN_ATOM_NAME, atom->name,
	 COLUMN_ATOM_SYMBOL, atom->symbol,
	 COLUMN_ATOM_AVG_MASS, avg_mass,
	 COLUMN_ISOTOPE_MASS, "",
	 COLUMN_ISOTOPE_ABUNDANCE, "",

	 COLUMN_ATOM_POINTER, atom,
	 
	 COLUMN_ATOM_VISIBLE, TRUE,
	 COLUMN_ATOM_EDITABLE, TRUE,
	 -1);
      
      /* Add children, namely abundances and monoisotopic masses. The
	 two arrays MUST have the same number of elements, of course.
      */
      g_assert (atom->massGPA->len == atom->abundGPA->len);
      
      for (jter = 0 ; jter < atom->massGPA->len ; jter++)
	{
	  GtkTreeIter tree_jter;
	  
	  mass  = g_ptr_array_index (atom->massGPA, jter);
	  g_assert (mass != NULL);

	  abund  = g_ptr_array_index (atom->abundGPA, jter);
	  g_assert (abund != NULL);

	  mono_mass = libpolyxmass_globals_dtoa (*mass, 10);
      
	  abundance = libpolyxmass_globals_dtoa (*abund, 10);
	  
	  gtk_tree_store_append (model, &tree_jter, &tree_iter);
	  gtk_tree_store_set 
	    (model, &tree_jter,

	     COLUMN_ATOM_NAME, "",
	     COLUMN_ATOM_SYMBOL, "",
	     COLUMN_ATOM_AVG_MASS, "",
	     COLUMN_ISOTOPE_MASS, mono_mass,
	     COLUMN_ISOTOPE_ABUNDANCE, abundance,

	     COLUMN_ATOM_POINTER, "",
	     
	     COLUMN_ATOM_VISIBLE, TRUE,
	     COLUMN_ATOM_EDITABLE, TRUE,
	     -1);
	  
	  g_free (mono_mass);
	  g_free (abundance);
	}
    }

  return GTK_TREE_MODEL (model);
}


GtkWidget *
polyxdef_ui_atomdef_wnd_create_atoms_treeview (PxmDefCtxt *defctxt)
{
  GtkTreeModel *model = NULL;
  GtkWidget *treeview = NULL;

  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;

  GtkWidget *sw = NULL;
  GtkWidget *vbox = NULL;

  gint col_offset = 0;

  
  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  

  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			     "atom_treeview_model");
  g_assert (model != NULL);
  

  /* Get the vbox where to put all the stuff... and same for the
     scrolled window...*/
  vbox = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			    "atoms_available_vbox");
  g_assert (vbox != NULL);
  
  sw = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			  "atoms_available_sw");
  g_assert (sw != NULL);
  

  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);
  g_assert (treeview != NULL);
  
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (defctxt->definition_wnd), 
		     "atom_treeview", treeview);
  
  
  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);
  
  /* Atom definition name.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_ATOM_NAME);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK (polyxdef_ui_atomdef_wnd_atom_name_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Name"),
						 renderer, 
						 "text",
						 
						 COLUMN_ATOM_NAME,
						 
						 "visible",
						 COLUMN_ATOM_VISIBLE,
						 
						 "editable",
						 COLUMN_ATOM_EDITABLE, 
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  

  /* Atom definition symbol.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_ATOM_SYMBOL);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK (polyxdef_ui_atomdef_wnd_atom_symbol_cell_edited),
     defctxt);
  
  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Symbol"),
						 renderer, 
						 "text",
						 
						 COLUMN_ATOM_SYMBOL,
						 
						 "visible",
						 COLUMN_ATOM_VISIBLE,
						 
						 "editable",
						 COLUMN_ATOM_EDITABLE, 
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  
  /* Atom average mass.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_ATOM_AVG_MASS);

  /* Note that we do not want that the user changes the average mass
     manually, as this mass has to be calculated from the isotopes each
     time an atom definition file is read from disk.
  */

  /*
    g_signal_connect 
    (G_OBJECT (renderer), "edited", 
    G_CALLBACK (polyxdef_ui_atomdef_wnd_atom_avg_mass_cell_edited),
    defctxt);
  */
  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Avg Mass"),
						 renderer, 
						 "text",
						 
						 COLUMN_ATOM_AVG_MASS,
						 
						 "visible",
						 COLUMN_ATOM_VISIBLE,
						 
						 /*
						   "editable",
						   COLUMN_ATOM_EDITABLE, 
						 */

						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  

  /* Isotope monoisotopic mass.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_ISOTOPE_MASS);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK (polyxdef_ui_atomdef_wnd_isotope_mass_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Mono Mass"),
						 renderer, 
						 "text",
						 
						 COLUMN_ISOTOPE_MASS,
						 
						 "visible",
						 COLUMN_ATOM_VISIBLE,
						 
						 "editable",
						 COLUMN_ATOM_EDITABLE, 
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  

  /* Isotope abundance.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_ISOTOPE_ABUNDANCE);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK (polyxdef_ui_atomdef_wnd_isotope_abund_cell_edited),
     defctxt);
  
  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Abundance"),
						 renderer, 
						 "text",
						 
						 COLUMN_ISOTOPE_ABUNDANCE,
						 
						 "visible",
						 COLUMN_ATOM_VISIBLE,
						 
						 "editable",
						 COLUMN_ATOM_EDITABLE, 
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  

  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);

  gtk_tree_view_columns_autosize (GTK_TREE_VIEW (treeview));


  /* End preparing the treeview displaying the available atoms.
   */

  return treeview ; 
}




/* TREEVIEW EDITED CELL CALLBACKS
 */
void
polyxdef_ui_atomdef_wnd_atom_name_cell_edited (GtkCellRendererText *cell,
					       const gchar *path_string,
					       const gchar *new_text,
					       gpointer data)
{
  PxmDefCtxt *defctxt = data;
  
  GtkTreeModel *model = NULL;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
  GtkTreeIter tree_iter;
  
  PxmAtom *atom = NULL;
  
  gint *column = NULL;
  gint idx_atom = -1;
  gint depth = -1;
  
  gint result = 0;
    
  gchar *old_text = NULL;
  gchar *help = NULL;

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->atomGPA != NULL);
  
  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			     "atom_treeview_model");
  g_assert (model != NULL);
  
  column = g_object_get_data (G_OBJECT (cell), "column");

  gtk_tree_model_get_iter (model, &tree_iter, path);

  depth = gtk_tree_path_get_depth (path);


  /* 
     We only do accept to change anything if the cell that is edited
     corresponds to a row which belongs to an atom row, and not an
     isotope. That is the depth cannot be higher than one.
  */
  if (depth > 1)
    {
      return;
    }
	
  /* 
     Before proceeding and accepting the new text, we should make sure
     that no other atom already exists by the same name.
  */
  result = pxmchem_atom_get_index_by_name ((gchar *) new_text, 
					   defctxt->atomGPA);
	
  if (-1 != result)
    {
      help = g_strdup_printf (_("An atom by the same name"
				" exists already: '%s'"),
			      new_text);

      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 help, 
	 POLYXMASS_MEDI_MSG_TIMEOUT);
      
      g_free (help);
      
      return ;
    }

  /* At this point, all is OK, we can make the replacement.
   */
  gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
  g_free (old_text);

  /* Get the index of the item that is edited.
   */
  idx_atom = gtk_tree_path_get_indices (path) [0];

  atom = g_ptr_array_index (defctxt->atomGPA, idx_atom);
	
  pxmchem_atom_set_name (atom, (gchar *) new_text);

  defctxt->is_modified = TRUE;
  polyxdef_ui_atomdef_wnd_update_status_label (defctxt);
  
  gtk_tree_store_set (GTK_TREE_STORE (model), 
		      &tree_iter, column,
		      (gchar *) new_text,
		      -1);
  return;
}

void
polyxdef_ui_atomdef_wnd_atom_symbol_cell_edited (GtkCellRendererText *cell,
						 const gchar *path_string,
						 const gchar *new_text,
						 gpointer data)
{



  PxmDefCtxt *defctxt = data;
  
  GtkTreeModel *model = NULL;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
  GtkTreeIter tree_iter;
  
  PxmAtom *atom = NULL;
  
  gint *column = NULL;
  gint idx_atom = -1;
  gint depth = -1;
  
  gint result = 0;
    
  gchar *old_text = NULL;
  gchar *help = NULL;


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->atomGPA != NULL);
  
  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			     "atom_treeview_model");
  g_assert (model != NULL);
  
  column = g_object_get_data (G_OBJECT (cell), "column");

  gtk_tree_model_get_iter (model, &tree_iter, path);

  depth = gtk_tree_path_get_depth (path);


  /* 
     We only do accept to change anything if the cell that is edited
     corresponds to a row which belongs to an atom row, and not an
     isotope. That is the depth cannot be higher than one.
  */
  if (depth > 1)
    {
      return;
    }
	
  /* 
     Before proceeding and accepting the new text, we should make sure
     that no other atom already exists by the same symbol.
  */
  result = pxmchem_atom_get_index_by_symbol ((gchar *) new_text, 
					     defctxt->atomGPA);
	
  if (-1 != result)
    {
      help = g_strdup_printf (_("An atom by the same symbol"
				" exists already: '%s'"),
			      new_text);
	    
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 help, 
	 POLYXMASS_MEDI_MSG_TIMEOUT);
	    
      g_free (help);
	    
      return ;
    }

  /* Further, we want to make sure that the syntax is correct: one
     uppercase character at the beginning and no more than 2 total.
  */
  if (FALSE == pxmchem_atom_check_symbol_syntax ((gchar *) new_text))
    {
      help = g_strdup_printf (_("Incorrect syntax for atom symbol: '%s'"),
			      new_text);
	    
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 help, 
	 POLYXMASS_MEDI_MSG_TIMEOUT);
      
      g_free (help);
      
      return ;
    }

  /* At this point, all is OK, we can make the replacement.
   */
  gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
  g_free (old_text);

  /* Get the index of the item that is edited.
   */
  idx_atom = gtk_tree_path_get_indices (path) [0];

  atom = g_ptr_array_index (defctxt->atomGPA, idx_atom);
	
  pxmchem_atom_set_symbol (atom, (gchar *) new_text);

  defctxt->is_modified = TRUE;
  polyxdef_ui_atomdef_wnd_update_status_label (defctxt);
  
  gtk_tree_store_set (GTK_TREE_STORE (model), 
		      &tree_iter, column,
		      (gchar *) new_text,
		      -1);
  return;
}

void
polyxdef_ui_atomdef_wnd_atom_avg_mass_cell_edited (GtkCellRendererText *cell,
						   const gchar *path_string,
						   const gchar *new_text,
						   gpointer data)
{
  /*
    debug_printf (("polyxdef_ui_atomdef_wnd_atom_avg_mass_cell_edited\n"));
  */
  
  return;
}

void
polyxdef_ui_atomdef_wnd_isotope_mass_cell_edited (GtkCellRendererText *cell,
						  const gchar *path_string,
						  const gchar *new_text,
						  gpointer data)
{
  PxmDefCtxt *defctxt = data;
  
  GtkTreeModel *model = NULL;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
  GtkTreeIter tree_iter;
  
  PxmAtom *atom = NULL;
  gdouble *mono = NULL;
  /*
    gdouble *abund = NULL;
  */
  
  gint *column = NULL;
  gint idx_atom = -1;
  gint idx_isotope = -1;
  gint depth = -1;
  
  gdouble value = 0;

  gchar *old_text = NULL;
  gchar *help = NULL;


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->atomGPA != NULL);
  
  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			     "atom_treeview_model");
  g_assert (model != NULL);
  
  column = g_object_get_data (G_OBJECT (cell), "column");
  
  gtk_tree_model_get_iter (model, &tree_iter, path);
  
  depth = gtk_tree_path_get_depth (path);

  /*
    debug_printf (("depth is %d\n", depth));
  */
  
  /* 
     We only do accept to change anything if the cell that is edited
     corresponds to a row which belongs to an isotope row, and not an
     atom. That is the depth cannot be lower than one.
  */
  if (depth != 2)
    {
      return;
    }
  
  /* We want to make sure that the mass that was entered as a string
     acutally can get transformed into a gdouble.
  */
  if (FALSE == libpolyxmass_globals_strtod ((gchar *) new_text, &value))
    {
      help = g_strdup_printf (_("Incorrect syntax for isotope mass: '%s'"),
			      new_text);
      
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 help, 
	 POLYXMASS_MEDI_MSG_TIMEOUT);
      
      g_free (help);
      
      return ;
    }
  
  /* At this point, all is OK, we can make the replacement.
   */
  gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
  g_free (old_text);
  
  /* 
     Get the index of the atom item of which the isotope item is
     edited.
  */
  idx_atom = gtk_tree_path_get_indices (path) [0];
  atom = g_ptr_array_index (defctxt->atomGPA, idx_atom);

  /* 
     Get the index of the isotope item that is edited.
  */
  idx_isotope = gtk_tree_path_get_indices (path) [1];

  /* Extract from the atom's massGPA and abundGPA arrays, the mass of
     the isotope and its abundance. These two arrays have to have the 
     same number of items ! Make a sanity check.
  */
  g_assert (atom->massGPA->len == atom->abundGPA->len);
  
  mono = g_ptr_array_index (atom->massGPA, idx_isotope);
  /*
    abund = g_ptr_array_index (atom->abundGPA, idx_isotope);
  */

  /* Actually make the value change now !!!
   */
  *mono = value;
  	
  defctxt->is_modified = TRUE;
  polyxdef_ui_atomdef_wnd_update_status_label (defctxt);
  
  gtk_tree_store_set (GTK_TREE_STORE (model), 
		      &tree_iter, column,
		      (gchar *) new_text,
		      -1);

  /* We now have to update the average mass of the atom !!!
   */
  polyxdef_ui_atomdef_wnd_atom_avg_mass_cell_update (defctxt,
						     idx_atom);
  
  return;
}

void
polyxdef_ui_atomdef_wnd_isotope_abund_cell_edited (GtkCellRendererText *cell,
						   const gchar *path_string,
						   const gchar *new_text,
						   gpointer data)
{
  PxmDefCtxt *defctxt = data;
  
  GtkTreeModel *model = NULL;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
  GtkTreeIter tree_iter;
  
  PxmAtom *atom = NULL;
  /*
    gdouble *mono = NULL;
  */
  gdouble *abund = NULL;
  

  gint *column = NULL;
  gint idx_atom = -1;
  gint idx_isotope = -1;
  gint depth = -1;
  
  gdouble value = 0;

  gchar *old_text = NULL;
  gchar *help = NULL;


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->atomGPA != NULL);
  
  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			     "atom_treeview_model");
  g_assert (model != NULL);
  
  column = g_object_get_data (G_OBJECT (cell), "column");
  
  gtk_tree_model_get_iter (model, &tree_iter, path);
  
  depth = gtk_tree_path_get_depth (path);
  
  /*
    debug_printf (("depth is %d\n", depth));
  */
  
  /* 
     We only do accept to change anything if the cell that is edited
     corresponds to a row which belongs to an isotope row, and not an
     atom. That is the depth cannot be lower than one.
  */
  if (depth != 2)
    {
      return;
    }
  
  /* We want to make sure that the mass that was entered as a string
     acutally can get transformed into a gdouble.
  */
  if (FALSE == libpolyxmass_globals_strtod ((gchar *) new_text, &value))
    {
      help = g_strdup_printf (_("Incorrect syntax for isotope "
				"abundance: '%s'"),
			      new_text);
      
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 help, 
	 POLYXMASS_MEDI_MSG_TIMEOUT);
      
      g_free (help);
      
      return ;
    }
  
  /* At this point, all is OK, we can make the replacement.
   */
  gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
  g_free (old_text);
  
  /* 
     Get the index of the atom item of which the isotope item is
     edited.
  */
  idx_atom = gtk_tree_path_get_indices (path) [0];
  atom = g_ptr_array_index (defctxt->atomGPA, idx_atom);

  /* 
     Get the index of the isotope item that is edited.
  */
  idx_isotope = gtk_tree_path_get_indices (path) [1];

  /* Extract from the atom's massGPA and abundGPA arrays, the mass of
     the isotope and its abundance. These two arrays have to have the 
     same number of items ! Make a sanity check.
  */
  g_assert (atom->massGPA->len == atom->abundGPA->len);
  
  /*
    mono = g_ptr_array_index (atom->massGPA, idx_isotope);
  */
  abund = g_ptr_array_index (atom->abundGPA, idx_isotope);

  /* Actually make the value change now !!!
   */
  *abund = value;
  	
  defctxt->is_modified = TRUE;
  polyxdef_ui_atomdef_wnd_update_status_label (defctxt);

  
  gtk_tree_store_set (GTK_TREE_STORE (model), 
		      &tree_iter, column,
		      (gchar *) new_text,
		      -1);

  /* We now have to update the average mass of the atom !!!
   */
  polyxdef_ui_atomdef_wnd_atom_avg_mass_cell_update (defctxt,
						     idx_atom);
  
  return;
}


/* ATOM / ISOTOPE ADD/REMOVE BUTTONS.
 */
void
polyxdef_ui_atomdef_wnd_add_atom_button (GtkWidget *widget,
					 gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeViewColumn* column = NULL;

  GtkTreeIter tree_iter;
  GtkTreeIter tree_iter_sel;
  GtkTreeIter tree_iter_parent;

  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;

  gint idx_atom = -1;

  gint depth = -1;
      
  PxmAtom *atom = NULL;
  PxmAtom *atom_test = NULL;
  


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->atomGPA != NULL);
  

  /****************** GENERAL PHILOSOPHY **************************/

  /* An atom is made with the following intrinsic members: name,
     symbol and monoisotopic and average mass.
   
     It may also contain any number of isotopes, which are just two
     gdouble values: mono mass and abundance; each value is stored int
     the atom->massGPA and atom->abundGPA GPtrArray of gdouble*
     instances.
    
     If an atom contains isotopes, they are represented in the atom
     treeview as items in a treenode at a depth incremented by one.
  */

  /* That means that depending on the treeview situation, the actions
     that we need to take in response to the clicking onto the ADD
     button are different.
  */

  /* If no selection is made at all, just append a new atom at the end
     of the treeview.
    
     If a selection is made:
     
     - if the selected item is an atom row, then a new atom row is to
     be added after this selected atom row.
   
     - if the selected row is an isotope row, then a new atom row is to
     be added after the atom row of which the isotope is currently
     selected.
  */


  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			     "atom_treeview_model");
  g_assert (model != NULL);
  
  treeview= g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			       "atom_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);

  /* If there is no current selection, that means that the user wants
     to add an atom item and not an isotope.
  */
  if (result == FALSE)
    {
      /* Allocate a new atom so that we can later fill its member
	 data.
      */
      atom = pxmchem_atom_new ();

      pxmchem_atom_set_name (atom, _("enter name here"));
      pxmchem_atom_set_symbol (atom, _("enter symbol here"));
      
      g_ptr_array_add (defctxt->atomGPA, atom);

      defctxt->is_modified = TRUE;
      polyxdef_ui_atomdef_wnd_update_status_label (defctxt);
      
      gtk_tree_store_append (GTK_TREE_STORE (model), &tree_iter, NULL);
      
      gtk_tree_store_set 
	((GtkTreeStore *) model, &tree_iter,

	 COLUMN_ATOM_NAME, atom->name,
	 COLUMN_ATOM_SYMBOL, atom->symbol,
	 COLUMN_ATOM_AVG_MASS, "",
	 COLUMN_ISOTOPE_MASS, "",
	 COLUMN_ISOTOPE_ABUNDANCE, "",

	 COLUMN_ATOM_POINTER, atom,
	 
	 COLUMN_ATOM_VISIBLE, TRUE,
	 COLUMN_ATOM_EDITABLE, TRUE,
	 -1);

      /* And now we want to ease the user's work on modifying the fake
	 atom name/symbol values into more useful ones: go to the place
	 where the item was stored in the tree store, and put the cursor
	 on the right row and put the keyboard focus on it also, so that
	 it get highlighted...
      */
      path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				      &tree_iter);
      g_assert (path != NULL);

      /*
	{
	gchar *help = NULL;
	help =  gtk_tree_path_to_string  (path);
	debug_printf (("path = %s\n", help));
	g_free (help);
	}
      */

      gtk_tree_view_expand_to_path (GTK_TREE_VIEW (treeview),
				    path);

      column = gtk_tree_view_get_column (treeview, COLUMN_ATOM_NAME);

      gtk_tree_selection_select_iter (tree_selection, &tree_iter);
  
      gtk_tree_view_set_cursor_on_cell (treeview,
					path,
					column,
					NULL,
					FALSE);
  
      gtk_widget_grab_focus (GTK_WIDGET (treeview));
  
      gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				    path,
				    column,
				    TRUE,
				    0.5,
				    0.5);
  
      gtk_tree_path_free (path);
  
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd,
	 _("Added new atom;"
	   " please, set its data!"), 
	 POLYXMASS_LONG_MSG_TIMEOUT); 
      
      return;
    }
  
  /* There is currently a selection, that means that the user wants to
     add an item. We'll have to decide if it is an isotope or an atom.
  */

  /* Get the path to the selected item, which can be an atom row or an
     isotope row, which are two very different cases that we must
     handle. Specifically, if depth == 1, the selected row is an atom,
     if depth == 2, then it is an isotope.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);
      
  /* Get the depth of the currently selected item, which is 1 if it is
   * an atom item or 2 if it is an isotope item.
   */
  depth = gtk_tree_path_get_depth (path);

  /*
    debug_printf (("depth=%d\n", depth));
  */

  /* Whatever the depth, we can get the index of the atom that is
     concerned, be it selected or one of its isotope children.
  */
  idx_atom = gtk_tree_path_get_indices (path) [0];
    
  /*
    debug_printf (("idx_atom=%d\n", idx_atom));
  */
  
  /* Finally we can free the path.
   */
  gtk_tree_path_free (path);

  /* Now, if the depth == 2, that means that the item that is selected
     is an isotope. Since we only want to add a new atom, we
     understand that the user wants to insert/add an atom object after
     the atom object that is actually the parent of the currently
     selected isotope.
  */
  if (depth == 2)
    {
      result = gtk_tree_model_iter_parent (GTK_TREE_MODEL (model), 
					   &tree_iter_parent, 
					   &tree_iter_sel);
      g_assert (result != FALSE);
      
      gtk_tree_store_insert_after (GTK_TREE_STORE (model),
				   &tree_iter,
				   NULL,
				   &tree_iter_parent);
    }
  else
    {
      gtk_tree_store_insert_after (GTK_TREE_STORE (model),
				   &tree_iter,
				   NULL,
				   &tree_iter_sel);
    }
  
  /* 
     Allocate a new atom so that we can later fill its member data.
  */
  atom = pxmchem_atom_new ();
  
  pxmchem_atom_set_name (atom, _("enter name here"));
  pxmchem_atom_set_symbol (atom, _("enter symbol here"));

  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);
      
  /* Get the index at which we will store the data. This index is the
     one that we will use in order to insert/add the newly allocated
     atom to atomGPA.
  */
  idx_atom = gtk_tree_path_get_indices (path) [0];

  if (idx_atom + 1 >= defctxt->atomGPA->len)
    {
      g_ptr_array_add (defctxt->atomGPA, atom);

      defctxt->is_modified = TRUE;
      polyxdef_ui_atomdef_wnd_update_status_label (defctxt);
    }
  else if (idx_atom + 1 < defctxt->atomGPA->len)
    {
      atom_test = g_ptr_array_insert_val (defctxt->atomGPA, idx_atom, atom);
      g_assert (atom == atom_test);

      defctxt->is_modified = TRUE;
      polyxdef_ui_atomdef_wnd_update_status_label (defctxt);
    }
  else
    {
      g_error (_("%s@%d: failed to insert/add atom item to array\n"),
	     __FILE__, __LINE__);
    }
  
  gtk_tree_path_free (path);

  gtk_tree_store_set 
    ((GtkTreeStore *) model, &tree_iter,

     COLUMN_ATOM_NAME, atom->name,
     COLUMN_ATOM_SYMBOL, atom->symbol,
     COLUMN_ATOM_AVG_MASS, "",
     COLUMN_ISOTOPE_MASS, "",
     COLUMN_ISOTOPE_ABUNDANCE, "",

     COLUMN_ATOM_POINTER, atom,
	 
     COLUMN_ATOM_VISIBLE, TRUE,
     COLUMN_ATOM_EDITABLE, TRUE,
     -1);


  /* And now we want to ease the user's work on modifying the fake
     atom name/symbol values into more useful ones: go to the place
     where the item was stored in the tree store, and put the cursor
     on the right row and put the keyboard focus on it also, so that
     it get highlighted...
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);

  /*
    {
    gchar *help = NULL;
    help =  gtk_tree_path_to_string  (path);
    debug_printf (("path = %s\n", help));
    g_free (help);
    }
  */

  gtk_tree_view_expand_to_path (GTK_TREE_VIEW (treeview),
				path);

  column = gtk_tree_view_get_column (treeview, COLUMN_ATOM_NAME);

  gtk_tree_selection_select_iter (tree_selection, &tree_iter);
  
  gtk_tree_view_set_cursor_on_cell (treeview,
				    path,
				    column,
				    NULL,
				    FALSE);
  
  gtk_widget_grab_focus (GTK_WIDGET (treeview));
  
  gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				path,
				column,
				TRUE,
				0.5,
				0.5);
  
  gtk_tree_path_free (path);
  
  polyxmass_timeoutmsg_message_set ((GtkWindow *) defctxt->definition_wnd,
				    _("Added new atom;"
				      " please, set its data!"), 
				    POLYXMASS_LONG_MSG_TIMEOUT); 
  return;
}


void
polyxdef_ui_atomdef_wnd_add_isotope_button (GtkWidget *widget,
					 gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeViewColumn* column = NULL;
  
  GtkTreeIter tree_iter;
  GtkTreeIter tree_iter_sel;

  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;

  gint idx_atom = -1;
  gint idx_isotope = -1;

  gint depth = -1;
      
  PxmAtom *atom = NULL;

  gdouble *mass = NULL;
  gdouble *abund = NULL;
  gdouble *value_test = NULL;


  
  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->atomGPA != NULL);
  


  /****************** GENERAL PHILOSOPHY **************************/

  /* An atom is made with the following intrinsic members: name,
     symbol and monoisotopic and average mass.
   
     It may also contain any number of isotopes, which are just two
     gdouble values: mono mass and abundance; each value is stored int
     the atom->massGPA and atom->abundGPA GPtrArray of gdouble*
     instances.
    
     If an atom contains isotopes, they are represented in the atom
     treeview as items in a treenode at a depth incremented by one.
  */

  /* That means that depending on the treeview situation, the actions
     that we need to take in response to the clicking onto the ADD
     button are different.
  */

  /* If no selection is made at all, just return.
    
  If a selection is made:
     
  - if the selected item is an atom row, then a new isotope row is
  to be added after the last isotope row of this selected atom
  row.
   
  - if the selected row is an isotope row, then a new isotope row is
  to be added after the isotope row that is currently selected.
  */

  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			     "atom_treeview_model");
  g_assert (model != NULL);

  treeview= g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			       "atom_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);

  /* If there is no current selection, return.
   */
  if (result == FALSE)
    return;

  /* There is currently a selection, that means that the user wants to
     add an item. We'll have to decide if it is an isotope or an atom.
  */

  /* Get the path to the selected item, which can be an atom row or an
     isotope row, which are two very different cases that we must
     handle. Specifically, if depth == 1, the selected row is an atom,
     if depth == 2, then it is an isotope.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);
      
  /* Get the depth of the currently selected item, which is 1 if it is
     an atom item or 2 if it is an isotope item.
  */
  depth = gtk_tree_path_get_depth (path);

  /*
    debug_printf (("depth is %d\n", depth));
  */

  /* Whatever the depth, we can get the index of the atom that is
     concerned, be it selected or one of its isotope children.
  */
  idx_atom = gtk_tree_path_get_indices (path) [0];
  /*
    debug_printf (("idx_atom is %d\n", idx_atom));
  */

  if (depth == 2)
    {
      idx_isotope = gtk_tree_path_get_indices (path) [1];
      /*
	debug_printf (("idx_isotope is %d\n", idx_isotope));
      */
    }
  
  /* Finally we can free the path.
   */
  gtk_tree_path_free (path);

  /* Now, if the depth == 2, that means that the item that is selected
     is an isotope. This means that we just want to insert/add a new
     isotope to the atom at idx_atom.
  */
  if (depth == 1)
    {
      /* The item that is selected is an atom item (at index
	 idx_atom). Since we want to add an isotope item, we must
	 insert an isotope item as the first child of the selected
	 atom item.
      */
      gtk_tree_store_prepend (GTK_TREE_STORE (model),
			      &tree_iter,
			      &tree_iter_sel);
    }
  else if (depth == 2)
    {
      /* An isotope item was selected. All we do is insert the new
	 isotope item after the currently selected one.
      */
      gtk_tree_store_insert_after (GTK_TREE_STORE (model),
				   &tree_iter,
				   NULL,
				   &tree_iter_sel);
    }
  else
    {
      g_error (_("%s@%d: failed to insert/add mass/abund "
		 "pair to atom treeview\n"),
	       __FILE__, __LINE__);
    }
  
  
  /* 
     Allocate new isotope's data (mono mass and abundance) so that we
     can later fill its member data.
  */
  /* Get the atom !
   */
  atom = g_ptr_array_index (defctxt->atomGPA, idx_atom);
  /*
    debug_printf (("atom is %s\n", atom->name));
  */

  mass = g_malloc0 (sizeof (gboolean));
  abund = g_malloc0 (sizeof (gboolean));
 
  /* But where do we have to put that stuff in the atom->massGPA and
     atom->abundGPA ? If an isotope was selected (depth == 2) we had
     determined the idx_isotope of the selected isotope. Otherwise
     idx_isotope is -1. Make sure both abundGPA and massGPA are of the
     same size, which is a sanity check.
  */
  g_assert (atom->massGPA->len == atom->abundGPA->len);

  if (depth == 1)
    {
      /* The selected item was an atom at index idx_atom, so we want
	 to insert at index 0 our new items.
      */
      /*
	debug_printf (("Insert the isotope's mass at index 0\n"));
      */

      value_test = 
	g_ptr_array_insert_val (atom->massGPA, 0, mass);
      g_assert (mass == value_test);
      
      defctxt->is_modified = TRUE;
      polyxdef_ui_atomdef_wnd_update_status_label (defctxt);

      /*
	debug_printf (("Insert the isotope's abundance at index 0\n"));
      */

      value_test = 
	g_ptr_array_insert_val (atom->abundGPA, 0, abund);
      g_assert (abund == value_test);
      
      defctxt->is_modified = TRUE;
      polyxdef_ui_atomdef_wnd_update_status_label (defctxt);
    }
  else /* depth == 2 */
    {
      /* 
	 We actually know that an isotope element was selected, at
	 index idx_isotope... If the selected item was the last one,
	 we append the new one otherwise we insert the new one at the
	 index of the currently selected one +1.
      */
      if (idx_isotope + 1 >= atom->abundGPA->len)
	{
	  /*
	    debug_printf (("Add the isotope's mass at end of array\n"));
	  */

	  g_ptr_array_add (atom->massGPA, mass);
	  defctxt->is_modified = TRUE;
	  polyxdef_ui_atomdef_wnd_update_status_label (defctxt);

	  g_ptr_array_add (atom->abundGPA, abund);	  
	  defctxt->is_modified = TRUE;
	  polyxdef_ui_atomdef_wnd_update_status_label (defctxt);
	}
      else if (idx_isotope + 1 < atom->abundGPA->len)
	{
	  /*
	    debug_printf (("Insert the isotope's abundance at index %d\n",
	    idx_isotope + 1));
	  */

	  value_test = 
	    g_ptr_array_insert_val (atom->massGPA, idx_isotope + 1, mass);
	  g_assert (mass == value_test);
	  
	  defctxt->is_modified = TRUE;
	  polyxdef_ui_atomdef_wnd_update_status_label (defctxt);

	  value_test = 
	    g_ptr_array_insert_val (atom->abundGPA, idx_isotope + 1, abund);
	  g_assert (abund == value_test);
	  
	  defctxt->is_modified = TRUE;
	  polyxdef_ui_atomdef_wnd_update_status_label (defctxt);
	}
      else
	{
	  g_error (_("%s@%d: failed to insert/add mass/abund "
		   "pair to array\n"),
		 __FILE__, __LINE__);
	}
    }
  
  /* We set fake 0.0000000 data for mass and abund in isotope item.
   */
  gtk_tree_store_set 
    ((GtkTreeStore *) model, &tree_iter,
     
     COLUMN_ATOM_NAME, "",
     COLUMN_ATOM_SYMBOL, "",
     COLUMN_ATOM_AVG_MASS, "",
     COLUMN_ISOTOPE_MASS, "0.0000000",
     COLUMN_ISOTOPE_ABUNDANCE, "0.0000000",
     
     COLUMN_ATOM_POINTER, atom,
     
     COLUMN_ATOM_VISIBLE, TRUE,
     COLUMN_ATOM_EDITABLE, TRUE,
     -1);
  
  /* And now we want to ease the user's work on modifying the fake
     mass/abundance values into more useful ones: go to the place
     where the item was stored in the tree store, and put the cursor
     on the right row and put the keyboard focus on it also, so that
     it get highlighted...
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);

  /*
    {
    gchar *help = NULL;
    help =  gtk_tree_path_to_string  (path);
    debug_printf (("path = %s\n", help));
    g_free (help);
    }
  */

  gtk_tree_view_expand_to_path (GTK_TREE_VIEW (treeview),
				path);

  column = gtk_tree_view_get_column (treeview, COLUMN_ISOTOPE_MASS);

  gtk_tree_selection_select_iter (tree_selection, &tree_iter);
  
  gtk_tree_view_set_cursor_on_cell (treeview,
				    path,
				    column,
				    NULL,
				    FALSE);
  
  gtk_widget_grab_focus (GTK_WIDGET (treeview));
  
  gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				path,
				column,
				TRUE,
				0.5,
				0.5);
  
  gtk_tree_path_free (path);
  
  /* We now have to update the average mass of the atom !!!
   */
  polyxdef_ui_atomdef_wnd_atom_avg_mass_cell_update (defctxt, idx_atom);

  polyxmass_timeoutmsg_message_set ((GtkWindow *) defctxt->definition_wnd,
				    _("Added new isotope;"
				      " please, set its data!"), 
				    POLYXMASS_LONG_MSG_TIMEOUT); 
  return;
}

void
polyxdef_ui_atomdef_wnd_remove_button (GtkWidget *widget,
					 gpointer data)
{
  PxmDefCtxt *defctxt = data;

  /* We only want to remove something if that something is actually
     selected.
  */
  GtkTreeView *treeview = NULL;
  GtkTreeModel *model = NULL;
  
  //  GtkTreeIter tree_iter;
  GtkTreeIter tree_iter_sel;

  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;

  gint idx_atom = -1;
  gint idx_isotope = -1;

  gint depth = -1;
      
  PxmAtom *atom = NULL;

  gdouble *mass = NULL;
  gdouble *abund = NULL;


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->atomGPA != NULL);
 


  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			     "atom_treeview_model");
  g_assert (model != NULL);
  
  treeview= g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			       "atom_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);

  /* If there is no current selection, return.
   */
  if (result == FALSE)
    return;

  /* There is currently a selection, that means that the user wants to
     remove an item. We'll have to decide if it is an isotope or an
     atom.
  */


  /* Get the path to the selected item, which can be an atom row or an
     isotope row, which are two very different cases that we must
     handle. Specifically, if depth == 1, the selected row is an atom,
     if depth == 2, then it is an isotope.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);
      
  /* Get the depth of the currently selected item, which is 1 if it is
   * an atom item or 2 if it is an isotope item.
   */
  depth = gtk_tree_path_get_depth (path);
  /*
    debug_printf (("depth is %d\n", depth));
  */

  /* Whatever the depth, we can get the index of the atom that is
     concerned, be it selected or one of its isotope children.
  */
  idx_atom = gtk_tree_path_get_indices (path) [0];
  /*
    debug_printf (("idx_atom is %d\n", idx_atom));
  */

  if (depth == 2)
    {
      idx_isotope = gtk_tree_path_get_indices (path) [1];
      /*
	debug_printf (("idx_isotope is %d\n", idx_isotope));
      */

      /* Remove this isotope !
       */
      atom = g_ptr_array_index (defctxt->atomGPA, idx_atom);
      g_assert (atom != NULL);
      
      mass = g_ptr_array_remove_index (atom->massGPA, idx_isotope);
      g_assert (mass != NULL);
      g_free (mass);

      abund = g_ptr_array_remove_index (atom->abundGPA, idx_isotope);
      g_assert (abund != NULL);
      g_free (abund);

      defctxt->is_modified = TRUE;
      polyxdef_ui_atomdef_wnd_update_status_label (defctxt);

      gtk_tree_store_remove (GTK_TREE_STORE (model),
			     &tree_iter_sel);

      /* We now have to update the average mass of the atom !!!
       */
      polyxdef_ui_atomdef_wnd_atom_avg_mass_cell_update (defctxt, idx_atom);
      

    }
  else if (depth == 1)
    {
      /* Remove this atom !
       */
      atom = g_ptr_array_remove_index (defctxt->atomGPA, idx_atom);
      g_assert (atom != NULL);
      pxmchem_atom_free (atom);

      defctxt->is_modified = TRUE;
      polyxdef_ui_atomdef_wnd_update_status_label (defctxt);

      gtk_tree_store_remove (GTK_TREE_STORE (model),
			     &tree_iter_sel);

      /* Note that we do not update the atom average mass cell because
	 what if we had deleted the last atom item ? It would crash
	 trying to find an atom at whatever index in the array !!!
      */
    }
  else
    {
      g_error (_("%s@%d: failed to remove item from treeview\n"),
	       __FILE__, __LINE__);
    }

  /* Finally we can free the path.
   */
  gtk_tree_path_free (path);

  polyxmass_timeoutmsg_message_set ((GtkWindow *) defctxt->definition_wnd,
				    _("Successfully removed item"), 
				    POLYXMASS_NORM_MSG_TIMEOUT); 
  return;
}



/*
  ATOM DEFINITION SAVING FUNCTIONS
*/
void
polyxdef_ui_atomdef_wnd_save_button (GtkWidget *widget,
				     gpointer data)
{
  PxmDefCtxt *defctxt = data;

  gint result = -1;

  gboolean valid_result = TRUE;

  gchar *xml = NULL;
  gchar *valid = NULL;

  FILE *filep = NULL;
 

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->atomGPA != NULL);
 
  /* We do not save anything if the array of atoms does contain 
     bogus data.
  */
  valid_result = pxmchem_atom_array_validate_all (defctxt->atomGPA, &valid);

  if (valid_result == FALSE)
    {
      /* There was at least one error, which means that the string in
	 valid has been allocated. We should show that string and
	 later free it.
      */
      g_assert (valid != NULL);
      
      if (defctxt->syntax_check_wnd != NULL)
	{
	  polyxdef_ui_syntaxcheck_wnd_reset (defctxt);
	}
      else
	{
	  /* We have to setup the window first.
	   */
	  defctxt->syntax_check_wnd = 
	    polyxdef_ui_syntaxcheck_wnd_setup_window (defctxt, NULL);
	  
	  if (defctxt->syntax_check_wnd == NULL)
	    {
	      g_message (_("%s@%d: failed to set up the syntax-checking "
		       "window for the following errors: '%s'\n\n"),
		     __FILE__, __LINE__, valid);
	      
	      g_free (valid);
	      
	      return;
	    }
	}

      /* We now have an empty textview where to set the errors' string.
       */
      polyxdef_ui_syntaxcheck_wnd_set_errors (defctxt, valid);

      g_free (valid);
      
      return;
    }
  else
    {
      /* Great, the validation succeeded, just inform the user.
       */
      g_message (_("%s@%d: the validation succeeded\n"),
	     __FILE__, __LINE__);
      
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 _("The validation succeeded"), 
	 POLYXMASS_NORM_MSG_TIMEOUT);
    }
  
  /* 
     Now we have to be sure that we have the filename of the atom
     definition file.
  */
  if (NULL == defctxt->filename)
    {
      /* 
	 The user should be made available a file selection window
	 so we'd better use the saveas button function!
      */
      return polyxdef_ui_atomdef_wnd_saveas_button (NULL, defctxt);
    }

  /* At this point we know that the data were from a file, so we just
     save those data in that same file.
  */

  /* First of all make sure we can open the file to write to it.
   */
  filep = fopen (defctxt->filename, "w");

  if (filep == NULL)
    {
      g_critical (_("%s@%d: failed to open file: '%s'\n"),
	     __FILE__, __LINE__, defctxt->filename);
      
      return;
    }
  
  /* Construct a string with all the xml-formatted data pertaining to
     the deftcxt->atomGPA.
  */
  xml = 
    pxmchem_atom_format_xml_string_atomdefdata (defctxt->atomGPA, "  ", 0);
  g_assert (xml != NULL);
  
  /* Copy the xml data string to the file 'defctxt->filename'.
   */
  result = fputs (xml, filep);
  
  fclose (filep);
  
  g_free (xml);
  
  if (result == EOF || result < 0)
    {
      g_critical (_("%s@%d: failed to save atom array to file: '%s'\n"),
	     __FILE__, __LINE__, defctxt->filename);
      
      return;
    }

  /* 
     Now we can tell that the file is unmodified. That goes to an
     allocated gboolean variable that is set as a datum to the window.
  */
  defctxt->is_modified = FALSE;
  polyxdef_ui_atomdef_wnd_update_status_label (defctxt);
  
  return;
}


void
polyxdef_ui_atomdef_wnd_saveas_button (GtkWidget *widget,
				     gpointer data)
{
  PxmDefCtxt *defctxt = data;

  gint result = -1;
  
  gboolean valid_result = FALSE;

  GtkWidget *dialog = NULL;

  gchar *xml = NULL;
  gchar *filename = NULL;
  gchar *help = NULL;
  gchar *valid = NULL;

  FILE *filep = NULL;
 

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->atomGPA != NULL);
  
  /* We do not save anything if the array of atoms does contain 
     bogus data.
  */
  valid_result = pxmchem_atom_array_validate_all (defctxt->atomGPA, &valid);

  if (valid_result == FALSE)
    {
      /* There was at least one error, which means that the string in
	 valid has been allocated. We should show that string and
	 later free it.
      */
      g_assert (valid != NULL);
      
      if (defctxt->syntax_check_wnd != NULL)
	{
	  polyxdef_ui_syntaxcheck_wnd_reset (defctxt);
	}
      else
	{
	  /* We have to setup the window first.
	   */
	  defctxt->syntax_check_wnd = 
	    polyxdef_ui_syntaxcheck_wnd_setup_window (defctxt, NULL);
	  
	  if (defctxt->syntax_check_wnd == NULL)
	    {
	      g_message (_("%s@%d: failed to validate the atom "
		       "definition with the following errors:\n\n %s.\n"),
		     __FILE__, __LINE__, valid);
	      
	      g_free (valid);
	      
	      return;
	    }
	}

      /* We now have an empty textview where to set the errors' string.
       */
      polyxdef_ui_syntaxcheck_wnd_set_errors (defctxt, valid);

      g_free (valid);
      
      return;
    }
  else
    {
      /* Great, the validation succeeded, just inform the user.
       */
      g_message (_("%s@%d: the validation succeeded\n"),
	     __FILE__, __LINE__);
      
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 _("The validation succeeded"), 
	 POLYXMASS_NORM_MSG_TIMEOUT);
    }
  

  dialog = 
    gtk_file_chooser_dialog_new (_("Save Atom Definition File"),
				 GTK_WINDOW (defctxt->definition_wnd),
				 GTK_FILE_CHOOSER_ACTION_SAVE,
				 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				 GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
				 NULL);
  
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      
      if (TRUE == g_file_test (filename, G_FILE_TEST_EXISTS))
	{
	  g_critical (_("%s@%d: file does exist already: '%s'\n"),
		 __FILE__, __LINE__, filename);
	  
	  g_free (filename);
	  
	  /* We can destroy the file chooser dialog.
	   */
	  gtk_widget_destroy (dialog);

	  return;
	}
    }
  else
    {
      g_free (filename);
      
      /* We can destroy the file chooser dialog.
       */
      gtk_widget_destroy (dialog);
      
      return;
    }
  
  /* Ok, we have a file to which the atom definition saving can be performed.
   */

  /* We can destroy the file chooser dialog.
   */
  gtk_widget_destroy (dialog);
  
  /* First of all make sure we can open the file to write to it.
   */
  filep = fopen (filename, "w");
  
  if (filep == NULL)
    {
      g_critical (_("%s@%d: failed to open file: '%s'\n"),
	     __FILE__, __LINE__, filename);
      
      g_free (filename);
      
      return;
    }

  /* Construct a string with all the xml-formatted data pertaining to
     the defctxt->atomGPA.
  */
  xml = 
    pxmchem_atom_format_xml_string_atomdefdata (defctxt->atomGPA, "  ", 0);
  g_assert (xml != NULL);
  
  /* Copy the xml data string to the file 'filename'.
   */
  result = fputs (xml, filep);
  
  fclose (filep);
  
  g_free (xml);
  
  if (result == EOF || result < 0)
    {
      g_critical (_("%s@%d: failed to save atom array to file: '%s'\n"),
	     __FILE__, __LINE__, filename);
      
      return;
    }

  /* Since the filename has changed, we set the new value. Thus we do not
     free the 'filename' initially returned by the chooser.
  */
  defctxt->filename = filename;
  
  /* The file is no more modified !
   */
  defctxt->is_modified = FALSE;
  polyxdef_ui_atomdef_wnd_update_status_label (defctxt);
  
  /* Now we should change the window title so that it reads the new
     filename.
  */
  help = g_strdup_printf (_("Atom Definition: %s"), defctxt->filename);
  gtk_window_set_title (GTK_WINDOW (defctxt->definition_wnd), help);
  g_free (help);

  return;
}



/* HELPERS
 */
void
polyxdef_ui_atomdef_wnd_update_status_label (PxmDefCtxt *defctxt)
{
  GtkWidget *widget = NULL;
  
  g_assert (defctxt->definition_wnd != NULL);
  

  widget = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			      "atomdef_status_label");
  g_assert (widget != NULL);
  

  gtk_label_set_text (GTK_LABEL (widget), 
		      defctxt->is_modified == TRUE ? 
		      _("Modified") : _("Unmodified"));

  return;
}


void
polyxdef_ui_atomdef_wnd_atom_avg_mass_cell_update (PxmDefCtxt *defctxt, 
						   gint idx)
{
  PxmAtom *atom = NULL;

  GtkTreeView *treeview = NULL;
  GtkTreeIter tree_iter;
  GtkTreeModel *model = NULL;
  GtkTreePath* path = NULL;
  
  gchar *help = NULL;
  gchar *avg_mass = NULL;

  
  /* An isotope mass or an isotope abundance have been changed. We
     thus have to recalculate the average mass of the atom. Next, we have
     to display that new value in the correct cell of the atoms treeview.
  */
  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->atomGPA != NULL);


  treeview = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
				"atom_treeview");
  g_assert (treeview != NULL);
  
  model =  g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			      "atom_treeview_model");
  g_assert (model != NULL);

  /*
    debug_printf (("the index of the atom item is %d\n",
    idx));
    debug_printf (("the length of the atom array is %d\n",
    defctxt->atomGPA->len));
  */

  g_assert (idx < defctxt->atomGPA->len);
  if (idx < 0)
    idx = 0;
  
  atom = g_ptr_array_index (defctxt->atomGPA, idx);
  g_assert (atom != NULL);
  
  /* We now have the column of the average mass for the atom. Get the
     path to the cell of interest, we have the index of the atom in
     the idx parameter.
  */
  path = gtk_tree_path_new_from_indices (idx, -1);
  
  gtk_tree_model_get_iter (model, &tree_iter, path);

  /* Time has come to make the calculation of the average mass...
   */

  if (FALSE == pxmchem_atom_calc_avg_in_atom (atom))
    {
      help = g_strdup_printf (_("Failed to calculate average mass"));
      
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 help, 
	 POLYXMASS_MEDI_MSG_TIMEOUT);
      
      g_free (help);
      gtk_tree_path_free (path);
      
      return ;
    }
  
  /* We now can update the cell. Convert the new average mass value
     into a string...
  */
  avg_mass = libpolyxmass_globals_dtoa (atom->avg, 10);

  gtk_tree_store_set (GTK_TREE_STORE (model), 
		      &tree_iter, 
		      COLUMN_ATOM_AVG_MASS, (gchar *) avg_mass,
		      -1);

  g_free (avg_mass);
  
  

  defctxt->is_modified = TRUE;
  polyxdef_ui_atomdef_wnd_update_status_label (defctxt);

  return;
}





/* WINDOW CLOSING FUNCTIONS
 */
gboolean
polyxdef_ui_atomdef_wnd_delete_event (GtkWidget *window,
				      GdkEvent *event,
				      gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkWidget *dialog = NULL;
  
  gint dialog_result = 0;
  
  gchar *help = NULL;
    


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->atomGPA != NULL);


  /* We only destroy the window if its corresponding atomdef
     instance is not modified.
  */
  if (FALSE == defctxt->is_modified)
    {
      /* Remove the window from the array of polyxdef contexts.
       */
      if (FALSE == g_ptr_array_remove (polyxdef_defctxtGPA, 
				       defctxt))
	{
	  g_critical (_("%s@%d: failed to remove context from array.\n"),
		 __FILE__, __LINE__);
	}

      polyxdef_defctxt_free (defctxt);  

      return TRUE;
    }

  /* The polyxdef context associated with this widget is modified, we
   * should ask if it should be saved before.
   */
  help = g_strdup_printf (_("The atom definition is modified, save it?\n"));
  
  dialog = gtk_message_dialog_new (GTK_WINDOW (defctxt->definition_wnd),
				   GTK_DIALOG_DESTROY_WITH_PARENT,
				   GTK_MESSAGE_QUESTION,
				   GTK_BUTTONS_YES_NO,
				   help);
  g_free (help);
  
  /* Add the CANCEL button, so that the user can simply cancel 
   * the operation.
   */
  gtk_dialog_add_button (GTK_DIALOG (dialog),
			 GTK_STOCK_CANCEL,
			 GTK_RESPONSE_CANCEL);

  dialog_result = gtk_dialog_run (GTK_DIALOG (dialog));

  switch (dialog_result)
    {
    case GTK_RESPONSE_YES:

      /* First destroy the message window.
       */
      gtk_widget_destroy (dialog);

      /* Next ask for a save of the atom definition.
       */
      polyxdef_ui_atomdef_wnd_save_button (NULL, defctxt);
      
      if (FALSE == defctxt->is_modified) 
	{
	  /* File was successfully saved.
	   */
	  /* Remove the window from the array of opened windows.
	   */
	  if (FALSE == g_ptr_array_remove (polyxdef_defctxtGPA, 
					   defctxt))
	    {
	      g_critical (_("%s@%d: failed to remove context from array.\n"),
		     __FILE__, __LINE__);
	    }
	  
	  polyxdef_defctxt_free (defctxt);  
	  
	  return TRUE;
	}
      
    case GTK_RESPONSE_NO:
      
      /* First destroy the message window.
       */
      gtk_widget_destroy (dialog);
      
      /* Remove the window from the array of opened windows.
       */
      if (FALSE == g_ptr_array_remove (polyxdef_defctxtGPA, 
				       defctxt))
	{
	  g_critical (_("%s@%d: failed to remove context from array.\n"),
		 __FILE__, __LINE__);
	}
      
      polyxdef_defctxt_free (defctxt);  
      
      return TRUE;
      
      
    case GTK_RESPONSE_CANCEL:
      
      /* First destroy the message window.
       */
      gtk_widget_destroy (dialog);
      
      return TRUE;
      
    default: 
      
      /* First destroy the message window.
       */
      gtk_widget_destroy (dialog);
      
      break;
    }

 return TRUE;
}


gboolean
polyxdef_ui_atomdef_wnd_destroy_event (GtkWidget *window,
				       GdkEvent *event,
				       gpointer data)
{
  return FALSE;
}

