/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxmass-ui-sequence-purification.h"
#include "polyxedit-ui-seqed-wnd.h"
#include "polyxedit-ui-masses-display-wnd.h"



GtkWidget *
polyxmass_sequence_purification_wnd_setup (GtkWidget *seqed_widget,
						   gchar *seq,
						   GPtrArray *errorsGPA)
{
  GtkWidget *widget = NULL;
  GtkWidget *window = NULL;

  GtkWidget *textview = NULL;
  GtkTextBuffer *buffer = NULL;
  GtkTextTag* error_tag = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;

  gint *index = NULL;
  
  
  /* The allocated gint objects and the errorsGPA array must be freed
     when no longer in use. The seq gchar string must be freed when no
     longer in use.
  */

  g_assert (seqed_widget != NULL);

  g_assert (seq != NULL);

  g_assert (errorsGPA != NULL);
  
  g_assert (PXM_SEQED_WIDGET (seqed_widget)->polymer != NULL);
  

  gui_file = 
    g_strdup_printf ("%s/polyxmass-sequence-purification.glade", userspec->gladedir);

  xml = glade_xml_new (gui_file, "sequence_purification_wnd", 
		       PACKAGE);
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      g_free (gui_file);
      
      return NULL;
    }
  
  g_free (gui_file);
  
  window = glade_xml_get_widget (xml, "sequence_purification_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to create the sequence purification window\n"),
		  __FILE__, __LINE__);
      
      g_object_unref (G_OBJECT (xml));
      
      return NULL;
    }
  

  g_object_set_data (G_OBJECT (window), "seqed_widget", seqed_widget);
  g_object_set_data (G_OBJECT (window), "errorsGPA", errorsGPA);
  g_object_set_data (G_OBJECT (window), "seq", seq);
  


  widget = glade_xml_get_widget (xml, "messages_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "messages_entry", widget);


  
  textview = glade_xml_get_widget (xml, "purified_sequence_textview"); 
  g_assert (textview != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "purified_sequence_textview", textview);

  /* We now should set up the seq string in textview. Specifically we
     should format the string so that each character located at any
     index present in the errorsGPA array is in bold and in red
     colour.
   */

  /* Get the buffer associated with the textview.
   */
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));

  /* We will format the erroneous positions in the sequence text in
     red. To do that we will have to create a new tag, that will belong
     to the buffer. We will reference later to tha tag using its name
     "error_tag".
  */
  error_tag = gtk_text_buffer_create_tag (buffer,
					  "error_tag",
					  "foreground", "red",
					  NULL);
  if (FALSE ==
      polyxmass_sequence_purification_wnd_sequence_format (window))
    {
      g_critical (_("%s@%d: failed to format the string: '%s' in the "
		    "textview.\n"),
		  __FILE__, __LINE__, seq);
      
      while (errorsGPA->len > 0)
	{
	  index = g_ptr_array_remove_index (errorsGPA, 0);
	  g_assert (index != NULL);
	  g_free (index);
	}
      
      g_ptr_array_free (errorsGPA, TRUE);
      
      g_free (seq);
      
      g_object_unref (G_OBJECT (xml));
      
      return NULL;
    }
  

  widget = glade_xml_get_widget (xml, "digit_checkbutton"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "digit_checkbutton", widget);

  widget = glade_xml_get_widget (xml, "punctuation_checkbutton"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "punctuation_checkbutton", widget);

  widget = glade_xml_get_widget (xml, "space_checkbutton"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "space_checkbutton", widget);

  widget = glade_xml_get_widget (xml, "remove_other_characters_entry"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "remove_other_characters_entry", widget);



  widget = glade_xml_get_widget (xml, "remove_all_tagged_chars_button"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "remove_all_tagged_chars_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxmass_sequence_purification_wnd_remove_all_tagged_chars_button), 
     window);


  widget = glade_xml_get_widget (xml, "purify_sequence_button"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "purify_sequence_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxmass_sequence_purification_wnd_purify_sequence_button), 
     window);


  widget = glade_xml_get_widget (xml, "check_sequence_button"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "check_sequence_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxmass_sequence_purification_wnd_check_sequence_button), 
     window);


  widget = glade_xml_get_widget (xml, "insert_sequence_button"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "insert_sequence_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxmass_sequence_purification_wnd_insert_sequence_at_point_button), 
     window);



  widget = glade_xml_get_widget (xml, "sequence_purification_cancel_button"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "sequence_purification_cancel_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxmass_sequence_purification_wnd_cancel_button), 
     window);

  widget = glade_xml_get_widget (xml, "sequence_purification_revert_button"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "sequence_purification_revert_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxmass_sequence_purification_wnd_revert_button), 
     window);



  g_signal_connect (G_OBJECT (window),
		      "delete_event",
		    G_CALLBACK 
		      (polyxmass_sequence_purification_wnd_delete_event), 
		      seqed_widget);

  g_signal_connect (G_OBJECT (window),
		      "delete_event",
		    G_CALLBACK 
		      (polyxmass_sequence_purification_wnd_destroy_event), 
		      seqed_widget);


  /* Finally we can show the whole sequence editor.
   */
  gtk_widget_show_all (window);


  /* We have finished setting up the window, and so also using
   * the xml data, unref them
   */
  g_object_unref (G_OBJECT (xml));

  /* Set this window pointer as a full datum to the sequence editing
     widget (seqed_widget), so that when its containing window is
     closed, this window is closed also.

     There might be more than one window of the present kind opened
     for a given polymer seqence editing widget, and we do not want
     that the second window destroys the datum name of the first
     window, so we create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("sequence-purification-wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (seqed_widget),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxmass_sequence_purification_wnd_really_close);

  return window;
}


gboolean
polyxmass_sequence_purification_wnd_sequence_format (GtkWidget *widget)
{
  GtkWidget *window = widget ;
  
  GtkTextView *textview = NULL;
  
  GtkTextBuffer *buffer = NULL;

  GtkTextIter start_iter;
  GtkTextIter end_iter;

  gint *index = NULL;
  gint iter = 0;
  
  gchar *seq = NULL;
  
  GPtrArray *errorsGPA = NULL;
  

  g_assert (window != NULL);

  /* Get all the pointers we are going to need from the window.
   */
  seq = g_object_get_data (G_OBJECT (window), "seq");
  g_assert (seq != NULL);

  
  errorsGPA = g_object_get_data (G_OBJECT (window), "errorsGPA");
  g_assert (errorsGPA != NULL);
  
  textview = g_object_get_data (G_OBJECT (window), 
				"purified_sequence_textview");
  g_assert (textview != NULL);
  

  /* OK, start the operations. Note that this function might be called
     by the function that handles the "revert" button, where the user
     asks that the textview be set as new, that is as it was before
     any changes were made to it. So we ought to first empty the
     textview here, even if it might be empty already, as this
     function is also called by the window setting up function.
   */
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));
  
  /* Empty the textview first.
   */
  gtk_text_buffer_get_start_iter (buffer, &start_iter);
  gtk_text_buffer_get_end_iter (buffer, &end_iter);

  gtk_text_buffer_delete (buffer, &start_iter, &end_iter);

  /* And now we can start afresh : we have to set the seq string to
     the textview.
   */

  /* Get end of buffer where we'll insert (that is append actually) 
     the seq string.
  */
  gtk_text_buffer_get_end_iter (buffer, &end_iter);
  
  gtk_text_buffer_insert (buffer, &end_iter, seq, -1);
  
  /* We now have to iterate in the array of errors and for each index
     iterated set the corresponding character to red.
  */
  for (iter = 0; iter < errorsGPA->len ; iter++)
    {
      index = g_ptr_array_index (errorsGPA, iter);
      g_assert (index != NULL);
      
      gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, *index);
      gtk_text_buffer_get_iter_at_offset (buffer, &end_iter, *index + 1);
      
      gtk_text_buffer_apply_tag_by_name (buffer,
					 "error_tag",
					 &start_iter,
					 &end_iter);
      
    }
  
  /* At this point all the positions in the seq string that were found
     erroneous are formatted so that they are easily visible.
  */

  /* And force the focus to the textview, so that all the characters
     that should be "error_tag"'ged are effectively tagged (is this a
     bug of GtkTextView) ?
  */
  gtk_widget_grab_focus (GTK_WIDGET (textview));


  return TRUE;
}


void
polyxmass_sequence_purification_wnd_remove_all_tagged_chars_button (GtkWidget *
							     widget,
							     gpointer data)
{
  GtkWidget *window = data;

  GtkTextView *textview = NULL;
  GtkTextBuffer *buffer = NULL;

  GtkTextIter start_iter;
  GtkTextIter end_iter;
  GtkTextIter moving_iter;

  GtkTextTagTable *table = NULL;
  GtkTextTag *error_tag = NULL;
  
  g_assert (window != NULL);

  textview = g_object_get_data (G_OBJECT (window), 
				"purified_sequence_textview");
  g_assert (textview != NULL);



  /* We are asked to iterate in the sequence buffer contained in the
     textview and remove from that sequence to remove all the
     characters that are currently tagged with the "error_tag"
     formatting tag. That is the user acknowledges that there are a
     number of erroneous characters in the sequence, and that these
     characters should be removed.
   */
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));

  gtk_text_buffer_get_start_iter (buffer, &start_iter);
  gtk_text_buffer_get_end_iter (buffer, &end_iter);

  table = gtk_text_buffer_get_tag_table (buffer);
  g_assert (table != NULL);
  
  error_tag = gtk_text_tag_table_lookup (table, "error_tag");
  g_assert (error_tag != NULL);
  
  /* Get an iterator to the first character in the buffer !
   */
  gtk_text_buffer_get_start_iter (buffer, &start_iter);
  moving_iter = start_iter;
  
  while (TRUE)
    {
      if (TRUE == gtk_text_iter_has_tag (&start_iter, error_tag))
	{
	  /* The first character in the sequence is tagged, so we have to
	     remove it. But we need an "end_iter" that points to the
	     forward characteur so that we can call the delete function.
	  */
	  moving_iter = start_iter;
	  gtk_text_iter_forward_char (&moving_iter);
	  
	  /* From Gtk+2 doc: the start and end will be re-initialized
	     to point to the location where text was deleted.
	  */
	  gtk_text_buffer_delete (buffer, &start_iter, &moving_iter);
	}
      else
	{
	  if (FALSE == gtk_text_iter_forward_char (&start_iter))
	    break;
	  else
	    continue;
	}      
    }
    
  return ;
}


void
polyxmass_sequence_purification_wnd_purify_sequence_button (GtkWidget *
							     widget,
							     gpointer data)
{
  GtkWidget *window = data;

  GtkWidget *entry = NULL;
  GtkWidget *checkbutton = NULL;
    
  GtkTextView *textview = NULL;
  GtkTextBuffer *buffer = NULL;
  gchar *bufferchars = NULL;

  gchar *remove_chars_pattern = NULL;
  
  gboolean remove_digit_chars = FALSE;
  gboolean remove_space_chars = FALSE;
  gboolean remove_punctuation_chars = FALSE;
  
  GtkTextIter start_iter;
  GtkTextIter end_iter;

  gsize size = 0;
  
  gint iter = 0;
  gint bufferchars_length =0;
  
  GPtrArray *errorsGPA = NULL;


  g_assert (window != NULL);


  /* Get all the pointers we are going to need from the window.
   */
  errorsGPA = g_object_get_data (G_OBJECT (window), "errorsGPA");
  g_assert (errorsGPA != NULL);
  
  textview = g_object_get_data (G_OBJECT (window), 
				"purified_sequence_textview");
  g_assert (textview != NULL);


  /* We are asked to purify the sequence that is displayed in the
     textview. The criteria that govern the purification process are
     set by the user in the form of three checkbutton widgets and a
     text entry widget.

     The checkbutton widgets tell if digits (0--9) and/or spaces
     (space, tab, newline...) and/or punctuation (:;,...) must be
     removed from the text.

     The text entry widget lists more characters that should be
     removed from the sequence text.
  */

  /* We thus have to iterate in sequence text and for each character
     we have to check the requested purification criteria. Each time a
     character in the sequence text happens to correspond to the
     purification criteria, then it should be removed.
  */

  /* OK, start the operations.
   */
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));

  gtk_text_buffer_get_start_iter (buffer, &start_iter);
  gtk_text_buffer_get_end_iter (buffer, &end_iter);


  /* Free this allocated char string.
   */
  bufferchars = gtk_text_buffer_get_text (buffer,
					  &start_iter,
					  &end_iter,
					  /*include_hidden_chars*/ FALSE);
  g_assert (bufferchars != NULL);

  
  /* Now we will iterate in the string and check its individual 
     characters. But first we have to see what's required by the user
     to be removed from that string!
  */
  checkbutton = g_object_get_data (G_OBJECT (window), 
				   "digit_checkbutton"); 
  g_assert (checkbutton != NULL);
  remove_digit_chars = 
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton));
  


  checkbutton = g_object_get_data (G_OBJECT (window), 
				   "punctuation_checkbutton"); 
  g_assert (checkbutton != NULL);
  remove_punctuation_chars = 
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton));
  


  checkbutton = g_object_get_data (G_OBJECT (window), 
				   "space_checkbutton"); 
  g_assert (checkbutton != NULL);
  remove_space_chars = 
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton));
  


  entry = g_object_get_data (G_OBJECT (window), 
			     "remove_other_characters_entry");
  g_assert (entry != NULL);
  remove_chars_pattern = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  /* OK, we are now set for the iteration in the sequence string.
   */
  size = strlen (bufferchars);
  g_assert (size < G_MAXINT) ;
  bufferchars_length = (gint) size;
  
  /* We process the string from its last character up to its first
     character, so that if we have to remove invalid characters, the
     indices between the string's characters and the actual buffer
     contained in the textview won't change. Do not forget the
     bufferchars is an allocated string that is nothing but a snapshot
     of the textview's buffer data at the beginning of the
     process. When we remove characters from the textview's buffer
     text, we do not remove characters from bufferchars, which is why
     to maintain the colinearity between indices, we start working
     from the end of the string.
  */
  for (iter = bufferchars_length - 1 ; iter >= 0; iter--)
    {
      /* We now can check the character for all what's been asked by
	 the user.
       */
      if (TRUE == remove_digit_chars 
	  && TRUE == g_ascii_isdigit (bufferchars [iter]))
	{
	  gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, iter);
	  gtk_text_buffer_get_iter_at_offset (buffer, &end_iter, iter + 1);
	  
	  gtk_text_buffer_delete (buffer, &start_iter, &end_iter);
	  
	  continue;
	}
      
      if (TRUE == remove_space_chars 
	  && TRUE == g_ascii_isspace (bufferchars [iter]))
	{
	  gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, iter);
	  gtk_text_buffer_get_iter_at_offset (buffer, &end_iter, iter + 1);
	  
	  gtk_text_buffer_delete (buffer, &start_iter, &end_iter);
	  
	  continue;
	}
      
      if (TRUE == remove_punctuation_chars 
	  && TRUE == g_ascii_ispunct (bufferchars [iter]))
	{
	  gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, iter);
	  gtk_text_buffer_get_iter_at_offset (buffer, &end_iter, iter + 1);
	  
	  gtk_text_buffer_delete (buffer, &start_iter, &end_iter);
	  
	  continue;
	}
      
      if (NULL != strchr (remove_chars_pattern, bufferchars [iter]))
	{
	  gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, iter);
	  gtk_text_buffer_get_iter_at_offset (buffer, &end_iter, iter + 1);
	  
	  gtk_text_buffer_delete (buffer, &start_iter, &end_iter);
	  
	  continue;
	}
    }

  /* We know this was allocated, and we do not need it anymore.
   */
  g_free (bufferchars);
  
  return ;
}

gboolean
polyxmass_sequence_purification_wnd_check_sequence_button (GtkWidget *
								   widget,
								   gpointer data)
{
  /* 'widget' param is the button. 
   */
  GtkWidget *window = data;
  PxmSeqedWidget *seqed_widget = NULL;
  
  
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;

  GtkTextView *textview = NULL;
  GtkTextBuffer *buffer = NULL;
  gchar *bufferchars = NULL;

  GtkTextIter start_iter;
  GtkTextIter end_iter;

  gint iter = 0;
  gint *index = NULL;

  gboolean success = FALSE;
    
  GPtrArray *fillGPA = NULL;
  GPtrArray *errorsGPA = NULL;

  GString *failed_indices = NULL;
  


  g_assert (window != NULL);

  seqed_widget = g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);
  
  polchemdefctxt = seqed_widget->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);


  /* We are asked to check the sequence that is presented in the
     textview. Most probably, this is because the user edited it to
     remove a number of invalid characters, and would like to ensure
     that it is now correct and that it will successfully be pasted
     into the sequence editor.

     So, what we do is find the buffer in the textview, get a copy of
     the text string therein and analyze it.
   */
  textview = g_object_get_data (G_OBJECT (window), 
				"purified_sequence_textview");
  g_assert (textview != NULL);

  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));

  gtk_text_buffer_get_start_iter (buffer, &start_iter);
  gtk_text_buffer_get_end_iter (buffer, &end_iter);

  /* Free this allocated char string when no more needed.
   */
  bufferchars = gtk_text_buffer_get_text (buffer,
					  &start_iter,
					  &end_iter,
					  /*include_hidden_chars*/ FALSE);
  g_assert (bufferchars != NULL);

  /* At this point we have the sequence text that is currently displayed
     in the textview. Let's allocate the arrays we'll need...
  */
  fillGPA = g_ptr_array_new ();
  errorsGPA = g_ptr_array_new ();
  
  pxmchem_monomer_fill_array_from_string_check_errors 
    (fillGPA,
     errorsGPA,
     bufferchars, 
     polchemdef->codelen,
     polchemdef->monomerGPA,
     FALSE /*empty_first*/);

  /* We now can check it there is something in errorsGPA. We just
     construct a string where we list all the positions that
     failed. 

     In the mean time, we take advantage of this work to re-tag all
     the erroneous characters in the textview. We first remove all
     tags from the sequence text, because we are going to add new ones
     if some erroneous characters are still found in it.
  */
  gtk_text_buffer_remove_tag_by_name (buffer, "error_tag",
				      &start_iter, &end_iter);
  
  if (errorsGPA->len > 0)
    {
      failed_indices = 
	g_string_new (_("Currently, the following positions are wrong: "));
      
      for (iter = 0 ; iter < errorsGPA->len ; iter++)
	{
	  index = g_ptr_array_index (errorsGPA, iter);
	  g_assert (index != NULL);
	  
	  g_string_append_printf (failed_indices, "%d ", *index + 1);
	  
	  gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, *index);
	  gtk_text_buffer_get_iter_at_offset (buffer, &end_iter, *index + 1);
	  
	  gtk_text_buffer_apply_tag_by_name (buffer,
					     "error_tag",
					     &start_iter,
					     &end_iter);
	}
      
      /* At this point we can display the wrong positions...
       */
      polyxmass_timeoutmsg_message_set (GTK_WINDOW (window),
					failed_indices->str,
					POLYXMASS_LONG_MSG_TIMEOUT);
      
      /* Free the GString object, since its contents are now in the
	 error_messages GtkEntry widget.
      */
      g_string_free (failed_indices, TRUE);

      success = FALSE;
    }
  else
    {
      polyxmass_timeoutmsg_message_set (GTK_WINDOW (window),
					_("No erroneous characters found"),
					POLYXMASS_LONG_MSG_TIMEOUT);

      success = TRUE;
    }
  
        
  /* Remember that two GPtrArray arrays were allocated locally to
     perform the checking operation. We now have to free all that
     stuff:
  */
  pxmchem_monomer_GPA_free (fillGPA);
  
  while (errorsGPA->len > 0)
    {
      index = g_ptr_array_remove_index (errorsGPA, 0);
      g_assert (index != NULL);
      g_free (index);
    }
  
  g_ptr_array_free (errorsGPA, TRUE);

  /* And remember that we also have to free the sequence text string
     that was copied from the textview's buffer:
  */
  g_free (bufferchars);

  gtk_widget_grab_focus (GTK_WIDGET (textview));
  
  return success;
}



void
polyxmass_sequence_purification_wnd_insert_sequence_at_point_button (GtkWidget *widget,
									     gpointer data)
{
  /* The sequence that is located in the textview should be inserted
     at the current cursor location. First force a check that the
     sequence currently located in the textview has no error in
     it. Next, get that sequence and make a "paste" in the current
     cursor location.
  */
  /* 'widget' param is the button. 
   */
  GtkWidget *window = data;
  PxmSeqedWidget *seqed_widget = NULL;
  
  
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;

  GtkTextView *textview = NULL;
  GtkTextBuffer *buffer = NULL;
  gchar *seq = NULL;

  GtkTextIter start_iter;
  GtkTextIter end_iter;

  GPtrArray *fillGPA = NULL;
  GPtrArray *errorsGPA = NULL;



  g_assert (window != NULL);

  seqed_widget = g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);
  
  polchemdefctxt = seqed_widget->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);


  if (FALSE == polyxmass_sequence_purification_wnd_check_sequence_button (NULL,
										  (gpointer) data))
    return;
  
  /* At this point we know that the sequence in the textview is OK.  Get it !
   */
  textview = g_object_get_data (G_OBJECT (window), 
				"purified_sequence_textview");
  g_assert (textview != NULL);

  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));

  gtk_text_buffer_get_start_iter (buffer, &start_iter);
  gtk_text_buffer_get_end_iter (buffer, &end_iter);


  /* Free this allocated char string when no more needed.
   */
  seq = gtk_text_buffer_get_text (buffer,
					  &start_iter,
					  &end_iter,
					  /*include_hidden_chars*/ FALSE);
  g_assert (seq != NULL);
  
  /* We now have our string that we have to interpret and to insert in
     the form of monomer icons in the editor at the insertion
     point. If there is a selection, that selection has to be erased
     and replaced.
  */

  /* First off, check that the sequence in the selection is correct by
     just asking to fill an array of monomer with that sequence string.
  */

  /* Allocate the array where the PxmMonomer objects are going to be
     stored as the seq is parsed.
  */
  fillGPA = g_ptr_array_new ();
  errorsGPA = g_ptr_array_new ();

  /* We now have to convert the string into an array of monomers.  We
     do not need to check for errors, because this has been done
     already in the calling function.
  */
  pxmchem_monomer_fill_array_from_string 
    (fillGPA,
     seq, 
     polchemdef->codelen,
     polchemdef->monomerGPA,
     TRUE /*empty_first*/);


  /* We have an array filled with all the PxmMonomer instances
     corresponding to each monomer code in the seq string. Of course,
     the number of monomer instances in the fillGPA MUST be indentical
     to the number of monomer codes in the sequence string. We do not
     check that because we know all the monomer codes could be
     validated in the calling function when the sequence in the
     textview was first checked for validity.
  */

  /* And now the tough part: the graphical part, and also rememember
     that the PxmMonomer instances that we got are not yet in the
     polymer sequence !
  */

  /* OK, let's first check if some sequence portion is selected right
     now. If so, that means that we'll first have to remove that
     sequence portion.
  */
  if (TRUE == polyxmass_seqed_widget_get_selection_indices (GTK_WIDGET (seqed_widget),
							    NULL,
							    NULL))
    {      
      if (-1 == 
	  polyxmass_seqed_widget_remove_selected_oligomer (GTK_WIDGET (seqed_widget)))
	{
	  g_critical (_("%s@%d: failed to remove the selected "
			"sequence portion.\n"),
		      __FILE__, __LINE__);
	  
	  g_free (seq);
	  
	  pxmchem_monomer_GPA_free (fillGPA);
	  
	  return;
	}
    }
  

  /* Now we should be able to insert the seq in the polymer sequence
     (paste, that is).
  */
  if (-1 == 
      polyxmass_seqed_widget_integrate_sequence_at_point (fillGPA,
							  GTK_WIDGET (seqed_widget),
							  FALSE /*render_modif*/))
    {
      g_critical (_("%s@%d: failed to integrate the selected "
		    "sequence portion.\n"),
		  __FILE__, __LINE__);
      
      g_free (seq);
      
      pxmchem_monomer_GPA_free (fillGPA);
      
      return;
    }
  
  /* The sequence and the selection.
   */
  polyxmass_seqed_widget_signal_mass_update_required (seqed_widget, 
						      PXM_MASSCALC_TARGET_BOTH);
    
  /* At this point we should be done. 
   */
  pxmchem_monomer_GPA_free (fillGPA);

  /* And remember that we also have to free the sequence text string
     that was copied from the textview's buffer:
  */
  g_free (seq);
}




void
polyxmass_sequence_purification_wnd_cancel_button (GtkWidget *widget,
							   gpointer data)
{
  GtkWidget *window = data;
  
  gchar *help = NULL;
  
  PxmSeqedWidget *seqed_widget = NULL;
  

  g_assert (window != NULL);

  seqed_widget = g_object_get_data (G_OBJECT (window), "seqed_widget");
  
  help = g_strdup_printf ("sequence-purification-wnd-%p", window);


  /* We set the datum to NULL, then the GDestroyNotify callback that
     was specified when the datum was set first (that is
     xxx_wnd_really_close () function above) will be called,
     thus destroying *this window, which is exactly what we want !
  */
  g_object_set_data (G_OBJECT (seqed_widget), help, NULL);

  g_free (help);

  return ;

}

void
polyxmass_sequence_purification_wnd_revert_button (GtkWidget *widget,
						    gpointer data)
{
  GtkWidget *window = data;



  /* We are asked that the textview be put to its initial state, as it
     was before any change. This task is performed when the window is
     opened in the first place and is coded in a function that is
     called during the setting up of the textview.
  */
  polyxmass_sequence_purification_wnd_sequence_format (window);
  
  return ;
}



/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxmass_sequence_purification_wnd_really_close (GtkWidget *window)
{
  gchar *seq = NULL;
  
  GPtrArray *errorsGPA = NULL;

  gint *index = NULL;

  g_assert (window != NULL);

  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  /* We have two allocated data that were shipped to this window when
     it was created and that we have to free before closing: the array
     of allocated error index integers and the sequence string.
  */
  /* Get all the pointers we are going to need from the window.
   */
  seq = g_object_get_data (G_OBJECT (window), "seq");
  g_assert (seq != NULL);
  
  g_free (seq);

  errorsGPA = g_object_get_data (G_OBJECT (window), "errorsGPA");
  g_assert (errorsGPA != NULL);
  
  while (errorsGPA->len > 0)
    {
      index = g_ptr_array_remove_index (errorsGPA, 0);
      g_assert (index != NULL);
      g_free (index);
    }
  
  g_ptr_array_free (errorsGPA, TRUE);

  gtk_widget_destroy (window);
}

gint
polyxmass_sequence_purification_wnd_delete_event (GtkWidget *window, 
						   GdkEventAny *event, 
						   gpointer data)
{
  GtkWidget *seqed_widget = data;

  gchar *help = NULL;
  

  g_assert (window != NULL);
  g_assert (seqed_widget != NULL);

  help = g_strdup_printf ("sequence-purification-wnd-%p", window);


  /* We set the datum to NULL, then the GDestroyNotify callback that
     was specified when the datum was set first (that is
     xxx_wnd_really_close () function above) will be called,
     thus destroying *this window, which is exactly what we want !
  */
  g_object_set_data (G_OBJECT (seqed_widget), help, NULL);
  g_free (help);
    
  return TRUE;
}


gint
polyxmass_sequence_purification_wnd_destroy_event (GtkWidget *window, 
						   GdkEventAny *event, 
						    gpointer data)
{
  return FALSE;
}



