/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-seqed-clipboard.h"
#include "polyxedit-ui-seqed-wnd.h"
#include "polyxedit-ui-masses-display-wnd.h"



GtkWidget *
polyxedit_seqed_clipboard_import_wnd_setup (PxmEditCtxt *editctxt,
					    gchar *seq,
					    GPtrArray *errorsGPA)
{
  GtkWidget *widget = NULL;
  GtkWidget *window = NULL;

  GtkWidget *textview = NULL;
  GtkTextBuffer *buffer = NULL;
  GtkTextTag* error_tag = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;

  gint *index = NULL;
  
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  
  /* The allocated gint objects and the errorsGPA array must be freed
     when no longer in use. The seq gchar string must be freed when no
     longer in use.
  */

  g_assert (editctxt != NULL);

  g_assert (seq != NULL);

  g_assert (errorsGPA != NULL);
  

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
   

  gui_file = 
    g_strdup_printf ("%s/polyxedit-seqeditor.glade", userspec->gladedir);

  xml = glade_xml_new (gui_file, "polseq_check_import_wnd", 
		       PACKAGE);
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed loading the interface\n"),
	     __FILE__, __LINE__);

      g_free (gui_file);
      
      return NULL;
    }
  
  g_free (gui_file);
  
  window = glade_xml_get_widget (xml, "polseq_check_import_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed creating the sequence import window\n"),
		  __FILE__, __LINE__);
      
      g_object_unref (G_OBJECT (xml));
      
      return NULL;
    }
  

  g_object_set_data (G_OBJECT (window), "editctxt", editctxt);
  g_object_set_data (G_OBJECT (window), "errorsGPA", errorsGPA);
  g_object_set_data (G_OBJECT (window), "seq", seq);
  


  widget = glade_xml_get_widget (xml, "messages_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "messages_entry", widget);



  /* The frame where the polymer sequence data are displayed:
   */
  widget = glade_xml_get_widget (xml, "sequence_data_frame");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "sequence_data_frame", widget);

  /* The vertical box where the polymer sequence data widgets are packed, 
     so that we make them invisible in one go.
  */
  widget = glade_xml_get_widget (xml, "polymer_sequence_data_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polymer_sequence_data_vbox", widget);
  
  /* The button in the polymer sequence data frame, thanks to which we
     can hide/show the sequence data.
  */
  widget = glade_xml_get_widget (xml, "polymer_sequence_data_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polymer_sequence_data_button", widget);

  /* Let's connect the button to a "clicked" signal.
  */
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxedit_seqed_clipboard_import_wnd_sequence_data_button_clicked), 
     window);
  
  /* Set the name of the protein to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "sequence_entry"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "sequence_entry", widget);

  gtk_entry_set_text (GTK_ENTRY (widget), 
		      editctxt->polymer->plminfo->name);

  /* Set the identity number (actually the editctxt pointer) to 
   * its GtkEntry (this is useful when cleavages are made to identify
   * unambiguously the polymer from which results are displayed).
   */
  widget = glade_xml_get_widget (xml, "identity_entry");
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "identity_entry",
		     widget);
  help = g_strdup_printf ("%p", editctxt);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  
  textview = glade_xml_get_widget (xml, "imported_sequence_textview"); 
  g_assert (textview != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "imported_sequence_textview", textview);

  /* We now should set up the seq string in textview. Specifically we
     should format the string so that each character located at any
     index present in the errorsGPA array is in bold and in red
     colour.
   */

  /* Get the buffer associated with the textview.
   */
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));

  /* We will format the erroneous positions in the sequence text in
     red. To do that we will have to create a new tag, that will belong
     to the buffer. We will reference later to tha tag using its name
     "error_tag".
  */
  error_tag = gtk_text_buffer_create_tag (buffer,
					  "error_tag",
					  "foreground", "red",
					  NULL);
  if (FALSE ==
      polyxedit_seqed_clipboard_import_wnd_sequence_format (window))
    {
      g_critical (_("%s@%d: failed formatting the string %s in the "
		    "textview.\n"),
		  __FILE__, __LINE__, seq);
      
      while (errorsGPA->len > 0)
	{
	  index = g_ptr_array_remove_index (errorsGPA, 0);
	  g_assert (index != NULL);
	  g_free (index);
	}
      
      g_ptr_array_free (errorsGPA, TRUE);
      
      g_free (seq);
      
      g_object_unref (G_OBJECT (xml));
      
      return NULL;
    }
  

  widget = glade_xml_get_widget (xml, "digit_checkbutton"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "digit_checkbutton", widget);

  widget = glade_xml_get_widget (xml, "punctuation_checkbutton"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "punctuation_checkbutton", widget);

  widget = glade_xml_get_widget (xml, "space_checkbutton"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "space_checkbutton", widget);

  widget = glade_xml_get_widget (xml, "remove_other_characters_entry"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "remove_other_characters_entry", widget);



  widget = glade_xml_get_widget (xml, "remove_all_tagged_chars_button"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "remove_all_tagged_chars_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxedit_seqed_clipboard_import_wnd_remove_all_tagged_chars_button), 
     window);


  widget = glade_xml_get_widget (xml, "purify_sequence_button"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "purify_sequence_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxedit_seqed_clipboard_import_wnd_purify_sequence_button), 
     window);


  widget = glade_xml_get_widget (xml, "check_sequence_button"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "check_sequence_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxedit_seqed_clipboard_import_wnd_check_sequence_button), 
     window);


  widget = glade_xml_get_widget (xml, "sequence_import_cancel_button"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "sequence_import_cancel_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxedit_seqed_clipboard_import_wnd_cancel_button), 
     window);

  widget = glade_xml_get_widget (xml, "sequence_import_revert_button"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "sequence_import_revert_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxedit_seqed_clipboard_import_wnd_revert_button), 
     window);



  g_signal_connect (G_OBJECT (window),
		      "delete_event",
		    G_CALLBACK 
		      (polyxedit_seqed_clipboard_import_wnd_delete_event), 
		      editctxt);

  g_signal_connect (G_OBJECT (window),
		      "delete_event",
		    G_CALLBACK 
		      (polyxedit_seqed_clipboard_import_wnd_destroy_event), 
		      editctxt);


  /* Finally we can show the whole sequence editor.
   */
  gtk_widget_show_all (window);


  /* We have finished setting up the window, and so also using
   * the xml data, unref them
   */
  g_object_unref (G_OBJECT (xml));


  /* Set this window pointer as a full datum to the polymer sequence
     editor window, so that when it is closed this window is closed
     also. 

     There might be more than one window of the present kind opened
     for a given polymer seqence editing window, and we do not want
     that the second window destroys the datum name of the first
     window, so we create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("clipboard_import_wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (editctxt->seqeditorctxt->sequence_editor_wnd),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxedit_seqed_clipboard_import_wnd_really_close);


  return window;
}


void
polyxedit_seqed_clipboard_import_wnd_sequence_data_button_clicked (GtkWidget *
								   widget,
								   gpointer 
								   data)
{
  GtkWidget *window = data;
  GtkWidget *vbox = NULL;

  PxmEditCtxt *editctxt = NULL;

  g_assert (widget != NULL);
  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  

  /* Get the vertical box where the polymer sequence data-displaying
     widgets are packed.
  */
  vbox = g_object_get_data (G_OBJECT (window),
			    "polymer_sequence_data_vbox");
  g_assert (vbox != NULL);

  /* If the vbox is visible we make it invisible, and vice-versa.
   */
  if (GTK_WIDGET_VISIBLE (vbox))
    gtk_widget_hide (vbox);
  else
    gtk_widget_show (vbox);
  
  return ;
}


gboolean
polyxedit_seqed_clipboard_import_wnd_sequence_format (GtkWidget *widget)
{
  GtkWidget *window = widget ;
  
  GtkTextView *textview = NULL;
  
  GtkTextBuffer *buffer = NULL;

  GtkTextIter start_iter;
  GtkTextIter end_iter;

  gint *index = NULL;
  gint iter = 0;
  
  gchar *seq = NULL;
  
  GPtrArray *errorsGPA = NULL;
  

  g_assert (window != NULL);

  /* Get all the pointers we are going to need from the window.
   */
  seq = g_object_get_data (G_OBJECT (window), "seq");
  g_assert (seq != NULL);

  
  errorsGPA = g_object_get_data (G_OBJECT (window), "errorsGPA");
  g_assert (errorsGPA != NULL);
  
  textview = g_object_get_data (G_OBJECT (window), 
				"imported_sequence_textview");
  g_assert (textview != NULL);
  

  /* OK, start the operations. Note that this function might be called
     by the function that handles the "revert" button, where the user
     asks that the textview be set as new, that is as it was before
     any changes were made to it. So we ought to first empty the
     textview here, even if it might be empty already, as this
     function is also called by the window setting up function.
   */
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));
  
  /* Empty the textview first.
   */
  gtk_text_buffer_get_start_iter (buffer, &start_iter);
  gtk_text_buffer_get_end_iter (buffer, &end_iter);

  gtk_text_buffer_delete (buffer, &start_iter, &end_iter);

  /* And now we can start afresh : we have to set the seq string to
     the textview.
   */

  /* Get end of buffer where we'll insert (that is append actually) 
     the seq string.
  */
  gtk_text_buffer_get_end_iter (buffer, &end_iter);
  
  gtk_text_buffer_insert (buffer, &end_iter, seq, -1);
  
  /* We now have to iterate in the array of errors and for each index
     iterated set the corresponding character to red.
  */
  for (iter = 0; iter < errorsGPA->len ; iter++)
    {
      index = g_ptr_array_index (errorsGPA, iter);
      g_assert (index != NULL);
      
      gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, *index);
      gtk_text_buffer_get_iter_at_offset (buffer, &end_iter, *index + 1);
      
      gtk_text_buffer_apply_tag_by_name (buffer,
					 "error_tag",
					 &start_iter,
					 &end_iter);
      
    }
  
  /* At this point all the positions in the seq string that were found
     erroneous are formatted so that they are easily visible.
  */

  /* And force the focus to the textview, so that all the characters
     that should be "error_tag"'ged are effectively tagged (is this a
     bug of GtkTextView) ?
  */
  gtk_widget_grab_focus (GTK_WIDGET (textview));


  return TRUE;
}


void
polyxedit_seqed_clipboard_import_wnd_remove_all_tagged_chars_button (GtkWidget *
							     widget,
							     gpointer data)
{
  GtkWidget *window = data;

  GtkTextView *textview = NULL;
  GtkTextBuffer *buffer = NULL;

  GtkTextIter start_iter;
  GtkTextIter end_iter;
  GtkTextIter moving_iter;

  GtkTextTagTable *table = NULL;
  GtkTextTag *error_tag = NULL;
  
  g_assert (window != NULL);

  textview = g_object_get_data (G_OBJECT (window), 
				"imported_sequence_textview");
  g_assert (textview != NULL);



  /* We are asked to iterate in the sequence buffer contained in the
     textview and remove from that sequence to remove all the
     characters that are currently tagged with the "error_tag"
     formatting tag. That is the user acknowledges that there are a
     number of erroneous characters in the sequence, and that these
     characters should be removed.
   */
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));

  gtk_text_buffer_get_start_iter (buffer, &start_iter);
  gtk_text_buffer_get_end_iter (buffer, &end_iter);

  table = gtk_text_buffer_get_tag_table (buffer);
  g_assert (table != NULL);
  
  error_tag = gtk_text_tag_table_lookup (table, "error_tag");
  g_assert (error_tag != NULL);
  
  /* Get an iterator to the first character in the buffer !
   */
  gtk_text_buffer_get_start_iter (buffer, &start_iter);
  moving_iter = start_iter;
  
  while (TRUE)
    {
      if (TRUE == gtk_text_iter_has_tag (&start_iter, error_tag))
	{
	  /* The first character in the sequence is tagged, so we have to
	     remove it. But we need an "end_iter" that points to the
	     forward characteur so that we can call the delete function.
	  */
	  moving_iter = start_iter;
	  gtk_text_iter_forward_char (&moving_iter);
	  
	  /* From Gtk+2 doc: the start and end will be re-initialized
	     to point to the location where text was deleted.
	  */
	  gtk_text_buffer_delete (buffer, &start_iter, &moving_iter);
	}
      else
	{
	  if (FALSE == gtk_text_iter_forward_char (&start_iter))
	    break;
	  else
	    continue;
	}      
    }
    
  return ;
}


void
polyxedit_seqed_clipboard_import_wnd_purify_sequence_button (GtkWidget *
							     widget,
							     gpointer data)
{
  GtkWidget *window = data;

  GtkWidget *entry = NULL;
  GtkWidget *checkbutton = NULL;
    
  GtkTextView *textview = NULL;
  GtkTextBuffer *buffer = NULL;
  gchar *bufferchars = NULL;

  gchar *remove_chars_pattern = NULL;
  
  gboolean remove_digit_chars = FALSE;
  gboolean remove_space_chars = FALSE;
  gboolean remove_punctuation_chars = FALSE;
  
  GtkTextIter start_iter;
  GtkTextIter end_iter;

  gint iter = 0;
  gint bufferchars_length =0;
  
  GPtrArray *errorsGPA = NULL;


  g_assert (window != NULL);


  /* Get all the pointers we are going to need from the window.
   */
  errorsGPA = g_object_get_data (G_OBJECT (window), "errorsGPA");
  g_assert (errorsGPA != NULL);
  
  textview = g_object_get_data (G_OBJECT (window), 
				"imported_sequence_textview");
  g_assert (textview != NULL);


  /* We are asked to purify the sequence that is displayed in the
     textview. The criteria that govern the purification process are
     set by the user in the form of three checkbutton widgets and a
     text entry widget.

     The checkbutton widgets tell if digits (0--9) and/or spaces
     (space, tab, newline...) and/or punctuation (:;,...) must be
     removed from the text.

     The text entry widget lists more characters that should be
     removed from the sequence text.
  */

  /* We thus have to iterate in sequence text and for each character
     we have to check the requested purification criteria. Each time a
     character in the sequence text happens to correspond to the
     purification criteria, then it should be removed.
  */

  /* OK, start the operations.
   */
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));

  gtk_text_buffer_get_start_iter (buffer, &start_iter);
  gtk_text_buffer_get_end_iter (buffer, &end_iter);


  /* Free this allocated char string.
   */
  bufferchars = gtk_text_buffer_get_text (buffer,
					  &start_iter,
					  &end_iter,
					  /*include_hidden_chars*/ FALSE);
  g_assert (bufferchars != NULL);

  
  /* Now we will iterate in the string and check its individual 
     characters. But first we have to see what's required by the user
     to be removed from that string!
  */
  checkbutton = g_object_get_data (G_OBJECT (window), 
				   "digit_checkbutton"); 
  g_assert (checkbutton != NULL);
  remove_digit_chars = 
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton));
  


  checkbutton = g_object_get_data (G_OBJECT (window), 
				   "punctuation_checkbutton"); 
  g_assert (checkbutton != NULL);
  remove_punctuation_chars = 
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton));
  


  checkbutton = g_object_get_data (G_OBJECT (window), 
				   "space_checkbutton"); 
  g_assert (checkbutton != NULL);
  remove_space_chars = 
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton));
  


  entry = g_object_get_data (G_OBJECT (window), 
			     "remove_other_characters_entry");
  g_assert (entry != NULL);
  remove_chars_pattern = gtk_entry_get_text (GTK_ENTRY (entry));
  
  /* OK, we are now set for the iteration in the sequence string.
   */
  bufferchars_length = strlen (bufferchars);
  

  /* We process the string from its last character up to its first
     character, so that if we have to remove invalid characters, the
     indices between the string's characters and the actual buffer
     contained in the textview won't change. Do not forget the
     bufferchars is an allocated string that is nothing but a snapshot
     of the textview's buffer data at the beginning of the
     process. When we remove characters from the textview's buffer
     text, we do not remove characters from bufferchars, which is why
     to maintain the colinearity between indices, we start working
     from the end of the string.
  */
  for (iter = bufferchars_length - 1 ; iter >= 0; iter--)
    {
      /* We now can check the character for all what's been asked by
	 the user.
       */
      if (TRUE == remove_digit_chars 
	  && TRUE == g_ascii_isdigit (bufferchars [iter]))
	{
	  gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, iter);
	  gtk_text_buffer_get_iter_at_offset (buffer, &end_iter, iter + 1);
	  
	  gtk_text_buffer_delete (buffer, &start_iter, &end_iter);
	  
	  continue;
	}
      
      if (TRUE == remove_space_chars 
	  && TRUE == g_ascii_isspace (bufferchars [iter]))
	{
	  gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, iter);
	  gtk_text_buffer_get_iter_at_offset (buffer, &end_iter, iter + 1);
	  
	  gtk_text_buffer_delete (buffer, &start_iter, &end_iter);
	  
	  continue;
	}
      
      if (TRUE == remove_punctuation_chars 
	  && TRUE == g_ascii_ispunct (bufferchars [iter]))
	{
	  gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, iter);
	  gtk_text_buffer_get_iter_at_offset (buffer, &end_iter, iter + 1);
	  
	  gtk_text_buffer_delete (buffer, &start_iter, &end_iter);
	  
	  continue;
	}
      
      if (NULL != strchr (remove_chars_pattern, bufferchars [iter]))
	{
	  gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, iter);
	  gtk_text_buffer_get_iter_at_offset (buffer, &end_iter, iter + 1);
	  
	  gtk_text_buffer_delete (buffer, &start_iter, &end_iter);
	  
	  continue;
	}
    }

  /* We know this was allocated, and we do not need it anymore.
   */
  g_free (bufferchars);
  
  return ;
}

void
polyxedit_seqed_clipboard_import_wnd_check_sequence_button (GtkWidget *
							     widget,
							     gpointer data)
{
  GtkWidget *window = data;

  PxmEditCtxt *editctxt = NULL;
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;

  GtkTextView *textview = NULL;
  GtkTextBuffer *buffer = NULL;
  gchar *bufferchars = NULL;

  GtkTextIter start_iter;
  GtkTextIter end_iter;

  gint iter = 0;
  gint *index = NULL;
  
  GPtrArray *fillGPA = NULL;
  GPtrArray *errorsGPA = NULL;

  GString *failed_indices = NULL;
  


  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);

  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);


  /* We are asked to check the sequence that is presented in the
     textview. Most probably, this is because the user edited it to
     remove a number of invalid characters, and would like to ensure
     that it is now correct and that it will successfully be pasted
     into the sequence editor.

     So, what we do is find the buffer in the textview, get a copy of
     the text string therein and analyze it.
   */
  textview = g_object_get_data (G_OBJECT (window), 
				"imported_sequence_textview");
  g_assert (textview != NULL);

  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));

  gtk_text_buffer_get_start_iter (buffer, &start_iter);
  gtk_text_buffer_get_end_iter (buffer, &end_iter);


  /* Free this allocated char string.
   */
  bufferchars = gtk_text_buffer_get_text (buffer,
					  &start_iter,
					  &end_iter,
					  /*include_hidden_chars*/ FALSE);
  g_assert (bufferchars != NULL);

  /* At this point we have the sequence text that is currently displayed
     in the textview. Let's allocate the arrays we'll need...
  */
  fillGPA = g_ptr_array_new ();
  errorsGPA = g_ptr_array_new ();
  
  pxmchem_monomer_fill_array_from_string_check_errors 
    (fillGPA,
     errorsGPA,
     bufferchars, 
     polchemdef->codelen,
     polchemdef->monomerGPA,
     FALSE /*empty_first*/);

  /* We now can check it there is something in errorsGPA. We just
     construct a string where we list all the positions that
     failed. 

     In the mean time, we take advantage of this work to re-tag all
     the erroneous characters in the textview. We first remove all
     tags from the sequence text, because we are going to add new ones
     if some erroneous characters are still found in it.
  */
  gtk_text_buffer_remove_tag_by_name (buffer, "error_tag",
				      &start_iter, &end_iter);
  
  if (errorsGPA->len > 0)
    {
      failed_indices = 
	g_string_new (_("Currently, the following positions are wrong: "));
      
      for (iter = 0 ; iter < errorsGPA->len ; iter++)
	{
	  index = g_ptr_array_index (errorsGPA, iter);
	  g_assert (index != NULL);
	  
	  g_string_append_printf (failed_indices, "%d ", *index + 1);
	  
	  gtk_text_buffer_get_iter_at_offset (buffer, &start_iter, *index);
	  gtk_text_buffer_get_iter_at_offset (buffer, &end_iter, *index + 1);
	  
	  gtk_text_buffer_apply_tag_by_name (buffer,
					     "error_tag",
					     &start_iter,
					     &end_iter);
	}
      
      /* At this point we can display the wrong positions...
       */
      polyxmass_timeoutmsg_message_set (GTK_WINDOW (window),
					failed_indices->str,
					POLYXMASS_LONG_MSG_TIMEOUT);
      
      /* Free the GString object, since its contents are now in the
	 error_messages GtkEntry widget.
      */
      g_string_free (failed_indices, TRUE);
    }
  else
    {
      polyxmass_timeoutmsg_message_set (GTK_WINDOW (window),
					_("No erroneous characters found"),
					POLYXMASS_LONG_MSG_TIMEOUT);
    }
  
        
  /* Remember that two GPtrArray arrays were allocated locally to
     perform the checking operation. We now have to free all that
     stuff:
  */
  pxmchem_monomer_GPA_free (fillGPA);
  
  while (errorsGPA->len > 0)
    {
      index = g_ptr_array_remove_index (errorsGPA, 0);
      g_assert (index != NULL);
      g_free (index);
    }
  
  g_ptr_array_free (errorsGPA, TRUE);

  /* And remember that we also have to free the sequence text string
     that was copied from the textview's buffer:
  */
  g_free (bufferchars);

  gtk_widget_grab_focus (GTK_WIDGET (textview));
  
  return ;
}

void
polyxedit_seqed_clipboard_import_wnd_cancel_button (GtkWidget *widget,
						    gpointer data)
{
  GtkWidget *window = data;
  
  PxmEditCtxt *editctxt = NULL;
  
  gchar *help = NULL;
  

  g_assert (window != NULL);

  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);
  

  help = g_strdup_printf ("clipboard_import_wnd-%p", window);


  /* We set the datum to NULL, then the GDestroyNotify callback that
     was specified when the datum was set first (that is
     xxx_wnd_really_close () function above) will be called,
     thus destroying *this window, which is exactly what we want !
  */
  g_object_set_data (G_OBJECT
		     (editctxt->seqeditorctxt->sequence_editor_wnd), 
		     help, NULL);
  g_free (help);
    
  return ;

}

void
polyxedit_seqed_clipboard_import_wnd_revert_button (GtkWidget *widget,
						    gpointer data)
{
  GtkWidget *window = data;



  /* We are asked that the textview be put to its initial state, as it
     was before any change. This task is performed when the window is
     opened in the first place and is coded in a function that is
     called during the setting up of the textview.
  */
  polyxedit_seqed_clipboard_import_wnd_sequence_format (window);
  
  return ;
}



/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxedit_seqed_clipboard_import_wnd_really_close (GtkWidget *window)
{
  gchar *seq = NULL;
  
  GPtrArray *errorsGPA = NULL;

  gint *index = NULL;

  g_assert (window != NULL);

  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  /* We have two allocated data that were shipped to this window when
     it was created and that we have to free before closing: the array
     of allocated error index integers and the sequence string.
  */
  /* Get all the pointers we are going to need from the window.
   */
  seq = g_object_get_data (G_OBJECT (window), "seq");
  g_assert (seq != NULL);
  
  g_free (seq);

  errorsGPA = g_object_get_data (G_OBJECT (window), "errorsGPA");
  g_assert (errorsGPA != NULL);
  
  while (errorsGPA->len > 0)
    {
      index = g_ptr_array_remove_index (errorsGPA, 0);
      g_assert (index != NULL);
      g_free (index);
    }
  
  g_ptr_array_free (errorsGPA, TRUE);

  gtk_widget_destroy (window);
}

gint
polyxedit_seqed_clipboard_import_wnd_delete_event (GtkWidget *window, 
						   GdkEventAny *event, 
						   gpointer data)
{
  PxmEditCtxt *editctxt = data;

  gchar *help = NULL;
  

  g_assert (window != NULL);
  g_assert (editctxt != NULL);

  help = g_strdup_printf ("clipboard_import_wnd-%p", window);


  /* We set the datum to NULL, then the GDestroyNotify callback that
     was specified when the datum was set first (that is
     xxx_wnd_really_close () function above) will be called,
     thus destroying *this window, which is exactly what we want !
  */
  g_object_set_data (G_OBJECT
		     (editctxt->seqeditorctxt->sequence_editor_wnd), 
		     help, NULL);
  g_free (help);
    
  return TRUE;
}


gint
polyxedit_seqed_clipboard_import_wnd_destroy_event (GtkWidget *window, 
						   GdkEventAny *event, 
						    gpointer data)
{
  return FALSE;
}






/************* CLIPBOARD LOW-LEVEL FUNCTIONS **********************/

void
polyxedit_seqed_clipboard_selection_get (GtkWidget *widget,
					 GtkSelectionData *selection_data,
					 guint info,
					 guint time_stamp,
					 gpointer data)
{
  PxmEditCtxt *editctxt = data;

  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  PxmProp *prop = NULL;
  
  gint start_idx = -1;
  gint end_idx = -1;

  gchar *seq;

  /* This function is called as a callback to a "selection_get" signal
     that our process gets from an application (or itself) that would
     like to paste what whe may have previously put in either the
     GDK_SELECTION_PRIMARY or the GDK_SELECTION_CLIPBOARD selections
     !. Whe have to make the difference between the two, because:

     1. if the GDK_SELECTION_PRIMARY is asked, then we real-time have
     to check if a sequence is currently selected. That is "HOT"
     selection.

     2. if the GDK_SELECTION_CLIPBOARD is asked, then we have to check
     if the canvas has a prop object named "CLIPBOARD_SELECTION"
     containing the sequence that was allocated when the user last did
     Ctrl + C. If such prop is there, then its data have to be
     provided to the caller with the gtk_selection_data_set () call
     below.
  */

  g_assert (editctxt != NULL);

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  
  if (selection_data->selection == GDK_SELECTION_PRIMARY)
    {
      /* We first check if there is a selection in the editctxt. If
	 there is such a selection, we just put it in the clipboard.
      */
      if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							      &start_idx,
							      &end_idx))
	{
	  g_message (_("%s@%d: no selection in the sequence editor\n"),
		 __FILE__, __LINE__);
      
	  return;
	}
  
      /* Note that due to sequence editor-specific reasons, the end_idx
	 that is returned above contains one more monomer, so we should
	 decrement this once before using the function above.
      */
      seq = pxmchem_polymer_make_codes_string (editctxt->polymer,
					       start_idx, end_idx - 1);

      g_assert (seq != NULL);
  
      /* When we return a single string, it should not be NULL-terminated,
	 that will be done for us.
      */
      gtk_selection_data_set (selection_data, GDK_SELECTION_TYPE_STRING,
			      8 * sizeof (gchar), 
			      seq, strlen (seq));

      return;
    }
  
  if (selection_data->selection == GDK_SELECTION_CLIPBOARD)
    {
      prop = libpolyxmass_prop_find_prop (seqeditorctxt->propGPA,
				      NULL,
				      NULL,
				      "CLIPBOARD_SELECTION",
				      NULL,
				      PXM_UNDEF_PROP_CMP);
      
      if (prop == NULL)
	return ;
      
      g_assert (prop->data != NULL);
      
      gtk_selection_data_set (selection_data, GDK_SELECTION_TYPE_STRING,
			      8 * sizeof (gchar), 
			      prop->data, strlen (prop->data));
  
      return;
    }
  
  return;
}


void
polyxedit_seqed_clipboard_selection_received (GtkWidget *widget,
					      GtkSelectionData *
					      selection_data,
					      guint time,
					      gpointer data)
{
  GtkWidget *window = NULL;
  
  PxmEditCtxt *editctxt = data;
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;
  
  GPtrArray *fillGPA = NULL;
  GPtrArray *errorsGPA = NULL;
  
  gchar *seq = NULL;

  

  g_assert (editctxt != NULL);

  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);
  

  /* This function is called when the data that were asked
     in the function call xxx are actually arrived on the local system.
     Remember that X can be distributed and that there may be network
     slugginess between the moment the data are asked and the moment 
     they are available (received).
  */

  /* IMPORTANT: check to see if retrieval succeeded.
   */
  if (selection_data->length < 0)
    {
      g_critical (_("%s@%d: selection retrieval failed.\n"),
	     __FILE__, __LINE__);

      return;
    }

  if (selection_data->type != GDK_SELECTION_TYPE_STRING)
    {
      g_critical (_("%s@%d: selection \"STRING\" was not "
		    "returned correctly.\n"),
		  __FILE__, __LINE__);

      return;
    }

  seq = g_strdup (selection_data->data);
  g_assert (seq != NULL);

  /* We now have our string that we have to interpret and to insert in
     the form of monomer icons in the editor at the insertion
     point. If there is a selection, that selection has to be erased
     and replaced.
  */

  /* First off, check that the sequence in the selection is correct by
     just asking to fill an array of monomer with that sequence string.
  */

  /* Allocate the array where the PxmMonomer objects are going to be
     stored as the seq is parsed.
  */
  fillGPA = g_ptr_array_new ();
  errorsGPA = g_ptr_array_new ();

  /* At this point we have to make sure that the string we got
     contains characters that effectively correctly encode
     monomers. That is, we must ensure that by parsing of the sequence
     we can convert the characters parsed into monomer codes. If this
     is not possible, then that means that there are errors and we
     have to make sure the user knows this. We will provide the user
     with the possibility to edit the sequence so that it gets
     purified in order to be pasted into the sequence editor.
  */

  /* The array errorsGPA will contain allocated gint objects
     representing the index of invalid characters/codes in seq.
  */
  pxmchem_monomer_fill_array_from_string_check_errors 
    (fillGPA,
     errorsGPA,
     seq, 
     polchemdef->codelen,
     polchemdef->monomerGPA,
     FALSE /*empty_first*/);
  
  if (errorsGPA->len == 0)
    {
      /* There was not a single error during the parsing of the
	 sequence. We can free the array right away.
       */
      g_ptr_array_free (errorsGPA, TRUE);
      errorsGPA = NULL;
    }
  else
    {
      /* Apparently there were errors in the parsing process. We
	 should give the user the proper feedback. This will be
	 performed by opening a window were the user will see the seq
	 string formatted in such a way that the invalid characters
	 will be highlighted to ease correction.
      */

      /* We can immediately free the fillGPA array that contains the valid 
	 monomer codes as we are not going to use it.
      */
      pxmchem_monomer_GPA_free (fillGPA);

      /* OK, now we have to setup the window where the seq string is
	 going to be displayed. We pass to the function the array of
	 indices where errors were encountered so that the seq string
	 will be formatted so that the user will find it easy to see
	 where the errors are located. The seq string will be
	 displayed in a textview widget were formatting is possible.
      */
      window = polyxedit_seqed_clipboard_import_wnd_setup (editctxt,
							   seq,
							   errorsGPA);
      
      if (window == NULL)
	{
	  g_critical (_("%s@%d: failed setting up the clipboard import  "
			"window.\n"),
		      __FILE__, __LINE__);
	  
	  /* NOTE that we do not free the 'errorsGPA' nor the 'seq'
	     objects because that freeing took place in the called
	     function above.
	  */
	  return;
	}
      
      /* At this point we should have setup the window and we can consider
	 that we have finished doing our work here. Let the work go on
	 if the seq string contained no error whatsoever.
      */
      
      /* NOTE that we do not free the errorsGPA nor the seq objects
	 because that freeing took place in the called function above.
      */
      
      return;
    }
  

  /* At this point we know that no error was found during the seq
     string parsing. We thus have an array filled with all the
     PxmMonomer instances corresponding to each monomer code in the
     seq string. Of course, the number of monomer instances in the
     fillGPA MUST be indentical to the number of monomer codes in the
     sequence string. We do not check that because we know all the
     monomer codes could be validated in the function call
     above. Otherwise the errorsGPA->len value would have been != 0.
  */

  /* And now the tough part: the graphical part, and also rememember
     that the PxmMonomer instances that we got are not yet in the
     polymer sequence !
  */

  /* OK, let's first check if some sequence portion is selected right
     now. If so, that means that we'll first have to remove that
     sequence portion.
  */
  if (TRUE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							 NULL,
							 NULL))
    {
      if (-1 == polyxedit_seqed_wnd_remove_selected_oligomer (editctxt))
	{
	  g_critical (_("%s@%d: failed removing the selected "
			"sequence portion.\n"),
		 __FILE__, __LINE__);
	  
	  g_free (seq);
	  
	  pxmchem_monomer_GPA_free (fillGPA);
	  
	  return;
	}
    }
     

  /* Now we should be able to insert the seq in the polymer sequence
     (paste, that is).
  */
  if (-1 == polyxedit_seqed_wnd_integrate_sequence_at_point (fillGPA,
							     editctxt))
    {
      g_critical (_("%s@%d: failed integrating the selected "
		    "sequence portion.\n"),
	     __FILE__, __LINE__);

      g_free (seq);

      pxmchem_monomer_GPA_free (fillGPA);
      
      return;
    }
      
  /* Update both the selected and whole sequence masses.
   */
  polyxedit_ui_masses_display_wnd_update_sequence_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  /* At this point we should be done. 
   */
  pxmchem_monomer_GPA_free (fillGPA);

  g_free (seq);

  return;
}


gboolean
polyxedit_seqed_clipboard_selection_clear (GtkWidget *widget,
				     GdkEventSelection *event,
				     gpointer data)
{
  PxmEditCtxt *editctxt = data;

  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  PxmProp *prop = NULL;

  gboolean result = FALSE;
  

  /* There are two cases. If the clear event is for the CLIPBOARD
     selection, then we remove from the poledtctx->propGPA the
     "CLIPBOARD_SELECTION"-named prop object (if it exists) and we
     free it (all data are text). If the clear event is for the
     PRIMARY selection, then we just remove the selection polygon.
  */


  g_assert (editctxt != NULL);

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);



  if (event->selection == GDK_SELECTION_PRIMARY)
    {
      if (TRUE == 
	  polyxedit_seqed_wnd_get_selection_indices (editctxt,
						     NULL,
						     NULL))
	{
	  polyxedit_seqed_wnd_remove_selection_polygon (editctxt->
							seqeditorctxt);

	  return TRUE;
	}
      else
	{
	  /* Propagate the signal because we did not perform anything.
	   */
	  return FALSE;
	}
    }
  
  
  if (event->selection == GDK_SELECTION_CLIPBOARD)
    {
      prop = libpolyxmass_prop_find_prop (seqeditorctxt->propGPA,
				      NULL,
				      NULL,
				      "CLIPBOARD_SELECTION",
				      NULL,
				      PXM_UNDEF_PROP_CMP);
      
      if (prop == NULL)
	{
	  /* No sequence is presently allocated.
	   */
	  /* Propagate the signal because we did not perform anything.
	   */
	  return FALSE;
	}
      else
	{
	  result = g_ptr_array_remove (seqeditorctxt->propGPA,
				       prop);
	  
	  g_assert (result != FALSE);
	  
	  libpolyxmass_prop_free (prop);
	  
	  return TRUE;
	}
    }
  
  return FALSE;
}




void
polyxedit_seqed_clipboard_primary_paste (PxmEditCtxt *editctxt)
{
  GdkAtom target_atom = GDK_NONE;
  

  /* This function is called by the middle-mouse-button click handler
     in this same file. This handler simply say the traditional X
     selection protocol that we would like to get the current
     selection contents. When we will acutally receive these data,
     we'll get a "selection_received" signal for which we have written
     a handler that will actually perform the real paste.
  */

  g_assert (editctxt != NULL);
  
  /* The data we are trying to get have to be in their string
     form. These data are a polymer sequence stretch in string form.
  */
  target_atom = gdk_atom_intern ("STRING", FALSE);
  g_assert (target_atom != GDK_NONE);
  
  gtk_selection_convert (GTK_WIDGET (editctxt->seqeditorctxt->
				     sequence_canvas),
			 GDK_SELECTION_PRIMARY,
			 target_atom,
			 GDK_CURRENT_TIME);
  return;
}



gboolean
polyxedit_seqed_clipboard_primary_copy (PxmEditCtxt *editctxt)
{

  g_assert (editctxt != NULL);
  
  /* This function is called each time a new selection is performed in
     the sequence editor (the canvas, actually). That selection owner
     setting, below, corresponds to the X selection mechanism that
     will set asynchronously the selected portion of the sequence as
     the PRIMARY selection, which is the one that --according to X
     tradition-- is pasted using the mouse middle button. No clipboard
     stuff here. Check the corresponding functions if you are
     interested in clipboard stuff.
  */

 
  /* We first check if there is a selection in the editctxt. If
     there is such a selection, we just put it in the
     clipboard. Otherwise we do nothing, so that we do not change the
     current selection in another process if any.
  */
  if (TRUE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							  NULL,
							  NULL))
    {
      gtk_selection_owner_set (GTK_WIDGET (editctxt->seqeditorctxt->
					   sequence_canvas),
			       GDK_SELECTION_PRIMARY,
			       GDK_CURRENT_TIME);
    }
  
  /*
    debug_printf (("current selection (indices) to copy to PRIMARY is"
    "%d-%d\n", start_idx, end_idx));
  */

  return TRUE;
}


gboolean
polyxedit_seqed_clipboard_copy (PxmEditCtxt *editctxt)
{
  gint start_idx = -1;
  gint end_idx = -1;

  gchar *seq = NULL;
  
  
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  PxmProp *prop = NULL;
  

  /* We first check if there is a selection in the editctxt. If
     there is such a selection, we make a string with that we set a 
     a datum to prop object that we'll create if we do not find it in
     the array of prop instances of the seqeditorctxt object. This way, 
     that sequence string will be available to other processes willing to 
     paste it through the Ctrl + V paste mechanism.
  */


  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);


  if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							  &start_idx,
							  &end_idx))
    {
      gtk_selection_owner_set (NULL,
			       GDK_SELECTION_CLIPBOARD,
			       GDK_CURRENT_TIME);
      return TRUE;
    }
  

  /*
    debug_printf (("current selection (indices) to copy to CLIPBOARD is"
    "%d-%d\n", start_idx, end_idx));
  */

  /* We now know that there is a selection. We have to put a copy of
     that current selection in the datum of a prop object that is
     going to stay in the seqeditorctxt->propGPA array of prop instances,
     so that we know where to search for it when a process requires
     that content.
  */
  prop = libpolyxmass_prop_find_prop (seqeditorctxt->propGPA,
				  NULL,
				  NULL,
				  "CLIPBOARD_SELECTION",
				  NULL,
				  PXM_UNDEF_PROP_CMP);
  
  if (prop == NULL)
    {
      prop = libpolyxmass_prop_new ();
      
      libpolyxmass_prop_set_name (prop, "CLIPBOARD_SELECTION");

      g_ptr_array_add (seqeditorctxt->propGPA, prop);
    }
  else
    {
      g_assert (prop->data != NULL);
      g_free (prop->data);
    }
  
  /* The indices of the selection are in the two parameters passed to
     the function. Note that due to sequence editor-specific reasons,
     the end_idx that is returned above contains one more monomer, so
     we should decrement this once before using the function above.
  */
  seq = pxmchem_polymer_make_codes_string (editctxt->polymer,
					   start_idx, end_idx - 1);
  
  g_assert (seq != NULL);
  
  prop->data = seq;
  
  /* Now that we have a prop object that contains a copy of the
     currently selected sequence portion, we can claim ownership of
     the clipboard selection.
  */

  gtk_selection_owner_set (GTK_WIDGET (editctxt->seqeditorctxt->
				       sequence_canvas),
			   GDK_SELECTION_CLIPBOARD,
			   GDK_CURRENT_TIME);

  return TRUE;
}


gboolean
polyxedit_seqed_clipboard_cut (PxmEditCtxt *editctxt)
{
  
  g_assert (editctxt != NULL);
  
  if (FALSE == polyxedit_seqed_wnd_get_selection_indices (editctxt,
							  NULL, NULL))
    return TRUE;
  
  /* OK, there is a selection. We first put it in the clipboard and next
     we cut it from the polymer sequence (remove the selection.
  */
  if (TRUE == polyxedit_seqed_clipboard_copy (editctxt))
    {
      /*
	debug_printf (("first index %d second index %d",
	seqeditorctxt->sel_mnm_idx1, seqeditorctxt->sel_mnm_idx2));
      */

      if (-1 == polyxedit_seqed_wnd_remove_selected_oligomer (editctxt))
	return FALSE;
      else 
	{
	  /* The sequence and the selection, since the selected
	     oligomer was remove, the selection and the sequence
	     actually were changed.
	   */
	  polyxedit_ui_masses_display_wnd_update_sequence_masses
	    ((PxmEditCtxt *) polyxedit_last_editctxt);
	  
	  polyxedit_ui_masses_display_wnd_update_selection_masses
	    ((PxmEditCtxt *) polyxedit_last_editctxt);
	  
	  return TRUE;
	}
    }
  else
    return FALSE;
}


gboolean
polyxedit_seqed_clipboard_paste (PxmEditCtxt *editctxt)
{
  GdkAtom target_atom = GDK_NONE;
  

  /* This function is called by the Ctrl + V handler in the keyboard
     handling functions. What we do here is not to paste the contents
     of the clipboard selection, what we is ask for the data in that
     clipboard selection. When that data are arrived from their owner
     to our process, then the "selection_received" signal is emitted
     to our process and we have written a handler for that
     signal.
  */

  g_assert (editctxt != NULL);
  
  /* The data we are trying to get have to be in their string
     form. These data are a polymer sequence stretch in string form.
  */
  target_atom = gdk_atom_intern ("STRING", FALSE);
  g_assert (target_atom != GDK_NONE);
  
  gtk_selection_convert (GTK_WIDGET (editctxt->
				     seqeditorctxt->sequence_canvas),
			 GDK_SELECTION_CLIPBOARD,
			 target_atom,
			 GDK_CURRENT_TIME);
  return TRUE;
}


