/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-findmass-results.h"

/* columns */

enum
{
  /* The mass that was found. */
  COLUMN_OLIG_TO_FIND_MASS,
  /* the difference between found and found masses */
  COLUMN_OLIG_MASS_ERROR,
  /* The name of the oligo ! */
  COLUMN_OLIG_NAME,
  /* The index + 1 in the array of oligos ! */
  COLUMN_OLIG_NUMBER,
  /* MONO or AVG: the mass type of the find */
  COLUMN_OLIG_MASS_TYPE,
  /* Mono mass */
  COLUMN_OLIG_MONO,
  /* Average mass */
  COLUMN_OLIG_AVG,
  /* Pointer to the oligomer */
  COLUMN_OLIG_POINTER,

  COLUMN_OLIG_COL_COUNT
};



GtkWidget *
polyxedit_findmass_res_wnd_setup (GtkWidget *parent,
				  GPtrArray *alloligsGPA)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;
  GtkWidget *grand_parent = NULL;
  

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;


  g_assert (parent != NULL);
  g_assert (alloligsGPA != NULL);

  gui_file = 
    g_strdup_printf ("%s/polyxedit-findmass-results.glade", 
		     userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "findmass_results_wnd", 
		       PACKAGE);
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed loading the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "findmass_results_wnd");
  
  if (window == NULL)
    {
      g_error (_("%s@%d: failed creating the findmass results window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* There is a GtkEntry in the window that is commonly used to
     display messages. We need to set this pointer to window
     immediately because we'll need this entry when displaying
     messages, in particular when oligomers are to be displayed in the
     treeview.
   */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window),
		     "messages_entry", widget);


  /* Set the parent window pointer to the relative entry.
   */
  /* The window from which _this_ window was asked to setup has
     a datum that corresponds to the former's parent, which is the
     window in which the mass data were in the first place.
     See the polyxedit-ui-findmass-options.c file at the place where
     the options window is setup.
  */
  grand_parent = g_object_get_data (G_OBJECT (parent), "grand_parent");
  g_assert (grand_parent != NULL);
  
  
  widget = glade_xml_get_widget (xml, "results_set_identity_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "results_set_identity_number_entry", 
		     widget);
  
  help = g_strdup_printf ("%p", grand_parent);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);
  

  /* The oligomers' frame must be correctly set up:
   */
  widget = glade_xml_get_widget (xml, "findmass_results_oligomers_frame");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "findmass_results_oligomers_frame", widget);

  /* In the first frame we'll later setup a treeview with all the
     oligomers arranged depending on the findmass specification
     from which they originated.
  */
  widget = glade_xml_get_widget (xml, "findmass_results_oligomers_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "findmass_results_oligomers_vbox", widget);


  /* We have to fill the treeview with the oligomers passed to this 
     function in the alloligsGPA parameter. There is a function to 
     do just that:
  */
  if (alloligsGPA)
    {
      g_object_set_data (G_OBJECT (window), "alloligsGPA", alloligsGPA);
      
      if (FALSE == polyxedit_findmass_res_wnd_show_oligs (window,
							    alloligsGPA))
	return NULL;
    }
  
  /* Finally the button of the window.
   */
  widget = glade_xml_get_widget (xml, "findmass_results_wnd_close_button");
  g_object_set_data (G_OBJECT (window),
		     "findmass_results_wnd_close_button", widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK (polyxedit_findmass_results_wnd_close_button),  
     window);
  

  /* We don't need the GladeXML object any more, so unref it
     to save some memory 
   */
  g_object_unref (G_OBJECT (xml));

  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxedit_findmass_res_wnd_delete_event),
		    window);
  
  g_signal_connect (G_OBJECT (window),
		    "destroy_event",
		    G_CALLBACK (polyxedit_findmass_res_wnd_destroy_event),
		    window);

  g_free (gui_file);


  return window;
}



gboolean
polyxedit_findmass_res_wnd_show_oligs (GtkWidget *window,
					 GPtrArray *alloligsGPA)
{
  GtkWidget *vbox = NULL;
  GtkWidget *messages_entry = NULL;
  
  GtkTreeModel *model = NULL;

  gint *count = NULL;

  gchar *help;
  
  gboolean res = FALSE;
  
  
  g_assert (window != NULL);
  g_assert (alloligsGPA != NULL);
  
  vbox = g_object_get_data (G_OBJECT (window),
			    "findmass_results_oligomers_vbox");
  g_assert (vbox != NULL);
  
  res = polyxedit_findmass_res_setup_oligs_treeview (window, vbox);

  /* By now, model should have the number of oligomers in the 
     treeview's store.
  */
  model = (GtkTreeModel *) g_object_get_data (G_OBJECT (window), 
					      "oligomers_treeview_model");
  g_assert (model != NULL);
  
  count = g_object_get_data (G_OBJECT (model), "total_oligos");
  g_assert (count != NULL);
  
  messages_entry = g_object_get_data (G_OBJECT (window),  
				      "messages_entry");
  g_assert (messages_entry != NULL);

  help = g_strdup_printf (_("Total number of oligomers: %d"),
			  *count);
  
  gtk_entry_set_text (GTK_ENTRY (messages_entry),
		      help);
  g_free (help);
  
  return res;
}


GtkTreeModel *
polyxedit_findmass_res_create_oligs_treeview_model (GPtrArray *alloligsGPA)
{
  GtkTreeStore *model;
  GtkTreeIter treeiter;
  
  gint iter = 0;
  gint jter = 0;
  gint *count = NULL;
    
  gdouble error = 0;

  gchar *mass_type = NULL;
  gchar *coordinates = NULL;
    
  GPtrArray *oligGPA = NULL;
  
  PxmProp *prop = NULL;
  PxmFindmassOpt *fmopt = NULL;
  PxmOligomer *oligomer = NULL;

  

  g_assert (alloligsGPA != NULL);
  

  model = gtk_tree_store_new (COLUMN_OLIG_COL_COUNT,
			      /* The mass that was found. */
			      G_TYPE_DOUBLE,
			      /* Difference (found - found) */
			      G_TYPE_DOUBLE,
			      /* The name of the oligo ! */
			      G_TYPE_STRING,
			      /* The index + 1 in the array of oligos ! */
			      G_TYPE_INT,
			      /* MONO or AVG: the mass type of the find */
			      G_TYPE_STRING,
			      /* Mono mass */
			      G_TYPE_DOUBLE,
			      /* Average mass */
			      G_TYPE_DOUBLE,
			      /* Pointer to the oligomer */
			      G_TYPE_POINTER);
  
  count = g_malloc0 (sizeof (gint));

  /* Remember that the array that we get is an array of arrays.  This
     array contains an array for each group of oligomers that were
     found by finding a given mass.
  */
  for (iter = 0; iter < alloligsGPA->len; iter++)
    {
      oligGPA = g_ptr_array_index (alloligsGPA, iter);
      g_assert (oligGPA != NULL);
            
      /* And now iterate in the array and analyze each oligomer in it.
       */
      for (jter = 0 ; jter < oligGPA->len ; jter++)
	{
	  oligomer = g_ptr_array_index (oligGPA, jter);

	  g_assert (oligomer != NULL);
	  g_assert (oligomer->masspair != NULL);
	  
	  /* Get the PxmFindmassOpt instance as a prop object.
	   */
	  prop = libpolyxmass_prop_find_prop (oligomer->propGPA,
					  NULL,
					  NULL,
					  "FINDMASS_OPT",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
	  g_assert (prop != NULL);
	  
	  fmopt = prop->data;
	  g_assert (fmopt != NULL);

	  /* Now we can compute the error:
	   */
	  if (fmopt->mass_type == PXM_MASS_MONO)
	    {
	      error = oligomer->masspair->mono - fmopt->mass;

	      /* Take advantage to set the mass type string:
	       */
	      mass_type = g_strdup_printf ("MONO");
	    }
	  else /*  (fmopt->mass_type == PXM_MASS_AVG) */
	    {
	      error = oligomer->masspair->avg - fmopt->mass;

	      /* Take advantage to set the mass type string:
	       */
	      mass_type = g_strdup_printf ("AVG");
	    }
	  
	  gtk_tree_store_append (model, &treeiter, NULL);
	  gtk_tree_store_set 
	    (model, &treeiter,

	     /* The mass that was found. */
	     COLUMN_OLIG_TO_FIND_MASS, fmopt->mass,
	     /* Difference (found - found) */
	     COLUMN_OLIG_MASS_ERROR, error,
	     /* The name of the oligo ! */
	     COLUMN_OLIG_NAME, oligomer->name,
	     /* The absolute index + 1 in the arrays of oligos ! */
	     COLUMN_OLIG_NUMBER, (*count) + 1,
	     /* MONO or AVG: the mass type of the find */
	     COLUMN_OLIG_MASS_TYPE, mass_type,
	     /* Mono mass */
	     COLUMN_OLIG_MONO, oligomer->masspair->mono,
	     /* Average mass */
	     COLUMN_OLIG_AVG, oligomer->masspair->avg,
	     /* Pointer to the oligomer */
	     COLUMN_OLIG_POINTER, oligomer,

	     -1);

	  g_free (mass_type);
	  g_free (coordinates);
	  
	  /* Store the number of oligomer treated, all partial cleavages
	     included.
	  */
	  (*count)++;
	}
      /* end of 
	 for (jter = 0 ; jter < oligGPA->len ; jter++)
      */
    }
  /* end of 
     for (iter = 0; iter < GPA->len; iter++)
  */
  
  /* Before returning the model set to it the number of items we have
     set to it. The integer that was allocated will be automatically
     freed upon destruction of the model.
  */
  g_object_set_data_full (G_OBJECT (model), "total_oligos", count,
			  (GDestroyNotify) free);

  return GTK_TREE_MODEL (model);
}


gboolean
polyxedit_findmass_res_setup_oligs_treeview (GtkWidget *window, 
					     GtkWidget *vbox)
{
  GPtrArray *alloligsGPA = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;

  GtkWidget *sw = NULL;


  g_assert (window != NULL);
  g_assert (vbox != NULL);
  
  alloligsGPA = g_object_get_data (G_OBJECT (window), "alloligsGPA");
  g_assert (alloligsGPA != NULL);


  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  
  

  /* Create the treeview model.
   */
  model = polyxedit_findmass_res_create_oligs_treeview_model (alloligsGPA);


  /* Set to the model a datum with a pointer to alloligsGPA (the array
     of oligomer-containing arrays) and to fragoptGPA (the array of
     fragopt instances), so that the array of oligomers' array that
     has been used to fill the model is accessible later along with
     the corresponding fragopt instances stored in the array
     'fragoptGPA'. Also a pointer to the window!
   */
  g_object_set_data (G_OBJECT (model), "alloligsGPA", alloligsGPA);
  g_object_set_data (G_OBJECT (model), "window", window);

  /* And now set the window a datum with a pointer to the mode,
   * so that later the model is accessible (add/remove button
   * handlers).
   */
  g_object_set_data (G_OBJECT (window), 
		     "oligomers_treeview_model", model);
    
  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);
  
  g_object_unref (G_OBJECT (model));

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (window), "oligomers_treeview", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_MULTIPLE);

  /* The found mass column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes ("To Find",
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_TO_FIND_MASS,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_TO_FIND_MASS);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* The mass error column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes ("Error",
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_MASS_ERROR,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_MASS_ERROR);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes ("Name",
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_NAME,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_NAME);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer number column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes ("Number",
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_NUMBER,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_NUMBER);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer mass type column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes ("Mass Type",
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_MASS_TYPE,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_MASS_TYPE);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer mono mass column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes ("Mono Mass",
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_MONO,

					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_OLIG_MONO);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer avg mass column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes ("Avg Mass",
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_AVG,

					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_OLIG_AVG);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer pointer column.
   */
  renderer = gtk_cell_renderer_text_new ();
  
  column = 
    gtk_tree_view_column_new_with_attributes ("Oligo. Pointer",
					      
					      renderer,
					      
					      "text", 
					      COLUMN_OLIG_POINTER,
					      
					      NULL);
  
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_visible (column, FALSE),
  g_object_set (G_OBJECT (renderer), "visible", FALSE, NULL);

  /* Finally add the treeview to the scrooll window.
   */
  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);
  
  return TRUE;
}



void
polyxedit_findmass_results_wnd_close_button (GtkWidget *widget, 
					    gpointer data)
{
  GtkWidget *window = data;

  GPtrArray *alloligsGPA = NULL;
  
  g_assert (window != NULL);
  

  /* We have a full array of arrays of oligomers to free before
     leaving this window! Also we have to free the array of fragopt
     objects.
  */
  alloligsGPA = g_object_get_data (G_OBJECT (window), "alloligsGPA");
  g_assert (alloligsGPA != NULL);
  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);

  /* Now we can safely destroy this window.
   */
  gtk_object_destroy (GTK_OBJECT (window));

  return;
}




gboolean
polyxedit_findmass_res_wnd_delete_event (GtkWidget *widget,
					   GdkEvent *event,
					   gpointer data)
{
  GPtrArray *alloligsGPA = NULL;

  /* We have a full array of arrays of oligomers to free before
     leaving this window! Also we have to free the array of fragopt
     objects.
  */
  g_assert (widget != NULL);
  
  alloligsGPA = g_object_get_data (G_OBJECT (widget), "alloligsGPA");
  g_assert (alloligsGPA != NULL);
  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
  
  /* Let Gtk+ do the rest of the work.
   */
  return FALSE;
}




gboolean
polyxedit_findmass_res_wnd_destroy_event (GtkWidget *widget,
					   GdkEvent *event,
					    gpointer data)
{
  return FALSE;
}


