/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-cleave-results.h"
#include "polyxedit-ui-findmass-options.h"
#include "polyxedit-ui-seqed-wnd.h"


/* columns */

enum
{
  /* Partial cleavage */
  COLUMN_OLIG_PARTIAL = 0,
  /* Oligomer name, "p0-n4", for example */
  COLUMN_OLIG_NAME,
  /* Coordinates */
  COLUMN_OLIG_COORDINATES,
  /* Masses */
  COLUMN_OLIG_MONO,
  COLUMN_OLIG_AVG,
  /* Contains modified monomers*/
  COLUMN_OLIG_MODIFIED,
  /* Pointer to oligomer */
  COLUMN_OLIG_POINTER,

  COLUMN_OLIG_COL_COUNT
};

enum
{
  COLUMN_OLIG_MDF_MNM_POS = 0,
  COLUMN_OLIG_MDF_MNM_CODE,
  
  COLUMN_OLIG_MDF_LE_MODIF,
  COLUMN_OLIG_MDF_MNM_MODIF,
  COLUMN_OLIG_MDF_RE_MODIF,

  COLUMN_OLIG_MDF_COL_COUNT
};



GtkWidget *
polyxedit_cleave_res_wnd_setup (PxmEditCtxt *editctxt,
				    GPtrArray *partialsGPA)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;


  g_assert (editctxt != NULL);

  gui_file = 
    g_strdup_printf ("%s/polyxedit-cleave-results.glade", 
		     userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "cleave_results_wnd", 
		       PACKAGE);
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed loading the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "cleave_results_wnd");
  
  if (window == NULL)
    {
      g_error (_("%s@%d: failed creating the cleavage results window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* Immediately set to the window a pointer to the editctxt:
   */
  g_object_set_data (G_OBJECT (window), "editctxt", editctxt);

  /* There is a GtkEntry in the window that is commonly used to
     display messages. We need to set this pointer to window
     immediately because we'll need this entry when displaying
     messages, in particular when oligomers are to be displayed in the
     treeview.
   */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window),
		     "messages_entry", widget);


  /* Set the polymer sequence name to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polseq_name_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_name_entry", widget);
  
  if (editctxt->polymer->plminfo->name != NULL)
    gtk_entry_set_text (GTK_ENTRY (widget), 
			editctxt->polymer->plminfo->name);
  else 
    gtk_entry_set_text (GTK_ENTRY (widget), _("not set"));

  /* Set the polymer sequence context id number (its pointer) to its
     correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "identity_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),  "identity_number_entry", widget);

  /* Set the polymer id number (the pointer to editctxt to the
     relative entry).
   */
  help = g_strdup_printf ("%p", editctxt);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  /* We are going to handle the oligomers' data in two panes:
   */
  widget = glade_xml_get_widget (xml, "cleave_results_oligomers_vpaned");
    g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "cleave_results_oligomers_vpaned", widget);

  /* In this pane, we'll have two frames:
   */

  /* The oligomers' frame must be correctly set up:
   */
  widget = glade_xml_get_widget (xml, "cleave_results_oligomers_frame");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "cleave_results_oligomers_frame", widget);

  /* In the first frame we'll later setup a treeview with all the
     oligomers arranged depending on the partial cleavage from which
     they originated.
  */
  widget = glade_xml_get_widget (xml, "cleave_results_oligomers_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "cleave_results_oligomers_vbox", widget);


  /* We have to fill the treeview with the oligomers passed to this 
     function in the partialsGPA parameter. There is a function to 
     do just that:
  */
  if (partialsGPA != NULL)
    {
      g_object_set_data (G_OBJECT (window), "partialsGPA", partialsGPA);
      
      if (FALSE == polyxedit_cleave_res_wnd_show_oligs (window,
							partialsGPA))
	return NULL;
    }
  
  /* In the second frame we'll have to set some data pertaining to the
     currently selected oligomer in the oligomers's treeview and also
     all the information that we got about how the polymer was
     cleaved, how the masses were calculated and how the ionization
     was performed.
  */
  widget = glade_xml_get_widget (xml, 
				 "cleave_results_oligomers_data_frame");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "cleave_results_oligomers_data_frame", widget);

  /* The identity number of the results' set.
   */
  widget = glade_xml_get_widget (xml, 
				 "oligodata_results_identity_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "oligodata_results_identity_number_entry", widget);

  /* The number of oligomers in the results' set.
   */
  widget = glade_xml_get_widget (xml, 
				 "oligodata_oligomer_count_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "oligodata_oligomer_count_entry", widget);

  /* Cleaving agent.
   */
  widget = glade_xml_get_widget (xml, 
				 "oligodata_cleaving_agent_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "oligodata_cleaving_agent_entry", widget);

  /* Partial cleavages asked.
   */
  widget = glade_xml_get_widget (xml, 
				 "oligodata_partial_cleavages_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "oligodata_partial_cleavages_entry", widget);


  /* Polymer is left-capped?
   */
  widget = glade_xml_get_widget (xml, 
				 "left_capped_polymer_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "left_capped_polymer_checkbutton", widget);


  /* Polymer is right-capped?
   */
  widget = glade_xml_get_widget (xml, 
				 "right_capped_polymer_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "right_capped_polymer_checkbutton", widget);


  /* Left end modification of the polymer taken into account?
   */
  widget = glade_xml_get_widget (xml, 
				 "account_left_end_modif_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "account_left_end_modif_checkbutton", widget);


  /* Right end modification of the polymer taken into account?
   */
  widget = glade_xml_get_widget (xml, 
				 "account_right_end_modif_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "account_right_end_modif_checkbutton", widget);


  /* Monomer modifications in the oligomer taken into account?
   */
  widget = glade_xml_get_widget (xml, 
				 "account_monomer_modifs_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "account_monomer_modifs_checkbutton", widget);


  /* Ionizerule stuff.
   */
  widget = 
    glade_xml_get_widget (xml, 
			  "calc_mass_options_ionizerule_actform_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "calc_mass_options_ionizerule_actform_entry", widget);
  
  widget = 
    glade_xml_get_widget (xml, 
			  "calc_mass_options_ionizerule_charge_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "calc_mass_options_ionizerule_charge_entry", widget);
  
  widget = 
    glade_xml_get_widget (xml, 
			  "calc_mass_options_ionizerule_level_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "calc_mass_options_ionizerule_level_entry", widget);
  
  /* The textview where the sequence of a given oligomer are to be 
     displayed.
  */
  widget = 
    glade_xml_get_widget (xml, 
			  "oligomer_sequence_textview");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "oligomer_sequence_textview", widget);
  

  /* The vertical box where the treeview listing the modification
     objects found in any selected oligomers:
  */
  widget = 
    glade_xml_get_widget (xml, 
			  "oligodata_modifications_treeview_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "oligodata_modifications_treeview_vbox", widget);
  
  polyxedit_cleave_res_setup_oligomer_modifs_treeview (window,
						       widget);
  
  /* At this point, the treeview which will be hosting a description
     of the modified monomers in any given selected oligomer is there,
     but empty. When the user will select an oligomer from the
     treeview of oligomers, data will be displayed in this treeview
     only if at least one monomer in the oligomer is modified, or if
     the oligomer is either left-end or right-end modified.
  */

  /* Finally the buttons of the window.
   */
  widget = glade_xml_get_widget (xml, "cleave_results_wnd_close_button");
  g_object_set_data (G_OBJECT (window),
		     "cleave_results_wnd_close_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxedit_cleave_results_wnd_close_button),
		    window);

  widget = glade_xml_get_widget (xml, "cleave_results_wnd_find_button");
  g_object_set_data (G_OBJECT (window),
		     "cleave_results_wnd_find_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxedit_cleave_results_wnd_find_button),
		    window);


  /* We don't need the GladeXML object any more, so unref it
     to save some memory 
   */
  g_object_unref (G_OBJECT (xml));

  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxedit_cleave_res_wnd_delete_event),
		    window);
  
  g_signal_connect (G_OBJECT (window),
		    "destroy_event",
		    G_CALLBACK (polyxedit_cleave_res_wnd_destroy_event),
		    window);

  g_free (gui_file);


  return window;
}


void
polyxedit_cleave_res_wnd_update_cleavage_data (GtkWidget *window)
{
  PxmCleaveOpt *cleaveopt = NULL;
  PxmCalcOpt *calcopt = NULL;
  PxmIonizerule *ionizerule = NULL;
  
  GtkWidget *widget = NULL;
  
  gint *count = NULL;
  
  gchar *help = NULL;

  GtkTreeModel *model = NULL;


  g_assert (window != NULL);
  


  /* Get pointers to the objects that we will need later and which
     we set to 'window' in the caller function.
  */
  calcopt = g_object_get_data (G_OBJECT (window), "calcopt");
  g_assert (calcopt != NULL);

  cleaveopt = g_object_get_data (G_OBJECT (window), "cleaveopt");
  g_assert (cleaveopt != NULL);
  
  ionizerule = g_object_get_data (G_OBJECT (window), "ionizerule");
  g_assert (ionizerule != NULL);




  /* The results' set identity number (is actually the pointer to the
     window in which the results are displayed).
   */
  widget = g_object_get_data (G_OBJECT (window), 
				 "oligodata_results_identity_number_entry");
  g_assert (widget != NULL);

  help = g_strdup_printf ("%p", window);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);
  
  /* Number of oligomers that were found in the results' set.
   */
  /* By now, model should have the number of oligomers in the 
     treeview's store.
  */
  model = (GtkTreeModel *) g_object_get_data (G_OBJECT (window), 
					      "oligomers_treeview_model");
  g_assert (model != NULL);
  
  count = g_object_get_data (G_OBJECT (model), "total_oligos");
  g_assert (count != NULL);

  widget = g_object_get_data (G_OBJECT (window), 
				 "oligodata_oligomer_count_entry");
  g_assert (widget != NULL);

  help = g_strdup_printf ("%d", *count);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  /* Cleaving agent.
   */
  widget = g_object_get_data (G_OBJECT (window), 
				 "oligodata_cleaving_agent_entry");
  g_assert (widget != NULL);

  gtk_entry_set_text (GTK_ENTRY (widget), cleaveopt->cleavespec->name);

  /* Partial cleavages asked.
   */
  help = g_strdup_printf ("%d",  cleaveopt->partial);
  
  widget = g_object_get_data (G_OBJECT (window), 
				 "oligodata_partial_cleavages_entry");
  g_assert (widget != NULL);

  gtk_entry_set_text (GTK_ENTRY (widget), help);

  g_free (help);
  

  /* Polymer is left-capped?
   */
  widget = g_object_get_data (G_OBJECT (window), 
				 "left_capped_polymer_checkbutton");
  g_assert (widget != NULL);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget),
				calcopt->capping & PXM_CAPPING_LEFT);


  /* Polymer is right-capped?
   */
  widget = g_object_get_data (G_OBJECT (window), 
				 "right_capped_polymer_checkbutton");
  g_assert (widget != NULL);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget),
				calcopt->capping & PXM_CAPPING_RIGHT);


  /* Left end modification of the polymer taken into account?
   */
  widget = g_object_get_data (G_OBJECT (window), 
				 "account_left_end_modif_checkbutton");
  g_assert (widget != NULL);

  gtk_toggle_button_set_active 
    (GTK_TOGGLE_BUTTON (widget),
     calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF);


  /* Right end modification of the polymer taken into account?
   */
  widget = g_object_get_data (G_OBJECT (window), 
				 "account_right_end_modif_checkbutton");
  g_assert (widget != NULL);

  gtk_toggle_button_set_active 
    (GTK_TOGGLE_BUTTON (widget),
     calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF);

  /* Monomer modifications in the oligomer taken into account?
   */
  widget = g_object_get_data (G_OBJECT (window), 
			      "account_monomer_modifs_checkbutton");
  g_assert (widget != NULL);
  
  gtk_toggle_button_set_active 
    (GTK_TOGGLE_BUTTON (widget), calcopt->mnm_chement);



  /* Ionizerule stuff.
   */
  widget = 
    g_object_get_data (G_OBJECT (window), 
			  "calc_mass_options_ionizerule_actform_entry");
  g_assert (widget != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), ionizerule->actform);

  help = g_strdup_printf ("%d", ionizerule->charge);
  
  widget = 
    g_object_get_data (G_OBJECT (window), 
			  "calc_mass_options_ionizerule_charge_entry");
  g_assert (widget != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), help);

  g_free (help);
  

  help = g_strdup_printf ("%d", ionizerule->level);
  
  widget = 
    g_object_get_data (G_OBJECT (window), 
			  "calc_mass_options_ionizerule_level_entry");
  g_assert (widget != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), help);

  g_free (help);

  return;
}



gboolean
polyxedit_cleave_res_wnd_show_oligs (GtkWidget *window,
				     GPtrArray *partialsGPA)
{
  GtkWidget *vbox = NULL;
  
  GtkTreeModel *model = NULL;

  gint *count = NULL;

  gchar *help;
  
  gboolean res = FALSE;
  
  
  g_assert (window != NULL);
  g_assert (partialsGPA != NULL);
  
  vbox = g_object_get_data (G_OBJECT (window),
			    "cleave_results_oligomers_vbox");
  g_assert (vbox != NULL);
  
  res = polyxedit_cleave_res_setup_oligs_treeview (window, 
							   vbox);

  /* By now, model should have the number of oligomers in the 
     treeview's store.
  */
  model = (GtkTreeModel *) g_object_get_data (G_OBJECT (window), 
					      "oligomers_treeview_model");
  g_assert (model != NULL);
  
  count = g_object_get_data (G_OBJECT (model), "total_oligos");
  g_assert (count != NULL);
  
  help = g_strdup_printf (_("Total number of oligomers: %d"),
			  *count);
  
  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					 help, 
					 POLYXMASS_LONG_MSG_TIMEOUT);
  
  g_free (help);
  
  return res;
}


GtkTreeModel *
polyxedit_cleave_res_create_oligs_treeview_model (GPtrArray *GPA)
{
  GtkTreeStore *model;
  GtkTreeIter treeiter;

  gint iter = 0;
  gint jter = 0;
  gint *count = NULL;
  
  gchar *coordinates = NULL;

  gboolean modified = FALSE;
  
  GPtrArray *oligGPA = NULL;
  
  PxmProp *prop = NULL;
  PxmOligomer *oligomer = NULL;


  g_assert (GPA != NULL);
  

  model = gtk_tree_store_new (COLUMN_OLIG_COL_COUNT,
			      /* Partial cleavage */
			      G_TYPE_INT,
			      /* Oligo name */
			      G_TYPE_STRING,
			      /* Coordinates */
			      G_TYPE_STRING,
			      /* Masses */
			      G_TYPE_DOUBLE,
			      G_TYPE_DOUBLE,
			      /* Modified */
			      G_TYPE_BOOLEAN,
			      /* Pointer to oligomer */
			      G_TYPE_POINTER);

  count = g_malloc0 (sizeof (gint));

  /* Remember that the array that we get is an array of arrays.  This
     array contains an array for each group of partial oligomers. Note
     that if at least one oligomer is created, as the result of
     cleaving a polymer sequence, it must be of partial cleavage 0,
     because that partial cleavage is the default one.
  */
  for (iter = 0; iter < GPA->len; iter++)
    {
      oligGPA = g_ptr_array_index (GPA, iter);
      g_assert (oligGPA != NULL);
      
      /* And now iterate in the array and analyze each oligomer in it.
       */
      for (jter = 0 ; jter < oligGPA->len ; jter++)
	{
	  oligomer = g_ptr_array_index (oligGPA, jter);

	  g_assert (oligomer != NULL);
	  g_assert (oligomer->masspair != NULL);
	  
	  /* Now synthesize all the strings that we are going to need
	     to display data in the treeview row. We'll free all that
	     stuff when the display is finished.f
	  */
	  coordinates = g_strdup_printf ("[%d-%d]",
					 oligomer->start_idx + 1,
					 oligomer->end_idx + 1);

	  /* See if the oligomer contains a modification... maybe not.
	   */
	  prop = libpolyxmass_prop_find_prop (oligomer->propGPA,
					  NULL,
					  NULL,
					  "POS/IDX/MODIF",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
	  if (prop != NULL)
	    modified = TRUE;
	  else
	    modified = FALSE;


	  gtk_tree_store_append (model, &treeiter, NULL);
	  gtk_tree_store_set 
	    (model, &treeiter,

	     COLUMN_OLIG_PARTIAL, iter,

	     COLUMN_OLIG_NAME, oligomer->name,
	     
	     COLUMN_OLIG_COORDINATES, coordinates,
	     
	     COLUMN_OLIG_MONO, oligomer->masspair->mono,
	     
	     COLUMN_OLIG_AVG, oligomer->masspair->avg,
	     
	     COLUMN_OLIG_MODIFIED, modified,

	     COLUMN_OLIG_POINTER, oligomer,
	     -1);
	  
#if 0
	  /* Generate a latex string so that printing is feasible
	   */
	  {
	    GString *printgs = NULL;
	    
	    printgs = g_string_new ("");
	    g_string_append_printf (printgs, "%d & %s & %s & "
				    "%lf & %lf & %d \\\\\n", 
				    iter, oligomer->name, coordinates,
				    oligomer->masspair->mono,
				    oligomer->masspair->avg,
				    modified);
	    printf (printgs->str);
	    
	    printgs = g_string_truncate (printgs, 0);
	  }
#endif  

	  /* Now that the data have been set to the store, we can free
	     the string that were all allocated.
	  */
	  g_free (coordinates);

	  /* Store the number of oligomer treated, all partial cleavages
	     included.
	  */
	  (*count)++;
	}
      /* end of 
	 for (jter = 0 ; jter < oligGPA->len ; jter++)
      */
    }
  /* end of 
     for (iter = 0; iter < GPA->len; iter++)
  */
  
  /* Before returning the model set to it the number of items we have
     set to it. The integer that was allocated will be automatically
     freed upon destruction of the model.
  */
  g_object_set_data_full (G_OBJECT (model), "total_oligos", count,
			  (GDestroyNotify) g_free);
  
  return GTK_TREE_MODEL (model);
}


gboolean
polyxedit_cleave_res_setup_oligs_treeview (GtkWidget *window, 
					     GtkWidget *vbox)
{
  PxmEditCtxt *editctxt = NULL;
  
  GPtrArray *GPA = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;
  GtkTreeSelection* selection = NULL;

  GtkWidget *sw = NULL;

  
  

  g_assert (window != NULL);
  g_assert (vbox != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);
  
  GPA = g_object_get_data (G_OBJECT (window), "partialsGPA");
  g_assert (GPA != NULL);
  

  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  
  

  /* Create the treeview model.
   */
  model = polyxedit_cleave_res_create_oligs_treeview_model (GPA);


  /* Set to the model a datum with a pointer to GPA, so that the array
   * of oligomers' array that has been used to fill the model is
   * accessible later. Also a pointer to the window!
   */
  g_object_set_data (G_OBJECT (model), "partialsGPA", GPA);
  g_object_set_data (G_OBJECT (model), "window", window);

  /* And now set the window a datum with a pointer to the mode,
   * so that later the model is accessible (add/remove button
   * handlers).
   */
  g_object_set_data (G_OBJECT (window), 
		     "oligomers_treeview_model", model);
  
  
  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);

  g_object_unref (G_OBJECT (model));

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
  gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (window), "oligomers_treeview", treeview);


  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  gtk_tree_selection_set_mode (selection,
			       GTK_SELECTION_MULTIPLE);

  /* "changed" is emitted by the selection object in the treeview each
     time the selection changes.
   */

  g_signal_connect (G_OBJECT (selection), "changed", 
		    G_CALLBACK 
		    (polyxedit_cleave_res_oligs_treeview_selection_changed),
		    window);
  
  /* "row-activated" is emitted when a row is
     double-clicked. (space-bar makes the same after the row is
     highlighted).
   */
  g_signal_connect (G_OBJECT (treeview), "row-activated", 
		    G_CALLBACK 
		    (polyxedit_cleave_res_oligs_treeview_row_activated),
		    window);
  

  /* Partial cleavage number column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_OLIG_PARTIAL);

  column = 
    gtk_tree_view_column_new_with_attributes ("Part. Cleav.",

						 renderer,

						 "text", 
						 COLUMN_OLIG_PARTIAL,
						 
						 NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_PARTIAL);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer number column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column",
		      (gint *) COLUMN_OLIG_NAME);

  column = 
    gtk_tree_view_column_new_with_attributes ("Number",

						 renderer,

						 "text", 
						 COLUMN_OLIG_NAME,
						 
						 NULL);

  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_NAME);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer coordinates column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_OLIG_COORDINATES);

  column = 
    gtk_tree_view_column_new_with_attributes ("Coordinates",
					      
					      renderer,
					      
					      "text", 
					      COLUMN_OLIG_COORDINATES,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_COORDINATES);
  
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  gtk_tree_sortable_set_sort_func 
    (GTK_TREE_SORTABLE (model),
     COLUMN_OLIG_COORDINATES,
     polyxedit_cleave_res_wnd_sort_coordinates_column,
     NULL,
     NULL);

  /* Oligomer mono mass column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_OLIG_MONO);

  column = 
    gtk_tree_view_column_new_with_attributes ("Mono Mass",

						 renderer,

						 "text", 
						 COLUMN_OLIG_MONO,

						 NULL);
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_MONO);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  gtk_tree_view_column_set_sort_column_id (column, COLUMN_OLIG_MONO);

  /* Oligomer avg mass column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_OLIG_AVG);

  column = 
    gtk_tree_view_column_new_with_attributes ("Avg Mass",

						 renderer,
						 
						 "text", 
						 COLUMN_OLIG_AVG,
						 
						 NULL);  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_AVG);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  gtk_tree_view_column_set_sort_column_id (column, COLUMN_OLIG_AVG);

  /* Oligomer modified column.
   */
  renderer = gtk_cell_renderer_toggle_new ();
  
  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_OLIG_MODIFIED);

  column = 
    gtk_tree_view_column_new_with_attributes ("Modified",

						 renderer,

						 "active", 
						 COLUMN_OLIG_MODIFIED,

						 NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer pointer column.
   */
  renderer = gtk_cell_renderer_text_new ();
  
  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_OLIG_POINTER);

  column = 
    gtk_tree_view_column_new_with_attributes ("Oligo. Pointer",

						 renderer,

						 "text", 
						 COLUMN_OLIG_POINTER,

						 NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_visible (column, FALSE),

  /* Finally add the treeview to the scrooll window.
   */
  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);
  
  return TRUE;
}


gboolean
polyxedit_cleave_res_setup_oligomer_modifs_treeview (GtkWidget *window, 
						     GtkWidget *vbox)
{
  GtkWidget *sw = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;
  GtkTreeSelection* selection = NULL;


  g_assert (window != NULL);
  g_assert (vbox != NULL);
  
  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  
  

  /* Create the treeview model.
   */
  model = 
    (GtkTreeModel *) gtk_tree_store_new (COLUMN_OLIG_MDF_COL_COUNT,
					 G_TYPE_INT, /* position */
					 G_TYPE_STRING, /* code */
					 G_TYPE_STRING, /* modif */
					 G_TYPE_STRING, /* LE modif */
					 G_TYPE_STRING); /* RE modif */
  
  g_object_set_data (G_OBJECT (model), "window", window);

  /* And now set the window a datum with a pointer to the mode,
   * so that later the model is accessible (add/remove button
   * handlers).
   */
  g_object_set_data (G_OBJECT (window), 
		     "oligomers_modifs_treeview_model", model);
  
  
  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);
  
  g_object_unref (G_OBJECT (model));

  /* Set alternating color shades for the treeview rows to ease reading
     them.
  */
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (window), "oligomers_modifs_treeview", 
		     treeview);
  
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  gtk_tree_selection_set_mode (selection,
			       GTK_SELECTION_MULTIPLE);

  /* Position at which the monomer in the oligomer is modified.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_OLIG_MDF_MNM_POS);

  column = 
    gtk_tree_view_column_new_with_attributes ("Position",

						 renderer,

						 "text",
						 COLUMN_OLIG_MDF_MNM_POS,

						 NULL);
  
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  
  /* Monomer code column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_OLIG_MDF_MNM_CODE);

  column = 
    gtk_tree_view_column_new_with_attributes ("Code",

						 renderer, 

						 "text",
						 COLUMN_OLIG_MDF_MNM_CODE,

						 NULL);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Monomer modifications name name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_OLIG_MDF_MNM_MODIF);

  column = 
    gtk_tree_view_column_new_with_attributes ("Monomer Modif.",

						 renderer, 

						 "text",
						 COLUMN_OLIG_MDF_MNM_MODIF,

						 NULL);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Left end modif column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *)COLUMN_OLIG_MDF_LE_MODIF );

  column = 
    gtk_tree_view_column_new_with_attributes ("Left End Modif.",

						 renderer, 

						 "text",
						 COLUMN_OLIG_MDF_LE_MODIF,

						 NULL);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Right end modif column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_OLIG_MDF_RE_MODIF);

  column = 
    gtk_tree_view_column_new_with_attributes ("Right End Modif.",

						 renderer, 

						 "text",
						 COLUMN_OLIG_MDF_RE_MODIF,

						 NULL);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);
  
  return TRUE;
}


void
polyxedit_cleave_res_olig_show_modification_data (PxmOligomer *oligomer,
						  GtkWidget *window)
{
  gint index = 0;
  gint offset = 0;
  gint idx = 0;
  
  gchar *modif = NULL;
  
  PxmMonomer *monomer = NULL;
  
  PxmProp *prop = NULL;
  
  GtkWidget *treeview = NULL;
  GtkTreeStore *model = NULL;
  GtkTreeIter treeiter;

  gboolean valid = FALSE;
  
  
  g_assert (window != NULL);
  
  treeview = g_object_get_data (G_OBJECT (window), 
				"oligomers_modifs_treeview");
  g_assert (treeview != NULL);
  
  model = 
    (GtkTreeStore *) gtk_tree_view_get_model ((GtkTreeView *) treeview);
  g_assert (model != NULL);
  

  gtk_tree_store_clear (model);

  /* If 'oligomer' is NULL, that means that all we want is to clear
     the store, and return without bothering, since that means that
     there is no currently selected oligomer the data of which to be
     displayed here.
  */
  if (oligomer == NULL)
    return;
  

  /* Now that we have removed all the items from the treeview, 
     we can start repopulating it.
  */
  valid = gtk_tree_model_get_iter_first ((GtkTreeModel *) model, &treeiter);

  g_assert (valid == FALSE); /* the tree has to be empty by now.*/
  
  /* First of all we check if there is a left end modif, so that it
     is listed first in the treeview.
  */
  prop = libpolyxmass_prop_find_prop (oligomer->propGPA,
				  NULL,
				  NULL,
				  "LEFT_END_MODIF",
				  NULL,
				  PXM_UNDEF_PROP_CMP);
  if (prop != NULL)
    {
      gtk_tree_store_append (model, &treeiter, NULL);

      gtk_tree_store_set 
	(model, &treeiter,

	 COLUMN_OLIG_MDF_LE_MODIF, prop->data,

	 -1);
    }

  /* Second we check if there is are monomer modifications in the
     oligomer. These are thus listed between the left end modif and
     the right end modif.
  */
  prop = libpolyxmass_prop_find_prop (oligomer->propGPA,
				  &idx,
				  NULL,
				  "POS/IDX/MODIF",
				  NULL,
				  PXM_UNDEF_PROP_CMP);

  /* Increment the index at which the next prop finding opearation has
     to be performed, otherwise we do not move in the
     oligomer->propGPA, and we enter an infinite loop.
  */
  idx++;
  
  while (prop != NULL)
    {
      /* Allocate a string long enough to accomodate the name 
	 of the modification.
      */
      modif = g_malloc0 (sizeof (gchar) + strlen (prop->data));
      
      /* The string that we get from the prop's data is something like
	 "10/25/Acetylation", where 10 is the offset (pos and not index)
	 with respect to the oligomer itself, while 25 is the index of
	 the oligomer's start in the polymer whence it came.
      */
      sscanf (prop->data, "%d/%d/%s", &offset, &index, modif);

      monomer = g_ptr_array_index (oligomer->polymer->monomerGPA,
				   index);
      g_assert (monomer != NULL);
      
      /*
	debug_printf (("the index is %d and the modif name is %s\n", 
	index, modif));
      */

      /* At this point we can store the property in the treeview.
       */
      gtk_tree_store_append (model, &treeiter, NULL);

      gtk_tree_store_set 
	(model, &treeiter,

	 COLUMN_OLIG_MDF_MNM_POS, offset,

	 COLUMN_OLIG_MDF_MNM_CODE, monomer->code,
	 COLUMN_OLIG_MDF_MNM_MODIF, modif,

	 -1);
      
      g_free (modif);
      
      
      prop = libpolyxmass_prop_find_prop (oligomer->propGPA,
				      &idx,
				      NULL,
				      "POS/IDX/MODIF",
				      NULL,
				      PXM_UNDEF_PROP_CMP);
      /* Increment the index at which the next prop finding opearation
	 has to be performed, otherwise we do not move in the
	 oligomer->propGPA, and we enter an infinite loop.
      */
      idx++;
    }

  /* Finally, we check if there is a right end modif, and this is going
     to be listed last.
  */
  prop = libpolyxmass_prop_find_prop (oligomer->propGPA,
				  NULL,
				  NULL,
				  "RIGHT_END_MODIF",
				  NULL,
				  PXM_UNDEF_PROP_CMP);
  if (prop != NULL)
    {
      gtk_tree_store_append (model, &treeiter, NULL);

      gtk_tree_store_set 
	(model, &treeiter,

	 COLUMN_OLIG_MDF_RE_MODIF, prop->data,

	 -1);
    }
  
  return;
}




void
polyxedit_cleave_res_oligs_treeview_selection_changed (GtkTreeSelection 
						       *selection,
						       gpointer data)
{
  GtkWindow * window = data;
  GtkWidget *messages_entry = data;
  
  GtkTreeModel *model = NULL;
  GtkTreeView *treeview;

  GtkTreeIter treeiter;

  PxmOligomer *oligomer = NULL;
  
  gboolean valid = FALSE;

  gboolean selected = FALSE;

  gint sel_count = 0;
  
  g_assert (selection != NULL);
  g_assert (window != NULL);
  

  /* We certainly will have to display some messages.
   */
  messages_entry = g_object_get_data (G_OBJECT (window),  
				       "messages_entry");
  g_assert (messages_entry != NULL);


  /* The selection object has a pointer to its related treeview.
   */
  treeview = gtk_tree_selection_get_tree_view (selection);
  
  /* Get the model from the treeview, so that we can iterate later
     in the treeview.
  */
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);


  /* Get the first treeiter in the list.
   */
  valid = gtk_tree_model_get_iter_first (model, &treeiter);

  while (valid == TRUE)
    {
      /* Walk through the treeview list of oligomers, reading each
	 row, only if actually selected.
      */
      selected = gtk_tree_selection_iter_is_selected (selection, 
						      &treeiter);
	  
      if (selected != TRUE)
	{
	  /* Go the the next treeview item.
	   */
	  valid = 
	    gtk_tree_model_iter_next (model, &treeiter);
	      
	  continue;
	}
      else
	sel_count++;
	  	  
      /* If there are more than one item selected, we cannot display
	 the data, because: what oligomer choose?
      */
      if (sel_count > 1)
	return;
	  
      /* The row is selected, thus we can ask that the contents of
	 the CODE column be allocated in 'code'. Make sure to
	 terminate the call to gtk_tree_model_get() * with a '-1'
	 value.
      */
      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_OLIG_POINTER, &oligomer,
			  -1);
	  
      valid = gtk_tree_model_iter_next (model, &treeiter);
    }
      
  /* Note that passing a NULL oligomer to the function below will cause
     the data of the previously selected oligomer to be erase, which
     is exactly what we want.
  */

  /* We now have the address of the sole selected oligomer,
     which means that we can display its data in the 
     oligodata pane widgets.
  */
  polyxedit_cleave_res_olig_show_data (oligomer,
				       GTK_WIDGET (window));
  
  /* 
     debug_printf (("the selected oligomer: [%d-%d]\n",
     oligomer->start_idx + 1, oligomer->end_idx + 1));
  */  
  return;
}


void
polyxedit_cleave_res_oligs_treeview_row_activated (GtkTreeView *treeview,
						   GtkTreePath *arg1,
						   GtkTreeViewColumn *arg2,
						   gpointer data)
{
  PxmEditCtxt *editctxt = NULL;

  PxmOligomer *oligomer = NULL;
  
  GtkWidget *window = data;

  GtkTreeIter treeiter;
  GtkTreeModel *model;

  
  
  /* We would like to select the corresponding sequence in the polymer
     sequence editor. We should get the start_idx and the end_idx for
     the current oligomer. Then we should make sure that these values
     are still OK for the currently displayed sequence (what if the
     user edited -by removing monomers- the polymer sequence between
     obtaining the oligomers and activating a row ? We might well be
     trying to access monomers out of the bounds of the polymer
     sequence array. Finally, we can draw the selection polygon and
     perform a X selection with its contents.
  */
  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");  
  g_assert (editctxt != NULL);
  
  /* We now should get the activated item.
   */

  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);
  
  if (FALSE == gtk_tree_model_get_iter (model,
					&treeiter,
					arg1))
    return;
  
  /* This point we have the item iter that was double-clicked in the
     treeiter variable. Get the oligomer pointer.
  */
  gtk_tree_model_get (model, &treeiter, 
		      COLUMN_OLIG_POINTER, &oligomer,
		      -1);

  polyxedit_seqed_wnd_select_sequence (editctxt,
				       oligomer->start_idx,
				       oligomer->end_idx);
  
  return;
}

void   
polyxedit_cleave_res_oligs_treeview_column_clicked (GtkTreeViewColumn *
						    treeviewcolumn,
						    gpointer data)
{
  debug_printf (("polyxedit_cleave_res_oligs_treeview_column_clicked\n"));
  
  return;
}


void
polyxedit_cleave_res_olig_show_data (PxmOligomer *oligomer,
				     GtkWidget *window)
{
  GtkTextView *textview = NULL;
  GtkTextIter text_iter_start;
  GtkTextIter text_iter_end;
  GtkTextBuffer *buffer = NULL;


  PxmProp * prop = NULL;
  
  g_assert (window != NULL);
  
  textview = g_object_get_data (G_OBJECT (window),
				"oligomer_sequence_textview");
  g_assert (textview != NULL);
  
  
  buffer = gtk_text_view_get_buffer (textview);
      


  /* If the 'oligomer' is NULL, that means that we must clear 
     the previously selected oligomer's data (if any), since that means
     that no oligomer is being selected in the treeview by now.
  */

  /* This treatment is performed whatever the 'oligomer' value. Get
     start of buffer and its end. Remove all text contained between
     these borders (which is all).
   */
  gtk_text_buffer_get_iter_at_offset (buffer, &text_iter_start, 0);
  gtk_text_buffer_get_iter_at_offset (buffer, &text_iter_end, -1);
  
  gtk_text_buffer_delete (buffer, &text_iter_start, &text_iter_end);
  

  /* Now the treatement depends on the 'oligomer' value.
   */
  if (oligomer != NULL)
    {
      /* We have to display a number of data for the current oligomer.
       */
      prop = libpolyxmass_prop_find_prop (oligomer->propGPA,
				      NULL,
				      NULL,
				      "SEQUENCE",
				      NULL,
				      PXM_UNDEF_PROP_CMP);
      
      if (prop != NULL)
	{
	  /* We want to display the sequence in the textview widget.
	   */
	  
	  
	  
	  gtk_text_buffer_insert (buffer, &text_iter_start, 
				  (gchar *) prop->data, -1);
	}
    }
  
  /* We also want to display the modifications that are encountered in
     the oligomer at the level of the monomers and of the left/right
     ends. We get a vbox in which we have to set up the treeview in
     which the data are to be displayed.
  */
  
  polyxedit_cleave_res_olig_show_modification_data (oligomer,
						    GTK_WIDGET (window));
  
  
  return;
}


gint
polyxedit_cleave_results_wnd_sort_mass_column (GtkTreeModel *model,
					      GtkTreeIter *a,
					      GtkTreeIter *b,
					      gpointer data)
{
  gint column = *((gint *) data);
  
  gchar *a_string = NULL;
  gchar *b_string = NULL;
  
  gdouble a_double = 0;
  gdouble b_double = 0;
  


  gtk_tree_model_get (model, a, 
		      column, a_string,
		      -1);
  
  gtk_tree_model_get (model, b, 
		      column, b_string,
		      -1);

  g_assert (a_string != NULL);
  g_assert (b_string != NULL);
  
  libpolyxmass_globals_strtod (a_string, &a_double);
  libpolyxmass_globals_strtod (b_string, &b_double);
  
  /*
    debug_printf (("a_double = %.5f - b_double = %.5f\n", 
    a_double, b_double));
  */
  
  g_free (a_string);
  g_free (b_string);
  
  if (a_double < b_double)
    return -1;
  else if (a_double > b_double)
    return 1;
  return 0; /* Values are identical */
}



void
polyxedit_cleave_results_wnd_close_button (GtkWidget *widget, gpointer data)
{
  GtkWidget *window = data;

  GPtrArray *GPA = NULL;
  
  g_assert (window != NULL);
  

  /* We have a full array of arrays of oligomers to free before
     leaving this window!
  */
  GPA = g_object_get_data (G_OBJECT (window), "partialsGPA");
  g_assert (GPA != NULL);
  
  pxmchem_oligomer_GPA_of_GPA_free (GPA);
  
  /* Now we can safely destroy this window.
   */
  gtk_object_destroy (GTK_OBJECT (window));
  
  return;
}


void
polyxedit_cleave_results_wnd_find_button (GtkWidget *widget, gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *window_new = NULL;
  
  GPtrArray *GPA = NULL;
  
  g_assert (window != NULL);
  

  /* The user wants to perform a mass finding operation in the oligomer
     data that are displayed in this results window. We must set
     up a window in which the user will enter the mass(es) to find.
  */

  /* We have a full array of arrays of oligomers to process in order
     find the mass data!
  */
  GPA = g_object_get_data (G_OBJECT (window), "partialsGPA");
  g_assert (GPA != NULL);
  
  window_new = polyxedit_findmass_opt_setup_wnd (window, GPA);
  
  if (window_new == NULL)
    {
      g_critical (_("%s@%d: failed setting up the find mass options window\n"),
	     __FILE__, __LINE__);
    }
  
  return;
}


gint
polyxedit_cleave_res_wnd_sort_coordinates_column (GtkTreeModel *model,
						  GtkTreeIter *a,
						  GtkTreeIter *b,
						  gpointer user_data)
{
  
  gchar *coords_a = NULL;
  gchar *coords_b = NULL;
  
  gint start_a = -1;
  gint end_a = -1;
  
  gint start_b = -1;
  gint end_b = -1;
  
  

  /* We get the pointer to two different items in the treeview (the
     model actually) and we need to access the data in them and then
     tell which is greater to the other.
  */
  gtk_tree_model_get (model, a, 
		      COLUMN_OLIG_COORDINATES, &coords_a,
		      -1);

  libpolyxmass_globals_parse_range_coordinates (coords_a,
						&start_a, &end_a);
  

  gtk_tree_model_get (model, b, 
		      COLUMN_OLIG_COORDINATES, &coords_b,
		      -1);

  libpolyxmass_globals_parse_range_coordinates (coords_b,
						&start_b, &end_b);
  
  /*
    debug_printf (("start_a is %d and end_a is %d \n", start_a, end_a));
    debug_printf (("start_b is %d and end_b is %d \n", start_b, end_b));
  */

  if (start_a == start_b)
    return 0;
  
  return (start_a > start_b) ? 1 : -1;
}


gboolean
polyxedit_cleave_res_wnd_delete_event (GtkWidget *window,
				       GdkEvent *event,
				       gpointer data)
{
  gint count = 0;
  GPtrArray *GPA = NULL;

  /* We have a full array of arrays of oligomers to free before
     leaving this window!
  */
  g_assert (window != NULL);
  
  GPA = g_object_get_data (G_OBJECT (window), "partialsGPA");
  g_assert (GPA != NULL);
  
  pxmchem_oligomer_GPA_of_GPA_free (GPA);
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  count = polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  /*
    printf ("polyxedit_elemcompos_wnd_delete_event, count is %d\n",
    count);
  */

  /* Let Gtk+ do the rest of the work.
   */
  return FALSE;
}




gboolean
polyxedit_cleave_res_wnd_destroy_event (GtkWidget *window,
					   GdkEvent *event,
					    gpointer data)
{
  return FALSE;
}


