/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-cleave-options.h"
#include "polyxedit-ui-cleave-results.h"



GtkWidget *
polyxedit_cleave_opt_wnd_setup (PxmEditCtxt *editctxt)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;
  

  g_assert (editctxt != NULL);

  gui_file = 
    g_strdup_printf ("%s/polyxedit-cleave-options.glade", 
		     userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "cleave_options_wnd", 
		       PACKAGE);
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed loading the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "cleave_options_wnd");
  
  if (window == NULL)
    {
      g_error (_("%s@%d: failed creating the cleavage options window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* Immediately set to the window a pointer to the editctxt:
   */
  g_object_set_data (G_OBJECT (window), "editctxt", editctxt);


  /* Set the polymer sequence name to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polseq_name_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_name_entry", widget);
  
  if (editctxt->polymer->plminfo->name != NULL)
    gtk_entry_set_text (GTK_ENTRY (widget), 
			editctxt->polymer->plminfo->name);
  else 
    gtk_entry_set_text (GTK_ENTRY (widget), _("not set"));

  /* Set the polymer sequence context id number (its pointer) to its
     correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "identity_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),  "identity_number_entry", widget);

  /* Set the polymer id number (the pointer to editctxt to the
     relative entry).
   */
  help = g_strdup_printf ("%p", editctxt);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);


  widget = glade_xml_get_widget (xml, "partial_cleavages_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),  "partial_cleavages_entry", widget);


  /* The cleavespecs' frame must be correctly set up:
   */
  widget = glade_xml_get_widget (xml, "cleavespecs_available_vbox");
  g_object_set_data (G_OBJECT (window),
		     "cleavespecs_available_vbox", widget);

  if (FALSE == 
      polyxedit_cleave_opt_setup_cls_treeview (window, widget))
    {
      g_error (_("%s@%d: failed setting up the cleavespecs' material\n"),
	     __FILE__, __LINE__);
      
      g_object_unref (G_OBJECT (xml));
      g_object_unref (G_OBJECT (window));
      g_free (gui_file);

      return NULL;
    }
  

  /* There is a GtkEntry in the window that is commonly used to 
   * display messages.
   */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window),
		     "messages_entry", widget);


  widget = glade_xml_get_widget (xml, "cleave_option_wnd_cleave_button");
  g_object_set_data (G_OBJECT (window),
		     "cleave_option_wnd_cleave_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxedit_cleave_opt_wnd_cleave_button),
		    window);


  /* We don't need the GladeXML object any more, so unref it
   * to save some memory 
   */
  g_object_unref (G_OBJECT (xml));

  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxedit_cleave_opt_wnd_delete_event),
		    window);
  
  g_signal_connect (G_OBJECT (window),
		    "destroy_event",
		    G_CALLBACK (polyxedit_cleave_opt_wnd_destroy_event),
		    window);

  g_free (gui_file);



  return window;
}



gboolean
polyxedit_cleave_opt_setup_cls_treeview (GtkWidget *window, 
					     GtkWidget *vbox)
{
  PxmEditCtxt *editctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;
  
  GPtrArray *GPA = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;

  GtkWidget *sw = NULL;

  gint col_offset;


  g_assert (vbox != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");


  polchemdef = editctxt->polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  GPA = polchemdef->cleavespecGPA;
  g_assert (GPA != NULL);
  

  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  
  

  /* Create the treeview model.
   */
  model = polyxedit_cleave_opt_create_cls_treeview_model (GPA);


  /* Set to the model a datum with a pointer to GPA, so that
   * the array of cleavespecs that has been used to fill the model
   * is accessible later. Also a pointer to the window!
   */
  g_object_set_data (G_OBJECT (model), "cleavespec_GPA", GPA);
  g_object_set_data (G_OBJECT (model), "window", window);

  /* And now set the window a datum with a pointer to the mode,
   * so that later the model is accessible (add/remove button
   * handlers).
   */
  g_object_set_data (G_OBJECT (window), 
		     "cleavespec_treeview_model", model);
  
  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);

  g_object_unref (G_OBJECT (model));

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (window), "cleavespec_treeview", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);
  
  /* Cleavespec name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_CLS_NAME);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, "Name",
						 renderer, 
						 "text",
						 
						 COLUMN_CLS_NAME,
						 
						 "visible",
						 COLUMN_CLS_VISIBLE,
						 
						 "editable",
						 COLUMN_CLS_EDITABLE, 
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  
  /* Cleavespec pattern column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", (gint *) 
		     COLUMN_CLS_PATTERN);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, "Pattern",
						 renderer, "text",

						 COLUMN_CLS_PATTERN,

						 "visible",
						 COLUMN_CLS_VISIBLE,

						 "editable",
						 COLUMN_CLS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Leftrightrule left code column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_LRR_LEFT_CODE);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, "Left code",
						 renderer, "text",
						 
						 COLUMN_LRR_LEFT_CODE,
						 
						 "visible",
						 COLUMN_CLS_VISIBLE,
						 
						 "editable",
						 COLUMN_CLS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Leftrightrule left actform column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_LRR_LEFT_ACTFORM);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, "Left actform",
						 renderer, "text",

						 COLUMN_LRR_LEFT_ACTFORM,

						 "visible",
						 COLUMN_CLS_VISIBLE,

						 "editable",
						 COLUMN_CLS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Leftrightrule right code column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_LRR_RIGHT_CODE);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, "Right code",
						 renderer, "text",

						 COLUMN_LRR_RIGHT_CODE,

						 "visible",
						 COLUMN_CLS_VISIBLE,

						 "editable",
						 COLUMN_CLS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Leftrightrule right actform column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_LRR_RIGHT_ACTFORM);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, "Right actform",
						 renderer, "text",

						 COLUMN_LRR_RIGHT_ACTFORM,

						 "visible",
						 COLUMN_CLS_VISIBLE,

						 "editable",
						 COLUMN_CLS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);
  
  return TRUE;
}



GtkTreeModel *
polyxedit_cleave_opt_create_cls_treeview_model (GPtrArray *GPA)
{
  GtkTreeStore *model;
  GtkTreeIter tree_iter;

  gint iter = 0;
  gint jter = 0;

  PxmCleaveSpec *cls = NULL;
  PxmCleaveRule *clr = NULL;
  
  model = gtk_tree_store_new (COLUMN_CLS_COL_COUNT,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_BOOLEAN,
			      G_TYPE_BOOLEAN);


  /* Add data for each cleavespec to the tree store.
   */
  for (iter = 0 ; iter < GPA->len ; iter++)
    {
      cls = g_ptr_array_index (GPA, iter);
      g_assert (cls != NULL);
      
      gtk_tree_store_append (model, &tree_iter, NULL);

      gtk_tree_store_set 
	(model, &tree_iter,
	 COLUMN_CLS_NAME, cls->name,
	 COLUMN_CLS_PATTERN, cls->pattern,
	 COLUMN_LRR_LEFT_CODE, "",
	 COLUMN_LRR_LEFT_ACTFORM, "",
	 COLUMN_LRR_RIGHT_CODE, "",
	 COLUMN_LRR_RIGHT_ACTFORM, "",
	 COLUMN_CLS_VISIBLE, TRUE,
	 COLUMN_CLS_EDITABLE, FALSE,
	 -1);

      /* Add children, namely cleaverules of current cleavespec, if any.
       */
      for (jter = 0 ; jter < cls->clrGPA->len ; jter++)
	{
	  GtkTreeIter tree_jter;
	  
	  clr = g_ptr_array_index (cls->clrGPA, jter);
	  g_assert (clr != NULL);

	  
	  gtk_tree_store_append (model, &tree_jter, &tree_iter);
	  gtk_tree_store_set 
	    (model, &tree_jter,
	     /* cleavespec data */
	     COLUMN_CLS_NAME, "",
	     COLUMN_CLS_PATTERN, "",

	     /* cleaverule data */
	     COLUMN_LRR_LEFT_CODE, 
	     (clr->left_code != NULL)? clr->left_code : "",

	     COLUMN_LRR_LEFT_ACTFORM,
	     (clr->left_actform != NULL)? clr->left_actform : "",

	     /* cleaverule data */
	     COLUMN_LRR_RIGHT_CODE, 
	     (clr->right_code != NULL)? clr->right_code : "",

	     COLUMN_LRR_RIGHT_ACTFORM,
	     (clr->right_actform != NULL)? clr->right_actform : "",

	     /* column behaviour */
	     COLUMN_CLS_VISIBLE, TRUE,
	     COLUMN_CLS_EDITABLE, TRUE,
	     -1);
	}
    }

  return GTK_TREE_MODEL (model);
}



void
polyxedit_cleave_opt_wnd_cleave_button (GtkWidget *widget, gpointer data)
{
  PxmEditCtxt *editctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;

  PxmCleaveSpec *cleavespec = NULL;
  PxmCleaveOpt *cleaveopt = NULL;
  PxmCalcOpt *calcopt = NULL;
  PxmIonizerule *ionizerule = NULL;
  
  GtkWidget *window = data;
  GtkWidget *display_window = NULL;
  GtkWidget *entry = NULL;
  GtkWidget *messages_entry = NULL;

  GtkTreeView *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;
  GtkTreeIter tree_iter_sel;

  GPtrArray *partialsGPA = NULL;
  

  gboolean res = 0;
  
  gint idx = -1;

  gchar *help = NULL;

  

  /* The mass calculation options are taken from the editctxt's
     calcopt member. The user can modify these at will by selecting
     the appropriate menu.
  */
  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");

  polchemdef = editctxt->polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  /* Allocate the PxmCleaveOpt object that will be needed.
   */
  cleaveopt = pxmchem_cleaveopt_new ();

  /* Make sure that we set the monomer modifications to the oligomers.
   */
  cleaveopt->mnm_chement = PXMCHEMENT_MNM_MODIF;

  /* Same for the polymer modifications.
   */
  cleaveopt->plm_chement = PXMCHEMENT_PLM_BOTH_MODIF;
    
  
  /* Duplicate the editctxt's calcopt (because its start_idx, end_idx
     members are going to be modified during the cleavage operation):
   */
  calcopt = pxmchem_calcopt_dup (editctxt->calcopt);

  /* Duplicate the polchemdef's ionizerule structure, so that we are 
     on our own during this polymer sequence cleavage.
  */
  ionizerule = pxmchem_ionizerule_dup (editctxt->ionizerule);
  
  /* We certainly will have to display some messages.
   */
  messages_entry = g_object_get_data (G_OBJECT (window),  
				       "messages_entry");
  g_assert (messages_entry != NULL);
  
  /* Get the number of partial cleavages asked (integer >= 0).
   */
  entry = g_object_get_data (G_OBJECT (window),  
			      "partial_cleavages_entry");
  g_assert (entry != NULL);

  help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  g_assert (help != NULL);
  
  if (strlen (help) <= 0)
    help = g_strdup ("0");
  
  help = libpolyxmass_globals_unspacify_string (&help);
  
  /* Now convert this string into an integer.
   */
  res = libpolyxmass_globals_strtoi (help, &cleaveopt->partial, 10);
  
  g_free (help);
  
  if (res == FALSE && cleaveopt->partial < 0)
    {
      gtk_entry_set_text (GTK_ENTRY (messages_entry),
			  _("Error with partial cleavages' value"));
      
      pxmchem_cleaveopt_free (cleaveopt);
      pxmchem_calcopt_free (calcopt);
      pxmchem_ionizerule_free (ionizerule);
      
      return ;
    }
  


  /* Get a pointer to the currently selected cleavage specification 
     in the treeview and duplicate it to the cleaveopt instance.
  */  

  /* Get a pointer to the treeview in the window.
   */
  treeview = (GtkTreeView *) g_object_get_data (G_OBJECT (window), 
						"cleavespec_treeview");
  g_assert (treeview != NULL);
  
  /* And to the related model:
   */
  /* Get the cleavespec treeview model from the window pointer passed
   * a data parameter.
   */
  model = (GtkTreeModel *) g_object_get_data (G_OBJECT (window), 
					      "cleavespec_treeview_model");

  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  res = gtk_tree_selection_get_selected (tree_selection,
					 &model,
					 &tree_iter_sel);
  
  /* If there is no current selection, return immediately.
   */
  if (res == FALSE)
    {
      pxmchem_cleaveopt_free (cleaveopt);
      pxmchem_calcopt_free (calcopt);
      pxmchem_ionizerule_free (ionizerule);

      return;
    }
  
  /* Get the path to the selected item, which can be a cleavespec row
   * or a cleaverule row, which are two very different cases that
   * we must handle. Specifically, if depth == 1, the selected row
   * is a cleavespec, if depth == 2, then it is a cleaverule.
   */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);

  /* We can get the index of the cleavespec that is concerned, be it
     selected or one of its cleaverule children.
  */
  idx = gtk_tree_path_get_indices (path) [0];

  /* Finally we can free the path.
   */
  gtk_tree_path_free (path);

  /* Since the treeview store was filled using the
     editctxt->polchemdefctxt->polchemdef->cleavespecGPA, we can get to 
     the cleavespec selected using this array and idx.
  */
  cleavespec = g_ptr_array_index (polchemdef->cleavespecGPA, idx);
  g_assert (cleavespec != NULL);
  
  /* Duplicate the cleavespec into the cleaveopt.
   */
  cleaveopt->cleavespec = pxmchem_cleavespec_dup (cleavespec);
  
  /* We want that each oligomer has its own sequence so that
     we will be able to display it in the results window.
  */
  cleaveopt->put_sequence = TRUE;
  
  /* We seem to be ready to perform the polymer sequence cleavage.
   */

  /* Allocate the array that will hold all the different oligomers'
     arrays (1 oligomers' array per partial cleavage).
  */
  partialsGPA = g_ptr_array_new ();
  
  if (-1 ==
      pxmchem_cleave_polymer (editctxt->polymer,
			      partialsGPA,
			      polchemdef,
			      cleaveopt,
			      calcopt,
			      ionizerule))
    {
      g_critical (_("%s@%d: failed cleaving polymer\n"),
	     __FILE__, __LINE__);
      
      gtk_entry_set_text (GTK_ENTRY (messages_entry),
			  _("failed cleaving polymer."));

      pxmchem_oligomer_GPA_of_GPA_free (partialsGPA);
      
      pxmchem_cleaveopt_free (cleaveopt);
      pxmchem_calcopt_free (calcopt);
      pxmchem_ionizerule_free (ionizerule);
      
      return;
    }

  
  /* And now make sure we can display the data: make the window where
     the oligomers are to be displayed. The function call below will
     manage to display in the window all the oligomers that are in the
     partialsGPA array. If something fails, just free the array of
     partials and return.
  */
  display_window = polyxedit_cleave_res_wnd_setup (editctxt,
						   partialsGPA);
  if (display_window == NULL)
    {
      gtk_entry_set_text (GTK_ENTRY (messages_entry),
			  _("Failed displaying the oligomers in a window"));
      
      pxmchem_oligomer_GPA_of_GPA_free (partialsGPA);
      
      pxmchem_cleaveopt_free (cleaveopt);
      pxmchem_calcopt_free (calcopt);
      pxmchem_ionizerule_free (ionizerule);
      
      return;
    }

  /* Now that we have the certainty that the oligomers' displaying window
     could be created and populated correctly with the oligomers, than
     we can set the three cleavage-specifying structures to it so that
     their data can be displayed if the user so wishes. All we do is 
     just set the data with the freeing function so that when the
     window is closed their data are freed automatically.
  */
  g_object_set_data_full (G_OBJECT (display_window),
			  "cleaveopt", cleaveopt,
			  (GDestroyNotify) pxmchem_cleaveopt_free);
  
  g_object_set_data_full (G_OBJECT (display_window),
			  "calcopt", calcopt,
			  (GDestroyNotify) pxmchem_calcopt_free);
  
  g_object_set_data_full (G_OBJECT (display_window),
			  "ionizerule", ionizerule,
			  (GDestroyNotify) pxmchem_ionizerule_free);

  /* And now we can ask that all the informations contained in these
     structures be updated in the results' window at proper places.
  */
  polyxedit_cleave_res_wnd_update_cleavage_data (display_window);
  
  
  /* Finally we can return, our work here is finished.
   */
  return;
}




gboolean
polyxedit_cleave_opt_wnd_delete_event (GtkWidget *widget,
					   GdkEvent *event,
					   gpointer data)
{
  return FALSE;
}




gboolean
polyxedit_cleave_opt_wnd_destroy_event (GtkWidget *widget,
					   GdkEvent *event,
					    gpointer data)
{
  return FALSE;
}

