/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxdef-globals.h"

#include "polyxdef-ui-cleavespecs.h"
#include "polyxdef-ui-polchemdef.h"




gboolean
polyxdef_ui_polchemdef_wnd_setup_cleavespecs_notebook_page (PxmDefCtxt *defctxt,
							    GladeXML *xml)
{
  GtkWidget *widget = NULL;


  g_assert (defctxt != NULL);
  g_assert (defctxt->polchemdef != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  
  
  /* The cleavespecs' notebook page must be correctly set up:
   */
  widget = glade_xml_get_widget (xml, "cleavespecs_available_vbox");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "cleavespecs_available_vbox", widget);

  if (FALSE == 
      polyxdef_ui_polchemdef_wnd_setup_cleavespecs_treeview (defctxt))
    {
      g_critical (_("%s@%d: failed setting up the cleavespecs' material\n"),
	     __FILE__, __LINE__);
      
      return FALSE;
    }
  
  widget = glade_xml_get_widget (xml, "polyxdef_cleavespec_add_cls_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polyxdef_cleavespec_add_cls_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_cleavespec_add_cls_button),
		    defctxt);

  widget = glade_xml_get_widget (xml, "polyxdef_cleavespec_add_clr_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polyxdef_cleavespec_add_clr_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_cleavespec_add_clr_button),
		    defctxt);

  widget = glade_xml_get_widget (xml, 
				 "polyxdef_cleavespec_remove_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polyxdef_cleavespec_remove_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_cleavespec_remove_button),
		    defctxt);
  
  return TRUE;
}


gboolean
polyxdef_ui_polchemdef_wnd_setup_cleavespecs_treeview (PxmDefCtxt *defctxt)
{
  GtkWidget *vbox = NULL;

  GPtrArray *cleavespecGPA = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;

  GtkWidget *sw = NULL;

  gint col_offset;


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  g_assert (defctxt->polchemdef != NULL);

  cleavespecGPA = defctxt->polchemdef->cleavespecGPA;
  g_assert (cleavespecGPA != NULL);
  
  vbox = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			    "cleavespecs_available_vbox");
  g_assert (vbox != NULL);
  


  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  
  

  /* Create the treeview model.
   */
  model = polyxdef_ui_polchemdef_wnd_create_cleavespecs_treeview_model 
    (cleavespecGPA);


  /* Set to the model a datum with a pointer to the window!
   */
  g_object_set_data (G_OBJECT (model), "window", defctxt->definition_wnd);

  /* And now set the window a datum with a pointer to the mode,
   * so that later the model is accessible (add/remove button
   * handlers).
   */
  g_object_set_data (G_OBJECT (defctxt->definition_wnd), 
		     "cleavespec_treeview_model", model);
  
  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);

  g_object_unref (G_OBJECT (model));

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (defctxt->definition_wnd), 
		     "cleavespec_treeview", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);
  
  /* Cleavespec name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_CLS_NAME);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_cleavespecs_treeview_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Name"),
						 renderer, 
						 "text",
						 
						 COLUMN_CLS_NAME,
						 
						 "visible",
						 COLUMN_CLS_VISIBLE,
						 
						 "editable",
						 COLUMN_CLS_EDITABLE, 
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  
  /* Cleavespec pattern column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", (gint *) 
		     COLUMN_CLS_PATTERN);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_cleavespecs_treeview_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Pattern"),
						 renderer, "text",

						 COLUMN_CLS_PATTERN,

						 "visible",
						 COLUMN_CLS_VISIBLE,

						 "editable",
						 COLUMN_CLS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Leftrightrule left code column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_CLR_LEFT_CODE);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_cleavespecs_treeview_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Left Code"),
						 renderer, "text",
						 
						 COLUMN_CLR_LEFT_CODE,
						 
						 "visible",
						 COLUMN_CLS_VISIBLE,
						 
						 "editable",
						 COLUMN_CLS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* Leftrightrule left actform column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_CLR_LEFT_ACTFORM);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_cleavespecs_treeview_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Left Actform"),
						 renderer, "text",

						 COLUMN_CLR_LEFT_ACTFORM,

						 "visible",
						 COLUMN_CLS_VISIBLE,

						 "editable",
						 COLUMN_CLS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);




  /* Leftrightrule right code column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_CLR_RIGHT_CODE);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_cleavespecs_treeview_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Right Code"),
						 renderer, "text",

						 COLUMN_CLR_RIGHT_CODE,

						 "visible",
						 COLUMN_CLS_VISIBLE,

						 "editable",
						 COLUMN_CLS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* Leftrightrule right actform column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_CLR_RIGHT_ACTFORM);

  g_signal_connect 
    (G_OBJECT (renderer), "edited", 
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_cleavespecs_treeview_cell_edited),
     defctxt);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Right Actform"),
						 renderer, "text",

						 COLUMN_CLR_RIGHT_ACTFORM,

						 "visible",
						 COLUMN_CLS_VISIBLE,

						 "editable",
						 COLUMN_CLS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);
  
  return TRUE;
}



GtkTreeModel *
polyxdef_ui_polchemdef_wnd_create_cleavespecs_treeview_model (GPtrArray *cleavespecGPA)
{
  GtkTreeStore *model;
  GtkTreeIter tree_iter;

  gint iter = 0;
  gint jter = 0;

  PxmCleaveSpec *cls = NULL;
  PxmCleaveRule *clr = NULL;
  
  model = gtk_tree_store_new (COLUMN_CLS_COL_COUNT,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_BOOLEAN,
			      G_TYPE_BOOLEAN);


  /* Add data for each cleavespec to the tree store.
   */
  for (iter = 0 ; iter < cleavespecGPA->len ; iter++)
    {
      cls = g_ptr_array_index (cleavespecGPA, iter);
      g_assert (cls != NULL);
      
      gtk_tree_store_append (model, &tree_iter, NULL);

      gtk_tree_store_set 
	(model, &tree_iter,
	 COLUMN_CLS_NAME, cls->name,
	 COLUMN_CLS_PATTERN, cls->pattern,
	 COLUMN_CLR_LEFT_CODE, "",
	 COLUMN_CLR_LEFT_ACTFORM, "",
	 COLUMN_CLR_RIGHT_CODE, "",
	 COLUMN_CLR_RIGHT_ACTFORM, "",
	 COLUMN_CLS_VISIBLE, TRUE,
	 COLUMN_CLS_EDITABLE, TRUE,
	 -1);

      /* Add children, namely cleaverules of current cleavespec, if any.
       */
      for (jter = 0 ; jter < cls->clrGPA->len ; jter++)
	{
	  GtkTreeIter tree_jter;
	  
	  clr = g_ptr_array_index (cls->clrGPA, jter);
	  g_assert (clr != NULL);

	  
	  gtk_tree_store_append (model, &tree_jter, &tree_iter);
	  gtk_tree_store_set 
	    (model, &tree_jter,
	     /* cleavespec data */
	     COLUMN_CLS_NAME, "",
	     COLUMN_CLS_PATTERN, "",

	     /* cleaverule data */
	     COLUMN_CLR_LEFT_CODE, 
	     (clr->left_code != NULL)? clr->left_code : "",

	     COLUMN_CLR_LEFT_ACTFORM,
	     (clr->left_actform != NULL)? clr->left_actform : "",

	     /* cleaverule data */
	     COLUMN_CLR_RIGHT_CODE, 
	     (clr->right_code != NULL)? clr->right_code : "",

	     COLUMN_CLR_RIGHT_ACTFORM,
	     (clr->right_actform != NULL)? clr->right_actform : "",

	     /* column behaviour */
	     COLUMN_CLS_VISIBLE, TRUE,
	     COLUMN_CLS_EDITABLE, TRUE,
	     -1);
	}
    }

  return GTK_TREE_MODEL (model);
}


void
polyxdef_ui_polchemdef_wnd_cleavespecs_treeview_cell_edited (GtkCellRendererText *cell,
							     const gchar *path_string,
							     const gchar *new_text,
							     gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeModel *model = NULL;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
  GtkTreeIter tree_iter;
  
  PxmCleaveSpec *cls = NULL;
  PxmCleaveSpec *cls_test = NULL;
  PxmCleaveRule *clr = NULL;
  
  GPtrArray *cleavespecGPA = NULL;

  gint *column;
  gint idx_cls = -1;
  gint idx_clr = -1;
  gint depth = -1;
    
  gint result = 0;
  
  gchar *old_text;
  gchar *help = NULL;
  

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  cleavespecGPA = defctxt->polchemdef->cleavespecGPA;
  g_assert (cleavespecGPA != NULL);

  g_assert (defctxt->polchemdef->atomGPA != NULL);

  


  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			     "cleavespec_treeview_model");
  g_assert (model != NULL);

  column = g_object_get_data (G_OBJECT (cell), "column");

  gtk_tree_model_get_iter (model, &tree_iter, path);

  depth = gtk_tree_path_get_depth (path);

  switch (GPOINTER_TO_INT (column))
    {
    case COLUMN_CLS_NAME:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a cleavespec row,
	 * and not a cleaverule row.
	 */
	if (depth > 1)
	  {
	    help = 
	      g_strdup_printf (_("select or create a cleavespec first"));
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);
	    
	    return ;
	  }
	
	/* Before proceeding and accepting the new text, we should 
	 * make sure that no other cleavespec already exists by the same
	 * name.
	 */
	result = 
	  pxmchem_cleavespec_get_index_by_name ((gchar *) new_text, 
						cleavespecGPA);

	if (-1 != result)
	  {
	    help = g_strdup_printf (_("error, a cleavespec by the same name"
				      " exists already: %s"),
				    new_text);
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);
	    
	    return ;
	  }

	/* At this point, all is OK, we can make the replacement.
	 */
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the item that is edited.
	 */
	idx_cls = gtk_tree_path_get_indices (path) [0];

	cls = g_ptr_array_index (cleavespecGPA, idx_cls);
	g_assert (cls != NULL);
	
	pxmchem_cleavespec_set_name (cls, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    cls->name,
			    -1);
      }
      break;

    case COLUMN_CLS_PATTERN:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a cleavespec row,
	 * and not a cleaverule row.
	 */
	if (depth > 1)
	  {
	    help = 
	      g_strdup_printf (_("select or create a cleavespec first"));
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);
	    
	    return ;
	  }
	

	/* We can ensure that the cleavespec's pattern is correct by
	   duplicating the cls that was being edited, changing its pattern
	   with the new_text and then test it against the polchemdef.
	*/

	/* Get the index of the item that is edited.
	 */
	idx_cls = gtk_tree_path_get_indices (path) [0];

	cls = g_ptr_array_index (cleavespecGPA, idx_cls);
	g_assert (cls != NULL);
	
	/* Make a copy so that we operate on a copy only.
	 */
	cls_test = pxmchem_cleavespec_dup (cls);
	
	/* And now make the validity check on the pattern.
	 */
	pxmchem_cleavespec_set_pattern (cls_test, (gchar *) new_text);

	if (-1 == 
	    pxmchem_cleave_parse_cleavespec_pattern (cls_test,
						     defctxt->polchemdef->
						     delim_codes,
						     defctxt->polchemdef->
						     codelen))
	  {
	    help = 
	      g_strdup_printf (_("error, the cleavespec pattern "
				 "is wrong: %s"), new_text);
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd, 
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);

	    pxmchem_cleavespec_free (cls_test);
	    
	    return ;
	  }

	/* Finally make the change, since the pattern is OK. But first
	   free the cleavespec that was used to make the test.
	*/
	pxmchem_cleavespec_free (cls_test);

	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	pxmchem_cleavespec_set_pattern (cls, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
	
	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    cls->pattern,
			    -1);
      }
      break;

    case COLUMN_CLR_LEFT_CODE:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a cleaverule row,
	 * and not a cleavespec row. 
	 */
	if (depth != 2)
	  {
	    help = 
	      g_strdup_printf (_("select or create a cleaverule first"));
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);
	    
	    return ;
	  }
	
	/* But first yet, verify that the monomer code is existing 
	 * in the polymer definition ! This ONLY if the new_text is 
	 * of non-0 length, because the DTD stipulates that this
	 * monomer code may be empty if no corresponding actform is
	 * found. This translates into: "no cleaverule is required,
	 * but if there it must be complete with monomer code and
	 * actform".
	 */
	if (strlen (new_text) > 0 
	    &&
	    -1 == 
	    pxmchem_monomer_get_index_by_code ((gchar *) new_text, 
					       defctxt->polchemdef->
					       monomerGPA))
	  {
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
	       _("error: code unknown"), 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    return ;
	  }

	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the cleavespec (main) item that is edited.
	 */
	idx_cls = gtk_tree_path_get_indices (path) [0];
	
	cls = g_ptr_array_index (cleavespecGPA, idx_cls);
	g_assert (cls != NULL);
		
	/* Get the index of the cleaverule (sub) item that is edited.
	 */
	idx_clr = gtk_tree_path_get_indices (path) [1];

	clr = g_ptr_array_index (cls->clrGPA, idx_clr);
	g_assert (clr != NULL);
	
	pxmchem_cleaverule_set_left_code (clr, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
	
	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    new_text,
			    -1);
	
      }
      break;

    case COLUMN_CLR_LEFT_ACTFORM:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a cleaverule row,
	 * and not a cleavespec row.
	 */
	if (depth != 2)
	  {
	    help = 
	      g_strdup_printf (_("select or create a cleaverule first"));
	    
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);
	    
	    return ;
	  }
	
	/* Before proceeding and accepting the new text, we should
	 * check that the actform in it is valid. Further, the
	 * document type definition stipulates that there MUST be an
	 * actform only if the corresponding monomer code is
	 * there. This translates into: "no cleaverule is required,
	 * but if there it must be complete with monomer code and
	 * actform".
	 */

	/* So, first manage to get the string that is located in the
	 * related monomer code column. Since the column that holds
	 * the monomer code is located left of this one, we can say
	 * that it is the third one, and that its column index is 2. 
	 */
	gtk_tree_model_get (GTK_TREE_MODEL (model),
			    &tree_iter,
			    2, &help,
			    -1);	
	
	g_assert (help != NULL);

	/* Now the different situations:
	 */
	if (strlen (new_text) > 0)
	  {
	    /* The new actform is non-empty, so there MUST be a
	     * monomer code for the related monomer code.
	     */
	        
	    /* We now have to check if the string contained in help is
	     * a valid monomer code. If it contains an invalid monomer
	     * code, we return immediately.
	     */
	    if (-1 == 
		pxmchem_monomer_get_index_by_code (help,
						   defctxt->polchemdef->
						   monomerGPA))
	      {
		
		polyxmass_timeoutmsg_message_set 
		  ((GtkWindow *) defctxt->definition_wnd,
		   _("error: related code is unknown"), 
		   POLYXMASS_MEDI_MSG_TIMEOUT);

		g_free (help);
		
		return ;
	      }
	    else
	      {
		/* There is a valid related monomer code in the 
		 * preceding column, so this actform has to be valid.
		 */
		result = 
		  pxmchem_actform_check ((gchar *) new_text,
					 defctxt->polchemdef->atomGPA);
		
		if (FALSE == result)
		  {
		    help = g_strdup_printf (_("error, the new actform"
					      " is invalid: %s"),
					    new_text);
		    
		    polyxmass_timeoutmsg_message_set 
		      ((GtkWindow *) defctxt->definition_wnd,
		       help, 
		       POLYXMASS_MEDI_MSG_TIMEOUT);
		    
		    g_free (help);
		    
		    return ;
		  }
	      }
	  }
	else
	  {
	    /* The new actform is empty, so the related monomer code 
	     * MUST be empty also.
	     */
	    if (strlen (help) > 0)
	      {
		polyxmass_timeoutmsg_message_set 
		  ((GtkWindow *) defctxt->definition_wnd,
		   _("error: remove related code, first"), 
		   POLYXMASS_MEDI_MSG_TIMEOUT);
		
		g_free (help);
		
		return ;
	      }
	    else
	      {
		/* For this special case, we do accept that the
		 * actform be empty, since it is authorized that it be
		 * empty if the related monomer code is also empty,
		 * which we have successfully verified. So no actform
		 * check here. Let the flow go on until the new
		 * empty string can replace the older one.
		 */
	      }
	  }
	
	g_free (help);
	

	/* At this point, all is OK, we can make the replacement.
	 */
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the item that is edited.
	 */
	idx_cls = gtk_tree_path_get_indices (path) [0];

	cls = g_ptr_array_index (cleavespecGPA, idx_cls);
	g_assert (cls != NULL);

	/* Get the index of the cleaverule (sub) item that is edited.
	 */
	idx_clr = gtk_tree_path_get_indices (path) [1];
	
	clr = g_ptr_array_index (cls->clrGPA, idx_clr);
	g_assert (clr != NULL);
	
	pxmchem_cleaverule_set_left_actform (clr, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    new_text,
			    -1);
      }
      break;

    case COLUMN_CLR_RIGHT_CODE:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a cleaverule row,
	 * and not a cleavespec row.
	 */
	if (depth != 2)
	  {
	    help = 
	      g_strdup_printf (_("select or create a cleaverule first"));
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd, 
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);
	    
	    return ;
	  }
	
	/* But first yet, verify that the monomer code is existing 
	 * in the polymer definition ! This ONLY if the new_text is 
	 * of non-0 length, because the DTD stipulates that this
	 * monomer code may be empty if no corresponding actform is
	 * found. This translates into: "no cleaverule is required,
	 * but if there it must be complete with monomer code and
	 * actform".
	 */
	if (strlen (new_text) > 0 
	    &&
	    -1 == 
	    pxmchem_monomer_get_index_by_code ((gchar *) new_text,
					       defctxt->polchemdef->
					       monomerGPA))
	  {
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd, 
	       _("error: code unknown"), 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    return ;
	  }
	
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the cleavespec (main) item that is edited.
	 */
	idx_cls = gtk_tree_path_get_indices (path) [0];
	
	cls = g_ptr_array_index (cleavespecGPA, idx_cls);
	g_assert (cls != NULL);
	
	/* Get the index of the cleaverule (sub) item that is edited.
	 */
	idx_clr = gtk_tree_path_get_indices (path) [1];

	clr = g_ptr_array_index (cls->clrGPA, idx_clr);
	g_assert (clr != NULL);
	
	pxmchem_cleaverule_set_right_code (clr, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
	
	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    new_text,
			    -1);
      }
      break;

    case COLUMN_CLR_RIGHT_ACTFORM:
      {
	/* We only do accept to change anything if the cell that is
	 * edited corresponds to a row which belongs to a cleaverule row,
	 * and not a cleavespec row.
	 */
	if (depth != 2)
	  {
	    help = 
	      g_strdup_printf (_("select or create a cleaverule first"));
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
	       help, 
	       POLYXMASS_MEDI_MSG_TIMEOUT);
	    
	    g_free (help);
	    
	    return ;
	  }
	
	/* Before proceeding and accepting the new text, we should
	 * check that the actform in it is valid. Further, the
	 * document type definition stipulates that there MUST be an
	 * actform only if the corresponding monomer code is
	 * there. This translates into: "no cleaverule is required,
	 * but if there it must be complete with monomer code and
	 * actform".
	 */

	/* So, first manage to get the string that is located in the
	 * related monomer code column. Since the column that holds
	 * the monomer code is located left of this one, we can say
	 * that it is the third one, and that its column index is 4. 
	 */
	gtk_tree_model_get (GTK_TREE_MODEL (model),
			    &tree_iter,
			    4, &help,
			    -1);	
	
	g_assert (help != NULL);

	/* Now the different situations:
	 */
	if (strlen (new_text) > 0)
	  {
	    /* The new actform is non-empty, so there MUST be a
	     * monomer code for the related monomer code.
	     */
	        
	    /* We now have to check if the string contained in help is
	     * a valid monomer code. If it contains an invalid monomer
	     * code, we return immediately.
	     */
	    if (-1 == 
		pxmchem_monomer_get_index_by_code (help,
						   defctxt->polchemdef->
						   monomerGPA))
	      {
		polyxmass_timeoutmsg_message_set 
		  ((GtkWindow *) defctxt->definition_wnd,
		   _("error: related code is unknown"), 
		   POLYXMASS_MEDI_MSG_TIMEOUT);
		
		g_free (help);
		
		return ;
	      }
	    else
	      {
		/* There is a valid related monomer code in the 
		 * preceding column, so this actform has to be valid.
		 */
		result = 
		  pxmchem_actform_check ((gchar *) new_text,
					 defctxt->polchemdef->atomGPA);
		
		if (FALSE == result)
		  {
		    help = g_strdup_printf (_("error, the new actform"
					      " is invalid: %s"),
					    new_text);
		    
		    polyxmass_timeoutmsg_message_set 
		      ((GtkWindow *) defctxt->definition_wnd,
		       help, 
		       POLYXMASS_MEDI_MSG_TIMEOUT);
		    
		    g_free (help);
		    
		    return ;
		  }
	      }
	  }
	else
	  {
	    /* The new actform is empty, so the related monomer code 
	     * MUST be empty also.
	     */
	    if (strlen (help) > 0)
	      {
		polyxmass_timeoutmsg_message_set 
		  ((GtkWindow *) defctxt->definition_wnd,
		   _("error: remove related code, first"), 
		   POLYXMASS_MEDI_MSG_TIMEOUT);
		
		g_free (help);
		
		return ;
	      }
	    else
	      {
		/* For this special case, we do accept that the
		 * actform be empty, since it is authorized that it be
		 * empty if the related monomer code is also empty,
		 * which we have successfully verified. So no actform
		 * check here. Let the flow go on until the new
		 * empty string can replace the older one.
		 */
	      }
	  }
	
	g_free (help);
	

	/* At this point, all is OK, we can make the replacement.
	 */
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the item that is edited.
	 */
	idx_cls = gtk_tree_path_get_indices (path) [0];

	cls = g_ptr_array_index (cleavespecGPA, idx_cls);
	g_assert (cls != NULL);

	/* Get the index of the cleaverule (sub) item that is edited.
	 */
	idx_clr = gtk_tree_path_get_indices (path) [1];
	
	clr = g_ptr_array_index (cls->clrGPA, idx_clr);
	g_assert (clr != NULL);
	
	pxmchem_cleaverule_set_right_actform (clr, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

	gtk_tree_store_set (GTK_TREE_STORE (model), 
			    &tree_iter, column,
			    new_text,
			    -1);
      }
      break;
    }

  gtk_tree_path_free (path);
}


void
polyxdef_ui_polchemdef_wnd_cleavespec_add_cls_button (GtkWidget *widget, 
						      gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeView *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreeViewColumn* column = NULL;

  GtkTreeIter tree_iter;
  GtkTreeIter tree_iter_sel;
  GtkTreeIter tree_iter_parent;

  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;

  gint idx_cls = -1;
  gint depth = -1;


  PxmCleaveSpec *cls = NULL;
  PxmCleaveSpec *cls_test = NULL;
  
  GPtrArray *cleavespecGPA = NULL;


  /****************** GENERAL PHILOSOPHY **************************/

  /* A cleavespec is made with two intrinsic members: name and pattern.
   * It may also contain any number of cleaverules. 
   * If a cleavespec contains cleaverule objects, they are represented
   * as items on a lower treenode. This way.
   * 
   * |   name  |  pattern | le_code | le_actform | re_code | re_actform |
   * | Trypsin |  R/;K/   |         |            |         |            |
   *      |
   *      +------------------ L     |     LAC    |         |            |
   */

  /* That means that depending on the treeview situation, the actions
     that we need to take in response to the clicking onto the ADD
     button are different.
  */

  /* If no selection is made at all, just append a new cleavespec at
     the end of the treeview.
    
     If a selection is made:
   
     - if the selected item is a cleavespec row, then a new cleavespec
     row is to be added after this selected cleavespec row.
   
     - if the selected row is a cleaverule row, then a new
     cleavespec row is to be added after the cleavespec row of which
     the cleaverule is currently selected.
  */

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  cleavespecGPA = defctxt->polchemdef->cleavespecGPA;
  g_assert (cleavespecGPA != NULL);


  /* Get the cleavespec treeview model from the window pointer passed
     a data parameter.
  */
  model = (GtkTreeModel *) 
    g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
		       "cleavespec_treeview_model");
  g_assert (model != NULL);
  
  
  treeview= g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			       "cleavespec_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);

  /* If there is no current selection, that means that the user wants
     to add a cleavespec item and not a cleaverule.
  */
  if (result == FALSE)
    {
      /* Allocate a new cleavespec so that we can later fill its
	 member data.
      */
      cls = pxmchem_cleavespec_new ();
      
      pxmchem_cleavespec_set_name (cls, "name here");
      pxmchem_cleavespec_set_pattern (cls, "pattern here");
      
      g_ptr_array_add (cleavespecGPA, cls);

      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
      
      gtk_tree_store_append (GTK_TREE_STORE (model), &tree_iter, NULL);
      
      gtk_tree_store_set 
	((GtkTreeStore *) model, &tree_iter,
	 
	 /* cleavespec data */
	 COLUMN_CLS_NAME, cls->name,
	 COLUMN_CLS_PATTERN, cls->pattern,
	 
	 /* cleaverule data */
	 COLUMN_CLR_LEFT_CODE, "",
	 COLUMN_CLR_LEFT_ACTFORM, "",
	 COLUMN_CLR_RIGHT_CODE, "",
	 COLUMN_CLR_RIGHT_ACTFORM, "",
	 
	 /* column behaviour */
	 COLUMN_CLS_VISIBLE, TRUE,
	 COLUMN_CLS_EDITABLE, TRUE,
	 -1);
      
      /* And now we want to ease the user's work on modifying the fake
	 cleavespec values into more useful ones: go to the place where
	 the item was stored in the tree store, and put the cursor on the
	 right row and put the keyboard focus on it also, so that it get
	 highlighted...
      */
      path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				      &tree_iter);
      g_assert (path != NULL);
      
      /*
	{
	gchar *help = NULL;
	help =  gtk_tree_path_to_string  (path);
	debug_printf (("path = %s\n", help));
	g_free (help);
	}
      */
      
      gtk_tree_view_expand_to_path (GTK_TREE_VIEW (treeview),
				    path);
      
      column = gtk_tree_view_get_column (treeview, COLUMN_CLS_NAME);
      
      gtk_tree_selection_select_iter (tree_selection, &tree_iter);
  
      gtk_tree_view_set_cursor_on_cell (treeview,
					path,
					column,
					NULL,
					FALSE);
      
      gtk_widget_grab_focus (GTK_WIDGET (treeview));
      
      gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				    path,
				    column,
				    TRUE,
				    0.5,
				    0.5);
      gtk_tree_path_free (path);
      
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 _("added new cleavespec;"
	   " set its data!"), 
	 POLYXMASS_LONG_MSG_TIMEOUT); 
      
      return;
    }
  
  /* There is currently a selection, that means that the user wants to
     add an item. We'll have to decide if it is a cleavespec or a
     cleaverule.
  */

  /* Get the path to the selected item, which can be a cleavespec row
     or a cleaverule row, which are two very different cases that
     we must handle. Specifically, if depth == 1, the selected row is
     a cleavespec, if depth == 2, then it is a cleaverule.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);
      
  /* Get the depth of the currently selected item, which is 1 if it is
     a cleavespec item or 2 if it is a cleaverule item.
  */
  depth = gtk_tree_path_get_depth (path);

  /*
    debug_printf (("depth=%d\n", depth));
  */
  
  /* Whatever the depth, we can get the index of the cleavespec that
     is concerned, be it selected or one of its cleaverule
     children.
  */
  idx_cls = gtk_tree_path_get_indices (path) [0];
   
  /*
    debug_printf (("idx_cls=%d\n", idx_cls);
  */
    
  /* Finally we can free the path.
   */
  gtk_tree_path_free (path);

  /* Now, if the depth == 2, that means that the item that is selected
     is a cleaverule. Since we only want to add a new cleavespec,
     we understand that the user wants to insert/add a cls object
     after the cleavespec object that is actually the parent of the
     currently selected cleaverule.
  */
  if (depth == 2)
    {
      result = gtk_tree_model_iter_parent (GTK_TREE_MODEL (model), 
					   &tree_iter_parent, 
					   &tree_iter_sel);
      g_assert (result != FALSE);

      gtk_tree_store_insert_after (GTK_TREE_STORE (model),
				   &tree_iter,
				   NULL,
				   &tree_iter_parent);
    }
  else
    {
      gtk_tree_store_insert_after (GTK_TREE_STORE (model),
				   &tree_iter,
				   NULL,
				   &tree_iter_sel);
    }
  
  /* Allocate a new cleavespec so that we can later fill its member
     data.
  */
  cls = pxmchem_cleavespec_new ();
  
  pxmchem_cleavespec_set_name (cls, "name here!");
  pxmchem_cleavespec_set_pattern (cls, "pattern here!");

  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);
      
  /* Get the index at which we will store the data. This index is the
     one that we will use in order to insert/add the newly allocated
     cls to cleavespecGPA.
  */
  idx_cls = gtk_tree_path_get_indices (path) [0];

  if (idx_cls + 1 >= cleavespecGPA->len)
    {
      g_ptr_array_add (cleavespecGPA, cls);

      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
    }
  else if (idx_cls + 1 < cleavespecGPA->len)
    {
      cls_test = g_ptr_array_insert_val (cleavespecGPA, idx_cls, cls);
      g_assert (cls == cls_test);

      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
    }
  else
    {
      g_error (_("%s@%d: could not insert/add cleavespec item to array\n"),
	     __FILE__, __LINE__);
    }
  
  gtk_tree_path_free (path);

  gtk_tree_store_set 
    ((GtkTreeStore *) model, &tree_iter,

     /* cleavespecif data */
     COLUMN_CLS_NAME, cls->name,
     COLUMN_CLS_PATTERN, cls->pattern,

     /* cleaverule data */
     COLUMN_CLR_LEFT_CODE, "",
     COLUMN_CLR_LEFT_ACTFORM, "",
     COLUMN_CLR_RIGHT_CODE, "",
     COLUMN_CLR_RIGHT_ACTFORM, "",

     /* column behaviour */
     COLUMN_CLS_VISIBLE, TRUE,
     COLUMN_CLS_EDITABLE, TRUE,
     -1);
  
 
  /* And now we want to ease the user's work on modifying the fake
     name/pattern values into more useful ones: go to the place where
     the item was stored in the tree store, and put the cursor on the
     right row and put the keyboard focus on it also, so that it get
     highlighted...
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);

  /*
    {
    gchar *help = NULL;
    help =  gtk_tree_path_to_string  (path);
    debug_printf (("path = %s\n", help));
    g_free (help);
    }
  */

  gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				path,
				NULL,
				TRUE,
				0.5,
				0.5);

  gtk_tree_view_expand_to_path (GTK_TREE_VIEW (treeview),
				path);

  column = gtk_tree_view_get_column (treeview, COLUMN_CLS_NAME);

  gtk_tree_selection_select_iter (tree_selection, &tree_iter);

  gtk_tree_view_set_cursor_on_cell (treeview,
				    path,
				    column,
				    NULL,
				    FALSE);
  
  gtk_widget_grab_focus (GTK_WIDGET (treeview));
  
  gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				path,
				column,
				TRUE,
				0.5,
				0.5);

  gtk_tree_path_free (path);
  
  polyxmass_timeoutmsg_message_set 
    ((GtkWindow *) defctxt->definition_wnd, 
     _("added new cleavespec;"
       " set its data!"), 
     POLYXMASS_LONG_MSG_TIMEOUT); 
  return;
}


void
polyxdef_ui_polchemdef_wnd_cleavespec_add_clr_button (GtkWidget *widget, 
						      gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeView *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreeViewColumn* column = NULL;

  GtkTreeIter tree_iter;
  GtkTreeIter tree_iter_sel;

  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;

  gint idx_cls = -1;
  gint idx_clr = -1;
  gint depth = -1;
      
  PxmCleaveSpec *cls = NULL;
  PxmCleaveRule *clr = NULL;
  PxmCleaveRule *clr_test = NULL;
  
  GPtrArray *cleavespecGPA = NULL;



  /****************** GENERAL PHILOSOPHY **************************/

  /* A cleavespec is made with two intrinsic members: name and pattern.
     It may also contain any number of cleaverules. 
     If a cleavespec contains cleaverule objects, they are represented
     as items on a lower treenode. This way.
    
     |   name  |  pattern | le_code | le_actform | re_code | re_actform |
     | Trypsin |  R/;K/   |         |            |         |            |
     |
     +------------------ L     |     LAC    |         |            |
  */

  /* That means that depending on the treeview situation, the actions
     that we need to take in response to the clicking onto the ADD
     button are different.
  */

  /* If no selection is made at all, just return.
    
  If a selection is made:
   
  - if the selected item is a cleavespec row, then a new
  cleaverule row is to be added to this cleavespec row, and a
  new subtree may need to be created if no one exists already.
   
  - if the selected row is a cleaverule row, then a new
  cleaverule row is to be added to the array of cleaverules.
  */

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  cleavespecGPA = defctxt->polchemdef->cleavespecGPA; 
  g_assert (cleavespecGPA != NULL);



  /* Get the cleavespec treeview model from the window pointer passed
   * a data parameter.
   */
  model = (GtkTreeModel *) 
    g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
		       "cleavespec_treeview_model");
  g_assert (model != NULL);

  treeview= g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			       "cleavespec_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);

  /* If there is no current selection, return.
   */
  if (result == FALSE)
    return;
  
  /* There is currently a selection, that means that the user wants to
     add an item. We'll have to decide if it is a cleavespec or a
     cleaverule.
  */

  /* Get the path to the selected item, which can be a cleavespec row
     or a cleaverule row, which are two very different cases that
     we must handle. Specifically, if depth == 1, the selected row is
     a cleavespec, if depth == 2, then it is a cleaverule.
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter_sel);
  g_assert (path != NULL);
      
  /* Get the depth of the currently selected item, which is 1 if it is
     a cleavespec item or 2 if it is a cleaverule item.
  */
  depth = gtk_tree_path_get_depth (path);

  /*
    debug_printf (("depth is %d\n", depth));
  */

  /* Whatever the depth, we can get the index of the cleavespec that
     is concerned, be it selected or one of its cleaverule
     children.
  */
  idx_cls = gtk_tree_path_get_indices (path) [0];
  
  /*
    debug_printf (("idx_cls=%d\n", idx_cls));
  */

  if (depth == 2)
    {
      idx_clr = gtk_tree_path_get_indices (path) [1];
      /*
	debug_printf (("idx_clr is %d\n", idx_clr));
      */
    }

  /* Finally we can free the path.
   */
  gtk_tree_path_free (path);

  /* Now, if the depth == 2, that means that the item that is selected
     is a cleaverule. This means that we just want to insert/add a
     new cleaverule to the cleavespec at idx_cls.
  */
  if (depth == 1)
    {
      /* The item that is selected is a cleavespec item (at index
	 idx_cls). Since we want to add a cleaverule item, we must
	 insert a cleaverule item as the first child of the
	 selected cleavespec item.
      */
      gtk_tree_store_prepend (GTK_TREE_STORE (model),
			      &tree_iter,
			      &tree_iter_sel);
    }
  else if (depth == 2)
    {
      /* A cleaverule item was selected. All we do is insert the new 
	 cleaverule item after the currently selected one.
      */
      gtk_tree_store_insert_after (GTK_TREE_STORE (model),
				   &tree_iter,
				   NULL,
				   &tree_iter_sel);
    }
  else
    {
      g_error (_("%s@%d: could not insert/add cleaverule to treeview\n"),
	     __FILE__, __LINE__);
    }

  
  /* Allocate a new cleaverule so that we can later fill its member
     data.
  */
  /* Get the cleavespec !
   */
  cls = g_ptr_array_index (cleavespecGPA, idx_cls);
  g_assert (cls != NULL);
  /*
    debug_printf (("cleavespec is %s\n", cls->name));
  */

  clr = pxmchem_cleaverule_new ();

  pxmchem_cleaverule_set_left_code (clr, "left code here!");
  pxmchem_cleaverule_set_left_actform (clr, "left actform here!");
  pxmchem_cleaverule_set_right_code (clr, "right code here!");
  pxmchem_cleaverule_set_right_actform (clr, "right actform here!");

  /* But where do we have to put that stuff in cls->clrGPA? If a
     cleaverule was selected (depth == 2) we had determined the
     idx_clr of the selected cleaverule. Otherwise idx_clr is -1.
  */
  if (depth == 1)
    {
      /* The selected item was a cleavespec at index idx_cls, so we
	 want to insert at index 0 our new items.
      */
      /*
	debug_printf (("Insert the cleaverule at index 0\n"));
      */

      clr_test = g_ptr_array_insert_val (cls->clrGPA, 0, clr);
      g_assert (clr == clr_test);
      
      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
    }
  else /* depth == 2 */
    {
      /* 
	 We actually know that a cleaverule element was selected,
	 at index idx_clr... If the selected item was the last one, we
	 append the new one otherwise we insert the new one at the
	 index of the currently selected one +1.
      */
      if (idx_clr + 1 >= cls->clrGPA->len)
	{
	  /*
	    debug_printf (("Add the cleaverule at end of array\n"));
	  */

	  g_ptr_array_add (cls->clrGPA, clr);

	  defctxt->is_modified = TRUE;
	  polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
	}
      else if (idx_clr + 1 < cls->clrGPA->len)
	{
	  /*
	    debug_printf (("Insert the cleaverule at index %d\n",
	    idx_clr + 1));
	  */

	  clr_test = 
	    g_ptr_array_insert_val (cls->clrGPA, idx_clr + 1, clr);
	  g_assert (clr == clr_test);
	  
	  defctxt->is_modified = TRUE;
	  polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
	}
      else
	{
	  g_error (_("%s@%d: could not insert/add cleaverule to array\n"),
		 __FILE__, __LINE__);
	}
    }
      

  gtk_tree_store_set 
    ((GtkTreeStore *) model, &tree_iter,

     /* cleavespecif data */
     COLUMN_CLS_NAME, "",
     COLUMN_CLS_PATTERN, "",

     /* cleaverule data */
     COLUMN_CLR_LEFT_CODE, clr->left_code,
     COLUMN_CLR_LEFT_ACTFORM, clr->left_actform,
     COLUMN_CLR_RIGHT_CODE, clr->right_code,
     COLUMN_CLR_RIGHT_ACTFORM, clr->right_actform,

     /* column behaviour */
     COLUMN_CLS_VISIBLE, TRUE,
     COLUMN_CLS_EDITABLE, TRUE,
     -1);
  
 
  /* And now we want to ease the user's work on modifying the fake
     cleaverule values into more useful ones: go to the place where
     the item was stored in the tree store, and put the cursor on the
     right row and put the keyboard focus on it also, so that it get
     highlighted...
  */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);

  /*
    {
    gchar *help = NULL;
    help =  gtk_tree_path_to_string  (path);
    debug_printf (("path = %s\n", help));
    g_free (help);
    }
  */

  gtk_tree_view_expand_to_path (GTK_TREE_VIEW (treeview),
				path);

  column = gtk_tree_view_get_column (treeview, COLUMN_CLR_LEFT_CODE);

  gtk_tree_selection_select_iter (tree_selection, &tree_iter);
  
  gtk_tree_view_set_cursor_on_cell (treeview,
				    path,
				    column,
				    NULL,
				    FALSE);
  
  gtk_widget_grab_focus (GTK_WIDGET (treeview));
  
  gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				path,
				column,
				TRUE,
				0.5,
				0.5);
  
  gtk_tree_path_free (path);
  
  polyxmass_timeoutmsg_message_set 
    ((GtkWindow *) defctxt->definition_wnd, 
     _("added new cleaverule;"
       " set its data!"), 
     POLYXMASS_LONG_MSG_TIMEOUT); 
  return;
}


void
polyxdef_ui_polchemdef_wnd_cleavespec_remove_button (GtkWidget *widget, 
						     gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeView *treeview = NULL;
  GtkTreeModel *model = NULL;

  GtkTreeIter tree_iter;

  GtkTreePath* path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;

  gint idx_cls = -1;
  gint idx_clr = -1;
  gint depth = -1;
      
  PxmCleaveSpec *cls = NULL;
  PxmCleaveRule *clr = NULL;
  
  GPtrArray *cleavespecGPA = NULL;

  /* We only remove the item that is selected. If no item is selected,
   * we return.
   */

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  cleavespecGPA = defctxt->polchemdef->cleavespecGPA;
  g_assert (cleavespecGPA != NULL);



  /* Get the cleavespec treeview model from the window pointer passed
   * a data parameter.
   */
  model = (GtkTreeModel *) 
    g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
		       "cleavespec_treeview_model");
  g_assert (model != NULL);
  
  treeview= g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			       "cleavespec_treeview");
  g_assert (treeview != NULL);
  
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter);

  /* If there is no current selection, return.
   */
  if (result == FALSE)
    return;
  
  /* Get the path to the selected item, which can be a cleavespec row
   * or a cleaverule row, which are two very different cases that
   * we must handle. Specifically, if depth == 1, the selected row
   * is a cleavespec, if depth == 2, then it is a cleaverule.
   */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);
      
  /* Get the depth of the currently selected item, which is 1
   * if it is a cleavespec item or 2 if it is a cleaverule item.
   */
  depth = gtk_tree_path_get_depth (path);
  //printf ("depth=%d\n", depth);
  
  /* Whatever the depth, we can get the index of the cleavespec
   * that is concerned, be it selected or one of its cleaverule
   * children.
   */
  idx_cls = gtk_tree_path_get_indices (path) [0];
  //printf ("idx_cls=%d\n", idx_cls);
    
  /* If depth == 2, that means that a cleaverule is selected. We must
   * get to know at what index it is and we will remove it.
   */
  if (depth == 2)
    {
      idx_clr = gtk_tree_path_get_indices (path) [1];
      //printf ("idx_clr=%d\n", idx_clr);
      
      /* Remove this cleaverule !
       */
      cls = g_ptr_array_index (cleavespecGPA, idx_cls);
      g_assert (cls != NULL);
      
      clr = g_ptr_array_remove_index (cls->clrGPA, idx_clr);
      g_assert (clr != NULL);
      pxmchem_cleaverule_free (clr);

      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
      
      gtk_tree_store_remove (GTK_TREE_STORE (model),
			     &tree_iter);
    }
  else if (depth == 1)
    {
      /* Remove this cleavespec !
       */
      cls = g_ptr_array_remove_index (cleavespecGPA, idx_cls);
      g_assert (cls != NULL);
      
      pxmchem_cleavespec_free (cls);
      
      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

      gtk_tree_store_remove (GTK_TREE_STORE (model),
			     &tree_iter);
    }
  else
    {
      g_error (_("%s@%d: could not remove item from treeview\n"),
	     __FILE__, __LINE__);
    }

  /* Finally we can free the path.
   */
  gtk_tree_path_free (path);


  return ;
}




