/*  Splash_Screen

PIRL CVS ID: Splash_Screen.java,v 1.6 2012/04/16 06:22:59 castalia Exp

Copyright (C) 2007-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Viewers;

import javax.swing.JWindow;
import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JProgressBar;
import javax.swing.Timer;
import javax.swing.AbstractAction;
import java.awt.event.ActionEvent;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Toolkit;


/**	A <i>Splash_Screen</i> is an application start-up image window.
<p>
	The splash screen image will be displayed in a borderless window
	centered on the display. A progress bar will be provided across the
	bottom of the window. The image is optional.
<p>
	An optional parent window may be specified that will be displayed
	when the splash screen is {@link #Stop() stopped}. This is typically
	the application GUI.
<p>
	The splash screen may be set to remain displayed for a fixed
	duration, or allowed to run indefinately until it is explicitly
	stopped.
<p>
	@author	Bradford Castalia, UA/PIRL
	@version 1.6
*/
public class Splash_Screen
	extends JWindow
{
public static final String
	ID = "PIRL.Viewers.Splash_Screen (1.6 2012/04/16 06:22:59)";

/**	Duration value for indefinate display.
*/
public static final int
	INDEFINATE_DURATION	= -1;

private JLabel
	Splash				= new JLabel ();

private JProgressBar
	Progress_Bar		= new JProgressBar ();

private JFrame
	Parent				= null;

private int
	Priority			= Thread.currentThread ().getPriority (),
	Duration			= INDEFINATE_DURATION,
	Elapsed				= 0;

private Timer
	Activity_Timer		= new Timer (1000, new Splash_Progress ());


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTORS	= 1 << 0,
	DEBUG_ACCESSORS		= 1 << 1,
	DEBUG_MANIPULATORS	= 1 << 2,
	DEBUG_ALL			= -1,

	DEBUG		    	= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a Splash_Screen.
<p>
	<b>Note</b>: When the splash screen image is {@link
	Icons#Load_Icon(String) loaded} from the image source its full
	pathname need not be specified if the Icons {@link
	Icons#Directory(String) directory} has already been specified. This
	is particularly useful if a {@link Icons#Directory(String, Class)
	class-relative directory} was specified.
<p>
	@param	image_source	The splash screen image source. This may be
		a file pathname or a URL. If null, or the source is not found,
		the screen will be displayed without an image; only a progress
		bar.
	@param	parent	A JFrame window, or null.
	@param	duration	The duration of the screen display, in seconds.
		If negative the duration is indefinate.
*/
public Splash_Screen
	(
	String	image_source,
	JFrame	parent,
	int		duration
	)
{
super (parent);

if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> Splash_Screen:\n"
		+"      image_source - " + image_source + '\n'
		+"    duration - " + duration);

Parent = parent;

getContentPane ().setLayout (new BorderLayout ());

ImageIcon
	image = Icons.Load_Icon (image_source);
if (image != null)
	{
	Splash.setIcon (image);
	getContentPane ().add (Splash);
	}

Progress_Bar.setStringPainted (true);
if ((Duration = duration) < 0)
	{
	Duration = INDEFINATE_DURATION;
	Progress_Bar.setIndeterminate (true);
	Progress_Bar.setString ("");
	}
else
	{
	Progress_Bar.setMaximum (Duration);
	Progress_Bar.setIndeterminate (false);
	Progress_Bar.setString (null);
	}
getContentPane ().add (Progress_Bar, BorderLayout.SOUTH);
pack ();

Dimension
	window_size = getPreferredSize (),
	screen_size = Toolkit.getDefaultToolkit ().getScreenSize ();

if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("    screen size = " + screen_size + '\n'
		+"    window size = " + window_size + '\n'
		+"       location = "
			+ ((screen_size.width  / 2) - (window_size.width  / 2)) + ','
			+ ((screen_size.height / 2) - (window_size.height / 2)));
setLocation
	((screen_size.width  / 2) - (window_size.width  / 2),
	 (screen_size.height / 2) - (window_size.height / 2));

//	Every second counts.
Activity_Timer.setCoalesce (false);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< Splash_Screen");
}

/**	Construct an indefinate duration Splash_Screen with a parent window.
<p>
	@param	image_source	The splash screen image source. This may be
		a file pathname or a URL. If null, or the source is not found,
		the screen will be displayed without an image; only a progress
		bar.
	@param	parent	A JFrame window, or null.
*/
public Splash_Screen
	(
	String	image_source,
	JFrame	parent
	)
{this (image_source, parent, -1);}

/**	Construct a Splash_Screen.
<p>
	@param	image_source	The splash screen image source. This may be
		a file pathname or a URL. If null, or the source is not found,
		the screen will be displayed without an image; only a progress
		bar.
	@param	duration	The duration of the screen display, in seconds.
		If negative the duration is indefinate.
*/
public Splash_Screen
	(
	String	image_source,
	int		duration
	)
{this (image_source, null, duration);}

/**	Construct an indefinate duration Splash_Screen.
<p>
	@param	image_source	The splash screen image source. This may be
		a file pathname or a URL. If null, or the source is not found,
		the screen will be displayed without an image; only a progress
		bar.
*/
public Splash_Screen
	(
	String	image_source
	)
{this (image_source, null, -1);}

/**	Construct an indefinate duration Splash_Screen.
<p>
	The splash screen display will only have a progress bar.
*/
public Splash_Screen ()
{this (null, null, -1);}

/*==============================================================================
	Accessors
*/
/**	Set the parent window.
<p>
	If the parent is non-null it will be displayed when the splash is
	{@link #Stop() stopped}.
<p>
	@param	parent	A JFrame window, or null.
	@return	This Splash_Screen
*/
public Splash_Screen Parent
	(
	JFrame	parent
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">-< Splash_Screen.Parent");
Parent = parent;
return this;
}

/**	Get the parent window.
<p>
	@return	The parent JFrame window, or null.
*/
public JFrame Parent ()
{return Parent;}

/**	Set the splash screen image.
<p>
	If the splash screen is currently being displayed, it is {@link #Stop()
	stopped} (but the parent window, if any, is first saved, set to null,
	and then restored after the image has been set).
<p>
	<b>Note</b>: When the splash screen image is {@link
	Icons#Load_Icon(String) loaded} from the image source its full
	pathname need not be specified if the Icons {@link
	Icons#Directory(String) directory} has already been specified. This
	is particularly useful if a {@link Icons#Directory(String, Class)
	class-relative directory} was specified.
<p>
	@param	image_source	The splash screen image source. This may be
		a file pathname or a URL. If null, or the source is not found,
		the screen will be displayed without an image; only a progress
		bar.
	@see	Icons#Load_Icon(String)
*/
public Splash_Screen Image
	(
	String	image_source
	)
{
JFrame
	parent = Parent;
Parent = null;
Stop ();

ImageIcon
	image = Icons.Load_Icon (image_source);
if (image != null)
	{
	if (Splash.getIcon () == null)
		getContentPane ().add (Splash, 0);
	}
else
	{
	if (Splash.getIcon () != null)
		getContentPane ().remove (0);
	}
Parent = parent;
Splash.setIcon (image);

validate ();
pack ();

Dimension
	window_size = getPreferredSize (),
	screen_size = Toolkit.getDefaultToolkit ().getScreenSize ();
setLocation
	((screen_size.width  / 2) - (window_size.width  / 2),
	 (screen_size.height / 2) - (window_size.height / 2));
return this;
}

/**	Set the duration of the screen display.
<p>
	If the duration is negative the screen will be displayed
	indefinately. In this case an indeterminate progress bar will be
	used. Otherwise a progress bar displaying the percent progress to the
	duration will be used.
<p>
	@param	duration	The duration of the screen display, in seconds.
		If negative the duration is indefinate.
	@return	This Splash_Screen.
*/
public Splash_Screen Duration
	(
	int		duration
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> Splash_Screen.Duration: " + duration);
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("    Duration = " + Duration);
if (duration < 0)
	duration = INDEFINATE_DURATION;
if (Progress_Bar.isIndeterminate ())
	{
	if (duration >= 0)
		{
		//	Change from indeterminate to determinate.
		if ((DEBUG & DEBUG_ACCESSORS) != 0)
			System.out.println
				("    Change from indeterminate to determinate");
		Progress_Bar.setMaximum (Duration = duration);
		Progress_Bar.setValue (Math.min (duration, Elapsed));
		Progress_Bar.setIndeterminate (false);
		Progress_Bar.setString (null);
		}
	}
else
	{
	if ((Duration = duration) < 0)
		{
		//	Change from determinate to indeterminate.
		if ((DEBUG & DEBUG_ACCESSORS) != 0)
			System.out.println
				("    Change from determinate to indeterminate");
		Progress_Bar.setIndeterminate (true);
		Progress_Bar.setString ("");
		}
	}
if (Activity_Timer.getDelay () == 500)
	{
	--Elapsed;
	Activity_Timer.setDelay (1000);
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< Splash_Screen.Duration: " + Duration);
return this;
}

/**	Get the screen display duration.
<p>
	@return	The duration (in seconds) of the screen display. If negative,
		the screen duration is indefinate.
*/
public int Duration ()
{return Duration;}

/**	Get the screen display elapsed time.
<p>
	@return	The elapsed time (in seconds) since the screen display was
		started. This will be zero if the screen is not being displayed
		or has just started being displayed.
*/
public int Elapsed ()
{return Elapsed;}

/*==============================================================================
	Manipulators
*/
/**	Start the splash screen display.
<p>
	The current thread priority is saved and then set to the maximum
	priority. The splash screen is then displayed. If the screen is
	running for a set {@link #Duration(int) duration} a timer is started.
<p>
	@see	#Stop()
*/
public Splash_Screen Start ()
{
if (! Activity_Timer.isRunning ())
	{
	Priority = Thread.currentThread ().getPriority ();
	if ((DEBUG & DEBUG_MANIPULATORS) != 0)
		System.out.println
			(">>> Splash_Screen.Start\n"
			+"    current thread group max priority: "
				+ Thread.currentThread ().getThreadGroup ().getMaxPriority () + '\n'
			+"    current thread priority: " + Priority + '\n'
			+"    new thread priority: " + Thread.MAX_PRIORITY);
	Thread.currentThread ().setPriority (Thread.MAX_PRIORITY);

	setVisible (true);
	Activity_Timer.setDelay (1000);
	Activity_Timer.start ();
	if ((DEBUG & DEBUG_MANIPULATORS) != 0)
		System.out.println
			("<<< Splash_Screen.Start");
	}
return this;
}

/**	Stop the splash screen.
<p>
	If the screen is running for a set {@link #Duration(int) duration} the
	timer is stopped. The screen is removed from the display and the
	current thread's priority is restored to its intitial value. If
	a non-null {@link #Parent(JFrame) parent window} has been bound to
	this splash screen it is displayed.
<p>
	@see	#Start()
*/
public void Stop ()
{
if (Activity_Timer.isRunning ())
	{
	if ((DEBUG & DEBUG_MANIPULATORS) != 0)
		System.out.println
			(">>> Splash_Screen.Stop");
	Activity_Timer.stop ();
	Activity_Timer.setDelay (1000);
	Elapsed = 0;
	setVisible (false);
	Thread.currentThread ().setPriority (Priority);
	if (Parent != null)
		Parent.setVisible (true);
	if ((DEBUG & DEBUG_MANIPULATORS) != 0)
		System.out.println
			("<<< Splash_Screen.Stop");
	}
}

/**	Splash screen progress Action attached to the Activity_Timer.
*/
private class Splash_Progress
	extends AbstractAction
{
public void actionPerformed (ActionEvent event)
{
++Elapsed;
if ((DEBUG & DEBUG_MANIPULATORS) != 0)
	System.out.println
		(">-< Splash_Screen.Activity_Timer: Elapsed = " + Elapsed);
if (Duration >= 0)
	{
	if (Elapsed <= Duration)
		{
		Progress_Bar.setValue (Elapsed);
		if (Elapsed == Duration)
			Activity_Timer.setDelay (500);
		}
	else
		Stop ();
	}
}
}	//	End of Splash_Progress class.


}
