/*	Kapellmeister

PIRL CVS ID: Kapellmeister.java,v 1.111 2012/04/16 06:04:11 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor.Maestro;

import	PIRL.Conductor.Conductor;
import	PIRL.Conductor.Manager;
import	PIRL.Conductor.Management;
import	PIRL.Conductor.Colors;
import	PIRL.Configuration.*;
import	PIRL.PVL.Parameter;
import	PIRL.PVL.PVL_Exception;
import	PIRL.Viewers.View_Locator;
import	PIRL.Viewers.Dialog_Box;
import	PIRL.Viewers.Password_Dialog;
import	PIRL.Viewers.Icons;
import	PIRL.Viewers.Stream_Monitor;
import	PIRL.Viewers.Memory_Chart_Panel;
import	PIRL.Viewers.Parameter_View;
import	PIRL.Messenger.*;
import	PIRL.PVL.*;
import	PIRL.Utilities.Host;
import	PIRL.Strings.String_Buffer;

import	javax.swing.JFrame;
import	javax.swing.JRootPane;
import	javax.swing.JMenuBar;
import	javax.swing.JMenu;
import	javax.swing.JPopupMenu;
import	javax.swing.JMenuItem;
import	javax.swing.JCheckBoxMenuItem;
import	javax.swing.JFileChooser;
import	javax.swing.JPanel;
import	javax.swing.JSplitPane;
import	javax.swing.JButton;
import	javax.swing.ImageIcon;
import	javax.swing.JComboBox;
import	javax.swing.JTable;
import	javax.swing.ListSelectionModel;
import	javax.swing.event.ListSelectionListener;
import	javax.swing.event.ListSelectionEvent;
import	javax.swing.JScrollPane;
import	javax.swing.BorderFactory;
import	javax.swing.Box;
import	javax.swing.text.SimpleAttributeSet;
import	javax.swing.text.StyleConstants;
import	javax.swing.KeyStroke;
import	javax.swing.ToolTipManager;
import	javax.swing.Action;
import	javax.swing.AbstractAction;
import	javax.swing.SwingUtilities;
import	javax.swing.UIManager;
import	java.awt.Toolkit;
import	java.awt.Point;
import	java.awt.Dimension;
import	java.awt.Rectangle;
import	java.awt.GridBagLayout;
import	java.awt.GridBagConstraints;
import	java.awt.BorderLayout;
import	java.awt.Insets;
import	java.awt.Cursor;
import	java.awt.event.ActionListener;
import	java.awt.event.ActionEvent;
import	java.awt.event.ItemListener;
import	java.awt.event.ItemEvent;
import	java.awt.event.WindowAdapter;
import	java.awt.event.WindowEvent;
import	java.awt.event.KeyEvent;
import	java.awt.event.MouseEvent;
import	java.awt.event.MouseAdapter;

import	java.net.InetSocketAddress;
import	java.io.File;
import	java.io.PrintStream;
import	java.io.IOException;
import	java.io.FileNotFoundException;
import	java.io.EOFException;
import	java.util.Collections;
import	java.util.List;
import	java.util.Vector;
import	java.util.Hashtable;


/**	A <i>Kapellmeister</i> is a management tool for Theaters where
	Conductors are located.
<p>
	A Kapellmeister provides a GUI for the management of Theater
	locations. The Kapelmeister maintains a list of all Theaters it knows
	about along with the connected and disconnected status for the
	Stage_Manager at the Theater location. A summary status report from
	any connected Stage_Manager may be obtained for a selected Theater
	location.
<p>
	The Stage_Manager at each Theater location is used to communicate
	with a set of Conductors assigned to the Theater. A Kapellmeister
	provides a table of all Conductors, and the Theater of each, along
	with the current processing status of each Conductor. A Profile table
	in the form of a matrix displays the counts of Conductors in each of
	the possible processing states, along with the total number of
	Conductors, for each unique combination of Theater and named
	Conductor, The name of a Conductor is typically the same as the name
	of the pipeline it is managing.
<p>
	A Manager window may be opened onto any Conductor. The Manager offers
	detailed monitoring and management of the Conductor operation. A
	Kapellmeister provides basic start, stop, error reset, and quit
	controls that may be applied to selections of Conductors. The
	Kapelmeister can be used to define new Conductor instances to be run
	at any Theater location.
<p>
	A file that defines the current Conductors and Theaters Profile may
	be generated by a Kapellmeister. A Profile file may be read by a
	Kapellmeister to produce the defined Conductor instances on all the
	specified Theaters. Defined Conductor instances that are not already
	present will be started; those that are already present will be quit,
	or stopped if they are currently running (they may optionally be
	aborted). Profile files, which are in Parameter Value Language text
	format, may be written by users as desired. Profile files that only
	specify Theater locations may be used to profide a Kapellmeister with
	a list of Theater locations to connect to and simply discover the
	Conductors that are operating there. A Profile file or a set of
	Theater locations may be specified on the Kapellmeister command line
	or interactively specified using the GUI.
<p>
	All Conductor state information is provided directory from the
	operating Conductors via the Stage_Mangers; no Database tables or
	other intermediary information repositories are used. Conductor state
	information arrives and is posted to the Kapellmeister and Manager
	displays in real-time; no data polling cycles are used.
<p>
	Connection to a Stage_Manager at a Theater location typically must be
	authenticated with a secure, public-private, encoded key exchange.
	The keys are based on a password - a character string of any length -
	that is known to the Stage_Manager and made known to a Kapellmeister.
<p>
	A Configuration file is used to provide parameter values, including a
	default Stage_Manager password, that initialize various Stage_Manager
	communication variables and initial GUI control states.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.111
	@see		Theater
	@see		Stage_Manager
	@see		Profile
*/
public class Kapellmeister
	extends JFrame
	implements Message_Delivered_Listener
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Maestro.Kapellmeister (1.111 2012/04/16 06:04:11)";


/**	The name included in the {@link #KAPELLMEISTER_IDENTITY} and used
	to {@link #Config_Pathname(String) find Configuration parameters}.
*/
public static final String
	KAPELLMEISTER_NAME			= "Kapellmeister";

/**	The Messenger client identity Message.
<p>
	The identity contains the {@link #KAPELLMEISTER_NAME}
	{@link Message#Identity(String) Identity}, a
	{@link Message#CLASS_ID_PARAMETER_NAME} parameter with the
	{@link #ID} value, and a {@link Configuration#HOST} parameter
	with the {@link Host#FULL_HOSTNAME} value.
*/
public static final Message
	KAPELLMEISTER_IDENTITY
		= Message
		.Identity (KAPELLMEISTER_NAME)
		.Set (Message.CLASS_ID_PARAMETER_NAME, ID)
		.Set (Configuration.HOST, Host.FULL_HOSTNAME);


//	Configuration.

/**	The default Configuration source filename.
*/
public static final String
	DEFAULT_CONFIGURATION_FILENAME	= "Kapellmeister.conf";

private Configuration
	Kapellmeister_Configuration		= null;

public static final String
	THEATERS_PARAMETER_NAME
		= Profile.THEATERS_PARAMETER_NAME,
		THEATER_LOCATION_PARAMETER_NAME
			= Theater.THEATER_LOCATION_PARAMETER_NAME,
		ADDRESS_PARAMETER_NAME
			= Theater.ADDRESS_PARAMETER_NAME,
		THEATER_KEY_PARAMETER_NAME
			= Theater.KEY_PARAMETER_NAME,
	CONDUCTORS_PARAMETER_NAME
		= "Conductors",
		PIPELINE_PARAMETER_NAME
			= Conductor_Definition.PIPELINE_PARAMETER_NAME,
		CATALOG_PARAMETER_NAME
			= Conductor_Definition.CATALOG_PARAMETER_NAME,
		//	Conductor identity uses a different name tban Conductor_Definition.
		CONFIGURATION_SOURCE_PARAMETER_NAME
			= Conductor_Definition.CONFIGURATION_SOURCE_PARAMETER_NAME,
		DATABASE_SERVER_PARAMETER_NAME
			= Conductor_Definition.SERVER_PARAMETER_NAME,
	CONFIRM_CONDUCTOR_ABORT_PARAMETER_NAME
		= "Confirm_Conductor_Abort",
	PROFILE_WAIT_TO_START_PARAMETER_NAME
		= "Profile_Wait_to_Start",
	PORT_PARAMETER_NAME
		= Stage_Manager.PORT_PARAMETER_NAME,
	RECEIVE_TIMEOUT_PARAMETER_NAME
		= "Receive_Timeout",
	PASSWORD_PARAMETER_NAME
		= Stage_Manager.PASSWORD_PARAMETER_NAME,
	TOOLTIPS_PARAMETER_NAME
		= Manager.TOOLTIPS_PARAMETER_NAME;

private Parameter_View
	Configuration_View				= null;


//	Main window.

public static String
	Kapellmeister_Icon_Source		= "Kapellmeister_Icon";
private static ImageIcon
	Kapellmeister_Icon				= null;

private static final int
	DEFAULT_WINDOW_LOCATION_X		= 100,
	DEFAULT_WINDOW_LOCATION_Y		= 50,
	DEFAULT_WINDOW_WIDTH			= 700,
	DEFAULT_WINDOW_HEIGHT			= 400,
	DEFAULT_LIST_WIDTH				= 250,
	DEFAULT_LIST_HEIGHT				= 100;

private JCheckBoxMenuItem
	Tooltips_Checkbox;
private static boolean
	DEFAULT_TOOLTIPS_ENABLED		= true;


//	Theaters.

private Hashtable<String, String>
	Theater_Keys					= new Hashtable<String, String> ();
private String
	Skeleton_Key					= null;

/**	Value used to display the Configuration password parameters.
*/
public static final String
	MASKED_PASSWORD					= "****";

//	Theater locations displayed by the Theater_List.
private Theater_List_Model
	Theaters						= new Theater_List_Model ();
private Theater_List
	Theater_List;
private ItemListener
	Theater_List_ItemListener;
private Action
	Theater_Status_Action			= null;

private static final int
	MAX_VISIBLE_THEATERS			= 20;

private JButton
	Open_Theater_Button;
private static final String
	OPEN_THEATER_ICON_NAME			= "Open_Theater",
	CLOSE_THEATER_ICON_NAME			= "Close_Theater";
private static ImageIcon
	Open_Theater_Icon				= null,
	Close_Theater_Icon				= null;


//	Conductors.

/**	The name of Conductor identities from the Stage_Manager.
*/
public static final String
	CONDUCTOR_NAME					= Conductor.CONDUCTOR_GROUP;

//	Conductor identities displayed by the Conductor_Table.
private Conductor_Table_Model
	Conductors						= new Conductor_Table_Model ();

//	Table of Conductors reported by the Stage_Manager.
private Conductor_Table
	Conductor_Table;
private Conductor_Table_ListSelectionListener
	Conductor_Table_Selection_Listener;

//	Conductors menu:
private JPopupMenu
	Conductor_Table_Popup_Menu;
private Action
	New_Conductor_Action			= null,
	Open_Conductor_Action			= null,
	Close_Conductor_Action			= null,
	Start_Conductor_Action			= null,
	Reset_Conductor_Action			= null,
	Stop_Conductor_Action			= null,
	Quit_Conductor_Action			= null;
private JCheckBoxMenuItem
	Confirm_Conductor_Abort_Checkbox,
	Wait_to_Start_Checkbox;

private boolean
	DEFAULT_CONFIRM_CONDUCTOR_ABORT	= true,
	DEFAULT_PROFILE_WAIT_TO_START	= true;

private New_Conductor_Dialog
	New_Conductor_Dialog;


//	Table of Theater by Conductor counts.
private Conductor_Matrix_Model
	Matrix;
private Conductor_Matrix
	Conductor_Matrix;

//	Profile menu:
private JPopupMenu
	Conductor_Matrix_Popup_Menu;
private JCheckBoxMenuItem
	Profile_Running_Count_Checkbox,
	Profile_Running_Count_Checkbox_Popup,
	Profile_Polling_Count_Checkbox,
	Profile_Polling_Count_Checkbox_Popup,
	Profile_Waiting_Count_Checkbox,
	Profile_Waiting_Count_Checkbox_Popup,
	Profile_Halted_Count_Checkbox,
	Profile_Halted_Count_Checkbox_Popup,
	Profile_Count_Annotate_Checkbox,
	Profile_Count_Annotate_Checkbox_Popup;

//	Current Conductor Matrix Profile.
private Profile
	Current_Profile					= null;

//	The pathname to the most recent Current_Profile.
private String
	Profile_Pathname				= null;

private JFileChooser
	File_Chooser					= new JFileChooser
										(System.getProperty ("user.dir"));


//	Monitor of communications and events with the Stage_Manager.

private JFrame
	Monitor_Window;
private Stream_Monitor
	Monitor;
private PrintStream
	Report;
private Memory_Chart_Panel
	Memory_Panel;
private Parameter_View
	Theater_Status_Report_View		= null;
private JCheckBoxMenuItem
	Monitor_Reports_Checkbox,
	Memory_Monitor_Checkbox;

/**	Message monitor auto-styling attributes.
*/
public static final SimpleAttributeSet
	MESSAGE_DELIVERED_STYLE			= new SimpleAttributeSet (),
	MESSAGE_SENT_STYLE				= new SimpleAttributeSet ();
static
	{
	StyleConstants.setFontFamily	(MESSAGE_DELIVERED_STYLE, "Monospaced");
	StyleConstants.setBold			(MESSAGE_DELIVERED_STYLE, true);
	StyleConstants.setBackground	(MESSAGE_DELIVERED_STYLE, Colors.MESSAGE_DELIVERED);

	StyleConstants.setFontFamily	(MESSAGE_SENT_STYLE, "Monospaced");
	StyleConstants.setBold			(MESSAGE_SENT_STYLE, true);
	StyleConstants.setBackground	(MESSAGE_SENT_STYLE, Colors.MESSAGE_SENT);
	}


/**	Success exit status (0).
*/
public static final int
	EXIT_SUCCESS					= 0;

/**	Command line syntax problem exit status (1).
*/
public static final int
	EXIT_COMMAND_LINE_SYNTAX		= 1;

/**	The log file could not be opened for writing exit status.
*/
public static final int
	EXIT_CONFIG_FILE_PROBLEM		= 2;

/**	A fatal unrecoverable error was encountered.
*/
public static final int
	EXIT_UNRECOVERABLE_ERROR		= 9;

/**	An unexpected exception occured.
*/
public static final int
	EXIT_UNEXPECTED_EXCEPTION		= 10;


//	Miscellaneous.

private static final Toolkit
	TOOLKIT							= Toolkit.getDefaultToolkit ();

/*	The menu accelerator key mask.
	This is the Control key by default.
	It is the Command key on Mac systems.
*/
private static final int
	MENU_SHORTCUT_KEY_MASK			= TOOLKIT.getMenuShortcutKeyMask ();
private static final Cursor
	DEFAULT_CURSOR
		= Cursor.getPredefinedCursor (Cursor.DEFAULT_CURSOR),
	WAIT_CURSOR	
		= Cursor.getPredefinedCursor (Cursor.WAIT_CURSOR);

private static String
	NL								= Stage_Manager.NL;

private static final char
	FILE_SEPARATOR
		= System.getProperty ("file.separator").charAt (0);


// Debug control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_CONSTRUCTOR			= 1 << 1,
	DEBUG_CONFIG				= 1 << 2,
	DEBUG_THEATER_CONNECT		= 1 << 3,
	DEBUG_MESSAGES				= 1 << 4,
	DEBUG_DONE					= 1 << 5,
	DEBUG_IDENTITIES			= 1 << 6,
	DEBUG_NEW_CONDUCTOR			= 1 << 7,
	DEBUG_CONDUCTOR_CONNECT		= 1 << 8,
	DEBUG_PROFILE				= 1 << 9,
	DEBUG_SELECTION				= 1 << 10,
	DEBUG_ICONS					= 1 << 11,
	DEBUG_MAIN					= 1 << 12,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a Kapellmeister using the default configuration.
<p>
	The {@link #DEFAULT_CONFIGURATION_FILENAME} will be 
*/
public Kapellmeister ()
	throws	Configuration_Exception
{this (null, null);}


public Kapellmeister
	(
	Vector<String>	sources
	)
	throws	Configuration_Exception
{this (sources, null);}


public Kapellmeister
	(
	Configuration 	configuration
	)
	throws	Configuration_Exception
{this (null, configuration);}


public Kapellmeister
	(
	Vector<String>	sources,
	Configuration 	configuration
	)
	throws	Configuration_Exception
{
super ("Kapellmeister");
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">>> Kapellmeister");

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Configure ...");
Configure (configuration);

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Load_Icons ...");
Load_Icons ();
if (Kapellmeister_Icon != null)
	setIconImage (Kapellmeister_Icon.getImage ());

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Menus ...");
setJMenuBar (Menus ());

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Panels ...");
setContentPane (Panels ());

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    New_Conductor_Dialog ...");
New_Conductor_Dialog =
	new New_Conductor_Dialog ("New Conductor", this);
New_Conductor_Dialog.Wait_to_Start
	(Wait_to_Start_Checkbox.isSelected ());

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Monitor_Window ...");
Monitor_Window ();

pack ();
setSize (new Dimension
	(DEFAULT_WINDOW_WIDTH, DEFAULT_WINDOW_HEIGHT));
setLocation
	(DEFAULT_WINDOW_LOCATION_X, DEFAULT_WINDOW_LOCATION_Y);
setDefaultCloseOperation (JFrame.DO_NOTHING_ON_CLOSE);
addWindowListener (new WindowAdapter ()
	{
	public void windowClosing (WindowEvent event)
		{Exit (EXIT_SUCCESS);}
	public void windowClosed (WindowEvent event)
		{Exit (EXIT_SUCCESS);}
	});

if ((DEBUG & DEBUG_MAIN) != 0)
	System.out.println
		("    Setting the Kapellmeister visible");
setVisible (true);

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Writing ID to Monitor ...");
Report (ID + NL + NL);

if (sources != null)
	{
	if (sources.size () == 1 &&
		new File ((String)sources.get (0)).isFile ())
		{
		if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
			System.out.println
				("    Load_Profile " + (String)sources.get (0) + " ...");
		Load_Profile ((String)sources.get (0));
		}
	else
		{
		if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
			System.out.println
				("    Open_Theaters ...");
		Open_Theaters (sources);
		}
	}
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("<<< Kapellmeister");
}
/*==============================================================================
	Accessors
*/
/**	Get the default Stage_Manager socket communications port number.
<p>
	@return	The communications port number.
	@see	#Default_Theater_Port(int)
*/
public int Default_Theater_Port ()
{return Theater.Default_Port ();}

/**	Set the default Stage_Manager socket communications port number.
<p>
	@param	port	The {@link Theater#Default_Port(int) default port}
		for Stage_Manager communications.
	@return	This Kapellmeister.
*/
public Kapellmeister Default_Theater_Port
	(
	int		port
	)
{Theater.Default_Port (port); return this;}

/**	Get the maximum amount of time, in seconds, that will be used when
	waiting for a Theater protocol Message to be received.
<p>
	@return	The timeout value.
	@see	#Receive_Timeout(int)
*/
public int Receive_Timeout ()
{return Theater.Default_Receive_Timeout ();}

/**	Set the maximum amount of time, in seconds, that will be used when
	waiting for a Theater protocol Message to be received.
<p>
	@param	timeout	The {@link Theater#Default_Receive_Timeout(int)
		default timeout value} which is also applied to all Theaters in
		the Theaters list.
	@return	This Kapellmeister.
*/
public Kapellmeister Receive_Timeout
	(
	int		timeout
	)
{
Theater.Default_Receive_Timeout (timeout);
Theaters.Receive_Timeout (timeout);
return this;
}

/*==============================================================================
	Configuration
*/
/**	Configure the Kapellmeister.
<p>
	Parameter names are case insensitive. All parameters are optional.
	Parameters will first be sought in the {@link #KAPELLMEISTER_NAME}
	group.
<p>
<dl>
<dt>{@link #PORT_PARAMETER_NAME}
<dd>The {@link #Default_Theater_Port(int) default Stage_Manager
	communcations socket port number}.

<dt>{@link #RECEIVE_TIMEOUT_PARAMETER_NAME}
<dd>The {@link #Receive_Timeout(int) maximum amount of time, in seconds,
	that will be used when waiting for a Theater protocol Message to be
	received}.

<dt>{@link Stage_Manager#PASSWORD_PARAMETER_NAME}
<dd>The password required by the Stage_Manager to authenticate
	client connections. If {@link
	Stage_Manager#UNAUTHENTICATED_CONNECTIONS_ALLOWED} is true
	unauthenticated connections with no password will be accepted.

<dt>{@link #TOOLTIPS_PARAMETER_NAME}
<dd>If true, enabled, yes, on, or 1 GUI tooltips will be enabled.
</dl>
<p>
	@param	configuration	A Configuration. If null, an attempt will be
		made to obtain a Configuration using the {@link
		#DEFAULT_CONFIGURATION_FILENAME} source.
*/
public void Configure
	(
	Configuration	configuration
	)
	throws	Configuration_Exception
{
if (configuration == null)
	{
	if ((DEBUG & DEBUG_CONFIG) != 0)
		System.out.println
			("    Trying " + DEFAULT_CONFIGURATION_FILENAME);
	try {configuration = new Configuration (DEFAULT_CONFIGURATION_FILENAME);}
	catch (IllegalArgumentException exception)
		{
		//	No configuration file; use a default.
		if ((DEBUG & DEBUG_CONFIG) != 0)
			System.out.println
				("    Using a default Configuration");
		configuration = new Configuration ();
		}
	}
Kapellmeister_Configuration = configuration;
Kapellmeister_Configuration.Case_Sensitive (false);
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("    Configuration -" + NL
		+ Kapellmeister_Configuration.Description ());

try {Kapellmeister_Configuration.Insert
		(new Parameter (Local_Theater.CLASS_ID_PARAMETER_NAME).Value (ID), 0);}
catch (PVL_Exception exception) {}

//	Stage_Manager

Default_Theater_Port ((int)Kapellmeister_Configuration.Get_Number
	(Config_Pathname (PORT_PARAMETER_NAME),
		Theater.Default_Port ()));
Config_Value (PORT_PARAMETER_NAME,
	new Integer (Theater.Default_Port ()));

Receive_Timeout ((int)Kapellmeister_Configuration.Get_Number
	(Config_Pathname (RECEIVE_TIMEOUT_PARAMETER_NAME),
		Theater.Default_Receive_Timeout ()));
Config_Value (RECEIVE_TIMEOUT_PARAMETER_NAME,
	new Integer (Theater.Default_Receive_Timeout ()));

Skeleton_Key = Config_Value (Stage_Manager.PASSWORD_PARAMETER_NAME);
if (Skeleton_Key != null &&
	Skeleton_Key.length () == 0)
	Skeleton_Key = null;

//	Mask out all password values.
Kapellmeister_Configuration.Set_All ("PASSWORD", MASKED_PASSWORD);
}

/**	Get the String value of a configuration parameter.
<p>
	@param	name	The name of the parameter from which to obtain the
		value. If the name is not an absolute pathname it will be {@link
		#Config_Pathname(String) qualified} before use.
	@return	A String value. This will be null if the parameter could
		not be found in the {@link #Configure(Configuration) configuration}.
*/
public String Config_Value
	(
	String	name
	)
{
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		(">-< Config_Value: " + name + " = "
			+ Kapellmeister_Configuration.Get_One (Config_Pathname (name)));
return Kapellmeister_Configuration.Get_One (Config_Pathname (name));
}

/**	Get the boolean value of a configuration parameter.
<p>
	@param	name	The name of the parameter from which to obtain the
		value. If the name is not an absolute pathname it will be {@link
		#Config_Pathname(String) qualified} before use.
	@param	default_value	The default value to be returned if the
		named parameter is not present in the Configuration.
	@return	The {@link Configuration#Enabled(String, boolean) enabled} state
		of the parameter, or the default value if the parameter is not
		present in the Configuration.
*/
public boolean Config_Flag
	(
	String	name,
	boolean	default_value
	)
{
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		(">-< Config_Flag: " + name + " = "
			+ Kapellmeister_Configuration.Enabled
				(Config_Pathname (name), default_value));
return Kapellmeister_Configuration.Enabled
	(Config_Pathname (name), default_value);
}

/**	Set a parameter in the configuration.
<p>
	The parameter is set in the {@link #Config_Pathname(String) parameter
	group} appropriate for the application.
<p>
	@param	name	The name of the Assignment parameter to have its
		value set. If the name is not an absolute pathname it will be {@link
		#Config_Pathname(String) qualified} before use.
	@param	value	An Object to use for the parameter's value.
		<b.N.B.</b>: If null, the parameter will have no value; it will
		be a Token.
	@return	true if an existing parameter by the same name was replaced;
		false if the parameter is being set for the first time.
	@throws	Configuration_Exception	If there was a problem setting
		the parameter.
*/
protected boolean Config_Value
	(
	String			name,
	Object			value
	)
throws Configuration_Exception
{
try {return Kapellmeister_Configuration.Set (Config_Pathname (name), value);}
catch (Configuration_Exception exception)
	{
	throw new Configuration_Exception (ID + NL
		+"Unable to set configuration parameter: " + name + " = " + value + NL
		+ exception.getMessage ());
	}
}

/**	Get the configuration pathname for a parameter name.
<p>
	If the name is not an {@link Configuration#Is_Absolute_Pathname(String)
	absolute pathname} the {@link #KAPELLMEISTER_NAME} group name is
	prepended to the name to form an absolute pathname.
<p>
	@param	name	A parameter name String.
	@return	The possibily modified abosulte pathname String.
*/
public String Config_Pathname
	(
	String	name
	)
{
if (name != null &&
	! Configuration.Is_Absolute_Pathname (name))
	return
		Kapellmeister_Configuration.Path_Delimiter () + KAPELLMEISTER_NAME +
		Kapellmeister_Configuration.Path_Delimiter () + name;
return name;
}

/*==============================================================================
	GUI
*/
private JMenuBar Menus ()
{
JMenuBar
	menu_bar = new JMenuBar ();

JMenu
	menu;
JMenuItem
	menu_item;

//	File.
menu = new JMenu ("File");
menu.setMnemonic (KeyEvent.VK_F);

menu_item = new JMenuItem ("Open Profile ...");
menu_item.setMnemonic (KeyEvent.VK_O);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_O, MENU_SHORTCUT_KEY_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Select_Profile ();}});
menu.add (menu_item);

menu_item = new JMenuItem ("Save");
menu_item.setMnemonic (KeyEvent.VK_S);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_S, MENU_SHORTCUT_KEY_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Save_Profile (false);}});
menu.add (menu_item);

menu_item = new JMenuItem ("Save As ...");
menu_item.setMnemonic (KeyEvent.VK_A);
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Save_Profile (true);}});
menu.add (menu_item);

File_Chooser.setFileSelectionMode (JFileChooser.FILES_ONLY);
File_Chooser.setFileHidingEnabled (false);

menu.addSeparator ();

menu_item = new JMenuItem ("Exit"); 
menu_item.setMnemonic (KeyEvent.VK_X);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_Q, MENU_SHORTCUT_KEY_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Exit (EXIT_SUCCESS);}});
menu.add (menu_item);

menu_bar.add (menu);

//	View.
menu = new JMenu ("View");
menu.setMnemonic (KeyEvent.VK_V);

menu_item = new JMenuItem ("Configuration");
menu_item.setMnemonic (KeyEvent.VK_C);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_C, ActionEvent.ALT_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{View_Configuration ();}});
menu.add (menu_item);

Theater_Status_Action = new AbstractAction ("Theater Status")
	{public void actionPerformed (ActionEvent event)
	{Theater_Status_Report ();}};
Theater_Status_Action.setEnabled (false);
Theater_Status_Action.putValue (Action.MNEMONIC_KEY, KeyEvent.VK_S);
Theater_Status_Action.putValue (Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke
	(KeyEvent.VK_S, ActionEvent.ALT_MASK));
menu.add (new JMenuItem (Theater_Status_Action));

menu_item = new JMenuItem ("Monitor");
menu_item.setMnemonic (KeyEvent.VK_M);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_M, ActionEvent.ALT_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Show_Monitor ();}});
menu.add (menu_item);

menu.addSeparator ();

Tooltips_Checkbox
	= new JCheckBoxMenuItem ("Tooltips",
		Config_Flag (TOOLTIPS_PARAMETER_NAME, DEFAULT_TOOLTIPS_ENABLED));
Tooltips_Checkbox.setMnemonic (KeyEvent.VK_T);
Tooltips_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{ToolTipManager.sharedInstance ().setEnabled
		(Tooltips_Checkbox.isSelected ());}});
menu.add (Tooltips_Checkbox);

menu_bar.add (menu);

//	Conductors.
menu = new JMenu ("Conductors");
menu.setMnemonic (KeyEvent.VK_C);

New_Conductor_Action = new AbstractAction ("New ...")
	{public void actionPerformed (ActionEvent event)
	{New_Conductor ();}};
New_Conductor_Action.setEnabled (false);
New_Conductor_Action.putValue (Action.MNEMONIC_KEY, KeyEvent.VK_N);
New_Conductor_Action.putValue (Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke
	(KeyEvent.VK_N, MENU_SHORTCUT_KEY_MASK));
menu.add (new JMenuItem (New_Conductor_Action));

menu.addSeparator ();

Open_Conductor_Action = new AbstractAction ("Open")
	{public void actionPerformed (ActionEvent event)
	{Open_Manager ();}};
Open_Conductor_Action.setEnabled (false);
Open_Conductor_Action.putValue (Action.MNEMONIC_KEY, KeyEvent.VK_O);
Open_Conductor_Action.putValue (Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke
	(KeyEvent.VK_O, ActionEvent.ALT_MASK));
menu.add (new JMenuItem (Open_Conductor_Action));

Close_Conductor_Action = new AbstractAction ("Close")
	{public void actionPerformed (ActionEvent event)
	{Close_Manager ();}};
Close_Conductor_Action.setEnabled (false);
Close_Conductor_Action.putValue (Action.MNEMONIC_KEY, KeyEvent.VK_C);
Close_Conductor_Action.putValue (Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke
	(KeyEvent.VK_W, ActionEvent.ALT_MASK));
menu.add (new JMenuItem (Close_Conductor_Action));

menu.addSeparator ();

Start_Conductor_Action = new AbstractAction ("Start")
	{public void actionPerformed (ActionEvent event)
	{Start_Conductor ();}};
Start_Conductor_Action.setEnabled (false);
Start_Conductor_Action.putValue (Action.MNEMONIC_KEY, KeyEvent.VK_T);
Start_Conductor_Action.putValue (Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke
	(KeyEvent.VK_T, ActionEvent.ALT_MASK));
menu.add (new JMenuItem (Start_Conductor_Action));

Reset_Conductor_Action = new AbstractAction ("Reset")
	{public void actionPerformed (ActionEvent event)
	{Reset_Conductor ();}};
Reset_Conductor_Action.setEnabled (false);
Reset_Conductor_Action.putValue (Action.MNEMONIC_KEY, KeyEvent.VK_R);
Reset_Conductor_Action.putValue (Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke
	(KeyEvent.VK_R, ActionEvent.ALT_MASK));
menu.add (new JMenuItem (Reset_Conductor_Action));

Stop_Conductor_Action = new AbstractAction ("Stop")
	{public void actionPerformed (ActionEvent event)
	{Stop_Conductor ();}};
Stop_Conductor_Action.setEnabled (false);
Stop_Conductor_Action.putValue (Action.MNEMONIC_KEY, KeyEvent.VK_P);
Stop_Conductor_Action.putValue (Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke
	(KeyEvent.VK_P, ActionEvent.ALT_MASK));
menu.add (new JMenuItem (Stop_Conductor_Action));

Quit_Conductor_Action = new AbstractAction ("Quit")
	{public void actionPerformed (ActionEvent event)
	{Quit_Conductor ();}};
Quit_Conductor_Action.setEnabled (false);
Quit_Conductor_Action.putValue (Action.MNEMONIC_KEY, KeyEvent.VK_Q);
Quit_Conductor_Action.putValue (Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke
	(KeyEvent.VK_Q, ActionEvent.ALT_MASK));
menu.add (new JMenuItem (Quit_Conductor_Action));

menu.addSeparator ();

Confirm_Conductor_Abort_Checkbox
	= new JCheckBoxMenuItem ("Confirm Abort",
		Config_Flag (CONFIRM_CONDUCTOR_ABORT_PARAMETER_NAME,
			DEFAULT_CONFIRM_CONDUCTOR_ABORT));
menu.add (Confirm_Conductor_Abort_Checkbox);

Wait_to_Start_Checkbox
	= new JCheckBoxMenuItem ("Wait-to-Start",
		Config_Flag (PROFILE_WAIT_TO_START_PARAMETER_NAME,
			DEFAULT_PROFILE_WAIT_TO_START));
Wait_to_Start_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{New_Conductor_Dialog.Wait_to_Start
		(Wait_to_Start_Checkbox.isSelected ());}});
menu.add (Wait_to_Start_Checkbox);

menu_bar.add (menu);

//	Profile.
menu = new JMenu ("Profile");
menu.setMnemonic (KeyEvent.VK_P);

Profile_Running_Count_Checkbox = new JCheckBoxMenuItem ("Running", true);
Profile_Running_Count_Checkbox.setBackground (Colors.RUNNING_STATE);
Profile_Running_Count_Checkbox.setMnemonic (KeyEvent.VK_R);
Profile_Running_Count_Checkbox.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_R, ActionEvent.ALT_MASK | ActionEvent.SHIFT_MASK));
Profile_Running_Count_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Profile_Running_Count_Checkbox_Popup.setState
	(Profile_Running_Count_Checkbox.getState ());
	 Profile_Counts ();}});
menu.add (Profile_Running_Count_Checkbox);

Profile_Polling_Count_Checkbox = new JCheckBoxMenuItem ("Polling", true);
Profile_Polling_Count_Checkbox.setBackground (Colors.POLLING_STATE);
Profile_Polling_Count_Checkbox.setMnemonic (KeyEvent.VK_P);
Profile_Polling_Count_Checkbox.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_P, ActionEvent.ALT_MASK | ActionEvent.SHIFT_MASK));
Profile_Polling_Count_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Profile_Polling_Count_Checkbox_Popup.setState
	(Profile_Polling_Count_Checkbox.getState ());
	 Profile_Counts ();}});
menu.add (Profile_Polling_Count_Checkbox);

Profile_Waiting_Count_Checkbox = new JCheckBoxMenuItem ("Waiting", true);
Profile_Waiting_Count_Checkbox.setBackground (Colors.WAITING_STATE);
Profile_Waiting_Count_Checkbox.setMnemonic (KeyEvent.VK_W);
Profile_Waiting_Count_Checkbox.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_W, ActionEvent.ALT_MASK | ActionEvent.SHIFT_MASK));
Profile_Waiting_Count_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Profile_Waiting_Count_Checkbox_Popup.setState
	(Profile_Waiting_Count_Checkbox.getState ());
	 Profile_Counts ();}});
menu.add (Profile_Waiting_Count_Checkbox);

Profile_Halted_Count_Checkbox = new JCheckBoxMenuItem ("Halted", true);
Profile_Halted_Count_Checkbox.setBackground (Colors.HALTED_STATE);
Profile_Halted_Count_Checkbox.setMnemonic (KeyEvent.VK_H);
Profile_Halted_Count_Checkbox.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_H, ActionEvent.ALT_MASK | ActionEvent.SHIFT_MASK));
Profile_Halted_Count_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Profile_Halted_Count_Checkbox_Popup.setState
	(Profile_Halted_Count_Checkbox.getState ());
	 Profile_Counts ();}});
menu.add (Profile_Halted_Count_Checkbox);

menu.addSeparator ();

Profile_Count_Annotate_Checkbox = new JCheckBoxMenuItem ("Annotate", false);
Profile_Count_Annotate_Checkbox.setMnemonic (KeyEvent.VK_A);
Profile_Count_Annotate_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Profile_Count_Annotate_Checkbox_Popup.setState
	(Profile_Count_Annotate_Checkbox.getState ());
	 Conductor_Matrix.State_Annotate
		(Profile_Count_Annotate_Checkbox.isSelected ());}});
menu.add (Profile_Count_Annotate_Checkbox);

menu_bar.add (menu);

return menu_bar;
}


private JPanel Panels ()
{
JPanel
	panel		= new JPanel (new GridBagLayout ()),
	table_panel	= new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.insets		= new Insets (5, 5, 5, 5);
table_panel.add (Theaters_Panel (), location);

location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.insets		= new Insets (0, 5, 5, 5);
table_panel.add (Conductor_Table_Panel (), location);

JSplitPane
	split_pane = new JSplitPane (JSplitPane.HORIZONTAL_SPLIT, 
 		table_panel, Conductor_Matrix_Panel ());
split_pane.setOneTouchExpandable (true);
panel.add (split_pane, location);

return panel;
}

/*------------------------------------------------------------------------------
	Theaters
*/
private JPanel Theaters_Panel ()
{
JPanel
	panel = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

panel.setBorder (BorderFactory.createTitledBorder (THEATERS_PARAMETER_NAME));

//	Open(ed)/Close(d) button:
Open_Theater_Button = new JButton ();
Open_Theater_Button.setBorderPainted (false);
Open_Theater_Button.setContentAreaFilled (false);
if (Open_Theater_Icon != null)
	Open_Theater_Button.setPreferredSize
		(new Dimension
			(Open_Theater_Icon.getIconWidth (),
			 Open_Theater_Icon.getIconHeight ()));
Theater_is_Open (false);
Open_Theater_Button.setEnabled (false);
Open_Theater_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Open_Theater ();}});
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 3, 0, 4);
panel.add (Open_Theater_Button, location);

//	List:
Theater_List		= new Theater_List (Theaters);
Theater_List.setMaximumRowCount (MAX_VISIBLE_THEATERS);
Theater_List.addItemListener (Theater_List_ItemListener = new ItemListener ()
	{public void itemStateChanged (ItemEvent event)
	{if (event.getStateChange () == ItemEvent.SELECTED)
		Theater_Selected ();}});
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.insets		= new Insets (0, 0, 0, 0);
panel.add (Theater_List, location);

return panel;
}

/*------------------------------------------------------------------------------
	Conductors
*/
private JPanel Conductor_Table_Panel ()
{
JPanel
	panel = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

panel.setBorder (BorderFactory.createTitledBorder (CONDUCTORS_PARAMETER_NAME));

Conductor_Table		= new Conductor_Table (Conductors);
Conductor_Table.addMouseListener (new Conductor_Table_Mouse_Listener ());
Conductor_Table.getSelectionModel ().addListSelectionListener
	(Conductor_Table_Selection_Listener =
		new Conductor_Table_ListSelectionListener ());
JScrollPane
	table_pane		= new JScrollPane (Conductor_Table);
table_pane.setMinimumSize (new Dimension (100, 100));
table_pane.setPreferredSize
	(new Dimension (DEFAULT_LIST_WIDTH, DEFAULT_LIST_HEIGHT));
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
panel.add (table_pane, location);

//	Popup menu:
Conductor_Table_Popup_Menu = new JPopupMenu ();
Conductor_Table_Popup_Menu.add (new JMenuItem (New_Conductor_Action));
Conductor_Table_Popup_Menu.addSeparator ();
Conductor_Table_Popup_Menu.add (new JMenuItem (Open_Conductor_Action));
Conductor_Table_Popup_Menu.add (new JMenuItem (Close_Conductor_Action));
Conductor_Table_Popup_Menu.addSeparator ();
Conductor_Table_Popup_Menu.add (new JMenuItem (Start_Conductor_Action));
Conductor_Table_Popup_Menu.add (new JMenuItem (Reset_Conductor_Action));
Conductor_Table_Popup_Menu.add (new JMenuItem (Stop_Conductor_Action));
Conductor_Table_Popup_Menu.add (new JMenuItem (Quit_Conductor_Action));
Conductor_Table_Popup_Menu.addSeparator ();
Conductor_Table_Popup_Menu.add (new JMenuItem (Theater_Status_Action));

return panel;
}


private class Conductor_Table_ListSelectionListener
	implements ListSelectionListener
{
public void valueChanged
	(
	ListSelectionEvent	event
	)
{
if (event.getValueIsAdjusting ())
	return;
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		(">>> Kapellmeister.Conductor_Table_ListSelectionListener.valueChanged");
int
	row = Conductor_Table.getSelectedRow ();
if (row >= 0)
	{
	row = Conductor_Table.convertRowIndexToModel (row);

	//	Select the corresponding Theater.
	if ((DEBUG & DEBUG_SELECTION) != 0)
		System.out.println
			("    Selecting theater at location: "
				+ Conductors.Location (row));
	Theater_Selected (Conductors.Location (row), false);
	}

Refresh_Conductor_Ops_Menus ();
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		("<<< Kapellmeister.Conductor_Table_ListSelectionListener.valueChanged");
}

}

/*..............................................................................
	Conductor_Table_Mouse_Listener
*/
/**	MouseListener for the Conductor_Table.
*/
private class Conductor_Table_Mouse_Listener
	extends MouseAdapter
{
public void mousePressed (MouseEvent event)
{Popup (event);}

public void mouseReleased (MouseEvent event)
{Popup (event);}

private void Popup
	(
	MouseEvent	event
	)
{
if (event.isPopupTrigger ())
	Conductor_Table_Popup_Menu.show
		(event.getComponent (), event.getX (), event.getY ());
}


public void mouseClicked
	(
	MouseEvent event
	)
{
if (event.getButton () == MouseEvent.BUTTON1)
	{
	if (event.getClickCount () == 2)
		Open_Manager ();
	else
	if (event.isShiftDown () &&
		event.isControlDown ())
		{
		Conductor_Table
			table = (Conductor_Table)event.getSource ();
		int
			row = table.rowAtPoint (event.getPoint ());
		if (row >= 0)
			{
			if (table.columnAtPoint (event.getPoint ()) == 0)
				Select_Conductors_at (Conductors.Location (row));
			else
				Select_Conductors_for (Conductors.Conductor_Name (row));
			}
		}
	}
}

}	//	Conductor_Table_Mouse_Listener

/*------------------------------------------------------------------------------
	Conductor Matrix
*/
private JPanel Conductor_Matrix_Panel ()
{
JPanel
	panel = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

panel.setBorder (BorderFactory.createTitledBorder ("Profile"));

Matrix				= new Conductor_Matrix_Model (Conductors);
Conductor_Matrix	= new Conductor_Matrix (Matrix);
Conductor_Matrix.addMouseListener (new Conductor_Matrix_Mouse_Listener ());
Conductor_Matrix.getSelectionModel ()
	.addListSelectionListener (new ListSelectionListener ()
	{public void valueChanged (ListSelectionEvent event)
	{if (! event.getValueIsAdjusting ())
		Matrix_Selected ();}});
JScrollPane
	table_pane		= new JScrollPane (Conductor_Matrix);
table_pane.setMinimumSize (new Dimension (100, 100));
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
panel.add (table_pane, location);

//	Popup menu:
Conductor_Matrix_Popup_Menu = new JPopupMenu ("Conductor State Counts");

Profile_Running_Count_Checkbox_Popup = new JCheckBoxMenuItem ("Running", true);
Profile_Running_Count_Checkbox_Popup.setBackground (Colors.RUNNING_STATE);
Profile_Running_Count_Checkbox_Popup.setMnemonic (KeyEvent.VK_R);
Profile_Running_Count_Checkbox_Popup.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_R, ActionEvent.ALT_MASK | ActionEvent.SHIFT_MASK));
Profile_Running_Count_Checkbox_Popup.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Profile_Running_Count_Checkbox.setState
	(Profile_Running_Count_Checkbox_Popup.getState ());
	 Profile_Counts ();}});
Conductor_Matrix_Popup_Menu.add (Profile_Running_Count_Checkbox_Popup);

Profile_Polling_Count_Checkbox_Popup = new JCheckBoxMenuItem ("Polling", true);
Profile_Polling_Count_Checkbox_Popup.setBackground (Colors.POLLING_STATE);
Profile_Polling_Count_Checkbox_Popup.setMnemonic (KeyEvent.VK_P);
Profile_Polling_Count_Checkbox_Popup.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_P, ActionEvent.ALT_MASK | ActionEvent.SHIFT_MASK));
Profile_Polling_Count_Checkbox_Popup.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Profile_Polling_Count_Checkbox.setState
	(Profile_Polling_Count_Checkbox_Popup.getState ());
	 Profile_Counts ();}});
Conductor_Matrix_Popup_Menu.add (Profile_Polling_Count_Checkbox_Popup);

Profile_Waiting_Count_Checkbox_Popup = new JCheckBoxMenuItem ("Waiting", true);
Profile_Waiting_Count_Checkbox_Popup.setBackground (Colors.WAITING_STATE);
Profile_Waiting_Count_Checkbox_Popup.setMnemonic (KeyEvent.VK_W);
Profile_Waiting_Count_Checkbox_Popup.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_W, ActionEvent.ALT_MASK | ActionEvent.SHIFT_MASK));
Profile_Waiting_Count_Checkbox_Popup.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Profile_Waiting_Count_Checkbox.setState
	(Profile_Waiting_Count_Checkbox_Popup.getState ());
	 Profile_Counts ();}});
Conductor_Matrix_Popup_Menu.add (Profile_Waiting_Count_Checkbox_Popup);

Profile_Halted_Count_Checkbox_Popup = new JCheckBoxMenuItem ("Halted", true);
Profile_Halted_Count_Checkbox_Popup.setBackground (Colors.HALTED_STATE);
Profile_Halted_Count_Checkbox_Popup.setMnemonic (KeyEvent.VK_H);
Profile_Halted_Count_Checkbox_Popup.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_H, ActionEvent.ALT_MASK | ActionEvent.SHIFT_MASK));
Profile_Halted_Count_Checkbox_Popup.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Profile_Halted_Count_Checkbox.setState
	(Profile_Halted_Count_Checkbox_Popup.getState ());
	 Profile_Counts ();}});
Conductor_Matrix_Popup_Menu.add (Profile_Halted_Count_Checkbox_Popup);

Conductor_Matrix_Popup_Menu.addSeparator ();

Profile_Count_Annotate_Checkbox_Popup = new JCheckBoxMenuItem ("Annotate", false);
Profile_Count_Annotate_Checkbox_Popup.setMnemonic (KeyEvent.VK_A);
Profile_Count_Annotate_Checkbox_Popup.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Profile_Count_Annotate_Checkbox.setState
	(Profile_Count_Annotate_Checkbox_Popup.getState ());
	 Conductor_Matrix.State_Annotate
		(Profile_Count_Annotate_Checkbox_Popup.isSelected ());}});

Conductor_Matrix_Popup_Menu.add (Profile_Count_Annotate_Checkbox_Popup);

return panel;
}


private void Profile_Counts ()
{
int
	counts_shown = 0;
if (Profile_Running_Count_Checkbox.isSelected ())
	counts_shown |= Conductor_Matrix.RUNNING_COUNT;
if (Profile_Polling_Count_Checkbox.isSelected ())
	counts_shown |= Conductor_Matrix.POLLING_COUNT;
if (Profile_Waiting_Count_Checkbox.isSelected ())
	counts_shown |= Conductor_Matrix.WAITING_COUNT;
if (Profile_Halted_Count_Checkbox.isSelected ())
	counts_shown |= Conductor_Matrix.HALTED_COUNT;
//	Always show Totals.
counts_shown |= Conductor_Matrix.TOTAL_COUNT;
Conductor_Matrix.Counts_Shown (counts_shown);
}

/*..............................................................................
	Conductor_Matrix_Mouse_Listener
*/
/**	MouseListener for the Conductor_Matrix table.
*/
private class Conductor_Matrix_Mouse_Listener
	extends MouseAdapter
{
public void mousePressed (MouseEvent event)
{Popup (event);}

public void mouseReleased (MouseEvent event)
{Popup (event);}

private void Popup
	(
	MouseEvent	event
	)
{
if (event.isPopupTrigger ())
	Conductor_Matrix_Popup_Menu.show
		(event.getComponent (), event.getX (), event.getY ());
}


public void mouseClicked
	(
	MouseEvent event
	)
{
if (event.getButton () == MouseEvent.BUTTON1)
	{
	}
}

}	//	Conductor_Matrix_Mouse_Listener


/*------------------------------------------------------------------------------
	Monitor
*/
private void Monitor_Window ()
{
Monitor_Window = new JFrame ("Kapellmeister Monitor");
JPanel
	monitor_panel	= new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

Monitor = new Stream_Monitor ()
	.Auto_Style (true)
	.Auto_Style ("==>", Monitor.HIGHLIGHT_STYLE)
	.Auto_Style ("!!!", Monitor.NOTICE_STYLE)
	.Auto_Style ("<<<", MESSAGE_DELIVERED_STYLE)
	.Auto_Style (">>>", MESSAGE_SENT_STYLE);
Monitor.setPreferredSize (new Dimension
	(DEFAULT_WINDOW_WIDTH, DEFAULT_WINDOW_HEIGHT));
Report = new PrintStream (Monitor.Stream (), true);

JMenuBar
	menu_bar = new JMenuBar ();
JMenu
	menu;

//	File menu.
menu = Monitor.File_Menu ();

menu.addSeparator ();

JMenuItem
	menu_item = new JMenuItem ("Close");
menu_item.setMnemonic (KeyEvent.VK_C);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_W, MENU_SHORTCUT_KEY_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Monitor_Window.setVisible (false);}});
menu.add (menu_item);

menu_bar.add (menu);

//	View menu.
menu = Monitor.View_Menu ();

menu.addSeparator ();

Monitor_Reports_Checkbox = new JCheckBoxMenuItem ("Reports", false);
menu.add (Monitor_Reports_Checkbox);

Memory_Monitor_Checkbox = new JCheckBoxMenuItem ("Memory Monitor", false);
Memory_Monitor_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Memory_Panel.Run
		(((JCheckBoxMenuItem)event.getSource ()).isSelected ());}});
menu.add (Memory_Monitor_Checkbox);

menu_bar.add (menu);

Monitor_Window.setJMenuBar (menu_bar);

//	Reports pane.
Monitor.setBorder (BorderFactory.createTitledBorder ("Reports"));
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
monitor_panel.add (Monitor, location);

//	Memory monitor pane.
Memory_Panel = new Memory_Chart_Panel (1, true);
Memory_Panel.Run (Memory_Monitor_Checkbox.isSelected ());
Memory_Panel.setBorder (BorderFactory.createTitledBorder ("Memory Monitor"));
location.weighty	= 0.0;
monitor_panel.add (Memory_Panel, location);

//	Tie the Start/Stop popup menu item to the View Memory Monitor menu item.
JPopupMenu
	popup_menu = Memory_Panel.Menu ();
menu_item = (JMenuItem)popup_menu.getComponent (0);
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Memory_Monitor_Checkbox.setSelected
		(! Memory_Monitor_Checkbox.isSelected ());}});

Monitor_Window.getContentPane ().add (monitor_panel, BorderLayout.CENTER);
Monitor_Window.pack ();

Dimension
	size = Memory_Panel.getPreferredSize ();
Memory_Panel.setPreferredSize (size);
size.width = 30;
Memory_Panel.setMinimumSize (size);


Monitor_Window.setDefaultCloseOperation (JFrame.DO_NOTHING_ON_CLOSE);
Monitor_Window.addWindowListener (new WindowAdapter ()
	{
	public void windowClosing (WindowEvent event)
		{Monitor_Window.setVisible (false);}
	public void windowClosed (WindowEvent event)
		{Monitor_Window.setVisible (false);}
	});
}


private void Show_Monitor ()
{
if (! Monitor_Window.isVisible ())
	{
	MONITOR_LOCATOR.Relocate (Monitor_Window, this);
	Monitor_Window.setVisible (true);
	}
Monitor_Window.toFront ();
}

private static final View_Locator
	MONITOR_LOCATOR	= new View_Locator ()
		.Offsets (0, 0)
		.Vertical   (View_Locator.BOTTOM	| View_Locator.OUTWARD)
		.Horizontal (View_Locator.LEFT		| View_Locator.INWARD);


private void Report
	(
	String	report
	)
{
if (Monitor_Reports_Checkbox.isSelected ())
	{
	final String
		reported = report;
	SwingUtilities.invokeLater (new Runnable ()
		{public void run () {Report.println (reported);}});
	}
}
	
/*==============================================================================
	Actions
*/
private void View_Configuration ()
{
if (Configuration_View == null)
	{
	Configuration_View = new Parameter_View
		("Kapellmeister",  Kapellmeister_Configuration);
	Configuration_View.setDefaultCloseOperation (JFrame.HIDE_ON_CLOSE);

	View_Locator
		locator = new View_Locator ()
			.Offsets (0, 0)
			.Vertical   (View_Locator.TOP   | View_Locator.INWARD)
			.Horizontal (View_Locator.RIGHT | View_Locator.OUTWARD);
	locator.Relocate (Configuration_View, this);
	Configuration_View.setVisible (true);
	}
else
	{
	Configuration_View.Parameter_Pane ()
		.Parameter (Kapellmeister_Configuration);
	Configuration_View.setVisible (true);
	Configuration_View.toFront ();
	}
}

/*------------------------------------------------------------------------------
	Profiles
*/
private void Select_Profile ()
{
if (File_Chooser.showOpenDialog (this) == JFileChooser.APPROVE_OPTION)
	{
	File
		file = File_Chooser.getSelectedFile ();
	if (! file.isFile ())
		{
		Dialog_Box.Notice
			("Can't open file " + file.getAbsolutePath () + '\n'
			+"No such file.",
			this);
		return;
		}
	try {CWD (file);}
	catch (FileNotFoundException exception) {}
	String
		pathname = null;
	try {pathname = file.getCanonicalPath ();}
	catch (Exception exception) {}

	Load_Profile (pathname);
	}
}


private void Load_Profile
	(
	String	source
	)
{
if ((DEBUG & DEBUG_PROFILE) != 0)
	System.out.println
		(">>> Kapellmeister.Load_Profile: " + source);
if (source == null)
	{
	if ((DEBUG & DEBUG_PROFILE) != 0)
		System.out.println
			("<<< Kapellmeister.Load_Profile");
	return;
	}

String
	report = null;
Exception
	exception = null;
try {Current_Profile = new Profile (source);}
catch (Configuration_Exception except)
	{
	report =
		"There is a configuration problem in the Profile -" + NL
		+ source + NL
		+ except.getMessage ();
	exception = except;
	}
catch (PVL_Exception except)
	{
	report =
		"There is a PVL syntax problem in the Profile -" + NL
		+ source + NL
		+ except + NL
		+ except.getMessage ();
	exception = except;
	}
if (report != null)
	{
	Report ("!!! " + report);
	if ((DEBUG & DEBUG_PROFILE) != 0)
		System.out.println ("    " + report);
	new Error_Report ("Load Profile Failure", report, exception, this);
	if ((DEBUG & DEBUG_PROFILE) != 0)
		System.out.println
			("<<< Kapellmeister.Load_Profile");
	return;
	}
if ((DEBUG & DEBUG_PROFILE) != 0)
	System.out.println
		("    Profile -" + NL
		+ Current_Profile);

//	Set the most recent Profile pathname and CWD.
File
	file = new File (source);
Profile_Pathname = file.getAbsolutePath ();
try {CWD (file);}
catch (Exception except) {/* Already found */}

Vector<Vector<Object>>
	theater_definitions = Current_Profile.Theater_Definitions ();
if ((DEBUG & DEBUG_PROFILE) != 0)
	System.out.println
		("    Theater_Definitions: " + theater_definitions);
while (theater_definitions.size () > 0)
	{
	String
		theater_location = 
			(String)theater_definitions.get (0)
				.get (Profile.THEATER_LOCATION_INDEX);
	if ((DEBUG & DEBUG_PROFILE) != 0)
		System.out.println
			("    " + theater_location + " -");

	//	Check for an existing, open theater at the location.
	Theater
		theater = Theaters.Theater (theater_location);
	if (theater == null ||
		! theater.Opened ())
		{
		//	Defer the load for this Theater until it opens.
		if ((DEBUG & DEBUG_PROFILE) != 0)
			System.out.println
				("    The Theater is not open; deferred profile load ...");
		Open_Theater (theater_location);
		}
	else
		Load_Theater_Profile (theater_location);

	//	Remove all definitions for the location.
	int
		index = theater_definitions.size ();
	while (--index >= 0)
		if (theater_location.equals (theater_definitions.get (index)
				.get (Profile.THEATER_LOCATION_INDEX)))
			theater_definitions.remove (index);
	}
if ((DEBUG & DEBUG_PROFILE) != 0)
	System.out.println
		("<<< Kapellmeister.Load_Profile");
}


private void Load_Theater_Profile
	(
	String	theater_location
	)
{
if ((DEBUG & DEBUG_PROFILE) != 0)
	System.out.println
		(">>> Kapellmeister.Load_Theater_Profile: " + theater_location);
if (Current_Profile == null ||
	theater_location == null ||
	theater_location.length () == 0)
	{
	if ((DEBUG & DEBUG_PROFILE) != 0)
		System.out.println
			("<<< Kapellmeister.Load_Theater_Profile");
	return;
	}

Vector<Object>
	theater_definition;
int
	entry = -1;
while ((entry = Current_Profile.Next_Theater_Index (theater_location, entry))
			>= 0)
	{
	//	Remove the theater definition so it will not be reprocessed.
	theater_definition = Current_Profile.Remove_Theater_Definition (entry--);
	if ((DEBUG & DEBUG_PROFILE) != 0)
		System.out.println
			("    theater_definition - " + theater_definition);

	Theater
		theater = Theaters.Theater (theater_location);
	if (theater != null &&
		theater_definition.size () > Profile.CONDUCTOR_NAME_INDEX)
		{
		String
			conductor_name = (String)theater_definition.get
				(Profile.CONDUCTOR_NAME_INDEX);
		if ((DEBUG & DEBUG_PROFILE) != 0)
			System.out.println
				("    conductor_name - " + conductor_name);
		if (conductor_name == null)
			continue;

		Conductor_Definition
			conductor_definition =
				Current_Profile.Conductor_Definition (conductor_name);
		if ((DEBUG & DEBUG_PROFILE) != 0)
			System.out.println
				("    conductor_definition -" + NL
				+ conductor_definition);
		if (conductor_definition == null)
			{
			//	Use a generic Conductor_Definition.
			conductor_definition = new Conductor_Definition (conductor_name);
			if ((DEBUG & DEBUG_PROFILE) != 0)
				System.out.println
					("    new Conductor_Definition -" + NL
					+ conductor_definition);
			}

		Integer
			conductor_count = null;
		if (theater_definition.size () > Profile.CONDUCTOR_COUNT_INDEX)
			conductor_count = (Integer)theater_definition.get
				(Profile.CONDUCTOR_COUNT_INDEX);
		if (conductor_count == null)
			conductor_count = new Integer (0);
		if ((DEBUG & DEBUG_PROFILE) != 0)
			System.out.println
				("      Count: " + conductor_count);

		int
			total =
				Conductors.Count (theater_location, conductor_definition);
		if ((DEBUG & DEBUG_PROFILE) != 0)
			System.out.println
				("    Current: " + total);

		total -= conductor_count;
		if (total < 0)
			{
			try
				{
				if ((DEBUG & DEBUG_PROFILE) != 0)
					System.out.println
						("      Start: " + -total + NL
						+"      Start_Conductor_Message -" + NL
						+ conductor_definition.Start_Conductor_Message
							(Wait_to_Start_Checkbox.isSelected (),
							-total));
				New_Conductor (theater,
						conductor_definition.Start_Conductor_Message
							(Wait_to_Start_Checkbox.isSelected (),
							-total));
				}
			catch (Exception exception)
				{
				String
					report =
					"The Conductor matrix Profile contains an invalid "
						+ "Conductor definition -" + NL
					+ conductor_definition + NL
					+ exception.getMessage ();
				Report ("!!! " + report);
				if ((DEBUG & DEBUG_PROFILE) != 0)
					System.out.println ("    " + report);
				Dialog_Box.Notice (report, Conductor_Table);
				}
			}
		else
		if (total > 0 &&
			Dialog_Box.Confirm
				("Quit or Stop " + total + ' ' + conductor_definition.Name ()
					+ " Conductor" + ((total == 1) ? "" : "s") + NL
				+"on the " + theater_location + " Theater?",
				Conductor_Table))
			{
			//	Stop the excess Conductors.
			if ((DEBUG & DEBUG_PROFILE) != 0)
				System.out.println
					("       Stop: " + total);
			//	Search for waiting Conductors that can be quit.
			Manager
				manager;
			int
				index = -1;
			while (total > 0 &&
				  (index = Conductors.Next_Index (theater_location, index))
					>= 0)
				{
				if (conductor_definition.Matches
						(Conductors.Identity (index)))
					{
					if (Conductors.Processing_State (index) < 0)
						{
						if ((DEBUG & DEBUG_PROFILE) != 0)
							System.out.println
								("    Quiting -" + NL
								+ Conductors.Identity (index));
						if ((manager = Conductors.Manager (index)) != null)
							manager.Quit_Conductor ();
						else
							Conductors.Management (index).Quit ();
						--index;
						--total;
						}
					}
				}

			//	Stop any remaining excess.
			if ((DEBUG & DEBUG_PROFILE) != 0)
				System.out.println
					("       Stop remaining: " + total);
			index = -1;
			while (total > 0 &&
				  (index = Conductors.Next_Index (theater_location, index))
					>= 0)
				{
				if (conductor_definition.Matches
						(Conductors.Identity (index)))
					{
					if ((DEBUG & DEBUG_PROFILE) != 0)
						System.out.println
							("    Stopping -" + NL
							+ Conductors.Identity (index));
					Conductors.Management (index).Stop ();
					--total;
					}
				}
			}
		}
	}
if ((DEBUG & DEBUG_PROFILE) != 0)
	System.out.println
		("<<< Kapellmeister.Load_Theater_Profile");
}


private void Save_Profile
	(
	boolean		inquire
	)
{
if (Theaters.Total_Opened () == 0)
	{
	Dialog_Box.Notice
		("There are no open theaters" + NL
		+"from which to create a profile that can be saved.");
	return;
	}

File
	file = null;
if (Profile_Pathname == null ||
	inquire)
	{
	if (File_Chooser.showSaveDialog (this) != JFileChooser.APPROVE_OPTION)
		return;
	file = File_Chooser.getSelectedFile ();
	}
else
	file = new File (Profile_Pathname);
try {CWD (file);}
catch (FileNotFoundException exception)
	{
	Dialog_Box.Notice
		("The pathname to " + file.getPath () + '\n'
		+"does not exist.",
		this);
	return;
	}
if (file.exists ())
	{
	if (file.isDirectory ())
		{
		Dialog_Box.Notice
			("Can't replace directory " + file.getAbsolutePath (),
			this);
		return;
		}
	if (! Dialog_Box.Confirm
			("Replace " + file.getAbsolutePath () + '?',
			this))
		return;
	}

try
	{
	Profile
		profile;
	if (Conductors.getRowCount () != 0)
		profile = Conductors.Profile ();
	else
		{
		profile = new Profile ();
		int
			index = -1,
			size = Theaters.getSize ();
		while (++index < size)
			if (Theaters.Opened (index))
				profile.Add_Theater_Definition (Theaters.Location (index));
		}
	profile.Write (file.getAbsolutePath ());
	Profile_Pathname = file.getAbsolutePath ();
	}
catch (Exception exception)
	{
	TOOLKIT.beep ();
	new Error_Report ("Profile Save Failed",
		"There was a problem while saving the Conductors Profile to -" + NL
		+ file.getAbsolutePath (),
		exception, this);
	}
}

/**	Sets the current working directory for finding Profile files.
<P>
	@param	pathname	A host filesystem pathname. If null, the
		"user.dir" System property will be used.
	@return	This Kapellmeister.
	@throws	FileNotFoundException	If neither the file nor the file's
		parent refers to an existing directory.
	@see	#CWD(File)
*/
public Kapellmeister CWD
	(
	String	pathname
	)
	throws FileNotFoundException
{
if (pathname == null)
	pathname = System.getProperty ("user.dir");
return CWD (new File (pathname));
}

/**	Sets the current working directory for finding Profile files.
<P>
	@param	file	A File to which to set the CWD. If null, the
		"user.dir" System property will be used. If the file does not
		refer to an existing directory, the file's parent will be used.
	@return	This Kapellmeister.
	@throws	FileNotFoundException	If neither the file nor the file's
		parent refers to an existing directory.
*/
public Kapellmeister CWD
	(
	File	file
	)
	throws FileNotFoundException
{
if (file == null)
	file = new File (System.getProperty ("user.dir"));
if (! file.isAbsolute ())
	file = file.getAbsoluteFile ();
if (! file.isDirectory ())
	//	Can only set directories.
	file = file.getParentFile ();
if (file.exists ())
	File_Chooser.setCurrentDirectory (file);
else
	throw new FileNotFoundException
		(ID + '\n'
		+"Can't set working directory to" + NL
		+ file.getAbsolutePath () + NL
		+"No such directory.");
return this;
}

/*------------------------------------------------------------------------------
	Theaters
*/
private Theater Open_Theater
	(
	String	location
	)
{
if (location == null ||
	location.length () == 0)
	return null;
if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		(">>> Kapellmeister.Open_Theater: " + location);

//	Check for an existing, open theater at the location.
Theater
	theater = Theaters.Theater (location);
if (theater == null ||
	! theater.Opened ())
	{
	//	Open the theater.
	String
		hostname = Theater.Host (location);
	if (hostname == null ||
		hostname.length () == 0)
		{
		if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
			System.out.println
				("    No hostname" + NL
				+"<<< Kapellmeister.Open_Theater");
		return null;
		}
	hostname = Host.IP_Address (hostname);
	if (hostname == null)
		{
		Dialog_Box.Notice
			("Unknown theater host for location" + NL
			+'"' + location + "\".",
			Theater_List);
		if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
			System.out.println
				("    Unknown theater host for location" + NL
				+"<<< Kapellmeister.Open_Theater");
		return null;
		}
	int
		port = Theater.Port (location);
	if (port < 0)
		{
		Dialog_Box.Error
			("Invalid theater host port number for location" + NL
			+'"' + location + "\".",
			Theater_List);
		if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
			System.out.println
				("    Invalid theater host port number for location" + NL
				+"<<< Kapellmeister.Open_Theater");
		return null;
		}
	location = Theater.Full_Location (location);

	String
		theater_key;
	if ((theater_key = Theater_Keys.get (location)) == null &&
		(theater_key = Skeleton_Key) == null &&
		(theater_key = Theater_Key (location)) == null)
		{
		if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
			System.out.println
				("    Password entry cancelled" + NL
				+"<<< Kapellmeister.Open_Theater");
		return null;
		}

	setCursor (WAIT_CURSOR);
	Exception
		thrown = null;
	try {
		if (theater == null)
			{
			//	New theater.
			if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
				System.out.println
					("    Opening a new Theater on host " + hostname
						+ " port " + port + '.');
			Report
				("==> Opening a new Theater on host " + hostname
					+ " port " + port + '.');
			theater = new Theater
				(hostname, port, Keyed_Identity (theater_key));
			}
		else
			{
			if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
				System.out.println
					("    Reopening a Theater -" + NL
					+ theater);
			Report
				("==> Reopening a Theater -" + NL
				+ theater);
			//	Re-open the existing theater.
			theater.Open (hostname, port, Keyed_Identity (theater_key));
			}
		}
	catch (IOException exception)
		{
		if (theater == null &&
			Theater_Keys.get (location) == null &&
			exception instanceof Theater_Protocol_Exception &&
			((Theater_Protocol_Exception)exception).Reason ()
				== Theater_Protocol_Exception.UNAUTHENTICATED)
			{
			//	The skeleton key did not open the new theater.
			if ((theater_key = Theater_Key (location)) == null)
				{
				if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
					System.out.println
						("    Password entry cancelled" + NL
						+"<<< Kapellmeister.Open_Theater");
				return null;
				}
			try {theater = new Theater
					(hostname, port, Keyed_Identity (theater_key));}
			catch (IOException except)
				{thrown = except;}
			}
		else
			thrown = exception;
		}
	setCursor (DEFAULT_CURSOR);
	if (thrown != null)
		{
		String
			report =
			"Unable to open the Theater"
				+ ((theater == null) ?
				(" at " + location + " (" + hostname + ':' + port + ").") :
				("." + NL + theater));
		Report
			("!!! " + report + NL
			+ thrown);
		new Error_Report ("Theater Open Failure", report, thrown, this);
		if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
			System.out.println
				("    Theater open failed." + NL
				+ report + NL
				+"<<< Kapellmeister.Open_Theater");
		return null;
		}
	Report
		("==> Opened Theater -" + NL
		+ theater + NL
		+"    with Stage Manager -" + NL
		+ theater.Stage_Manager_Identity ());

	if (Theater_Keys.get (location) == null)
		//	Cache the theater key.
		Theater_Keys.put (location, theater_key);
	if (Skeleton_Key == null)
		Skeleton_Key = theater_key;

	//	Add the theater to the list BEFORE listening for messages.
	if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
		System.out.println
			("    Add theater to List -" + NL
			+ theater);
	Theaters.Add (theater);
	Theater_Selected (theater.Location ());

	//	Begin listening for Messages.
	if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
		System.out.println
			("    Start listening for messages");
	theater.Employer (this);
	theater.Listen_for_Messages ();

	//	Request reports of Messengers connected to the Stage_Manager.
	if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
		System.out.println
			("    Requesting Messenger connection reports");
	try {theater.Send_Message
			(Message.Action (Theater.START_MESSENGER_REPORTING_ACTION));}
	catch (IOException exception)
		{
		// A Done message for the theater will be delivered.
		String
			report =
			"There was a problem sending the Theater Stage Manager the "
				+ Theater.START_MESSENGER_REPORTING_ACTION + " request:" + NL
			+ NL
			+ theater;
		Report ("!!! " + report + NL
			+ exception);
		new Error_Report ("Theater Open Failure", report, exception, this);
		if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
			System.out.println
				("    Theater open failed." + NL
				+ report + NL
				+ exception + NL
				+"<<< Kapellmeister.Open_Theater");
		return null;
		}
	}

if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		("    Theater -" + NL
		+ theater + NL
		+"<<< Kapellmeister.Open_Theater");
return theater;
}


private void Open_Theaters
	(
	Vector<String>	theater_locations
	)
{
if (theater_locations == null ||
	theater_locations.size () == 0)
	return;
int
	index = -1,
	size = theater_locations.size ();
while (++index < size)
	Open_Theater (theater_locations.get (index));
}


private void Open_Theater ()
{
if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		(">>> Kapellmeister.Open_Theater");
String
	location = (String)Theater_List.getSelectedItem ();
if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		("    Theater location - " + location);
if (location == null ||
	location.length () == 0)
	{
	Dialog_Box.Notice
		("A theater location is required.",
		Theater_List);
	if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
		System.out.println
			("    No location" + NL
			+"<<< Kapellmeister.Open_Theater");
	return;
	}

//	Check for an existing, opened Theater.
Theater
	theater = Theaters.Theater (location);
if (theater == null ||
	! theater.Opened ())
	theater = Open_Theater (location);
else
	Close_Theater (theater, null);
Theater_is_Open ((theater == null) ? false : theater.Opened ());

if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		("<<< Kapellmeister.Open_Theater");
}


private void Close_Theater
	(
	Theater	theater,
	Message	message
	)
{
if (theater == null)
	return;
if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		(">>> Kapellmeister.Close_Theater:" + NL
		+ theater);

String
	location = Theaters.Location (theater);
if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		("    Theater location - " + location);
if (location != null)
	{
	Report
		("==> Closing the theater located at " + location);
	if (location.equals
			(Theater.Full_Location ((String)Theater_List.getSelectedItem ())))
		Theater_is_Open (false);

	if (message != null)
		{
		String
			report = Message.Unescape
				(message.Get (Theater.EXCEPTION_PARAMETER_NAME));
		if (report != null)
			{
			report =
			"The theater at " + location + " unexpectedly closed." + NL
			+ NL
			+ report + NL
			+ theater;
			Report
				("!!! " + report);
			if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
				System.out.println ("    " + report);
			Dialog_Box.Notice (report, Theater_List);
			}
		}

	if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
		System.out.println
			("    Suspending Messenger connection reports");
	try {theater.Send_Message
			(Message.Action (Theater.STOP_MESSENGER_REPORTING_ACTION));}
	catch (IOException exception)
		{/* Remove all identies for the theater anyway */}

	//	Remove all Conductors at the theater location.
	Conductors.Remove (location);

	if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
		System.out.println
			("    Restart Messenger connection reports");
	try {theater.Send_Message
			(Message.Action (Theater.START_MESSENGER_REPORTING_ACTION));}
	catch (IOException exception) {}
	}

//	Make sure the theater is closed.
Theaters.Close (theater);
if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		("<<< Kapellmeister.Close_Theater");
}

/**	Theaters_List ItemList ItemEvent handler.
*/
private void Theater_Selected ()
{
if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		(">>> Kapellmeister.Theater_Selected");
String
	location = Theater.Full_Location ((String)Theater_List.getSelectedItem ());
if (location == null ||
	location.length () == 0)
	{
	if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
		System.out.println
			("    No location selected");
	location = Theater.Full_Location (Theaters.Previous_Location ());
	if (location != null &&
		location.length () != 0)
		{
		if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
			System.out.println
				("    Restoring previous location: " + location);
		Theater_Selected (location);
		TOOLKIT.beep ();
		}
	}
else
	{
	if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
		System.out.println
			("    Location selected: " + location);
	Theater
		theater = Theaters.Theater (location);
	if (theater != null &&
		theater.Opened ())
		Select_Conductors_at (location);
	else
		theater = Open_Theater (location);
	Theater_is_Open ((theater == null) ? false : theater.Opened ());
	}
if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		("<<< Kapellmeister.Theater_Selected");
}

/**	Select the Theater location to be selected in the Theaters_List.
<p>
	@param	location	The Theater location String in standard format
		as it is expected to appear in the Theaters_List. If null
		or the location is already selected nothing is done.
*/
private void Theater_Selected
	(
	String	location,
	boolean	update_conductor_selections
	)
{
location = Theater.Full_Location (location);
if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		(">>> Kapellmeister.Theater_Selected: " + location);
if (location != null &&
	! location.equals
		(Theater.Full_Location ((String)Theater_List.getSelectedItem ())))
	{
	if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
		System.out.println
			("    Selecting");
	//	Remove listener to prevent another selection event.
	Theater_List.removeItemListener (Theater_List_ItemListener);
	Theater_List.setSelectedItem (location);
	//	Restore listener.
	Theater_List.addItemListener (Theater_List_ItemListener);
	Theater_is_Open (Theaters.Opened (location));

	if (update_conductor_selections)
		Select_Conductors_at (location);
	}
if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		("<<< Kapellmeister.Theater_Selected");
}


private void Theater_Selected
	(
	String	location
	)
{Theater_Selected (location, true);}


private void Theater_is_Open
	(
	boolean	opened
	)
{
if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		(">>> Kapellmeister.Theater_is_Open: " + opened);
if (opened)
	{
	if (Close_Theater_Icon == null)
		{
		Open_Theater_Button.setText ("Close");
		Open_Theater_Button.setBackground (Colors.CLOSED_THEATER);
		}
	else
		Open_Theater_Button.setIcon (Close_Theater_Icon);
	Open_Theater_Button.setToolTipText
		("Press to close the theater");
	}
else
	{
	if (Open_Theater_Icon == null)
		{
		Open_Theater_Button.setText ("Open");
		Open_Theater_Button.setBackground (Colors.OPENED_THEATER);
		}
	else
		Open_Theater_Button.setIcon (Open_Theater_Icon);
	Open_Theater_Button.setToolTipText
		("Press to open the theater");
	}
Open_Theater_Button.setEnabled (true);

if (Theater_List != null)
	Theater_List.getEditor ().getEditorComponent ().setBackground
		(opened ?
			Colors.Selected_Color (true, Colors.OPENED_THEATER) :
			Colors.Selected_Color (true, Colors.CLOSED_THEATER));

if (New_Conductor_Action != null &&
	Conductor_Table != null)
	{
	if (! opened &&
		Conductor_Table.getSelectedRowCount () == 0)
		New_Conductor_Action.setEnabled (false);
	else
		New_Conductor_Action.setEnabled (true);
	}

if (Theater_Status_Action != null)
	Theater_Status_Action.setEnabled (opened);
		
if ((DEBUG & DEBUG_THEATER_CONNECT) != 0)
	System.out.println
		("<<< Kapellmeister.Theater_is_Open");
}


private void Theater_Status_Report ()
{
String
	location = (String)Theater_List.getSelectedItem ();
if (location == null ||
	location.length () == 0)
	{
	Dialog_Box.Notice
		("A theater location must be selected.",
		Monitor_Window);
	return;
	}

//	Check for an existing, opened Theater.
Theater
	theater = Theaters.Theater (location);
if (theater == null ||
	! theater.Opened ())
	{
	Dialog_Box.Notice
		("The " + location + " theater is not open.",
		Monitor_Window);
	return;
	}

Theater_Status_Report (new Message ()
		.Set (location, "Status requested ...")
		.Name (location));
Theater_Status_Report (theater);
}


private void Theater_Status_Report
	(
	Theater	theater
	)
{
if (theater == null)
	return;

String
	location = theater.Location ();
Theater_Status_Report (new Message ()
	.Set (location, "Status requested ...")
	.Name (location));

try {theater.Send_Message
		(Message.Action (Theater.STATUS_REPORT_ACTION));}
catch (IOException exception)
	{
	Theater_Status_Report (new Message ()
		.Set (location, "Unable to obtain the status")
		.Name (location));
	new Error_Report ("Theater Status Report Failure",
		"Unable to send the " + Theater.STATUS_REPORT_ACTION + " request.",
		exception, Theater_Status_Report_View);
	}
}


private void Theater_Status_Report
	(
	Messenger	messenger,
	Message		message
	)
{
if (messenger == null ||
	message == null)
	return;

message.Remove (Theater.ACTION_PARAMETER_NAME);
message.Name (Theaters.Location (messenger));

Theater_Status_Report (message);
}


private void Theater_Status_Report
	(
	Parameter	report
	)
{
if (report == null)
	return;

String
	location = report.Name ();
report.Name (Parser.CONTAINER_NAME);
if (Theater_Status_Report_View == null)
	{
	Theater_Status_Report_View = new Parameter_View
		(location + " Theater Status", report);
	Theater_Status_Report_View.setDefaultCloseOperation (JFrame.HIDE_ON_CLOSE);

	View_Locator
		locator = new View_Locator ()
			.Offsets (0, 0)
			.Vertical   (View_Locator.TOP   | View_Locator.INWARD)
			.Horizontal (View_Locator.RIGHT | View_Locator.OUTWARD);
	locator.Relocate (Theater_Status_Report_View, Monitor_Window);
	}
else
	{
	Theater_Status_Report_View.setTitle (location + " Theater Status");
	Theater_Status_Report_View.Parameter_Pane ().Parameter (report);
	}
Theater_Status_Report_View.setVisible (true);
Theater_Status_Report_View.toFront ();
}

/**	Get the Stage_Manager key for a Theater location.
<p>
	If a key for the location can be obtained from the Theater_Keys cache
	it is returned. If no key is obtained a Password dialog is used to
	obtain a key string. An empty key string will only be accepted if
	{@link Stage_Manager#UNAUTHENTICATED_CONNECTIONS_ALLOWED} is true.
	Otherwise a Notice is displayed at the Password dialog repeated.
<p>
	@param	location	A Theater location string. If null, null is
		returned.
	@return	A Theater Stage_Manager key (password) string. This will be
		null if a key could be obtained from the Theater_Keys cache and
		the user canceled the Password dialog. It will only be the emtpy
		string if {@link Stage_Manager#UNAUTHENTICATED_CONNECTIONS_ALLOWED}
		is true.
*/
private String Theater_Key
	(
	String	location
	)
{
if (location == null)
	return null;
String
	key = Theater_Keys.get (location);
while (key == null)
	{
	key = Password_Dialog.Get_Password
		("Kapellmeister Theater Opening", location + " Stage_Manager Password",
		this);
	/*
		Note: The Password will only be null if the user selected Canel.
		It will be empty if Accept was selected without entering a password.
	*/
	if (key != null &&
		key.length () == 0 &&
		! Stage_Manager.UNAUTHENTICATED_CONNECTIONS_ALLOWED)
		{
		Dialog_Box.Notice ("A Stage_Manager password is required.", this);
		key = null;
		}
	else
		break;
	}
return key;
}


private Message Keyed_Identity
	(
	String	theater_key
	)
{
Message
	identity = Listener_Identity ();
if (theater_key != null &&
	theater_key.length () != 0)
	identity.Set (THEATER_KEY_PARAMETER_NAME, theater_key);
return identity;
}

/*------------------------------------------------------------------------------
	Conductors
*/
private void New_Conductor ()
{
if ((DEBUG & DEBUG_NEW_CONDUCTOR) != 0)
	System.out.println
		(">>> Kapellmeister.New_Conductor");
String
	location = null;

//	Check for a selected Conductor.
int
	row = Conductor_Table.getSelectedRow ();
if (row >= 0)
	{
	//	Clone the selected Conductor on the same Theater.
	Message
		identity = Conductors.Identity
			(Conductor_Table.convertRowIndexToModel (row));
	if (identity != null)
		{
		New_Conductor_Dialog
			.Name (Conductors.Conductor_Name (identity))
			.Pipeline (identity.Get (PIPELINE_PARAMETER_NAME))
			.Catalog (identity.Get (CATALOG_PARAMETER_NAME))
			.Configuration (identity.Get (CONFIGURATION_SOURCE_PARAMETER_NAME))
			.Server (identity.Get (DATABASE_SERVER_PARAMETER_NAME));
		location = Conductors.Theater_Location (identity);
		}
	}

if (location == null)
	//	Use the selected Theater.
	location = Theater.Full_Location ((String)Theater_List.getSelectedItem ());
if ((DEBUG & DEBUG_NEW_CONDUCTOR) != 0)
	System.out.println
		("    Theater location: " + location);

Theater
	theater = Theaters.Theater (location);
if ((DEBUG & DEBUG_NEW_CONDUCTOR) != 0)
	System.out.println
		("    Theater -" + NL
		+ theater);
if (theater == null ||
	! theater.Opened ())
	{
	Dialog_Box.Notice
		("An opened theater location must be selected.",
		Theater_List);
	if ((DEBUG & DEBUG_NEW_CONDUCTOR) != 0)
		System.out.println
			("    No selected theater" + NL
			+"<<< Kapellmeister.New_Conductor");
	return;
	}
Theater_Selected (location);

New_Conductor_Dialog.Title ("New Conductor at " + location);
New_Conductor_Dialog.setLocation (Conductor_Table.getLocationOnScreen ());
New_Conductor_Dialog.setVisible (true);
New_Conductor (theater, New_Conductor_Dialog.Start_Conductor_Message ());
if ((DEBUG & DEBUG_NEW_CONDUCTOR) != 0)
	System.out.println
		("<<< Kapellmeister.New_Conductor");
}


private void New_Conductor
	(
	Theater	theater,
	Message	conductor_definition
	)
{
if (theater == null ||
	conductor_definition == null)
	return;
if ((DEBUG & DEBUG_NEW_CONDUCTOR) != 0)
	System.out.println
		(">>> Kapellmeister.New_Conductor:" + NL
		+"    Theater -" + NL
		+ theater + NL
		+"    Conductor definition -" + NL
		+ conductor_definition);

Report
	("==> Starting new Conductor(s):" + NL
	+"Theater -"
	+ theater + NL
	+"Conductor definition -" + NL
	+ conductor_definition);
try {theater.Send_Message (conductor_definition);}
catch (IOException exception)
	{
	String
		report =
		"There was a problem sending the new Conductor message:" + NL
		+ NL
		+ exception + NL
		+"Message -" + NL
		+ conductor_definition + NL
		+"Theater -" + NL
		+ theater + NL
		+ NL
		+ ID;
	Report
		("!!! " + report);
	if ((DEBUG & DEBUG_NEW_CONDUCTOR) != 0)
		System.out.println ("    " + report);
	Dialog_Box.Error (report, Conductor_Table);
	// A Done message for the theater will be delivered.
	}
if ((DEBUG & DEBUG_NEW_CONDUCTOR) != 0)
	System.out.println
		("<<< Kapellmeister.New_Conductor");
}


private void New_Conductor_Reply
	(
	Message	message
	)
{
if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
	System.out.println
		(">-< Kapellmeister.New_Conductor_Reply:" + NL
		+ message);
String
	explanation = Message.Unescape
		(message.Get (Theater.EXPLANATION_PARAMETER_NAME));
if (explanation != null)
	{
	String
		string = message.Get (Stage_Manager.COMMAND_PARAMETER_NAME);
	if (string != null &&
		explanation.indexOf (string) < 0)
		explanation += NL + NL + "Command line: " + string;

	new Error_Report ("Conductor startup failure",
		explanation,
		Message.Unescape (message.Get (Theater.EXCEPTION_PARAMETER_NAME)),
		this);
	}
}

/**	Get the Remote_Theater Management for a Conductor.
<p>
	If the Conductors table contains a {@link
	Conductor_Table_Model#Management(Message) Remote_Theater Management
	for the identity} which is {@link Remote_Theater#Opened() open},
	that is returned. Otherwise an attempt is made to obtain it.
<p>
	Remote_Theater Management is obtained by first getting the {@link
	Conductor_Table_Model#Theater_Location(Message) theater location
	from the identity}. This is used to obtain the {@link Open_Theater(String)
	Theater for the location}. The Theater, along with the Conductor identity
	and {@link #Listener_Identity() Kapellmeister identity 
*/
private Remote_Theater Management
	(
	Message	identity
	)
{
if (identity == null)
	return null;
if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
	System.out.println
		(">>> Kapellmeister.Management:" + NL
		+ identity);

Remote_Theater
	management = Conductors.Management (identity);
if (management != null &&
	! management.Opened ())
	management = null;

if (management == null)
	{
	String
		location = Conductors.Theater_Location (identity);
	if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
		System.out.println
			("    Conductor theater location: " + location);
	if (location != null)
		{
		Theater
			theater = Open_Theater (location);
		if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
			System.out.println
				("    Theater: " + theater);
		if (theater != null)
			{
			try {management =
					Conductors.Open_Management (identity, theater,
						Keyed_Identity (Theater_Key (location)));}
			catch (IOException exception)
				{
				String
					report =
					"Management for a Conductor located at " + location + NL
					+"could not be obtained." + NL
					+ NL
					+"Conductor identity -" + NL
					+ identity + NL
					+"Theater -" + NL
					+ theater;
				Report ("!!! " + report + NL
					+ exception);
				if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
					System.out.println
						("    " + report + NL
						+ exception);
				new Error_Report ("Conductor Management Failure",
					report, exception, this);
				}
			}
		}
	else
		{
		String
			report =
			"The " + Conductors.Identification (identity)
				+ " Conductor does not have a theater location!" + NL
			+ NL
			+"Conductor identity -" + NL
			+ identity + NL
			+ NL
			+ ID;
		Report ("!!! " + report);
		if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
			System.out.println
				("    " + report);
		new Error_Report ("Conductor Management Failure",
			report, this);
		}
	}

if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
	System.out.println
		("    Management -" + NL
		+ management + NL
		+"<<< Management");
return management;
}


private void Open_Manager ()
{
if (Conductor_Table.getSelectedRowCount () > 0)
	{
	int[]
		rows = Conductor_Table.getSelectedRows ();
	for (int
			count = 0;
			count < rows.length;
			count++)
		Open_Manager (Conductors.Identity
			(Conductor_Table.convertRowIndexToModel (rows[count])));
	}
}


private Manager Open_Manager
	(
	Message	identity
	)
{
if (identity == null)
	return null;
if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
	System.out.println
		(">>> Kapellmeister.Open_Manager:" + NL
		+ identity);

int
	index = Conductors.Index (identity);
if (index < 0)
	{
	Dialog_Box.Notice
		("The " + Conductors.Identification (identity)
			+ " Conductor can not be found!" + NL
		+"Conductor identity -" + NL
		+ identity + NL
		+ NL
		+ ID,
		Conductor_Table);
	if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
		System.out.println
			("    No Conductor identity in the data model!" + NL
			+"<<< Kapellmeister.Open_Manager");
	return null;
	}

Manager
	manager = Conductors.Manager (index);
if (manager == null)
	{
	if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
		System.out.println
			("    Opening new Manager");
	Remote_Theater
		remote_theater = Management (identity);
	if (remote_theater != null)
		{
		Report
			("==> Starting a new Conductor Manager." + NL
			+"Conductor identity -" + NL
			+ identity);
		try
			{
			manager = new Manager (remote_theater, Kapellmeister_Configuration);
			manager.addWindowListener (new WindowAdapter ()
				{public void windowClosed (WindowEvent event)
				{Close_Manager ((Manager)event.getWindow ());}});
			Conductors.Manager (index, manager);
			}
		catch (Exception exception)
			{
			String
				report =
				"A Manager could not be constructed for the Conductor -"+ NL
				+ NL
				+ identity;
			Report
				("!!! " + report + NL
				+ exception);
			new Error_Report ("New Manager Failure",
				report, exception, this);
			if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
				System.out.println
					("    Manager open failed" + NL
					+ report + NL
					+ exception);
			manager = null;
			}
		}
	}
else
	{
	manager.toFront ();
	if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
		System.out.println
			("    Manager already open.");
	}

if ((DEBUG & DEBUG_CONDUCTOR_CONNECT) != 0)
	System.out.println
		("<<< Open_Manager");
return manager;
}


private void Close_Manager
	(
	Manager	manager
	)
{Conductors.Close (manager);}


private void Close_Manager ()
{
if (Conductor_Table.getSelectedRowCount () > 0)
	{
	int[]
		rows = Conductor_Table.getSelectedRows ();
	/*
		An array of Managers are obtained
		rather the using the data model row indices
		because the model, and thus the validity of row indices,
		may change as the selected rows are processed.
	*/
	Manager[]
		managers = new Manager[rows.length];
	int
		index = rows.length;
	while (--index >= 0)
		managers[index] = Conductors.Manager
			(Conductor_Table.convertRowIndexToModel (rows[index]));
	while (++index < managers.length)
		if (managers[index] != null)
			managers[index].Close ();
	}
}


private void Start_Conductor ()
{
if (Conductor_Table.getSelectedRowCount () > 0)
	{
	int[]
		rows = Conductor_Table.getSelectedRows ();
	/*
		An array of Managements are obtained
		rather the using the data model row indices
		because the model, and thus the validity of row indices,
		may change as the selected rows are processed.
	*/
	Management[]
		managements = new Management[rows.length];
	int
		index = rows.length;
	while (--index >= 0)
		managements[index] = Conductors.Management
			(Conductor_Table.convertRowIndexToModel (rows[index]));
	while (++index < managements.length)
		{
		if (managements[index] != null)
			{
			try {managements[index].Start ();}
			catch (Remote_Management_Exception exception)
				{
				Dialog_Box.Error
					("Conductor management protocol failure" + NL
					+ NL
					+ Message.Unescape (exception.getMessage ()),
					Conductor_Table);
				}
			}
		}
	}
}


private void Reset_Conductor ()
{
if (Conductor_Table.getSelectedRowCount () > 0)
	{
	int[]
		rows = Conductor_Table.getSelectedRows ();
	/*
		An array of data model identity Message objects are obtained
		rather the using the data model row indices for each operation
		because the model, and thus the validity of row indices,
		is likely to change as the selected rows are processed.
	*/
	Message[]
		identities = new Message[rows.length];
	int
		index = rows.length;
	while (--index >= 0)
		identities[index] = Conductors.Identity
			(Conductor_Table.convertRowIndexToModel (rows[index]));

	Management
		management;
	while (++index < identities.length)
		{
		if (identities[index] == null ||
			(management = Conductors.Management (identities[index])) == null)
			continue;

		if (Conductors.Processing_State (identities[index])
				== Conductor_Table.HALTED_STATE)
			{
			try {management.Reset_Sequential_Failures ();}
			catch (Remote_Management_Exception exception)
				{
				Dialog_Box.Error
					("Conductor management protocol failure" + NL
					+ NL
					+ Message.Unescape (exception.getMessage ()),
					Conductor_Table);
				}
			}
		}
	}
}


private void Stop_Conductor ()
{
if (Conductor_Table.getSelectedRowCount () > 0)
	{
	int[]
		rows = Conductor_Table.getSelectedRows ();
	/*
		An array of Managements are obtained
		rather the using the data model row indices
		because the model, and thus the validity of row indices,
		may change as the selected rows are processed.
	*/
	Management[]
		managements = new Management[rows.length];
	int
		index = rows.length;
	while (--index >= 0)
		managements[index] = Conductors.Management
			(Conductor_Table.convertRowIndexToModel (rows[index]));
	while (++index < managements.length)
		{
		if (managements[index] != null)
			managements[index].Stop ();
		}
	}
}


private void Quit_Conductor ()
{
if (Conductor_Table.getSelectedRowCount () > 0)
	{
	boolean
		check = Confirm_Conductor_Abort_Checkbox.isSelected ();
	int[]
		rows = Conductor_Table.getSelectedRows ();
	/*
		An array of data model identity Message objects are obtained
		rather the using the data model row indices for each operation
		because the model, and thus the validity of row indices,
		is likely to change as the selected rows are processed.
	*/
	Message[]
		identities = new Message[rows.length];
	int
		index = rows.length;
	while (--index >= 0)
		identities[index] = Conductors.Identity
			(Conductor_Table.convertRowIndexToModel (rows[index]));

	Manager
		manager;
	Management
		management;
	int
		processing_state;
	while (++index < identities.length)
		{
		if (identities[index] == null)
			continue;

		if ((manager = Conductors.Manager (identities[index])) != null)
			manager.Quit_Conductor ();
		else
		if ((management = Conductors.Management (identities[index])) != null)
			{
			if (check)
				{
				processing_state =
					Conductors.Processing_State (identities[index]);
				if (processing_state == Conductor_Table.RUNNING_STATE ||
					processing_state == Conductor_Table.POLLING_STATE)
					{
					if (! Dialog_Box.Confirm
							("Abort the active "
								+ Conductors.Conductor_Name (identities[index])
								+ " Conductor" + NL
							+"at the " +
								Conductors.Theater_Location (identities[index])
								+ " Theater?",
							Conductor_Table))
						continue;
					}
				}
			management.Quit ();
			}
		}
	}
}


private void Select_Conductors_at
	(
	String	location
	)
{
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		(">>> Kapellmeister.Select_Conductors_at: " + location);
ListSelectionModel
	selector = Conductor_Table.getSelectionModel ();
int
	first_row = -1,
	last_row = -1,
	row,
	index = -1;
while((row = Conductors.Next_Index (location, index)) >= 0)
	{
	if (index < 0)
		{
		//	Remove the selection listener to prevent spurious updates.
		selector.removeListSelectionListener
			(Conductor_Table_Selection_Listener);
		//	Deselect all current selections.
		Conductor_Table.clearSelection ();
		}
	index = row;
	row = Conductor_Table.convertRowIndexToView (row);

	if (first_row < 0)
		//	Initialize the row range indices.
		first_row = last_row = row;
	else
	if (++last_row != row)
		{
		//	Start of new non-contiguous interval. Add previous interval.
		selector.addSelectionInterval (first_row, --last_row);
		first_row = last_row = row;
		}
	}
if (first_row >= 0)
	{
	//	Final interval.
	selector.addSelectionInterval (first_row, last_row);
	selector.addListSelectionListener (Conductor_Table_Selection_Listener);
	Refresh_Conductor_Ops_Menus ();
	}
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		("<<< Kapellmeister.Select_Conductors_at");
}


private void Select_Conductors_for
	(
	String	name
	)
{
if (name == null ||
	name.length () == 0)
	return;
int
	size = Conductors.getRowCount ();
if (size == 0)
	return;
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		(">>> Kapellmeister.Select_Conductors_for: " + name);
ListSelectionModel
	selector = Conductor_Table.getSelectionModel ();
//	Remove the selection listener to prevent spurious updates.
selector.removeListSelectionListener
	(Conductor_Table_Selection_Listener);
//	Deselect all current selections.
Conductor_Table.clearSelection ();

int
	first_row = -1,
	last_row = -1,
	row = -1;
while(++row < size)
	{
	if (! Conductors.Conductor_Name (row).equals (name))
		continue;

	row = Conductor_Table.convertRowIndexToView (row);
	if (first_row < 0)
		//	Initialize the row range indices.
		first_row = last_row = row;
	else
	if (++last_row != row)
		{
		//	Start of new non-contiguous interval. Add previous interval.
		selector.addSelectionInterval (first_row, --last_row);
		first_row = last_row = row;
		}
	}
if (first_row >= 0)
	{
	//	Final interval.
	selector.addSelectionInterval (first_row, last_row);
	selector.addListSelectionListener (Conductor_Table_Selection_Listener);
	Refresh_Conductor_Ops_Menus ();
	}
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		("<<< Kapellmeister.Select_Conductors_for");
}


private void Refresh_Conductor_Ops_Menus ()
{
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		(">>> Kapellmeister.Refresh_Conductor_Ops_Menus");
if (Conductor_Table.getSelectedRowCount () == 0)
	{
	//	Nothing selected.
	New_Conductor_Action.setEnabled
		(Theaters.Opened ((String)Theater_List.getSelectedItem ()));
	if ((DEBUG & DEBUG_SELECTION) != 0)
		System.out.println
			("      New: "
				+ Theaters.Opened ((String)Theater_List.getSelectedItem ()));
	Open_Conductor_Action.setEnabled (false);
	Close_Conductor_Action.setEnabled (false);
	Start_Conductor_Action.setEnabled (false);
	Reset_Conductor_Action.setEnabled (false);
	Stop_Conductor_Action.setEnabled (false);
	Quit_Conductor_Action.setEnabled (false);
	if ((DEBUG & DEBUG_SELECTION) != 0)
		System.out.println
			("     Open: false" + NL
			+"    Close: false" + NL
			+"    Start: false" + NL
			+"     Stop: false" + NL
			+"     Quit: false" + NL
			+"<<< Kapellmeister.Conductor_Selected");
	return;
	}

int[]
	rows = Conductor_Table.getSelectedRows ();
int
	index = rows.length;
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.print
		("    " + index + " table rows selected:");
while (--index >= 0)
	{
	if ((DEBUG & DEBUG_SELECTION) != 0)
		System.out.print
			(" " + rows[index]);
	rows[index] = Conductor_Table.convertRowIndexToModel (rows[index]);
	}
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println ();

//	New
New_Conductor_Action.setEnabled (true);
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		("      New: true");

//	Open
while (++index < rows.length)
	if (Conductors.Manager (rows[index]) == null)
		break;
Open_Conductor_Action.setEnabled (index < rows.length);
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		("     Open: " + (index < rows.length));

//	Close
index = -1;
while (++index < rows.length)
	if (Conductors.Manager (rows[index]) != null)
		break;
Close_Conductor_Action.setEnabled (index < rows.length);
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		("    Close: " + (index < rows.length));

//	Start
index = -1;
while (++index < rows.length)
	if (Conductors.Processing_State (rows[index]) <= 0)
		break;
Start_Conductor_Action.setEnabled (index < rows.length);
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		("    Start: " + (index < rows.length));

//	Reset
index = -1;
while (++index < rows.length)
	if (Conductors.Processing_State (rows[index])
		== Conductor_Table.HALTED_STATE)
		break;
Reset_Conductor_Action.setEnabled (index < rows.length);
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		("    Reset: " + (index < rows.length));

//	Stop
index = -1;
while (++index < rows.length)
	if (Conductors.Processing_State (rows[index]) > 0)
		break;
Stop_Conductor_Action.setEnabled (index < rows.length);
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		("     Stop: " + (index < rows.length));

//	Quit
Quit_Conductor_Action.setEnabled (true);
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		("     Quit: true");

if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		("<<< Kapellmeister.Refresh_Conductor_Ops_Menus");
}


private Message Uniquely_Named
	(
	Message	identity
	)
{
int
	index = -1,
	size = Conductors.getRowCount ();
return identity;
}

private void Matrix_Selected ()
{
}

//------------------------------------------------------------------------------
private void Exit
	(
	int		status
	)
{
setVisible (false);
Conductors.Close_All ();
Theaters.Close_All ();

//!!!	State saving operations.

System.exit (status);
}

/*==============================================================================
	Messages
*/
//	Message_Delivered_Listener
/**	Get the idenity of this Messenger client.
<p>
	@return	A copy of the {@link #KAPELLMEISTER_IDENTITY} Message.
*/
public Message Listener_Identity ()
{
try {return new Message (KAPELLMEISTER_IDENTITY);}
catch (PVL_Exception exception)
	{
	//	This can only happen if the Identity Message has been corrupted.
	Dialog_Box.Error
		("The required Kapellmeister Identity has become corrupted -" + NL
		+ KAPELLMEISTER_IDENTITY + NL
		+ exception + NL
		+ exception.getMessage () + NL
		+ NL
		+ "Unable to continue.",
		this);
	Exit (EXIT_UNRECOVERABLE_ERROR);
	}
return null;
}


private Vector<Message_Delivered_Event>
	Message_Event_Queue			= new Vector<Message_Delivered_Event> ();


//	Message_Delivered_Listener
/**	Take delivery of a Message and queue it for disposition.
<p>
	<b>N.B.</b>: This method is public as a side-effect of implementing
	the Message_Delivered_Listener interface.
<p>
	The Message_Delivered_Event is added to a queue and the message
	disposition method is scheduled for execution on the Swing event
	thread where the Message_Delivered_Event will be retrieved from the
	queue and examined for appropriate disposition.
<p>
	@param	event	The Message_Delivered_Event that arrived from
		a Messenger.
	@see	Message_Delivered_Event
*/
public void Message_Delivered
	(
	Message_Delivered_Event	event
	)
{
synchronized (Message_Event_Queue)
	{Message_Event_Queue.add (event);}
SwingUtilities.invokeLater (new Runnable ()
{public void run () {Message_Disposition ();}});
}


private void Message_Disposition ()
{
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(">>> Kapellmeister.Message_Disposition" + NL
		+"    Message_Event_Queue size: " + Message_Event_Queue.size ());

//	Pull the next Message_Delivered_Event from the front of the queue.
Message_Delivered_Event
	event;
synchronized (Message_Event_Queue)
	{
	/*
		>>> WARNING <<< No check is being done for an empty queue because
		Message_Disposition is run from the event queue after a
		Message_Delivered enqueues the delivered event, and only one
		event is removed from the queue each time Message_Disposition is
		run. If this one-to-one relationship between Message_Delivered
		and Message_Disposition is ever changed an empty queue check must
		be added.
	*/
	event = Message_Event_Queue.remove (0);
	}

if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(">>> Kapellmeister.Message_Delivered: " + event.Message.Action () + NL
		+"    From -" + NL
		+ (Messenger)event.getSource () + NL
		+ ((Messenger)event.getSource ()).Identity () + NL
		+"    Message -" + NL
		+ event.Message.Routing () + NL
		+ event.Message);
Report
	("<<< Message delivered:" + NL
	+ event.Message.Routing () + NL
	+ event.Message);

//	Message disposition depends on its Action.
switch (Theater.Action_Code (event.Message.Action ()))
	{
	case Theater.MESSENGERS_REPORT_CODE:
		Conductor_Identities
			((Messenger)event.getSource (), event.Message);
		break;
	case Theater.START_CONDUCTOR_CODE:
		New_Conductor_Reply (event.Message);
		break;
	case Theater.IDENTIFY_CODE:
		Send ((Messenger)event.getSource (), Listener_Identity ()
			.Reply_To (event.Message));
		break;
	case Theater.STATUS_REPORT_CODE:
		Theater_Status_Report ((Messenger)event.getSource (), event.Message);
		break;
	case Theater.DONE_CODE:
		if ((DEBUG & DEBUG_DONE) != 0)
			System.out.println
				("==> Kapellmeister.Message_Disposition: Done" + NL
				+ (Messenger)event.getSource () + NL
				+ event.Message);
		Close_Theater
			(Theaters.Theater ((Messenger)event.getSource ()), event.Message);
		break;
	case Theater.UNDELIVERABLE_CODE:
	case Theater.NACK_CODE:
	default:
		Report
			("!!! Unexpected Message -" + NL
			+ event.Message);
	}
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("<<< Kapellmeister.Message_Disposition");
}


private void Send
	(
	Messenger	messenger,
	Message		message
	)
{
if (messenger == null ||
	message == null)
	return;
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(">>> Kapellmeister.Send:" + NL
		+ messenger + NL
		+ message.Routing () + NL
		+ message);
Report
	(">>> Sending message:" + NL
	+ message.Routing () + NL
	+ message);

try {messenger.Send (message);}
catch (IOException exception)
	{
	if (! (exception instanceof EOFException))
		{
		String
			report =
			"A message could not be sent:" + NL
			+ exception + NL
			+"Message -" + NL
			+ message + NL
			+"Messenger -" + NL
			+ messenger + NL
			+ NL
			+ ID;
		Report
			("!!! " + report);
		message.Set (Theater.EXCEPTION_PARAMETER_NAME, exception.toString ());
		}
	Close_Theater (Theaters.Theater (messenger), message);
	}
catch (PVL_Exception exception)
	{
	String
		report =
		"A message to be sent was found to be corrupted:" + NL
		+ exception + NL
		+ exception.getMessage () + NL
		+"Message -" + NL
		+ message + NL
		+"Messenger -" + NL
		+ messenger + NL
		+ NL
		+ ID;
	Report
		("!!! " + report);
	if ((DEBUG & DEBUG_MESSAGES) != 0)
		System.out.println
			("    " + exception + NL
			+ report);
	}
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("<<< Kapellmeister.Send");
}

/**	Update the table Conductors table of identities from the client
	identities Message received from the Stage_Manager.
<p>
	A Vector of identity Messages is assembled from the Messenger's
	Message. The Message is expected to contain zero or more Parameter
	Groups with the {@link Theater#IDENTITY_ACTION} name. Each such
	group that contains a {@link Theater#NAME_PARAMETER_NAME} with the
	{@link #CONDUCTOR_NAME} value and a non-null {@link
	Theater#ROUTE_TO_PARAMETER_NAME} value will be stored in the
	identities Vector as a new Message copied from Group. This Message
	will also have its {@link Message#Route_To(Value) route-to} list set
	to the Value of the {@link Theater#ROUTE_TO_PARAMETER_NAME}, and the
	{@link #THEATER_LOCATION_PARAMETER_NAME} set to the {@link
	Theater_List_Model#Location(Messenger) theater location obtained
	from the Theaters list.
*/
private void Conductor_Identities
	(
	Messenger	messenger,
	Message		message
	)
{
if (messenger == null ||
	message == null)
	return;
if ((DEBUG & DEBUG_IDENTITIES) != 0)
	System.out.println
		(">>> Kapellmeister.Conductor_Identities:" + NL
		+ message);
Theater
	theater = Theaters.Theater (messenger);
if (theater == null)
	{
	Report
		("!!! A " + Theater.MESSENGERS_REPORT_ACTION
			+ " was received from an unknown theater messenger -" + NL
		+ messenger + NL
		+ message);
	if ((DEBUG & DEBUG_IDENTITIES) != 0)
		System.out.println
			("    Unknown theater!" + NL
			+"<<< Kapellmeister.Conductor_Identities");
	return;
	}

String
	error_report = null;
Vector<Message>
	identities = new Vector<Message> ();
Message
	identity;
Value
	route_to;
Parameter
	parameter = null;
while ((parameter = message.Find (Theater.IDENTITY_ACTION, parameter))
		!= null)
	{
	try
		{
		//	Copy the identity parameters into a new Message.
		identity = new Message (parameter);
		if (! CONDUCTOR_NAME
				.equals (identity.Get (Theater.NAME_PARAMETER_NAME)))
			//	Ignore anything but Conductors (for now).
			continue;
		route_to =
			identity.Value_of (Theater.ROUTE_TO_PARAMETER_NAME);
		if (route_to == null)
			{
			String
				report =
				"The " + Theater.MESSENGERS_REPORT_ACTION
					+ " from the theater at location "
					+ theater.Location () + NL
				+"contains an identity with no "
					+ Theater.ROUTE_TO_PARAMETER_NAME
					+ " parameter -" + NL
				+ identity;
			Report
				("!!! " + report);
			if ((DEBUG & DEBUG_IDENTITIES) != 0)
				System.out.println (report);
			if (error_report == null)
				error_report = report;
			else
				error_report += NL
					+ report;
			continue;
			}
		identity.Route_To (route_to);

		identities.add (identity);
		}
	catch (PVL_Exception exception)
		{
		//	Bad identity parameters!
		String
			report =
			"The " + Theater.MESSENGERS_REPORT_ACTION
				+ " from the theater at location "
				+ theater.Location () + NL
			+"contains an invalid identity -" + NL
			+ parameter.Description () + NL
			+ exception + NL
			+ exception.getMessage ();
		Report
			("!!! " + report);
		if ((DEBUG & DEBUG_IDENTITIES) != 0)
			System.out.println (report);
		if (error_report == null)
			error_report = report;
		else
			error_report += NL
				+ report;
		}
	}

if (error_report != null)
	Dialog_Box.Error (error_report, Conductor_Table);

//	Update the data model, which will update the Conductor_Table.
if ((DEBUG & DEBUG_IDENTITIES) != 0)
	System.out.println
		("    Replacing the Conductor identities -" + NL
		+ identities);
setCursor (WAIT_CURSOR);
try
	{
	Conductors.Replace
		(
		identities,
		theater,
		/*
			This theater should always have a valid key
			because the it is in the Theaters list
			and must be open to be able to receive this report
			and thus must have a key registered for it.
		*/
		Keyed_Identity (Theater_Key (Theaters.Location (theater)))
		);
	}
catch (IOException exception)
	{
	String
		report =
		"Unable to connect to one or more Conductors at the "
			+ theater.Location () + " theater -" + NL
		+ theater;
	Report
		("!!! " + report + NL
		+ NL
		+ exception.getMessage ());
	if ((DEBUG & DEBUG_IDENTITIES) != 0)
		System.out.println
			("    Kapellmeister.Conductor_Identities: IOException -" + NL
			+ report);
	new Error_Report
		("Conductor Connection Failure",
		report, exception, this);
	}
setCursor (DEFAULT_CURSOR);

if (Current_Profile != null &&
	Current_Profile.Total_Theater_Definitions () != 0)
	{
	if ((DEBUG & DEBUG_IDENTITIES) != 0)
		System.out.println
			("    Load_Theater_Profile ...");
	Load_Theater_Profile (theater.Location ());
	}
if ((DEBUG & DEBUG_IDENTITIES) != 0)
	System.out.println
		("<<< Kapellmeister.Conductor_Identities");
}

/*==============================================================================
	Helpers
*/
/**	Loads the application icons used by the GUI.
*/
private void Load_Icons ()
{
if (Open_Theater_Icon == null)
	{
	if ((DEBUG & DEBUG_ICONS) != 0)
		System.out.println
			(">>> Kapellmeister.Load_Icons");
	Icons.Directory ("Icons", this.getClass ());
	if ((DEBUG & DEBUG_ICONS) != 0)
		System.out.println
			("    Loading icons from " + Icons.Directory ());
	Kapellmeister_Icon
		= Icons.Load_Icon (Kapellmeister_Icon_Source);
	Open_Theater_Icon
		= Icons.Load_Icon (OPEN_THEATER_ICON_NAME);
	Close_Theater_Icon
		= Icons.Load_Icon (CLOSE_THEATER_ICON_NAME);
	if ((DEBUG & DEBUG_ICONS) != 0)
		System.out.println
			("    The Icons were "
				+ ((Open_Theater_Icon == null) ? "not " : "") + "loaded" + NL
			+"<<< Kapellmeister.Load_Icons");
	}
}

/*==============================================================================
	Application main
*/
public static void main
	(
	String[]	args
	)
{
if ((DEBUG & DEBUG_MAIN) != 0)
	System.out.println (ID + NL);

Vector<String>
	sources = null;
String
	configuration_filename = null;

for (int
		count = 0;
		count < args.length;
		count++)
	{
	if (args[count].length () > 1 &&
		args[count].charAt (0) == '-')
		{
		switch (args[count].charAt (1))
			{
			case 'C':	//	Configuration
			case 'c':
				if (++count == args.length ||
					args[count].length () == 0 ||
					args[count].charAt (0) == '-')
					{
					System.out.println ("Missing configuration filename");
					Usage ();
					}
				if (configuration_filename != null)
					{
					System.out.println ("Multiple configuration files -" + NL
						+ configuration_filename + NL
						+ "and" + NL
						+ args[count]);
					Usage ();
					}
				configuration_filename = args[count];
				break;

			case 'V':	//	Version
			case 'v':
				System.out.println (ID);
				System.exit (EXIT_SUCCESS);

			case 'H':	//	Help
			case 'h':
			default:
				Usage ();
			}
		}
	else
		{
		if (sources == null)
			sources = new Vector<String> ();
		if (! sources.contains (args[count]))
			sources.add (args[count]);
		}
	}


Configuration
	configuration = null;
if (configuration_filename != null)
	{
	if ((DEBUG & DEBUG_MAIN) != 0)
		System.out.println
			("    Loading the Configuration from " + configuration_filename);
	try {configuration = new Configuration (configuration_filename);}
	catch (IllegalArgumentException exception)
		{
		System.out.println ("Unable to find the configuration file - "
			+ configuration_filename);
		System.exit (EXIT_CONFIG_FILE_PROBLEM);
		}
	catch (Configuration_Exception exception)
		{
		System.out.println ("Could not load the configuration file - "
			+ configuration_filename + NL
			+ exception.getMessage ());
		System.exit (EXIT_CONFIG_FILE_PROBLEM);
		}
	}

//	Use the cross-platform LAF.
try {UIManager.setLookAndFeel
	(UIManager.getCrossPlatformLookAndFeelClassName ());}
catch (Exception exception) {/* Just leave the existing LAF. */}

if ((DEBUG & DEBUG_MAIN) != 0)
	System.out.println
		("    Constructing the Kapellmeister");
Kapellmeister
	kapellmeister = null;
try {kapellmeister = new Kapellmeister (sources, configuration);}
catch (Configuration_Exception exception)
	{
	System.out.println
		("Could not load the configuration." + NL
		+ exception.getMessage () + NL);
	System.exit (EXIT_CONFIG_FILE_PROBLEM);
	}
catch (Exception exception)
	{
	System.out.println
		("Unexpected exception!" + NL
		+ ID);
	exception.printStackTrace (System.out);
	System.exit (EXIT_UNEXPECTED_EXCEPTION);
	}
}


public static void Usage ()
{
System.out.println 
	("Usage: Kapellmeister <Options> [<profile> | <location> [...]]" + NL
	+"  Options -" + NL
	+"    -Configuration <source>"
		+ " (default: " + DEFAULT_CONFIGURATION_FILENAME + ')' + NL
	+"    -Help" + NL
	);
System.exit (EXIT_COMMAND_LINE_SYNTAX);
}

}
