/*
 * layout.cpp - A superclass for layouts.
 *
 * Copyright (c) 2004 by Alastair M. Robinson
 * Distributed under the terms of the GNU General Public License -
 * see the file named "COPYING" for more details.
 *
 */

#include <iostream>
#include <string.h>

#include "dialogs.h"
#include "support/generaldialogs.h"
#include "pixbufthumbnail/egg-pixbuf-thumbnail.h"
#include "imagesource/pixbuf_from_imagesource.h"
#include "support/rotatepixbuf.h"
#include "support/maskpixbuf.h"

#include "imagesource/imagesource.h"
#include "imagesource/imagesource_gdkpixbuf.h"
#include "imagesource/imagesource_cms.h"
#include "imagesource/imagesource_util.h"
#include "imagesource/imagesource_mask.h"
#include "imagesource/imagesource_rotate.h"
#include "imagesource/imagesource_promote.h"

#include "photoprint_state.h"
#include "support/progress.h"
#include "util.h"
#include "support/layoutrectangle.h"

#include "layout.h"

using namespace std;


ConfigTemplate LayoutDB::Template[]=
{
	ConfigTemplate("LayoutType","NUp"),
	ConfigTemplate()
};


Layout_ImageInfo::Layout_ImageInfo(Layout &layout, const char *filename, int page, bool allowcropping, PP_ROTATION rotation)
	: page(page), allowcropping(allowcropping), crop_hpan(CENTRE), crop_vpan(CENTRE),
	rotation(rotation), layout(layout), maskfilename(NULL), thumbnail(NULL), mask(NULL), transformed(NULL), selected(false),
	customprofile(NULL), customintent(INTENT_DEFAULT)
{
	bool relative=true;

	if(filename[0]=='/' || filename[1]==':')
		relative=false;

	if(filename[0]=='\\' && filename[1]=='\\')
		relative=false;

	if(relative)
		this->filename=BuildAbsoluteFilename(filename);
	else
		this->filename=strdup(filename);

	GetThumbnail();
}


Layout_ImageInfo::Layout_ImageInfo(Layout &layout, Layout_ImageInfo *ii, int page, bool allowcropping, PP_ROTATION rotation)
	: page(page), allowcropping(allowcropping), crop_hpan(CENTRE), crop_vpan(CENTRE),
	rotation(rotation), layout(layout), maskfilename(NULL), thumbnail(NULL), mask(NULL), transformed(NULL), selected(false),
	customprofile(NULL), customintent(INTENT_DEFAULT)
{
	cerr << "Stealing existing image!" << endl;

	if(ii)
	{
		thumbnail=ii->thumbnail;
		ii->thumbnail=NULL;  // We steal the thumbnail from the old image!
		mask=ii->mask;
		ii->mask=NULL;	// And the mask too!

		customprofile=ii->customprofile;
		ii->customprofile=NULL;	// And the custom profile!
		
		customintent=ii->customintent;
	
		if(ii->filename)
			this->filename=strdup(ii->filename);
		if(ii->maskfilename)
			this->maskfilename=strdup(ii->maskfilename);

		allowcropping=ii->allowcropping;
		crop_hpan=ii->crop_hpan;
		crop_vpan=ii->crop_vpan;
		rotation=ii->rotation;
	}
}


Layout_ImageInfo::~Layout_ImageInfo()
{
	if(thumbnail)
		g_object_unref(thumbnail);
	if(mask)
		g_object_unref(mask);
	if(transformed)
		g_object_unref(transformed);
	layout.imagelist=g_list_remove(layout.imagelist,this);
	if(customprofile)
		free(customprofile);
	free(filename);
}


void Layout_ImageInfo::DrawThumbnail(GtkWidget *widget,int xpos,int ypos,int width,int height)
{
	GdkPixbuf *thumbnail=GetThumbnail();

	if(transformed)
		g_object_unref(transformed);
	transformed=NULL;
	
	LayoutRectangle r(gdk_pixbuf_get_width(thumbnail),gdk_pixbuf_get_height(thumbnail));
	LayoutRectangle target(xpos,ypos,width,height);

	RectFit *fit=r.Fit(target,allowcropping,rotation,crop_hpan,crop_vpan);

	cerr << "Scaling thumbnail" << endl;

	GdkPixbuf *tmp;
	switch(fit->rotation)
	{
		case 0:
			transformed=gdk_pixbuf_scale_simple(thumbnail,fit->width,fit->height,GDK_INTERP_NEAREST);
			break;
		case 270:
			tmp=gdk_pixbuf_rotate_simple(thumbnail,GDK_PIXBUF_ROTATE_CLOCKWISE);
			transformed=gdk_pixbuf_scale_simple(tmp,fit->width,fit->height,GDK_INTERP_NEAREST);
			g_object_unref(G_OBJECT(tmp));
			break;
		case 180:
			tmp=gdk_pixbuf_rotate_simple(thumbnail,GDK_PIXBUF_ROTATE_UPSIDEDOWN);
			transformed=gdk_pixbuf_scale_simple(tmp,fit->width,fit->height,GDK_INTERP_NEAREST);
			g_object_unref(G_OBJECT(tmp));
			break;
		case 90:
			tmp=gdk_pixbuf_rotate_simple(thumbnail,GDK_PIXBUF_ROTATE_COUNTERCLOCKWISE);
			transformed=gdk_pixbuf_scale_simple(tmp,fit->width,fit->height,GDK_INTERP_NEAREST);
			g_object_unref(G_OBJECT(tmp));
			break;
	}

	int dw=fit->width;
	int dh=fit->height;
	
	if(dw > width)
		dw=width;

	if(dh > height)
		dh=height;
	
	if(dw>gdk_pixbuf_get_width(transformed))
	{
		cerr << "DW too high" << endl;
		dw=gdk_pixbuf_get_width(transformed);
	}

	if(dh>gdk_pixbuf_get_height(transformed))
	{
		cerr << "DH too high" << endl;
		dh=gdk_pixbuf_get_height(transformed);
	}

	cerr << "Done - drawing..." << endl;
	
	if(mask)
		maskpixbuf(transformed,fit->xoffset,fit->yoffset,dw,dh,mask);

	gdk_draw_pixbuf(widget->window,NULL,transformed,
		fit->xoffset,fit->yoffset,
		fit->xpos,fit->ypos,
		dw,dh,
		GDK_RGB_DITHER_NONE,0,0);

	cerr << "Done" << endl;

	delete fit;
}


void Layout_ImageInfo::SetMask(const char *filename)
{
	if(mask)
	{
		g_object_unref(mask);
		mask=NULL;
	}
	if(maskfilename)
	{
		free(maskfilename);
		maskfilename=NULL;
	}
	if(filename)
		maskfilename=strdup(filename);
}


GdkPixbuf *Layout_ImageInfo::GetThumbnail()
{
	GError *err=NULL;

	if(layout.state.batchmode)
		return(NULL);

	if(maskfilename && !mask)
	{
		mask=egg_pixbuf_get_thumbnail_for_file (maskfilename, EGG_PIXBUF_THUMBNAIL_LARGE, &err);
		cerr << "Attempting to load mask from: " << maskfilename << endl;
		if(!mask)
		{
			cerr << "Failed." << endl;
			try
			{
				ImageSource *src=ISLoadImage(maskfilename);
				if(src)
				{
					int w,h;
					w=(src->width*256)/src->height;
					h=256;
					if(w>256)
					{
						w=256;
						h=(src->height*256)/src->width;
					}
					src=ISScaleImageBySize(src,w,h,IS_SCALING_NEARESTNEIGHBOUR);
					mask=pixbuf_from_imagesource(src);
					delete src;
				}
			}
			catch(const char *err)
			{
				cerr << "Error: " << err << endl;
			}	
			if(!mask)
			{
				if(err && err->message)
					cerr << "Error: " << err->message << endl;
				else
					cerr << "Can't get mask thumbnail" << endl;
				free(maskfilename);
				maskfilename=NULL;
			}
		}
	}

	if(thumbnail)
		return(thumbnail);

	cerr << "Thumbnail not cached - loading..." << endl;

	ImageSource *src=NULL;
		
	thumbnail=egg_pixbuf_get_thumbnail_for_file (filename, EGG_PIXBUF_THUMBNAIL_LARGE, &err);

	if(!thumbnail)
	{
		cerr << "Can't get pixbuf - loading thumbnail via ImageSource..." << endl;
		src=ISLoadImage(filename);
		if(src)
		{
			int w,h;
			w=(src->width*256)/src->height;
			h=256;
			if(w>256)
			{
				w=256;
				h=(src->height*256)/src->width;
			}
			src=ISScaleImageBySize(src,w,h,IS_SCALING_NEARESTNEIGHBOUR);
			thumbnail=pixbuf_from_imagesource(src);
		}

		if(!thumbnail)
		{
			if(err && err->message)
				throw err->message;
			else
				throw "Can't get thumbnail";
		}
	}

	// If there's no display profile, then we can use the Default RGB profile instead...
	cerr << "Checking for Display Profile..." << endl;
	CMSProfile *targetprof;
	CMColourDevice target=CM_COLOURDEVICE_NONE;
	if((targetprof=layout.state.profilemanager.GetProfile(CM_COLOURDEVICE_DISPLAY)))
		target=CM_COLOURDEVICE_DISPLAY;
	else if((targetprof=layout.state.profilemanager.GetProfile(CM_COLOURDEVICE_DEFAULTRGB)))
		target=CM_COLOURDEVICE_DEFAULTRGB;
	if(targetprof)
		delete targetprof;

	if(target!=CM_COLOURDEVICE_NONE)
	{
		cerr << "Found - loading image to check for embedded profile..." << endl;
		if(!src)
		{
			src=ISLoadImage(filename);
		}

		CMSTransform *transform=NULL;
		if(src)
		{
			CMSProfile *emb;
			if(customprofile)
				emb=new CMSProfile(customprofile);  // FIXME: lifespan?
			else
				emb=src->GetEmbeddedProfile();
			if(emb)
			{
				cerr << "Creating embedded->monitor transform..." << endl;
				if(emb->GetColourSpace()==IS_TYPE_CMYK)
				{
//					Need to replace the RGB thumbnail with a CMYK version!
					cerr << "Creating new thumbnail - CMYK->monitor" << endl;
					int w,h;
					w=(src->width*256)/src->height;
					h=256;
					if(w>256)
					{
						w=256;
						h=(src->height*256)/src->width;
					}
					src=ISScaleImageBySize(src,w,h,IS_SCALING_NEARESTNEIGHBOUR);
					transform = layout.factory->GetTransform(target,emb,customintent);
					src=new ImageSource_CMS(src,transform);
					thumbnail=pixbuf_from_imagesource(src);
					transform=NULL; // Don't want to apply the transform a second time...
				}
				else
				{
					transform = layout.factory->GetTransform(target,emb,customintent);
				}
			}
			else
			{
				cerr << "Creating default->monitor transform..." << endl;
				transform = layout.factory->GetTransform(target,IS_TYPE_RGB,customintent);
			}
		}
		if(transform)
		{
			cerr << "Applying transform..." << endl;
			ImageSource *src2=new ImageSource_GdkPixbuf(thumbnail);
			src2=new ImageSource_CMS(src2,transform);
			GdkPixbuf *tn2=pixbuf_from_imagesource(src2);
			delete src2;
			g_object_unref(G_OBJECT(thumbnail));
			thumbnail=tn2;
		}
	}

	if(src)
	{
		delete src;		
	}

	return(thumbnail);
}


LayoutRectangle *Layout_ImageInfo::GetBounds()
{
	// Dummy function - override in subclasses!
	LayoutRectangle *result=new LayoutRectangle(0,0,100,100);
	return(result);
}


ImageSource *Layout_ImageInfo::GetImageSource(CMColourDevice target,CMTransformFactory *factory)
{
	ImageSource *result=NULL;
	try
	{
		cerr << "Loading image " << endl;
		ImageSource *is=ISLoadImage(filename);

		IS_TYPE colourspace=layout.GetColourSpace(target);

		if(STRIP_ALPHA(is->type)==IS_TYPE_GREY)
			is=new ImageSource_Promote(is,colourspace);

		if(STRIP_ALPHA(is->type)==IS_TYPE_BW)
			is=new ImageSource_Promote(is,colourspace);

		CMSTransform *transform=NULL;

		if(factory)
		{
			CMSProfile *emb;
			if(customprofile)
				emb=new CMSProfile(customprofile);  // FIXME: Lifespan!
			else
				emb=is->GetEmbeddedProfile();
			
			if(emb)
			{
				cerr << "Has embedded / assigned profile..." << endl;
				transform=factory->GetTransform(target,emb,customintent);  // FIXME: intent!
			}
			else
			{
				cerr << "No embedded profile - using default" << endl;
				transform=factory->GetTransform(target,IS_TYPE(STRIP_ALPHA(is->type)),customintent);
			}
	
			if(transform)
				is=new ImageSource_CMS(is,transform);
		}
		result=is;
	}
	catch(const char *err)
	{
		ErrorMessage_Dialog(err);
	}
	return(result);
}


ImageSource *Layout_ImageInfo::ApplyMask(ImageSource *is)
{
	if(maskfilename)
	{
		ImageSource *mask=ISLoadImage(maskfilename);
		if((is->width>is->height)^(mask->width>mask->height))
		{
			mask=new ImageSource_Rotate(mask,90);
			cerr << "Rotating mask" << endl;
		}
		mask=ISScaleImageBySize(mask,is->width,is->height,IS_SCALING_AUTOMATIC);
		is=new ImageSource_Mask(is,mask);
	}
	return(is);
}


bool Layout_ImageInfo::GetSelected()
{
	return(selected);
}


void Layout_ImageInfo::SetSelected(bool sel)
{
	selected=sel;
}


void Layout_ImageInfo::ToggleSelected()
{
	selected=!selected;
}


const char *Layout_ImageInfo::GetFilename()
{
	return(filename);
}


void Layout_ImageInfo::FlushThumbnail()
{
	g_object_unref(thumbnail);
	thumbnail=NULL;
}


void Layout_ImageInfo::AssignProfile(const char *filename)
{
	if(customprofile)
		free(customprofile);
	customprofile=NULL;
	if(filename)
		customprofile=strdup(filename);
	if(customprofile)
		cerr << "AssignProfile:  Custom Profile now set to " << customprofile << endl;
	else
		cerr << "AssignProfile:  No custom profile set" << endl;
	FlushThumbnail();
}


const char *Layout_ImageInfo::GetAssignedProfile()
{
	return(customprofile);
}


void Layout_ImageInfo::SetRenderingIntent(int intent)
{
	FlushThumbnail();
	customintent=intent;
}


int Layout_ImageInfo::GetRenderingIntent()
{
	return(customintent);
}


Layout::Layout(PhotoPrint_State &state,Layout *oldlayout)
	: xoffset(0), yoffset(0), pages(1), currentpage(0), backgroundfilename(NULL), background(NULL),
	backgroundtransformed(NULL), state(state), imagelist(NULL), iterator(NULL), factory(NULL)
{
	factory=state.profilemanager.GetTransformFactory();
}


Layout::~Layout()
{
	if(factory)
		delete factory;
	if(backgroundtransformed)
		g_object_unref(G_OBJECT(backgroundtransformed));
	backgroundtransformed=NULL;

	if(background)
		g_object_unref(G_OBJECT(background));
	background=NULL;

	if(backgroundfilename)
		free(backgroundfilename);
	backgroundfilename=NULL;

	if(imagelist)
	{
		for(unsigned int i=0;i<g_list_length(imagelist);++i)
		{
			GList *element=g_list_nth(imagelist,i);
			Layout_ImageInfo *ii=(Layout_ImageInfo *)element->data;
			delete ii;
		}
	}
	g_list_free(imagelist);
}


void Layout::Clear()
{
	SetBackground(NULL);
	if(imagelist)
	{
		for(unsigned int i=0;i<g_list_length(imagelist);++i)
		{
			GList *element=g_list_nth(imagelist,i);
			Layout_ImageInfo *ii=(Layout_ImageInfo *)element->data;
			delete ii;
		}
	}
	g_list_free(imagelist);
	imagelist=NULL;
	pages=1;
}


// DUMMY FUNCTION - should be overridden by subclasses
int Layout::AddImage(const char *filename,bool allowcropping,PP_ROTATION rotation)
{
	cerr << "AddImage: Dummy function - should be overridden" << endl;
	return(0);
}


// DUMMY FUNCTION - should be overridden by subclasses
void Layout::CopyImage(Layout_ImageInfo *ii)
{
	cerr << "CopyImage: Dummy function - should be overridden" << endl;
}


ImageSource *Layout::GetImageSource(int page,CMColourDevice target,CMTransformFactory *factory,int res)
{
	cerr << "GetImageSource: Dummy function - should be overridden" << endl;
	return(NULL);
}


IS_TYPE Layout::GetColourSpace(CMColourDevice target)
{
	enum IS_TYPE colourspace=IS_TYPE_RGB;
	CMSProfile *profile=state.profilemanager.GetProfile(target);
	if(profile)
	{
		colourspace=profile->GetColourSpace();
	}
	else
	{
		const char *cs=state.FindString("PrintColourSpace");
		if(strcmp(cs,"RGB")==0)
			colourspace=IS_TYPE_RGB;
		else if(strcmp(cs,"CMYK")==0)
			colourspace=IS_TYPE_CMYK;
		else
			cerr << "PrintColourSpace is set to an unknown colour space!" << endl;
	}
	return(colourspace);
}


void Layout::TransferImages(Layout *oldlayout,Progress *p)
{
	if(currentpage>=pages);
	if(oldlayout)
	{
		int count=0,img=0;
		Layout_ImageInfo *ii=oldlayout->FirstImage();
		while(ii)
		{
			++count;
			ii=oldlayout->NextImage();
		}
		ii=oldlayout->FirstImage();
		while(ii)
		{
			++img;
			CopyImage(ii);
			ii=oldlayout->NextImage();
			if(p)
			{
				if(!(p->DoProgress(img,count)))
					ii=NULL;
			}
		}
	}
	if(currentpage>=pages)
		currentpage=pages-1;
}


void Layout::DBToLayout(LayoutDB &db)
{
	state.printer.SetDriver(state.printoutput.FindString("Driver"));
}


void Layout::LayoutToDB(LayoutDB &db)
{
}


void Layout::Reflow()
{
}


GtkWidget *Layout::CreateWidget()
{
	return(NULL);
}


void Layout::RefreshWidget(GtkWidget *widget)
{
}


int Layout::GetCurrentPage()
{
	return(currentpage);
}


void Layout::SetCurrentPage(int page)
{
	currentpage=page;
}


void Layout::Print(Progress *p)
{
	state.printer.SetProgress(p);
	CMTransformFactory *factory=state.profilemanager.GetTransformFactory();
	for(int p=0;p<GetPages();++p)
	{
		ImageSource *is=GetImageSource(p,CM_COLOURDEVICE_PRINTER,factory);
		if(is)
		{
			state.printer.Print(is,xoffset,yoffset);
			delete is;
		}
	}
	delete factory;
	state.printer.SetProgress(NULL);
}

void (*Layout::SetUnitFunc())(GtkWidget *wid,enum Units unit)
{
	cerr << "This function should be overridden" << endl;
	return(NULL);
}


void Layout::DrawPreview(GtkWidget *widget,int xpos,int ypos,int width,int height)
{
	Layout_ImageInfo *ii=FirstImage();

	while(ii)
	{
		if(currentpage==ii->page)
			ii->DrawThumbnail(widget,xpos,ypos,width,height);
		ii=NextImage();
	}
}


Layout_ImageInfo *Layout::FirstImage()
{
	iterator=imagelist;
	if(iterator)
		return((Layout_ImageInfo *)iterator->data);
	else
		return(NULL);
}


Layout_ImageInfo *Layout::NextImage()
{
	if(iterator)
		iterator=g_list_next(iterator);
	if(iterator)
		return((Layout_ImageInfo *)iterator->data);
	else
		return(NULL);
}


int Layout::GetPages()
{
	return(pages);
}


Layout_ImageInfo *Layout::FirstSelected()
{
	Layout_ImageInfo *ii=FirstImage();
	while(ii)
	{
		if(ii->GetSelected())
			return(ii);
		ii=NextSelected();
	}
	return(NULL);
}


Layout_ImageInfo *Layout::NextSelected()
{
	Layout_ImageInfo *ii=NextImage();
	while(ii)
	{
		if(ii->GetSelected())
			return(ii);
		ii=NextSelected();
	}
	return(NULL);
}


void Layout::SelectAll()
{
	Layout_ImageInfo *ii=FirstImage();
	while(ii)
	{
		ii->SetSelected(true);
		ii=NextImage();
	}
}


void Layout::SelectNone()
{
	Layout_ImageInfo *ii=FirstSelected();
	while(ii)
	{
		ii->SetSelected(false);
		ii=NextSelected();
	}
}


Layout_ImageInfo *Layout::ImageAtCoord(int x,int y)
{
	return(NULL);
}


int Layout::GetCapabilities()
{
	return(0);
}


void Layout::SetBackground(const char *filename)
{
	if(backgroundtransformed)
		g_object_unref(G_OBJECT(backgroundtransformed));
	backgroundtransformed=NULL;

	if(background)
		g_object_unref(G_OBJECT(background));
	background=NULL;

	if(backgroundfilename)
		free(backgroundfilename);
	backgroundfilename=NULL;
	
	cerr << "Setting background..." << endl;
	if(filename && strlen(filename)>0)
	{
		GError *err=NULL;

		backgroundfilename=strdup(filename);

		cerr << "Attempting to load background from: " << backgroundfilename << endl;
		background=egg_pixbuf_get_thumbnail_for_file (backgroundfilename, EGG_PIXBUF_THUMBNAIL_LARGE, &err);
		if(!background)
		{
			cerr << "Failed." << endl;
			try
			{
				ImageSource *src=ISLoadImage(backgroundfilename);
				if(src)
				{
					int w,h;
					w=(src->width*256)/src->height;
					h=256;
					if(w>256)
					{
						w=256;
						h=(src->height*256)/src->width;
					}
					src=ISScaleImageBySize(src,w,h,IS_SCALING_NEARESTNEIGHBOUR);
					background=pixbuf_from_imagesource(src);
					delete src;
				}
			}
			catch(const char *err)
			{
				cerr << "Error: " << err << endl;
			}	
			if(!background)
			{
				if(err && err->message)
					cerr << "Error: " << err->message << endl;
				else
					cerr << "Can't get mask thumbnail" << endl;
				free(backgroundfilename);
				backgroundfilename=NULL;
			}
		}
	}
}


void Layout::FlushThumbnails()
{
	delete factory;
	factory=new CMTransformFactory(state.profilemanager);

	Layout_ImageInfo *ii=FirstImage();
	while(ii)
	{
		ii->FlushThumbnail();
		ii=NextImage();	
	}
}
