#!/usr/bin/python
#
# pgi-calc-deps - a program to fill out a package list with dependencies.
#
# $Progeny: pgi-calc-deps.py,v 1.9 2002/03/17 05:28:15 jlicquia Exp $
#
# Copyright 2001 Progeny Linux Systems, Inc.
#
# This program takes a list of packages in any order, either from
# stdin or from files listed on the command line.  It calculates the
# dependencies on the packages and outputs to stdout the concatenated
# package list, including dependencies, such that you could install
# each package with dpkg in the proper order without dependency
# problems.
#
# The basic algorithm bears some explanation, as it's a bit complex.
# When a package name is read, it's pushed onto a stack, which is
# initially empty.  Then, the last item on the stack is examined.  If
# it's already in the output list, it's popped off; otherwise, its
# deps are calculated with "apt-cache depends".  If all its deps can
# be met with packages already in the output list, then it's appended
# to the output list and popped off the stack.  Otherwise, any deps
# that aren't in the output list are also pushed on the stack.  After
# the deps are examined, the program loops, checking the last item on
# the stack again; this loop continues until the stack is empty.  Once
# the stack is empty, another package is read, which causes this
# entire process to start again.  Once the last package is read, we're
# done; we write our list to stdout and exit.
#
# This is obviously simplified; for example, to prevent dependency
# loops, we abort if the stack ever gets to 100.

import sys
import os
import string
import re

# Many packages have versioned conflicts with each other, which is
# fine (as long as the proper versions are available).  Unfortunately,
# "apt-cache depends" reports these as simple conflicts, which is
# usually wrong.  So, if "apt-cache depends" is reporting a conflict,
# we have to check the situation out a bit more closely.  This
# function does that.

all_versions = {}
all_conflicts = {}

def check_conflict(pkg1, pkg2):
    "Report whether pkg1 conflicts with pkg2."

    global all_versions
    global all_conflicts

##     sys.stderr.write("DEBUG: Checking conflict of %s with %s...\n"
##                      % (pkg1, pkg2))

    # Get the relevant information from both packages and store it.
    result = 0
    for pkg in (pkg1, pkg2):
        if not all_versions.has_key(pkg):
            version = ""
            conflicts = []
            pipe = os.popen("apt-cache show %s" % pkg)
            for line in pipe.readlines():
                if line[-1] == "\n":
                    line = line[:-1]
                if len(line) < 1:
                    continue
                if line[0] == " ":
                    continue
                m = re.match(r'(.+):\s*(.+)', line)
                if not m:
                    continue
                (field, value) = m.group(1, 2)
                if field == "Version":
                    version = value
                elif field == "Conflicts":
                    conflicts = re.split(r',\s*', value)
            pipe.close()
            all_versions[pkg] = version
            all_conflicts[pkg] = conflicts

    # Now look for the particular conflict that's causing trouble.
    for conflict in all_conflicts[pkg1]:
        conflict_match = re.search(r'^(\S+)\s+', conflict)
        if conflict_match:
            conflict_pkg = conflict_match.group(1)
        else:
            conflict_pkg = conflict
        if conflict_pkg == pkg2:
            # Jackpot!  Now parse any versioning information that
            # might be available.  If none is available, then there's
            # a conflict.
            m = re.search(r'\((\S+)\s+([^\)]+)\)', conflict)
            if not m:
                result = 1
                break

            (modifier, comp_version) = m.group(1, 2)

            # Now do the comparison, using the modifier found in the
            # conflict.  If the comparison succeeds, then there's a
            # conflict.
            retval = os.system("dpkg --compare-versions '%s' '%s' '%s'"
                               % (all_versions[pkg2], modifier, comp_version))
            if (retval >> 8) == 0:
                result = 1
                break

    # Whatever we found, return it.
    return result

# Figure out what files we're working with.
if len(sys.argv) > 1:
    filelist = sys.argv[1:]
else:
    filelist = [ "-" ]

# Run through all the files of packages...
package_list_out = []
conflicts_list = []
package_queue = []
package_bumped_list = []
pkg_relations = {}
for fn in filelist:
    if fn == "-":
        file = sys.stdin
    else:
        file = open(fn)

    # Run through all the packages in this file.
    for newpkg in file.readlines():
        # Skip comments and blank lines.
        if newpkg[0] == "#":
            continue
        if not re.search(r'\S+', newpkg):
            continue

        # Add the package to the stack for processing.
        package_queue.append(string.strip(newpkg))

        # Start processing the stack.
        while package_queue:
            # Whoa - if the stack is this large, we've got a loop
            # somewhere.  We need to freak out.
            if len(package_queue) > 100:
                sys.stderr.write(
"""ERROR: Package stack too big, probable loop.
  First ten elements of current stack:
""")
                for pkg in package_queue[-10:]:
                    sys.stderr.write("    %s\n" % pkg)
                sys.exit(1)

            # Get package on top of the stack.
            pkg = package_queue[-1]

            # Already looked at this package.  Move on...
            if pkg in package_list_out:
                package_queue = package_queue[:-1]
                continue

            # Have we already seen this package?
            if pkg_relations.has_key(pkg):
                # Yup - just grab the relationship info we saw before.
                relationships = pkg_relations[pkg]
            else:
                # Nope - need to get it from apt.
                deppipe = os.popen("apt-cache depends %s" % pkg)
                relationships = {}
                accumulator = []
                accumulator_field = ""
                concat_next = ""
                for line in deppipe.readlines():
                    # Skip the first line, which typically contains the
                    # package name.
                    if line[0] != " ":
                        continue

                    # Check if a new relationship is being declared.
                    m = re.search(r'\s+(\|?)([a-zA-Z]+):\s+(.+)', line)
                    if m:
                        # "Or relationships" are treated, for our
                        # purposes, exactly the same as alternatives.  So,
                        # if the previous relationship indicated an or
                        # with the next one (by prepending a "|" to the
                        # field name), then just tack this one on to the
                        # accumulator.  Otherwise, save the accumulated
                        # results from before and start a new relationship
                        # record.
                        if concat_next != "|":
                            if accumulator_field:
                                if not relationships.has_key(accumulator_field):
                                    relationships[accumulator_field] = []
                                relationships[accumulator_field].append(accumulator)
                            accumulator = []
                        (concat_next, accumulator_field, value) = \
                            m.group(1, 2, 3)

                        # Ignore virtual packages.
                        if re.match(r'<.*>', value):
                            value = ""
                    else:
                        value = line

                    # Add the value to the accumulator, if there's one to add.
                    if value:
                        value = string.strip(value)
                        if value[-1] == "\n":
                            value = value[:-1]
                        accumulator.append(value)

                # OK, we've read all the relationships.  Clean up and save
                # the last accumulated relationship.
                deppipe.close()
                if accumulator_field:
                    if not relationships.has_key(accumulator_field):
                        relationships[accumulator_field] = []
                    relationships[accumulator_field].append(accumulator)

                # Save the package's relationships in case we need to
                # resolve a problem later.
                pkg_relations[pkg] = relationships

            # Now check that all the relationships are OK.  We ignore
            # Suggests and Recommends for now, and only worry about
            # Depends and Conflicts.

            # Conflicts aren't currently handled very well.  If we've
            # already "approved" a package, and a later package
            # conflicts, we basically throw out the entire queue and
            # go on to the next package (after giving the appropriate
            # post-mortem to stderr, of course).  Other than a check
            # for versioned conflicts (which apt-cache depends reports
            # as simple conflicts), this is all we do.  At this stage,
            # I'm loath to overengineer the problem; since this is
            # originally intended to just build the minimal system
            # tarball, I'm not expecting any major conflicts.  If this
            # becomes a problem, though, we'll likely need to replace
            # this with something more sophisticated.
            pkg_conflicts = []
            try:
                if relationships.has_key("Conflicts"):
                    for pkglist in relationships["Conflicts"]:
                        # There should only be one package in any
                        # alternatives group here.
                        for deppkg in pkglist:
                            if deppkg in package_list_out:
                                if check_conflict(pkg, deppkg):
                                    sys.stderr.write(
"""BAD: %s conflicts with %s.
  Current stack contents:
""" % (pkg, deppkg))
                                    for queuepkg in package_queue:
                                        sys.stderr.write("    %s\n" % queuepkg)
                                    raise RuntimeError
                            pkg_conflicts.append(deppkg)
            except RuntimeError:
                package_queue = []
                continue

            # Depends:
            stack_changed = 0
            if relationships.has_key("Depends"):
                for pkglist in relationships["Depends"]:
                    # Check each alternative in turn in the order
                    # given by apt-cache, adding the
                    # already-approved ones to would_satisfy and
                    # setting found to the first such.
                    would_satisfy = []
                    found = None
                    for deppkg in pkglist:
                        if deppkg in package_list_out:
                            pass
                        elif deppkg in conflicts_list:
                            # Oops!  Check a little more closely.
                            conflict_approved = 1
                            for approved_pkg in package_list_out:
                                if check_conflict(approved_pkg, deppkg):
                                    conflict_approved = 0
                                    break
                            if not conflict_approved:
                                continue

                        if not found:
                            found = deppkg
                        would_satisfy.append(deppkg)

                    # If none of the alternatives passed muster, throw
                    # a fit.
                    if not would_satisfy:
                        sys.stderr.write(
"""BAD: Dep of %s conflicts with approved package.
  Current dep alternatives:
""" % pkg)
                        for deppkg in pkglist:
                            sys.stderr.write("    %s\n" % deppkg)
                        sys.stderr.write("  Current stack contents:\n")
                        for queuepkg in package_queue:
                            sys.stderr.write("    %s\n" % queuepkg)
                        sys.stderr.write("  Current approved packages:\n")
                        for approved_pkg in package_list_out:
                            sys.stderr.write("    %s\n" % approved_pkg)
                        raise RuntimeError

                    # If any of our potential winners are already
                    # in the list, use it, otherwise use the
                    # first one.
                    if found:
                        deppkg = found
                        for i in would_satisfy:
                            if i in package_list_out:
                                deppkg = i
                                break

##                    sys.stderr.write("DEBUG: Checking dep of %s on %s...\n"
##                                     % (pkg, deppkg))

                    # If the package depends on itself, just skip the
                    # dep and move on.
                    if pkg == deppkg:
##                        sys.stderr.write("DEBUG: Package depends on itself; skipping.\n")
                        continue

                    # If the package depends on the next package in
                    # the stack, that's most likely indicative of a
                    # loop, where two packages depend on each other.
                    # In this case, just assume that this particular
                    # dependency is met, and move on.
                    if len(package_queue) > 1:
                        if deppkg == package_queue[-2]:
##                            sys.stderr.write(
##                                "DEBUG: Possible loop, skipping.\n")
                            continue

                    # If the package is already in the queue, then
                    # there are several things that could be
                    # happening; the package itself and one of its
                    # deps could both depend on it, the package could
                    # depend on itself, or there could be some kind of
                    # a dependency loop.  The best thing to do is move
                    # the package up in the queue and reconsider the
                    # queue from the start.  If there's a loop, we
                    # break it by keeping track of packages we move
                    # up; if they come up a second time, ignore the
                    # dep.
                    if deppkg in package_queue:
                        if deppkg in package_bumped_list:
                            package_bumped_list.remove(deppkg)
                            continue
                        else:
##                             sys.stderr.write(
##                                 "DEBUG: Moving %s to top of stack.\n"
##                                 % deppkg)
                            package_queue.remove(deppkg)
                            package_queue.append(deppkg)
                            package_bumped_list.append(deppkg)
                            stack_changed = 1
                            break

                    # Now we have a clear dependency.  Is it already
                    # approved?  If so, go on to the next one.
                    if deppkg in package_list_out:
                        continue

                    # We're adding a new package to the equation.  We
                    # need to check its deps as well, so add it to the
                    # stack and note that the stack has changed.
##                    sys.stderr.write("DEBUG: Adding %s to the stack.\n"
##                                     % deppkg)
                    package_queue.append(deppkg)
                    stack_changed = 1
                    break

            # If this package hasn't caused the stack to change, then
            # all its deps are currently met.  Add it to the output
            # list and pop it off the stack.  Also, remember what it
            # conflicts with, so we can check for that in future
            # packages.
            if not stack_changed:
##                sys.stderr.write("DEBUG: Approved %s.\n" % pkg)
                package_list_out.append(pkg)
                package_queue = package_queue[:-1]
                conflicts_list = conflicts_list + pkg_conflicts

    # OK, we're back to the scope where we've just read all the
    # packages from one of the files in the list and put it through
    # the above torture test.
    file.close()

# Now we're done reading all the files we're going to read, and we
# supposedly have a complete package list in package_list_out.  Write
# that package list to stdout.
sys.stdout.write(string.join(package_list_out, "\n") + "\n")
