#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: meshDraw.c,v 1.00 2001/09/25 06:48:57 knepley Exp $";
#endif

/*
     Defines the interface to functions for drawing a mesh
*/

#include "src/mesh/meshimpl.h"    /*I "mesh.h" I*/

#undef  __FUNCT__
#define __FUNCT__ "MeshDrawLine"
/*@
  MeshDrawLine - This function draws a line, taking into account the
  periodicity of the mesh.

  Collective on Mesh

  Input Parameters:
+ mesh  - The mesh
. draw  - The PetscDraw context
. xA,yA - The coordinates of an endpoint
. xB,yB - The coordinates of the other endpoint
- color - The line color

  Level: intermediate

.keywords mesh, draw
.seealso MeshDrawTriangle()
@*/
int MeshDrawLine(Mesh mesh, PetscDraw draw, double xA, double yA, double xB, double yB, int color)
{
  int ierr;

  PetscFunctionBegin;
  if (mesh->isPeriodic == PETSC_TRUE) {
    ierr = PetscDrawLine(draw, xA, yA, MeshPeriodicRelativeX(mesh, xB, xA), MeshPeriodicRelativeY(mesh, yB, yA), color);
    CHKERRQ(ierr);
  } else {
    ierr = PetscDrawLine(draw, xA, yA, xB, yB, color);                                                    CHKERRQ(ierr);
  }
  PetscFunctionReturn(ierr);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshDrawTriangle"
/*@
  MeshDrawTriangle - This function draws a triangle, taking into account the
  periodicity of the mesh.

  Collective on Mesh

  Input Parameters:
+ mesh  - The mesh
. draw  - The PetscDraw context
. xA,yA - The coordinates of the first corner
. xB,yB - The coordinates of the second corner
. xC,yC - The coordinates of the third corner
. colorA, colorB, colorC - The colors of the corners

  Level: intermediate

.keywords mesh, draw
.seealso MeshDrawLine()
@*/
int MeshDrawTriangle(Mesh mesh, PetscDraw draw, double xA, double yA, double xB, double yB, double xC, double yC, int colorA,
                     int colorB, int colorC)
{
  int ierr;

  PetscFunctionBegin;
  if (mesh->isPeriodic == PETSC_TRUE) {
    ierr = PetscDrawTriangle(draw, xA, yA, MeshPeriodicRelativeX(mesh, xB, xA), MeshPeriodicRelativeY(mesh, yB, yA),
                        MeshPeriodicRelativeX(mesh, xC, xA), MeshPeriodicRelativeY(mesh, yC, yA), colorA, colorB, colorC);
    CHKERRQ(ierr);
  } else {
    ierr = PetscDrawTriangle(draw, xA, yA, xB, yB, xC, yC, colorA, colorB, colorC);                       CHKERRQ(ierr);
  }
  PetscFunctionReturn(ierr);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshSetHighlightElement"
/*@C
  MeshSetHighlightElement - This function highlights the given element when the mesh is displayed.

  Collective on Mesh

  Input Parameters:
+ mesh    - The mesh
- element - The local element index to highlight

  Level: intermediate

  Note: You can retrieve the local element index from the global element index using
  the Partition function PartitionGlobalToLocalElementIndex()

.keywords mesh, element
.seealso MeshGetHighlightElement(), MeshView()
@*/
int MeshSetHighlightElement(Mesh mesh, int element)
{
  PetscFunctionBegin;
  PetscValidHeaderSpecific(mesh, MESH_COOKIE);
  mesh->highlightElement = element;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshGetHighlightElement"
/*@C
  MeshGetHighlightElement - This function returns the element which is highlighted when the mesh is displayed.

  Collective on Mesh

  Input Parameter:
. mesh  - The mesh

  Output Parameter:
. element - The local element element to highlight, or -1 for no highlighting

  Level: intermediate

  Note: You can retrieve the global element index from the local element index using
  the Partition function PartitionLocalToGlobalElementIndex()

.keywords mesh, element
.seealso MeshSetHighlightElement(), MeshView()
@*/
int MeshGetHighlightElement(Mesh mesh, int *element)
{
  PetscFunctionBegin;
  PetscValidHeaderSpecific(mesh, MESH_COOKIE);
  PetscValidIntPointer(element);
  *element = mesh->highlightElement;
  PetscFunctionReturn(0);
}
