#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: fieldClassMapCreate.c,v 1.3 2000/01/10 03:54:17 knepley Exp $";
#endif

#include "src/grid/gridimpl.h"    /*I "grid.h" I*/

#undef  __FUNCT__
#define __FUNCT__ "FieldClassMapCreate"
/*@ 
  FieldClassMapCreate - This function creates an empty field class map.

  Collective on MPI_Comm

  Input Parameter:
. comm - The communicator for the object

  Output Parameter:
. map  - The map

  Level: beginner

.keywords: class, field class, class map, create
.seealso: FieldClassMapSetType()
@*/
int FieldClassMapCreate(MPI_Comm comm, FieldClassMap *map)
{
  FieldClassMap cm;
  int           ierr;

  PetscFunctionBegin;
  PetscValidPointer(map);
  *map = PETSC_NULL;
#ifndef PETSC_USE_DYNAMIC_LIBRARIES
  ierr = GridInitializePackage(PETSC_NULL);                                                               CHKERRQ(ierr);
#endif

  PetscHeaderCreate(cm, _FieldClassMap, FieldClassMapOps, CLASS_MAP_COOKIE, -1, "FieldClassMap", comm,
                    FieldClassMapDestroy, FieldClassMapView);
  PetscLogObjectCreate(cm);
  PetscLogObjectMemory(cm, sizeof(struct _FieldClassMap));
  ierr = PetscMemzero(cm->ops, sizeof(FieldClassMapOps));                                                 CHKERRQ(ierr);
  cm->bops->publish  = PETSC_NULL /* FieldClassMapPublish_Petsc */;
  cm->type_name      = PETSC_NULL;
  cm->serialize_name = PETSC_NULL;
  cm->data           = PETSC_NULL;

  /* Initialize fields */
  cm->numFields          = 0;
  cm->fields             = PETSC_NULL;
  cm->numNodes           = 0;
  cm->numGhostNodes      = 0;
  cm->numOverlapNodes    = 0;
  cm->numClasses         = 0;
  cm->fieldClasses       = PETSC_NULL;
  cm->classes            = PETSC_NULL;
  cm->classSizes         = PETSC_NULL;
  cm->isReduced          = PETSC_FALSE;
  cm->reduceFieldClasses = PETSC_NULL;
  cm->isConstrained      = PETSC_FALSE;
  cm->isClassConstrained = PETSC_NULL;
  cm->classSizeDiffs     = PETSC_NULL;
  cm->mapSize            = 0;
  cm->numOldClasses      = 0;
  cm->classMap           = PETSC_NULL;

  *map = cm;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "FieldClassMapSerialize"
/*@ 
  FieldClassMapSerialize - This function stores or recreates a field class map using a viewer for a binary file.

  Collective on MPI_Comm

  Input Parameters:
+ comm   - The communicator for the object
. viewer - The viewer context
- store  - This flag is PETSC_TRUE is data is being written, otherwise it will be read

  Output Parameter:
. map    - The map

  Level: beginner

.keywords: class, field class, class map, serialize
.seealso: MeshSerialize(), GridSerialize()
@*/
int FieldClassMapSerialize(MPI_Comm comm, FieldClassMap *map, PetscViewer viewer, PetscTruth store)
{
  int      (*serialize)(MPI_Comm, FieldClassMap *, PetscViewer, PetscTruth);
  int        fd, len;
  char      *name;
  PetscTruth match;
  int        ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(viewer, PETSC_VIEWER_COOKIE);
  PetscValidPointer(map);

  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_BINARY, &match);                             CHKERRQ(ierr);
  if (match == PETSC_FALSE) SETERRQ(PETSC_ERR_ARG_WRONG, "Must be binary viewer");
  ierr = PetscViewerBinaryGetDescriptor(viewer, &fd);                                                     CHKERRQ(ierr);

  if (store) {
    PetscValidHeaderSpecific(*map, CLASS_MAP_COOKIE);
    ierr = PetscStrlen((*map)->class_name, &len);                                                         CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &len,                    1,   PETSC_INT,  0);                             CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  (*map)->class_name,     len, PETSC_CHAR, 0);                             CHKERRQ(ierr);
    ierr = PetscStrlen((*map)->serialize_name, &len);                                                     CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &len,                    1,   PETSC_INT,  0);                             CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  (*map)->serialize_name, len, PETSC_CHAR, 0);                             CHKERRQ(ierr);
    ierr = PetscFListFind(comm, FieldClassMapSerializeList, (*map)->serialize_name, (void (**)(void)) &serialize);CHKERRQ(ierr);
    if (!serialize) SETERRQ(PETSC_ERR_ARG_WRONG, "Type cannot be serialized");
    ierr = (*serialize)(comm, map, viewer, store);                                                        CHKERRQ(ierr);
  } else {
    ierr = PetscBinaryRead(fd, &len,    1,   PETSC_INT);                                                  CHKERRQ(ierr);
    ierr = PetscMalloc((len+1) * sizeof(char), &name);                                                    CHKERRQ(ierr);
    name[len] = 0;
    ierr = PetscBinaryRead(fd,  name,   len, PETSC_CHAR);                                                 CHKERRQ(ierr);
    ierr = PetscStrcmp(name, "FieldClassMap", &match);                                                    CHKERRQ(ierr);
    ierr = PetscFree(name);                                                                               CHKERRQ(ierr);
    if (match == PETSC_FALSE) SETERRQ(PETSC_ERR_ARG_WRONG, "Non-fieldClassMap object");
    /* Dispatch to the correct routine */
    if (!FieldClassMapSerializeRegisterAllCalled) {
      ierr = FieldClassMapSerializeRegisterAll(PETSC_NULL);                                               CHKERRQ(ierr);
    }
    if (!FieldClassMapSerializeList) SETERRQ(PETSC_ERR_ARG_CORRUPT, "Could not find table of methods");
    ierr = PetscBinaryRead(fd, &len,    1,   PETSC_INT);                                                  CHKERRQ(ierr);
    ierr = PetscMalloc((len+1) * sizeof(char), &name);                                                    CHKERRQ(ierr);
    name[len] = 0;
    ierr = PetscBinaryRead(fd,  name,   len, PETSC_CHAR);                                                 CHKERRQ(ierr);
    ierr = PetscFListFind(comm, FieldClassMapSerializeList, name, (void (**)(void)) &serialize);          CHKERRQ(ierr);
    if (!serialize) SETERRQ(PETSC_ERR_ARG_WRONG, "Type cannot be serialized");
    ierr = (*serialize)(comm, map, viewer, store);                                                        CHKERRQ(ierr);
    ierr = PetscStrfree((*map)->serialize_name);                                                          CHKERRQ(ierr);
    (*map)->serialize_name = name;
  }

  PetscFunctionReturn(0);
}
