#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: linear.c,v 1.7 2000/01/10 03:54:15 knepley Exp $";
#endif

/*
   Defines piecewise linear function space on a two dimensional 
   grid. Suitable for finite element type discretization of a PDE.
*/

#include "src/grid/discretization/discimpl.h"         /*I "discretization.h" I*/
#include "src/mesh/impls/triangular/triimpl.h"

extern int DiscTransformCoords_Triangular_2D_Quadratic(double, double, double *, double *, double *);

/* For precomputed integrals, the table is structured as follows:

     precompInt[op,i,j] = \int_{SE} <op \phi^i(\xi,\eta), \phi^j(\xi,\eta)> |J^{-1}|

   where we recall that |J| is a constant for linear affine maps,
   and the map of any triangle to the standard element is linear.
   The numbering of the nodes in the standard element is

                 3
                 |\
                 | \
                 |  \
                 |   \
                 1----2
*/

#undef  __FUNCT__
#define __FUNCT__ "DiscDestroy_Triangular_2D_Linear"
static int DiscDestroy_Triangular_2D_Linear(Discretization disc) {
  PetscFunctionBegin;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscView_Triangular_2D_Linear_File"
static int DiscView_Triangular_2D_Linear_File(Discretization disc, PetscViewer viewer) {
  PetscFunctionBegin;
  PetscViewerASCIIPrintf(viewer, "Linear discretization\n");
  PetscViewerASCIIPrintf(viewer, "    %d shape functions per component\n", disc->funcs);
  PetscViewerASCIIPrintf(viewer, "    %d registered operators\n", disc->numOps);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscView_Triangular_2D_Linear"
static int DiscView_Triangular_2D_Linear(Discretization disc, PetscViewer viewer) {
  PetscTruth isascii;
  int        ierr;

  PetscFunctionBegin;
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_ASCII, &isascii);                            CHKERRQ(ierr);
  if (isascii == PETSC_TRUE) {
    ierr = DiscView_Triangular_2D_Linear_File(disc, viewer);                                              CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateFunctionGalerkin_Triangular_2D_Linear"
static int DiscEvaluateFunctionGalerkin_Triangular_2D_Linear(Discretization disc, Mesh mesh, PointFunction f, PetscScalar alpha,
                                                             int elem, PetscScalar *array, void *ctx)
{
  Mesh_Triangular *tri            = (Mesh_Triangular *) mesh->data;
  double          *nodes          = tri->nodes;
  int             *elements       = tri->faces;
  int              corners        = mesh->numCorners;
  int              dim            = disc->dim;
  int              comp           = disc->comp;           /* The number of components in this field */
  int              funcs          = disc->funcs;          /* The number of shape functions per component */
  PetscScalar     *funcVal        = disc->funcVal;        /* Function value at a quadrature point */
  int              numQuadPoints  = disc->numQuadPoints;  /* Number of points used for Gaussian quadrature */
  double          *quadPoints     = disc->quadPoints;     /* Points in the standard element for Gaussian quadrature */
  double          *quadWeights    = disc->quadWeights;    /* Weights in the standard element for Gaussian quadrature */
  double          *quadShapeFuncs = disc->quadShapeFuncs; /* Shape function evaluated at quadrature points */
  double           jac;                                   /* |J| for map to standard element */
  double           x, y;                                  /* The integration point */
  double           x11, y11, x21, y21, x31, y31;
  int              rank = -1;
  int              i, j, k, p;
#ifdef PETSC_USE_BOPT_g
  PetscTruth       opt;
#endif
  int              ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);

  /* For dummy collective calls */
  if (array == PETSC_NULL) {
    for(p = 0; p < numQuadPoints; p++) {
      ierr = (*f)(0, 0, PETSC_NULL, PETSC_NULL, PETSC_NULL, PETSC_NULL, ctx);                             CHKERRQ(ierr);
    }
    PetscFunctionReturn(0);
  }

#ifdef PETSC_USE_BOPT_g
  if ((elem < 0) || (elem >= mesh->part->numOverlapElements)) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE, "Invalid element");
#endif
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element
     which must be a constant for linear elements */
  x11 = nodes[elements[elem*corners]*dim];
  y11 = nodes[elements[elem*corners]*dim+1];
  if (mesh->isPeriodic == PETSC_TRUE) {
    x21 = MeshPeriodicDiffX(mesh, nodes[elements[elem*corners+1]*dim]   - x11);
    x31 = MeshPeriodicDiffX(mesh, nodes[elements[elem*corners+2]*dim]   - x11);
    y21 = MeshPeriodicDiffY(mesh, nodes[elements[elem*corners+1]*dim+1] - y11);
    y31 = MeshPeriodicDiffY(mesh, nodes[elements[elem*corners+2]*dim+1] - y11);
  } else {
    x21 = nodes[elements[elem*corners+1]*dim]   - x11;
    x31 = nodes[elements[elem*corners+2]*dim]   - x11;
    y21 = nodes[elements[elem*corners+1]*dim+1] - y11;
    y31 = nodes[elements[elem*corners+2]*dim+1] - y11;
  }
  jac = PetscAbsReal(x21*y31 - x31*y21);
  if (jac < 1.0e-14) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g y21: %g x31: %g y31: %g\n", rank, elem, x21, y21, x31, y31);
    SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  }
#ifdef PETSC_USE_BOPT_g
  ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                        CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g y21: %g x31: %g y31: %g jac: %g\n",
                rank, elem, x21, y21, x31, y31, jac);
  }
#endif

  /* Calculate element vector entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++) {
    x    = MeshPeriodicX(mesh, x21*quadPoints[p*dim] + x31*quadPoints[p*dim+1] + x11);
    y    = MeshPeriodicY(mesh, y21*quadPoints[p*dim] + y31*quadPoints[p*dim+1] + y11);
    ierr = (*f)(1, comp, &x, &y, PETSC_NULL, funcVal, ctx);                                               CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
    ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                      CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p:%d jac: %g", rank, p, jac);
      for(j = 0; j < comp; j++) PetscPrintf(PETSC_COMM_SELF, " func[%d]: %g", j, PetscRealPart(funcVal[j]));
      PetscPrintf(PETSC_COMM_SELF, "\n");
  }
#endif

    for(i = 0, k = 0; i < funcs; i++) {
      for(j = 0; j < comp; j++, k++) {
        array[k] += alpha*funcVal[j]*quadShapeFuncs[p*funcs+i]*jac*quadWeights[p];
#ifdef PETSC_USE_BOPT_g
        ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                  CHKERRQ(ierr);
        if (opt == PETSC_TRUE) {
          PetscPrintf(PETSC_COMM_SELF, "[%d]  array[%d]: %g\n", rank, k, PetscRealPart(array[k]));
        }
#endif
      }
    }
  }
  PetscLogFlops(7 + (8 + 5*funcs*comp) * numQuadPoints);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateOperatorGalerkin_Triangular_2D_Linear"
static int DiscEvaluateOperatorGalerkin_Triangular_2D_Linear(Discretization disc, Mesh mesh, int elemSize,
                                                             int rowStart, int colStart, int op, PetscScalar alpha,
                                                             int elem, PetscScalar *field, PetscScalar *mat, void *ctx)
{
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  double          *nodes      = tri->nodes;          /* The node coordinates */
  int             *elements   = tri->faces;          /* The element corners */
  int              numCorners = mesh->numCorners;    /* The number of corners per element */
  int              dim        = disc->dim;
  Operator         oper       = disc->operators[op]; /* The operator to discretize */
  Discretization   test       = oper->test;          /* The space of test functions */
  OperatorFunction opFunc     = oper->opFunc;        /* Integrals of operators which depend on J */
  PetscScalar     *precompInt = oper->precompInt;    /* Precomputed integrals of the operator on shape functions */
  int              rowSize    = test->size;          /* The number of shape functions per element */
  int              colSize    = disc->size;          /* The number of test  functions per element */
  double           x21, x31, y21, y31;               /* Coordinates of the element, with point 1 at the origin */
  double           jac;                              /* |J| for map to standard element */
  double           coords[MAX_CORNERS*2];            /* Coordinates of the element */
  int              rank;
  int              i, j, f;
  int              ierr; 

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
  /* Check for valid operator */
  if ((op < 0) || (op >= disc->numOps) || (!disc->operators[op])) SETERRQ(PETSC_ERR_ARG_WRONG, "Invalid operator");
#endif

  if (precompInt != PETSC_NULL) {
    /* Calculate the determinant of the inverse Jacobian of the map to the standard element
       which must be a constant for linear elements - 1/|x_{21} y_{31} - x_{31} y_{21}| */
    if (mesh->isPeriodic == PETSC_TRUE) {
      x21 = MeshPeriodicDiffX(mesh, nodes[elements[elem*numCorners+1]*dim]   - nodes[elements[elem*numCorners]*dim]);
      x31 = MeshPeriodicDiffX(mesh, nodes[elements[elem*numCorners+2]*dim]   - nodes[elements[elem*numCorners]*dim]);
      y21 = MeshPeriodicDiffY(mesh, nodes[elements[elem*numCorners+1]*dim+1] - nodes[elements[elem*numCorners]*dim+1]);
      y31 = MeshPeriodicDiffY(mesh, nodes[elements[elem*numCorners+2]*dim+1] - nodes[elements[elem*numCorners]*dim+1]);
    } else {
      x21 = nodes[elements[elem*numCorners+1]*dim]   - nodes[elements[elem*numCorners]*dim];
      x31 = nodes[elements[elem*numCorners+2]*dim]   - nodes[elements[elem*numCorners]*dim];
      y21 = nodes[elements[elem*numCorners+1]*dim+1] - nodes[elements[elem*numCorners]*dim+1];
      y31 = nodes[elements[elem*numCorners+2]*dim+1] - nodes[elements[elem*numCorners]*dim+1];
    }
    jac = PetscAbsReal(x21*y31 - x31*y21);
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]x21: %g y21: %g x31: %g y31: %g jac: %g\n", rank, x21, y21, x31, y31, jac);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }

    /* Calculate element matrix entries which are all precomputed */
    for(i = 0; i < rowSize; i++) {
      for(j = 0; j < colSize; j++) {
        mat[(i+rowStart)*elemSize + j+colStart] += alpha*precompInt[i*colSize + j]*jac;
      }
    }
    PetscLogFlops(7 + 2*rowSize*colSize);
  } else {
    if (opFunc == PETSC_NULL) SETERRQ(PETSC_ERR_ARG_CORRUPT, "Invalid function");
    if (mesh->isPeriodic == PETSC_TRUE) {
      coords[0*dim+0] = nodes[elements[elem*numCorners+0]*dim+0];
      coords[0*dim+1] = nodes[elements[elem*numCorners+0]*dim+1];
      for(f = 1; f < PetscMax(disc->funcs, test->funcs); f++) {
        coords[f*dim+0] = MeshPeriodicRelativeX(mesh, nodes[elements[elem*numCorners+f]*dim+0], coords[0*dim+0]);
        coords[f*dim+1] = MeshPeriodicRelativeY(mesh, nodes[elements[elem*numCorners+f]*dim+1], coords[0*dim+1]);
      }
    } else {
      for(f = 0; f < PetscMax(disc->funcs, test->funcs); f++) {
        coords[f*dim+0] = nodes[elements[elem*numCorners+f]*dim+0];
        coords[f*dim+1] = nodes[elements[elem*numCorners+f]*dim+1];
      }
    }

    ierr = (*opFunc)(disc, test, rowSize, colSize, rowStart, colStart, elemSize, coords, alpha, field, mat, ctx);
    CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateNonlinearOperatorGalerkin_Triangular_2D_Linear"
static int DiscEvaluateNonlinearOperatorGalerkin_Triangular_2D_Linear(Discretization disc, Mesh mesh, NonlinearOperator f,
                                                                      PetscScalar alpha, int elem, int numArgs, PetscScalar **field,
                                                                      PetscScalar *vec, void *ctx)
{
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  double          *nodes      = tri->nodes;      /* The node coordinates */
  int             *elements   = tri->faces;      /* The element corners */
  int              numCorners = mesh->numCorners; /* The number of corners per element */
  int              dim        = disc->dim;
  int              comp       = disc->comp;      /* The number of components in this field */
  int              funcs      = disc->funcs;     /* The number of shape functions per component */
  PetscScalar     *funcVal    = disc->funcVal;   /* Function value at a quadrature point */
  PetscScalar    **fieldVal   = disc->fieldVal;  /* Field value and derivatives at a quadrature point */
  double           jac;                          /* |J| for map to standard element */
  double           invjac;                       /* |J^{-1}| for map from standard element */
  int              numQuadPoints;                /* Number of points used for Gaussian quadrature */
  double          *quadPoints;                   /* Points in the standard element for Gaussian quadrature */
  double          *quadWeights;                  /* Weights in the standard element for Gaussian quadrature */
  double          *quadShapeFuncs;               /* Shape function evaluated at quadrature points */
  double          *quadShapeFuncDers;            /* Shape function derivatives evaluated at quadrature points */
  double           x, y;                         /* The integration point */
  double           dxix;                         /* \PartDer{\xi}{x}  */
  double           detx;                         /* \PartDer{\eta}{x} */
  double           dxiy;                         /* \PartDer{\xi}{y}  */
  double           dety;                         /* \PartDer{\eta}{y} */
  PetscScalar      dfxi;                         /* \PartDer{field}{\xi}  */
  PetscScalar      dfet;                         /* \PartDer{field}{\eta} */
  double           x11, y11, x21, y21, x31, y31;
  int              rank = -1;
  int              i, j, k, p, func, arg;
#ifdef PETSC_USE_BOPT_g
  PetscTruth       opt;
#endif
  int              ierr;

  PetscFunctionBegin;
  if (numArgs > 2) SETERRQ(PETSC_ERR_SUP, "Only configured to handle two nonlinear arguments");
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);
  numQuadPoints     = disc->numQuadPoints;
  quadPoints        = disc->quadPoints;
  quadWeights       = disc->quadWeights;
  quadShapeFuncs    = disc->quadShapeFuncs;
  quadShapeFuncDers = disc->quadShapeFuncDers;
  
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element
     which must be a constant for linear elements */
  x11 = nodes[elements[elem*numCorners]*dim];
  y11 = nodes[elements[elem*numCorners]*dim+1];
  if (mesh->isPeriodic == PETSC_TRUE) {
    x21 = MeshPeriodicDiffX(mesh, nodes[elements[elem*numCorners+1]*dim]   - x11);
    x31 = MeshPeriodicDiffX(mesh, nodes[elements[elem*numCorners+2]*dim]   - x11);
    y21 = MeshPeriodicDiffY(mesh, nodes[elements[elem*numCorners+1]*dim+1] - y11);
    y31 = MeshPeriodicDiffY(mesh, nodes[elements[elem*numCorners+2]*dim+1] - y11);
  } else {
    x21 = nodes[elements[elem*numCorners+1]*dim]   - x11;
    x31 = nodes[elements[elem*numCorners+2]*dim]   - x11;
    y21 = nodes[elements[elem*numCorners+1]*dim+1] - y11;
    y31 = nodes[elements[elem*numCorners+2]*dim+1] - y11;
  }
  jac = PetscAbsReal(x21*y31 - x31*y21);
  if (jac < 1.0e-14) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g y21: %g x31: %g y31: %g\n", rank, elem, x21, y21, x31, y31);
    SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  }
#ifdef PETSC_USE_BOPT_g
  ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                        CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g y21: %g x31: %g y31: %g jac: %g\n",
                rank, elem, x21, y21, x31, y31, jac);
  }
#endif

  /* These are the elements of the inverse matrix */
  invjac = 1/jac;
  dxix =  y31*invjac;
  dxiy = -x31*invjac;
  detx = -y21*invjac;
  dety =  x21*invjac;

  /* Calculate element vector entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++) {
    x = MeshPeriodicX(mesh, x21*quadPoints[p*dim] + x31*quadPoints[p*dim+1] + x11);
    y = MeshPeriodicY(mesh, y21*quadPoints[p*dim] + y31*quadPoints[p*dim+1] + y11);
    /* Can this be simplified? */
    for(arg = 0; arg < numArgs; arg++) {
      for(j = 0; j < comp*(dim+1); j++) fieldVal[arg][j] = 0.0;
      for(func = 0; func < funcs; func++) {
        for(j = 0; j < comp; j++) {
          fieldVal[arg][j*(dim+1)]   += field[arg][func*comp+j]*quadShapeFuncs[p*funcs+func];
          fieldVal[arg][j*(dim+1)+1] += field[arg][func*comp+j]*quadShapeFuncDers[p*funcs*dim+func*dim];
          fieldVal[arg][j*(dim+1)+2] += field[arg][func*comp+j]*quadShapeFuncDers[p*funcs*dim+func*dim+1];
        }
      }
    }

    /* Convert the field derivatives to old coordinates */
    for(arg = 0; arg < numArgs; arg++) {
      for(j = 0; j < comp; j++) {
        dfxi                       = fieldVal[arg][j*(dim+1)+1];
        dfet                       = fieldVal[arg][j*(dim+1)+2];
        fieldVal[arg][j*(dim+1)+1] = dfxi*dxix + dfet*detx;
        fieldVal[arg][j*(dim+1)+2] = dfxi*dxiy + dfet*dety;
      }
    }

    ierr = (*f)(1, comp, &x, &y, PETSC_NULL, numArgs, fieldVal, funcVal, ctx);                            CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
    ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                      CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p:%d jac: %g", rank, p, jac);
      for(j = 0; j < comp; j++)
        PetscPrintf(PETSC_COMM_SELF, " func[%d]: %g", j, PetscRealPart(funcVal[j]));
      PetscPrintf(PETSC_COMM_SELF, "\n");
    }
#endif

    for(i = 0, k = 0; i < funcs; i++) {
      for(j = 0; j < comp; j++, k++) {
        vec[k] += alpha*funcVal[j]*quadShapeFuncs[p*funcs+i]*jac*quadWeights[p];
#ifdef PETSC_USE_BOPT_g
        ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                  CHKERRQ(ierr);
        if (opt == PETSC_TRUE) {
          PetscPrintf(PETSC_COMM_SELF, "[%d]  vec[%d]: %g\n", rank, k, PetscRealPart(vec[k]));
        }
#endif
      }
    }
  }
  PetscLogFlops(12 + (8 + (6*numArgs + 5)*funcs*comp + 6*numArgs*comp) * numQuadPoints);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateALEOperatorGalerkin_Triangular_2D_Linear"
static int DiscEvaluateALEOperatorGalerkin_Triangular_2D_Linear(Discretization disc, Mesh mesh, int elemSize,
                                                                int rowStart, int colStart, int op, PetscScalar alpha,
                                                                int elem, PetscScalar *field, PetscScalar *ALEfield, PetscScalar *mat,
                                                                void *ctx)
{
  Mesh_Triangular    *tri        = (Mesh_Triangular *) mesh->data;
  double             *nodes      = tri->nodes;          /* The node coordinates */
  int                *elements   = tri->faces;          /* The element corners */
  int                 numCorners = mesh->numCorners;    /* The number of corners per element */
  int                 dim        = disc->dim;
  Operator            oper       = disc->operators[op]; /* The operator to discretize */
  Discretization      test       = oper->test;          /* The space of test functions */
  ALEOperatorFunction opFunc     = oper->ALEOpFunc;     /* Integrals of operators which depend on J */
  int                 rowSize    = test->size;          /* The number of shape functions per element */
  int                 colSize    = disc->size;          /* The number of test  functions per element */
  double              coords[MAX_CORNERS*2];            /* Coordinates of the element */
  int                 f;
  int                 ierr; 

  PetscFunctionBegin;
#ifdef PETSC_USE_BOPT_g
  /* Check for valid operator */
  if ((op < 0) || (op >= disc->numOps) || (!disc->operators[op])) SETERRQ(PETSC_ERR_ARG_WRONG, "Invalid operator");
#endif

  if (opFunc == PETSC_NULL) SETERRQ(PETSC_ERR_ARG_CORRUPT, "Invalid function");
  if (mesh->isPeriodic == PETSC_TRUE) {
    coords[0*dim+0] = nodes[elements[elem*numCorners+0]*dim+0];
    coords[0*dim+1] = nodes[elements[elem*numCorners+0]*dim+1];
    for(f = 1; f < PetscMax(disc->funcs, test->funcs); f++) {
      coords[f*dim+0] = MeshPeriodicRelativeX(mesh, nodes[elements[elem*numCorners+f]*dim+0], coords[0*dim+0]);
      coords[f*dim+1] = MeshPeriodicRelativeY(mesh, nodes[elements[elem*numCorners+f]*dim+1], coords[0*dim+1]);
    }
  } else {
    for(f = 0; f < PetscMax(disc->funcs, test->funcs); f++) {
      coords[f*dim+0] = nodes[elements[elem*numCorners+f]*dim+0];
      coords[f*dim+1] = nodes[elements[elem*numCorners+f]*dim+1];
    }
  }

  ierr = (*opFunc)(disc, test, rowSize, colSize, rowStart, colStart, elemSize, coords, alpha, field, ALEfield, mat, ctx);
  CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateNonlinearALEOperatorGalerkin_Triangular_2D_Linear"
static int DiscEvaluateNonlinearALEOperatorGalerkin_Triangular_2D_Linear(Discretization disc, Mesh mesh, NonlinearOperator f,
                                                                         PetscScalar alpha, int elem, int numArgs, PetscScalar **field,
                                                                         PetscScalar *ALEfield, PetscScalar *vec, void *ctx)
{
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  double          *nodes      = tri->nodes;      /* The node coordinates */
  int             *elements   = tri->faces;      /* The element corners */
  int              numCorners = mesh->numCorners; /* The number of corners per element */
  int              dim        = disc->dim;
  int              comp       = disc->comp;      /* The number of components in this field */
  int              funcs      = disc->funcs;     /* The number of shape functions per component */
  PetscScalar     *funcVal    = disc->funcVal;   /* Function value at a quadrature point */
  PetscScalar    **fieldVal   = disc->fieldVal;  /* Field value and derivatives at a quadrature point */
  double           jac;                          /* |J| for map to standard element */
  double           invjac;                       /* |J^{-1}| for map from standard element */
  int              numQuadPoints;                /* Number of points used for Gaussian quadrature */
  double          *quadPoints;                   /* Points in the standard element for Gaussian quadrature */
  double          *quadWeights;                  /* Weights in the standard element for Gaussian quadrature */
  double          *quadShapeFuncs;               /* Shape function evaluated at quadrature points */
  double          *quadShapeFuncDers;            /* Shape function derivatives evaluated at quadrature points */
  double           x, y;                         /* The integration point */
  double           dxix;                         /* \PartDer{\xi}{x}  */
  double           detx;                         /* \PartDer{\eta}{x} */
  double           dxiy;                         /* \PartDer{\xi}{y}  */
  double           dety;                         /* \PartDer{\eta}{y} */
  PetscScalar      dfxi;                         /* \PartDer{field}{\xi}  */
  PetscScalar      dfet;                         /* \PartDer{field}{\eta} */
  double           x11, y11, x21, y21, x31, y31;
  int              rank;
  int              i, j, k, p, func, arg;
#ifdef PETSC_USE_BOPT_g
  PetscTruth       opt;
#endif
  int              ierr;

  PetscFunctionBegin;
  if (numArgs > 2) SETERRQ(PETSC_ERR_SUP, "Only configured to handle two nonlinear arguments");
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);

  numQuadPoints     = disc->numQuadPoints;
  quadPoints        = disc->quadPoints;
  quadWeights       = disc->quadWeights;
  quadShapeFuncs    = disc->quadShapeFuncs;
  quadShapeFuncDers = disc->quadShapeFuncDers;
  
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element
     which must be a constant for linear elements */
  x11 = nodes[elements[elem*numCorners]*dim];
  y11 = nodes[elements[elem*numCorners]*dim+1];
  if (mesh->isPeriodic == PETSC_TRUE) {
    x21 = MeshPeriodicDiffX(mesh, nodes[elements[elem*numCorners+1]*dim]   - x11);
    x31 = MeshPeriodicDiffX(mesh, nodes[elements[elem*numCorners+2]*dim]   - x11);
    y21 = MeshPeriodicDiffY(mesh, nodes[elements[elem*numCorners+1]*dim+1] - y11);
    y31 = MeshPeriodicDiffY(mesh, nodes[elements[elem*numCorners+2]*dim+1] - y11);
  } else {
    x21 = nodes[elements[elem*numCorners+1]*dim]   - x11;
    x31 = nodes[elements[elem*numCorners+2]*dim]   - x11;
    y21 = nodes[elements[elem*numCorners+1]*dim+1] - y11;
    y31 = nodes[elements[elem*numCorners+2]*dim+1] - y11;
  }
  jac = PetscAbsReal(x21*y31 - x31*y21);
  if (jac < 1.0e-14) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g y21: %g x31: %g y31: %g\n", rank, elem, x21, y21, x31, y31);
    SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  }
#ifdef PETSC_USE_BOPT_g
  ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                        CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g y21: %g x31: %g y31: %g jac: %g\n",
                rank, elem, x21, y21, x31, y31, jac);
  }
#endif

  /* These are the elements of the inverse matrix */
  invjac = 1/jac;
  dxix =  y31*invjac;
  dxiy = -x31*invjac;
  detx = -y21*invjac;
  dety =  x21*invjac;

  /* Calculate element vector entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++) {
    x = MeshPeriodicX(mesh, x21*quadPoints[p*dim] + x31*quadPoints[p*dim+1] + x11);
    y = MeshPeriodicY(mesh, y21*quadPoints[p*dim] + y31*quadPoints[p*dim+1] + y11);
    /* Can this be simplified? */
    for(arg = 0; arg < numArgs; arg++) {
      for(j = 0; j < comp*(dim+1); j++) fieldVal[arg][j] = 0.0;
      for(func = 0; func < funcs; func++)
        for(j = 0; j < comp; j++) {
          fieldVal[arg][j*(dim+1)]   += (field[arg][func*comp+j] - ALEfield[func*comp+j])*quadShapeFuncs[p*funcs+func];
          fieldVal[arg][j*(dim+1)+1] += field[arg][func*comp+j]*quadShapeFuncDers[p*funcs*dim+func*dim];
          fieldVal[arg][j*(dim+1)+2] += field[arg][func*comp+j]*quadShapeFuncDers[p*funcs*dim+func*dim+1];
        }
    }

    /* Convert the field derivatives to old coordinates */
    for(arg = 0; arg < numArgs; arg++) {
      for(j = 0; j < comp; j++) {
        dfxi                       = fieldVal[arg][j*(dim+1)+1];
        dfet                       = fieldVal[arg][j*(dim+1)+2];
        fieldVal[arg][j*(dim+1)+1] = dfxi*dxix + dfet*detx;
        fieldVal[arg][j*(dim+1)+2] = dfxi*dxiy + dfet*dety;
      }
    }

    ierr = (*f)(1, comp, &x, &y, PETSC_NULL, numArgs, fieldVal, funcVal, ctx);                            CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
    ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                      CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p:%d jac: %g", rank, p, jac);
      for(j = 0; j < comp; j++)
        PetscPrintf(PETSC_COMM_SELF, " func[%d]: %g", j, PetscRealPart(funcVal[j]));
      PetscPrintf(PETSC_COMM_SELF, "\n");
    }
#endif

    for(i = 0, k = 0; i < funcs; i++) {
      for(j = 0; j < comp; j++, k++) {
        vec[k] += alpha*funcVal[j]*quadShapeFuncs[p*funcs+i]*jac*quadWeights[p];
#ifdef PETSC_USE_BOPT_g
        ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                  CHKERRQ(ierr);
        if (opt == PETSC_TRUE) {
          PetscPrintf(PETSC_COMM_SELF, "[%d]  vec[%d]: %g\n", rank, k, PetscRealPart(vec[k]));
        }
#endif
      }
    }
  }
  PetscLogFlops(12 + (8 + (7*numArgs + 5)*funcs*comp + 6*numArgs*comp) * numQuadPoints);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Laplacian_Triangular_2D_Linear"
int Laplacian_Triangular_2D_Linear(Discretization disc, Discretization test, int rowSize, int colSize,
                                   int globalRowStart, int globalColStart, int globalSize, double *coords,
                                   PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
  double      x21, x31, y21, y31; /* Coordinates of the element, with point 1 at the origin */
  double      jac;                /* |J| for map to standard element */
  PetscScalar w;                  /* 1/(2 jac) */
  int         comp;               /* Number of components */
  int         i;

  PetscFunctionBegin;
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element
     which must be a constant for linear elements - 1/|x_{21} y_{31} - x_{31} y_{21}| */
  x21 = coords[2] - coords[0];
  x31 = coords[4] - coords[0];
  y21 = coords[3] - coords[1];
  y31 = coords[5] - coords[1];
  jac = PetscAbsReal(x21*y31 - x31*y21);
#ifdef PETSC_USE_BOPT_g
  if (jac < 1.0e-14) {
    PetscPrintf(PETSC_COMM_SELF, "x21: %g y21: %g x31: %g y31: %g jac: %g\n", x21, y21, x31, y31, jac);
    SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  }
#endif

  comp = rowSize/disc->funcs;
  w    = 1.0/(2.0*jac);
  w   *= alpha;
  for(i = 0; i < comp; i++) {
    /* \phi^1 \phi^1 */
    array[(0*comp+i+globalRowStart)*globalSize + 0*comp+i+globalColStart] =
			(-x21*x21 + 2.0*x21*x31 - x31*x31 - (y21 - y31)*(y21 - y31))*w;
    /* \phi^1 \phi^2 */
    array[(0*comp+i+globalRowStart)*globalSize + 1*comp+i+globalColStart] = (-x21*x31 + x31*x31 + y31*(y31 - y21))*w;
    /* \phi^1 \phi^3 */
    array[(0*comp+i+globalRowStart)*globalSize + 2*comp+i+globalColStart] = (x21*x21 - x21*x31 + y21*(y21 - y31))*w;
    /* \phi^2 \phi^1 */
    array[(1*comp+i+globalRowStart)*globalSize + 0*comp+i+globalColStart] =
			array[(0*comp+i+globalRowStart)*globalSize + 1*comp+i+globalColStart];
    /* \phi^2 \phi^2 */
    array[(1*comp+i+globalRowStart)*globalSize + 1*comp+i+globalColStart] = (-x31*x31 - y31*y31)*w;
    /* \phi^2 \phi^3 */
    array[(1*comp+i+globalRowStart)*globalSize + 2*comp+i+globalColStart] = (x21*x31 + y21*y31)*w;
    /* \phi^3 \phi^1 */
    array[(2*comp+i+globalRowStart)*globalSize + 0*comp+i+globalColStart] =
			array[(0*comp+i+globalRowStart)*globalSize + 2*comp+i+globalColStart];
    /* \phi^3 \phi^2 */
    array[(2*comp+i+globalRowStart)*globalSize + 1*comp+i+globalColStart] =
			array[(1*comp+i+globalRowStart)*globalSize + 2*comp+i+globalColStart];
    /* \phi^3 \phi^3 */
    array[(2*comp+i+globalRowStart)*globalSize + 2*comp+i+globalColStart] = (-x21*x21 - y21*y21)*w;
  }
  PetscLogFlops(47);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Weighted_Laplacian_Triangular_2D_Linear"
int Weighted_Laplacian_Triangular_2D_Linear(Discretization disc, Discretization test, int rowSize, int colSize,
                                            int globalRowStart, int globalColStart, int globalSize, double *coords,
                                            PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
  double               x21, x31, y21, y31; /* Coordinates of the element, with point 1 at the origin */
#ifdef PETSC_USE_BOPT_g
  double               jac;                /* |J| for map to standard element */
#endif
  PetscScalar          w;                  /* 1/2 */
  int                  comp;               /* Number of components */
  int                  i;

  /* Each element is weighted by its Jacobian. This is supposed to make smaller elements "stiffer". */
  PetscFunctionBegin;
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element
     which must be a constant for linear elements - 1/|x_{21} y_{31} - x_{31} y_{21}| */
  x21 = coords[2] - coords[0];
  x31 = coords[4] - coords[0];
  y21 = coords[3] - coords[1];
  y31 = coords[5] - coords[1];
#ifdef PETSC_USE_BOPT_g
  jac = PetscAbsReal(x21*y31 - x31*y21);
  if (jac < 1.0e-14) {
    PetscPrintf(PETSC_COMM_SELF, "x21: %g y21: %g x31: %g y31: %g jac: %g\n", x21, y21, x31, y31, jac);
    SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  }
#endif

  comp = rowSize/3;
  w  = 1.0/(2.0);
  w *= alpha;
  for(i = 0; i < comp; i++)
  {
    /* \phi^1 \phi^1 */
    array[(0*comp+i+globalRowStart)*globalSize + 0*comp+i+globalColStart] =
			(-x21*x21 + 2.0*x21*x31 - x31*x31 - (y21 - y31)*(y21 - y31))*w;
    /* \phi^1 \phi^2 */
    array[(0*comp+i+globalRowStart)*globalSize + 1*comp+i+globalColStart] = (-x21*x31 + x31*x31 + y31*(y31 - y21))*w;
    /* \phi^1 \phi^3 */
    array[(0*comp+i+globalRowStart)*globalSize + 2*comp+i+globalColStart] = (x21*x21 - x21*x31 + y21*(y21 - y31))*w;
    /* \phi^2 \phi^1 */
    array[(1*comp+i+globalRowStart)*globalSize + 0*comp+i+globalColStart] =
			array[(0*comp+i+globalRowStart)*globalSize + 1*comp+i+globalColStart];
    /* \phi^2 \phi^2 */
    array[(1*comp+i+globalRowStart)*globalSize + 1*comp+i+globalColStart] = (-x31*x31 - y31*y31)*w;
    /* \phi^2 \phi^3 */
    array[(1*comp+i+globalRowStart)*globalSize + 2*comp+i+globalColStart] = (x21*x31 + y21*y31)*w;
    /* \phi^3 \phi^1 */
    array[(2*comp+i+globalRowStart)*globalSize + 0*comp+i+globalColStart] =
			array[(0*comp+i+globalRowStart)*globalSize + 2*comp+i+globalColStart];
    /* \phi^3 \phi^2 */
    array[(2*comp+i+globalRowStart)*globalSize + 1*comp+i+globalColStart] =
			array[(1*comp+i+globalRowStart)*globalSize + 2*comp+i+globalColStart];
    /* \phi^3 \phi^3 */
    array[(2*comp+i+globalRowStart)*globalSize + 2*comp+i+globalColStart] = (-x21*x21 - y21*y21)*w;
  }
  PetscLogFlops(47);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Gradient_Triangular_2D_Linear"
int Gradient_Triangular_2D_Linear(Discretization disc, Discretization test, int rowSize, int colSize,
                                  int globalRowStart, int globalColStart, int globalSize, double *coords,
                                  PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
  /* We are using the convention that

       \nabla \matrix{v_1 \cr v_2 \cr \vdots \cr v_n} =
         \matrix{v^{(1)}_1 \cr \vdots \cr v^{(d)}_1 \cr v^{(1)}_2 \cr \vdots \cr v^{(d)}_n}

     and

       \nabla \cdot \matrix{v^{(1)}_1 \cr \vdots \cr v^{(d)}_1 \cr v^{(1)}_2 \cr \vdots \cr v^{(d)}_n} =
         \matrix{v_1 \cr v_2 \cr \vdots \cr v_n}

     where $d$ is the number of space dimensions. This agrees with the convention which allows
     $\Delta \matrix{u_1 \cr u_2} = 0$ to denote a set of scalar equations. This also means that
     the dimension of the test function vector must be divisible by the number of space dimensions */
  int     numQuadPoints;     /* Number of points used for Gaussian quadrature */
  double *quadWeights;       /* Weights in the standard element for Gaussian quadrature */
  double *quadShapeFuncs;    /* Shape functions evaluated at quadrature points */
  double *quadTestFuncDers;  /* Test function derivatives evaluated at quadrature points */
  double  dxxi;              /* \PartDer{x}{\xi}  */
  double  dxet;              /* \PartDer{x}{\eta} */
  double  dyxi;              /* \PartDer{y}{\xi}  */
  double  dyet;              /* \PartDer{y}{\eta} */
  double  dxix;              /* \PartDer{\xi}{x}  */
  double  detx;              /* \PartDer{\eta}{x} */
  double  dxiy;              /* \PartDer{\xi}{y}  */
  double  dety;              /* \PartDer{\eta}{y} */
  double  dphix;             /* \PartDer{\phi_i}{x} \times \PartDer{\phi_j}{x} */
  double  dphiy;             /* \PartDer{\phi_i}{y} \times \PartDer{\phi_j}{y} */
  double  jac;               /* |J| for map to standard element */
  double  invjac;            /* |J^{-1}| for map from standard element */
  int     dim;               /* The problem dimension */
  int     comp;              /* The number of field components */
  int     tcomp;             /* The number of field components for the test field */
  int     funcs;             /* The number of shape functions */
  int     tfuncs;            /* The number of test functions */
  int     i, j, c, tc, f, p;

  PetscFunctionBegin;
  /* Calculate element matrix entries by Gaussian quadrature --
     Since we integrate by parts here, the test and shape functions are switched */
  dim              = disc->dim;
  comp             = disc->comp;
  tcomp            = test->comp;
  funcs            = disc->funcs;
  tfuncs           = test->funcs;
  numQuadPoints    = disc->numQuadPoints;
  quadWeights      = disc->quadWeights;
  quadShapeFuncs   = disc->quadShapeFuncs;
  quadTestFuncDers = test->quadShapeFuncDers;
  for(p = 0; p < numQuadPoints; p++) {
    /* \PartDer{x}{\xi}(p)  = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{x}{\eta}(p) = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\eta}
       \PartDer{y}{\xi}(p)  = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{y}{\eta}(p) = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\eta} */
    dxxi = 0.0; dxet = 0.0;
    dyxi = 0.0; dyet = 0.0;
    for(f = 0; f < tfuncs; f++) {
      dxxi += coords[f*dim]  *quadTestFuncDers[p*tfuncs*dim+f*dim];
      dxet += coords[f*dim]  *quadTestFuncDers[p*tfuncs*dim+f*dim+1];
      dyxi += coords[f*dim+1]*quadTestFuncDers[p*tfuncs*dim+f*dim];
      dyet += coords[f*dim+1]*quadTestFuncDers[p*tfuncs*dim+f*dim+1];
    }
    jac  = PetscAbsReal(dxxi*dyet - dxet*dyxi);
#ifdef PETSC_USE_BOPT_g
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "p: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                  p, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
#endif
    /* These are the elements of the inverse matrix */
    invjac =  1.0/jac;
    dxix   =  dyet*invjac;
    dxiy   = -dxet*invjac;
    detx   = -dyxi*invjac;
    dety   =  dxxi*invjac;

    /* The rows are test functions */
    for(i = 0; i < tfuncs; i++) {
      /* We divide by the space dimension */
      for(tc = 0; tc < tcomp/dim; tc++) {
        /* The columns are shape functions */
        for(j = 0; j < funcs; j++) {
          for(c = 0; c < comp; c++) {
            dphix = quadTestFuncDers[p*tfuncs*dim+i*dim]*dxix + quadTestFuncDers[p*tfuncs*dim+i*dim+1]*detx;
            dphiy = quadTestFuncDers[p*tfuncs*dim+i*dim]*dxiy + quadTestFuncDers[p*tfuncs*dim+i*dim+1]*dety;
            array[(i*tcomp+tc*dim+globalRowStart)*globalSize + j*comp+c+globalColStart] +=
              -alpha*dphix*quadShapeFuncs[p*funcs+j]*jac*quadWeights[p];
            array[(i*tcomp+tc*dim+1+globalRowStart)*globalSize + j*comp+c+globalColStart] +=
              -alpha*dphiy*quadShapeFuncs[p*funcs+j]*jac*quadWeights[p];
          }
        }
      }
    }
  }
  PetscLogFlops((8*tfuncs + 8 + 8*tfuncs*tcomp*funcs*comp) * numQuadPoints);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscInterpolateField_Triangular_2D_Linear"
int DiscInterpolateField_Triangular_2D_Linear(Discretization disc, Mesh oldMesh, int elem, double x, double y, double z,
                                              PetscScalar *oldFieldVal, PetscScalar *newFieldVal, InterpolationType type)
{
  Mesh_Triangular *tri        = (Mesh_Triangular *) oldMesh->data;
  Partition        p          = oldMesh->part;
  int              dim        = disc->dim;
  int              numCorners = oldMesh->numCorners;
  int             *elements   = tri->faces;
  int             *neighbors  = tri->neighbors;
  double          *nodes      = tri->nodes;
  double           coords[12];  /* Coordinates of our "big element" */
  double           x11, y11;    /* Coordinates of vertex 0 */
  double           xi, eta;     /* Canonical coordinates of the interpolation point */
  double           dxix;        /* \PartDer{\xi}{x}  */
  double           detx;        /* \PartDer{\eta}{x} */
  double           dxiy;        /* \PartDer{\xi}{y}  */
  double           dety;        /* \PartDer{\eta}{y} */
  double           dxxi;        /* \PartDer{x}{\xi}  */
  double           dxet;        /* \PartDer{x}{\eta} */
  double           dyxi;        /* \PartDer{y}{\xi}  */
  double           dyet;        /* \PartDer{y}{\eta} */
  double           jac, invjac; /* The Jacobian determinant and its inverse */
  int              comp = disc->comp;
  int              neighbor, corner, nelem, node, c;
#ifdef PETSC_USE_BOPT_g
  PetscTruth       opt;
#endif
  int              ierr;

  PetscFunctionBegin;
  /* No scheme in place for boundary elements */
  for(neighbor = 0; neighbor < 3; neighbor++)
    if (neighbors[elem*3+neighbor] < 0)
      type = INTERPOLATION_LOCAL;

  switch (type)
  {
  case INTERPOLATION_LOCAL:
    x11    = nodes[elements[elem*numCorners]*dim];
    y11    = nodes[elements[elem*numCorners]*dim+1];
    if (oldMesh->isPeriodic == PETSC_TRUE) {
      dxxi = MeshPeriodicDiffX(oldMesh, nodes[elements[elem*numCorners+1]*dim]   - x11);
      dxet = MeshPeriodicDiffX(oldMesh, nodes[elements[elem*numCorners+2]*dim]   - x11);
      dyxi = MeshPeriodicDiffY(oldMesh, nodes[elements[elem*numCorners+1]*dim+1] - y11);
      dyet = MeshPeriodicDiffY(oldMesh, nodes[elements[elem*numCorners+2]*dim+1] - y11);
    } else {
      dxxi = nodes[elements[elem*numCorners+1]*dim]   - x11;
      dxet = nodes[elements[elem*numCorners+2]*dim]   - x11;
      dyxi = nodes[elements[elem*numCorners+1]*dim+1] - y11;
      dyet = nodes[elements[elem*numCorners+2]*dim+1] - y11;
    }
    jac  = PetscAbsReal(dxxi*dyet - dxet*dyxi);
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g y21: %g x31: %g y31: %g\n", p->rank, elem, dxxi, dyxi, dxet, dyet);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
#ifdef PETSC_USE_BOPT_g
    ierr = PetscOptionsHasName(PETSC_NULL, "-trace_interpolation", &opt);                                 CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g y21: %g x31: %g y31: %g jac: %g\n",
                  p->rank, elem, dxxi, dyxi, dxet, dyet, jac);
    }
#endif

    /* These are the elements of the inverse matrix */
    invjac = 1/jac;
    dxix   =  dyet*invjac;
    dxiy   = -dxet*invjac;
    detx   = -dyxi*invjac;
    dety   =  dxxi*invjac;
    if (oldMesh->isPeriodic == PETSC_TRUE) {
      xi     = dxix*MeshPeriodicDiffX(oldMesh, x - x11) + dxiy*MeshPeriodicDiffY(oldMesh, y - y11);
      eta    = detx*MeshPeriodicDiffX(oldMesh, x - x11) + dety*MeshPeriodicDiffY(oldMesh, y - y11);
    } else {
      xi     = dxix*(x - x11) + dxiy*(y - y11);
      eta    = detx*(x - x11) + dety*(y - y11);
    }
    for(c = 0 ; c < comp; c++) {
      newFieldVal[c] = oldFieldVal[0*comp+c]*(1.0 - xi - eta) + oldFieldVal[1*comp+c]*xi + oldFieldVal[2*comp+c]*eta;
    }
    PetscLogFlops(7+15+7*comp);
    break;
  case INTERPOLATION_HALO:
    /* Here is our "big element" where numbers in parantheses represent
       the numbering on the old little element:

           2
           |\
           | \
           |  \
       (1) 4---3 (0)
           |\  |\
           | \ | \
           |  \|  \
           0---5---1
              (2)

       We search for the neighbor node by looking for the vertex not a member of the original element.
    */
    for(neighbor = 0; neighbor < 3; neighbor++)
    {
      nelem = neighbors[elem*3+neighbor];
      for(corner = 0; corner < 3; corner++)
      {
        node = elements[nelem*numCorners+corner];
        if ((node != elements[elem*numCorners+((neighbor+1)%3)]) && (node != elements[elem*numCorners+((neighbor+2)%3)]))
        {
          /* The neighboring elements give the vertices */
          coords[neighbor*2]   = MeshPeriodicRelativeX(oldMesh, nodes[node*2+0], x);
          coords[neighbor*2+1] = MeshPeriodicRelativeY(oldMesh, nodes[node*2+1], y);
          break;
        }
      }
    }
    /* Element vertices form midnodes */
    coords[3*2]   = MeshPeriodicRelativeX(oldMesh, nodes[elements[elem*numCorners+0]*2+0], x);
    coords[3*2+1] = MeshPeriodicRelativeY(oldMesh, nodes[elements[elem*numCorners+0]*2+1], y);
    coords[4*2]   = MeshPeriodicRelativeX(oldMesh, nodes[elements[elem*numCorners+1]*2+0], x);
    coords[4*2+1] = MeshPeriodicRelativeY(oldMesh, nodes[elements[elem*numCorners+1]*2+1], y);
    coords[5*2]   = MeshPeriodicRelativeX(oldMesh, nodes[elements[elem*numCorners+2]*2+0], x);
    coords[5*2+1] = MeshPeriodicRelativeY(oldMesh, nodes[elements[elem*numCorners+2]*2+1], y);
    /* Get the (\xi,\eta) coordinates of the point */
    ierr = DiscTransformCoords_Triangular_2D_Quadratic(x, y, coords, &xi, &eta);                          CHKERRQ(ierr);
    if ((xi < -1.0e-02) || (eta < -1.0e-02)) {
      PetscPrintf(PETSC_COMM_SELF, "Linear: elem: %d x: %g y: %g xi: %g eta: %g\n", elem, x, y, xi, eta);
      SETERRQ(PETSC_ERR_PLIB, "Standard element coordinates were negative");
    }
    for(c = 0 ; c < comp; c++) {
      newFieldVal[c] = oldFieldVal[0*comp+c]*(1.0 - xi - eta)*(1.0 - 2.0*xi - 2.0*eta) +
        oldFieldVal[1*comp+c]*xi *(2.0*xi  - 1.0)      +
        oldFieldVal[2*comp+c]*eta*(2.0*eta - 1.0)      +
        oldFieldVal[3*comp+c]*4.0*xi*eta               +
        oldFieldVal[4*comp+c]*4.0*eta*(1.0 - xi - eta) +
        oldFieldVal[5*comp+c]*4.0*xi *(1.0 - xi - eta);
    }
    PetscLogFlops(34*comp);
    break;
  default:
    SETERRQ1(PETSC_ERR_ARG_WRONG, "Unknown interpolation type %d", type);
  }
  
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscInterpolateElementVec_Triangular_2D_Linear"
int DiscInterpolateElementVec_Triangular_2D_Linear(Discretization disc, ElementVec vec, Discretization newDisc, ElementVec newVec)
{
  int          funcs = disc->funcs;
  int          comp  = disc->comp;
  int          size  = disc->size;
  PetscScalar *array, *newArray;
  PetscTruth   islin, isquad;
  int          f, c;
  int          ierr;

  PetscFunctionBegin;
  ierr = ElementVecGetArray(vec,    &array);                                                              CHKERRQ(ierr);
  ierr = ElementVecGetArray(newVec, &newArray);                                                           CHKERRQ(ierr);
  ierr = PetscTypeCompare((PetscObject) newDisc, DISCRETIZATION_TRIANGULAR_2D_LINEAR,    &islin);         CHKERRQ(ierr);
  ierr = PetscTypeCompare((PetscObject) newDisc, DISCRETIZATION_TRIANGULAR_2D_QUADRATIC, &isquad);        CHKERRQ(ierr);
  if (islin == PETSC_TRUE) {
    ierr = PetscMemcpy(newArray, array, size * sizeof(PetscScalar));                                      CHKERRQ(ierr);
  } else if (isquad == PETSC_TRUE) {
    for(f = 0; f < newDisc->funcs; f++) {
      for(c = 0; c < comp; c++) {
        if (f < funcs) {
          newArray[f*comp+c] = array[f*comp+c];
        } else {
          newArray[f*comp+c] = 0.5*(array[((f+1)%funcs)*comp+c] + array[((f+2)%funcs)*comp+c]);
        }
      }
    }
  } else {
    SETERRQ(PETSC_ERR_SUP, "Discretization not supported");
  }
  ierr = ElementVecRestoreArray(vec,    &array);                                                          CHKERRQ(ierr);
  ierr = ElementVecRestoreArray(newVec, &newArray);                                                       CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscSetupQuadrature_Triangular_2D_Linear"
/*
  DiscSetupQuadrature_Triangular_2D_Linear - Setup Gaussian quadrature with a 7 point integration rule

  Input Parameter:
. disc - The Discretization
*/
int DiscSetupQuadrature_Triangular_2D_Linear(Discretization disc) {
  int dim   = disc->dim;
  int funcs = disc->funcs;
  int p;
  int ierr;

  PetscFunctionBegin;
  disc->numQuadPoints = 7;
  ierr = PetscMalloc(disc->numQuadPoints*dim       * sizeof(double), &disc->quadPoints);                  CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints           * sizeof(double), &disc->quadWeights);                 CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints*funcs     * sizeof(double), &disc->quadShapeFuncs);              CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints*funcs*dim * sizeof(double), &disc->quadShapeFuncDers);           CHKERRQ(ierr);
  PetscLogObjectMemory(disc, (disc->numQuadPoints*(funcs*(dim+1) + dim+1)) * sizeof(double));
  disc->quadPoints[0]  = 1.0/3.0;
  disc->quadPoints[1]  = disc->quadPoints[0];
  disc->quadWeights[0] = 0.11250000000000;
  disc->quadPoints[2]  = 0.797426985353087;
  disc->quadPoints[3]  = 0.101286507323456;
  disc->quadWeights[1] = 0.0629695902724135;
  disc->quadPoints[4]  = disc->quadPoints[3];
  disc->quadPoints[5]  = disc->quadPoints[2];
  disc->quadWeights[2] = disc->quadWeights[1];
  disc->quadPoints[6]  = disc->quadPoints[4];
  disc->quadPoints[7]  = disc->quadPoints[3];
  disc->quadWeights[3] = disc->quadWeights[1];
  disc->quadPoints[8]  = 0.470142064105115;
  disc->quadPoints[9]  = 0.059715871789770;
  disc->quadWeights[4] = 0.066197076394253;
  disc->quadPoints[10] = disc->quadPoints[8];
  disc->quadPoints[11] = disc->quadPoints[8];
  disc->quadWeights[5] = disc->quadWeights[4];
  disc->quadPoints[12] = disc->quadPoints[9];
  disc->quadPoints[13] = disc->quadPoints[8];
  disc->quadWeights[6] = disc->quadWeights[4];
  for(p = 0; p < disc->numQuadPoints; p++) {
    /* \phi^0: 1 - \xi - \eta */
    disc->quadShapeFuncs[p*funcs]                =  1.0 - disc->quadPoints[p*dim] - disc->quadPoints[p*dim+1];
    disc->quadShapeFuncDers[p*funcs*dim+0*dim]   = -1.0;
    disc->quadShapeFuncDers[p*funcs*dim+0*dim+1] = -1.0;
    /* \phi^1: \xi */
    disc->quadShapeFuncs[p*funcs+1]              =  disc->quadPoints[p*dim];
    disc->quadShapeFuncDers[p*funcs*dim+1*dim]   =  1.0;
    disc->quadShapeFuncDers[p*funcs*dim+1*dim+1] =  0.0;
    /* \phi^2: \eta */
    disc->quadShapeFuncs[p*funcs+2]              =  disc->quadPoints[p*dim+1];
    disc->quadShapeFuncDers[p*funcs*dim+2*dim]   =  0.0;
    disc->quadShapeFuncDers[p*funcs*dim+2*dim+1] =  1.0;
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscSetupOperators_Triangular_2D_Linear"
/*
  DiscSetupOperators_Triangular_2D_Linear - Setup the default operators

  Input Parameter:
. disc - The Discretization
*/
int DiscSetupOperators_Triangular_2D_Linear(Discretization disc) {
  int          comp = disc->comp;
  int          size = disc->size;
  PetscScalar *precompInt;
  int          newOp;
  int          c, i, j;
  int          ierr;

  PetscFunctionBegin;
  /* The Identity operator I -- the matrix is symmetric */
  ierr = PetscMalloc(size*size * sizeof(PetscScalar), &precompInt);                                       CHKERRQ(ierr);
  PetscLogObjectMemory(disc, size*size * sizeof(PetscScalar));
  ierr = PetscMemzero(precompInt, size*size * sizeof(PetscScalar));                                       CHKERRQ(ierr);
  for(c = 0; c < comp; c++) {
    precompInt[(0*comp+c)*size + 0*comp+c] = 1.0/12.0;
    precompInt[(0*comp+c)*size + 1*comp+c] = 1.0/24.0;
    precompInt[(0*comp+c)*size + 2*comp+c] = 1.0/24.0;
    precompInt[(1*comp+c)*size + 1*comp+c] = 1.0/12.0;
    precompInt[(1*comp+c)*size + 2*comp+c] = 1.0/24.0;
    precompInt[(2*comp+c)*size + 2*comp+c] = 1.0/12.0;
  }
  for(i = 0; i < size; i++) {
    for(j = 0; j < i; j++) {
      precompInt[i*size + j] = precompInt[j*size + i];
    }
  }
  ierr = DiscretizationRegisterPrecomputedOperator(disc, precompInt, &newOp);                             CHKERRQ(ierr);
  if (newOp != IDENTITY) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", IDENTITY);
  /* The Laplacian operator \Delta -- the matrix is symmetric */
  ierr = DiscretizationRegisterOperator(disc, Laplacian_Triangular_2D_Linear, &newOp);                    CHKERRQ(ierr);
  if (newOp != LAPLACIAN) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", LAPLACIAN);
  /* The Gradient operator \nabla -- the matrix is rectangular */
  ierr = DiscretizationRegisterOperator(disc, Gradient_Triangular_2D_Linear, &newOp);                     CHKERRQ(ierr);
  if (newOp != GRADIENT) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", GRADIENT);
  /* The Divergence operator \nabla\cdot -- the matrix is rectangular */
  ierr = DiscretizationRegisterOperator(disc, PETSC_NULL, &newOp);                                        CHKERRQ(ierr);
  if (newOp != DIVERGENCE) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", DIVERGENCE);
  /* The weighted Laplacian operator -- the matrix is symmetric */
  ierr = DiscretizationRegisterOperator(disc, Weighted_Laplacian_Triangular_2D_Linear, &newOp);           CHKERRQ(ierr);
  if (newOp != WEIGHTED_LAP) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", WEIGHTED_LAP);
  PetscFunctionReturn(0);
}

static struct _DiscretizationOps DOps = {PETSC_NULL/* DiscretizationSetup */,
                                         DiscSetupOperators_Triangular_2D_Linear,
                                         PETSC_NULL/* DiscretizationSetFromOptions */,
                                         DiscView_Triangular_2D_Linear,
                                         DiscDestroy_Triangular_2D_Linear,
                                         DiscEvaluateFunctionGalerkin_Triangular_2D_Linear,
                                         DiscEvaluateOperatorGalerkin_Triangular_2D_Linear,
                                         DiscEvaluateALEOperatorGalerkin_Triangular_2D_Linear,
                                         DiscEvaluateNonlinearOperatorGalerkin_Triangular_2D_Linear,
                                         DiscEvaluateNonlinearALEOperatorGalerkin_Triangular_2D_Linear,
                                         DiscInterpolateField_Triangular_2D_Linear,
                                         DiscInterpolateElementVec_Triangular_2D_Linear};

EXTERN_C_BEGIN
#undef  __FUNCT__
#define __FUNCT__ "DiscCreate_Triangular_2D_Linear"
int DiscCreate_Triangular_2D_Linear(Discretization disc) {
  int arg;
  int ierr;

  PetscFunctionBegin;
  if (disc->comp <= 0) {
    SETERRQ(PETSC_ERR_ARG_WRONG, "Discretization must have at least 1 component. Call DiscretizationSetNumComponents() to set this.");
  }
  ierr = PetscMemcpy(disc->ops, &DOps, sizeof(struct _DiscretizationOps));                                CHKERRQ(ierr);
  disc->dim   = 2;
  disc->funcs = 3;
  disc->size  = disc->funcs*disc->comp;

  ierr = DiscretizationSetupDefaultOperators(disc);                                                       CHKERRQ(ierr);
  ierr = DiscSetupQuadrature_Triangular_2D_Linear(disc);                                                  CHKERRQ(ierr);

  ierr = DiscretizationCreate(disc->comm, &disc->bdDisc);                                                 CHKERRQ(ierr);
  ierr = DiscretizationSetNumComponents(disc->bdDisc, disc->comp);                                        CHKERRQ(ierr);
  ierr = DiscretizationSetType(disc->bdDisc, BD_DISCRETIZATION_TRIANGULAR_2D_LINEAR);                     CHKERRQ(ierr);

  /* Storage */
  ierr = PetscMalloc(disc->comp * sizeof(PetscScalar),   &disc->funcVal);                                 CHKERRQ(ierr);
  ierr = PetscMalloc(2          * sizeof(PetscScalar *), &disc->fieldVal);                                CHKERRQ(ierr);
  for(arg = 0; arg < 2; arg++) {
    ierr = PetscMalloc(disc->comp*(disc->dim+1) * sizeof(PetscScalar), &disc->fieldVal[arg]);             CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}
EXTERN_C_END
