#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: quadratic.c,v 1.7 2000/01/10 03:54:16 knepley Exp $";
#endif

/*
   Defines piecewise quadratic function space on a two dimensional 
   grid. Suitable for finite element type discretization of a PDE.
*/

#include "src/grid/discretization/discimpl.h"         /*I "discretization.h" I*/
#include "src/mesh/impls/triangular/triimpl.h"

/* For precomputed integrals, the table is structured as follows:

     precompInt[op,i,j] = \int_{SE} <op \phi^i(\xi,\eta), \phi^j(\xi,\eta)> |J^{-1}|

   The Jacobian in this case may not be constant over the element in question.
   The numbering of the nodes in the standard element is

                 3
                 |\
                 | \
                 5  4
                 |   \
                 |    \
                 1--6--2
*/

#undef  __FUNCT__
#define __FUNCT__ "DiscDestroy_Triangular_1D_Quadratic"
static int DiscDestroy_Triangular_1D_Quadratic(Discretization disc) {
  PetscFunctionBegin;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscView_Triangular_1D_Quadratic_File"
static int DiscView_Triangular_1D_Quadratic_File(Discretization disc, PetscViewer viewer) {
  PetscFunctionBegin;
  PetscViewerASCIIPrintf(viewer, "Quadratic discretization\n");
  PetscViewerASCIIPrintf(viewer, "    %d shape functions per component\n", disc->funcs);
  PetscViewerASCIIPrintf(viewer, "    %d registered operators\n", disc->numOps);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscView_Triangular_1D_Quadratic"
static int DiscView_Triangular_1D_Quadratic(Discretization disc, PetscViewer viewer) {
  PetscTruth isascii;
  int        ierr;

  PetscFunctionBegin;
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_ASCII, &isascii);                            CHKERRQ(ierr);
  if (isascii == PETSC_TRUE) {
    ierr = DiscView_Triangular_1D_Quadratic_File(disc, viewer);                                           CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateShapeFunctions_Triangular_1D_Quadratic_Private"
int DiscEvaluateShapeFunctions_Triangular_1D_Quadratic_Private(double xi, double eta, double *coords, double *x,
                                                               double *y, double *dxxi, double *dxet, double *dyxi, double *dyet)
{
  /* ASSUMPTION: The coordinates passed in are corrected for periodicity */
  PetscFunctionBegin;
  *x    = 0.0; *y    = 0.0;
  *dxxi = 0.0; *dxet = 0.0;
  *dyxi = 0.0; *dyet = 0.0;
  /* \phi^0: 1 - 3 (\xi + \eta) + 2 (\xi + \eta)^2 */
  *x    += coords[0*2+0]*(1.0 - (xi + eta))*(1.0 - 2.0*(xi + eta));
  *dxxi += coords[0*2+0]*(-3.0 + 4.0*(xi + eta));
  *dxet += coords[0*2+0]*(-3.0 + 4.0*(xi + eta));
  *y    += coords[0*2+1]*(1.0 - (xi + eta))*(1.0 - 2.0*(xi + eta));
  *dyxi += coords[0*2+1]*(-3.0 + 4.0*(xi + eta));
  *dyet += coords[0*2+1]*(-3.0 + 4.0*(xi + eta));
  /* \phi^1: \xi (2\xi - 1) */
  *x    += coords[1*2+0]*(xi*(2.0*xi - 1.0));
  *dxxi += coords[1*2+0]*(4.0*xi  - 1.0);
  *dxet += 0.0;
  *y    += coords[1*2+1]*(xi*(2.0*xi - 1.0));
  *dyxi += coords[1*2+1]*(4.0*xi  - 1.0);
  *dyet += 0.0;
  /* \phi^2: \eta (2\eta - 1) */
  *x    += coords[2*2+0]*(eta*(2.0*eta - 1.0));
  *dxxi += 0.0;
  *dxet += coords[2*2+0]*(4.0*eta - 1.0);
  *y    += coords[2*2+1]*(eta*(2.0*eta - 1.0));
  *dyxi += 0.0;
  *dyet += coords[2*2+1]*(4.0*eta - 1.0);
  /* \phi^3: 4 \xi \eta */
  *x    += coords[3*2+0]*(4.0*xi*eta);
  *dxxi += coords[3*2+0]*(4.0*eta);
  *dxet += coords[3*2+0]*(4.0*xi);
  *y    += coords[3*2+1]*(4.0*xi*eta);
  *dyxi += coords[3*2+1]*(4.0*eta);
  *dyet += coords[3*2+1]*(4.0*xi);
  /* \phi^4: 4 \eta (1 - \xi - \eta) */
  *x    += coords[4*2+0]*(4.0*eta*(1.0 - (xi + eta)));
  *dxxi += coords[4*2+0]*(-4.0*eta);
  *dxet += coords[4*2+0]*(-8.0*eta + 4.0*(1.0 - xi));
  *y    += coords[4*2+1]*(4.0*eta*(1.0 - (xi + eta)));
  *dyxi += coords[4*2+1]*(-4.0*eta);
  *dyet += coords[4*2+1]*(-8.0*eta + 4.0*(1.0 - xi));
  /* \phi^5: 4 \xi (1 - \xi - \eta) */
  *x    += coords[5*2+0]*(4.0*xi*(1.0 - (xi + eta)));
  *dxxi += coords[5*2+0]*(-8.0*xi  + 4.0*(1.0 - eta));
  *dxet += coords[5*2+0]*(-4.0*xi);
  *y    += coords[5*2+1]*(4.0*xi*(1.0 - (xi + eta)));
  *dyxi += coords[5*2+1]*(-8.0*xi  + 4.0*(1.0 - eta));
  *dyet += coords[5*2+1]*(-4.0*xi);
  PetscLogFlops(36+20+20+20+30+20);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscTransformCoords_Triangular_1D_Quadratic"
int DiscTransformCoords_Triangular_1D_Quadratic(double x, double y, double *coords, double *newXi, double *newEta)
{
  /* ASSUMPTION: The coordinates passed in are corrected for periodicity */
  double xi, eta;     /* Canonical coordinates of the point */
  double dxix;        /* \PartDer{\xi}{x}  */
  double detx;        /* \PartDer{\eta}{x} */
  double dxiy;        /* \PartDer{\xi}{y}  */
  double dety;        /* \PartDer{\eta}{y} */
  double dxxi;        /* \PartDer{x}{\xi}  */
  double dxet;        /* \PartDer{x}{\eta} */
  double dyxi;        /* \PartDer{y}{\xi}  */
  double dyet;        /* \PartDer{y}{\eta} */
  double new_x;       /* x(\xi,\eta) */
  double new_y;       /* x(\xi,\eta) */
  double residual_x;  /* x(\xi,\eta) - x */
  double residual_y;  /* x(\xi,\eta) - y */
  double jac, invjac; /* The Jacobian determinant and its inverse */
  int    maxIters = 100;
  int    iter;
  int    ierr;

  PetscFunctionBegin;
  /* We have to solve

       \sum_f x(f) \phi^f(\xi,\eta) = x
       \sum_f y(f) \phi^f(\xi,\eta) = y

     where f runs over nodes (each one has coordinates and a shape function). We
     will use Newton's method

     / \sum_f x(f) \PartDer{\phi^f}{\xi} \sum_f x(f) \PartDer{\phi^f}{\eta} \ / d\xi  \ = / \sum_f x(f) \phi^f(\xi,\eta) - x\
     \ \sum_f y(f) \PartDer{\phi^f}{\xi} \sum_f y(f) \PartDer{\phi^f}{\eta} / \ d\eta /   \ \sum_f y(f) \phi^f(\xi,\eta) - y/

     which can be rewritten more compactly as

     / \PartDer{x}{\xi} \PartDer{x}{\eta} \  / d\xi  \ = / x(\xi,\eta) - x \
     \ \PartDer{y}{\xi} \PartDer{y}{\eta} /  \ d\eta /   \ y(\xi,\eta) - y /

     The initial guess will be the linear solution.

     ASSUMPTION: The coordinates passed in are all on the same sheet as x,y
  */
  /* Form linear solution */
  dxxi = coords[1*2+0] - coords[0*2+0];
  dxet = coords[2*2+0] - coords[0*2+0];
  dyxi = coords[1*2+1] - coords[0*2+1];
  dyet = coords[2*2+1] - coords[0*2+1];
  jac  = PetscAbsReal(dxxi*dyet - dxet*dyxi);
  if (jac < 1.0e-14) SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  invjac = 1/jac;
  dxix   =  dyet*invjac;
  dxiy   = -dxet*invjac;
  detx   = -dyxi*invjac;
  dety   =  dxxi*invjac;
  xi     = dxix*(x - coords[0*2+0]) + dxiy*(y - coords[0*2+1]);
  eta    = detx*(x - coords[0*2+0]) + dety*(y - coords[0*2+1]);
  for(iter = 0; iter < maxIters; iter++) {
    /* This is clumsy, but I can't think of anything better right now */
    ierr = DiscEvaluateShapeFunctions_Triangular_1D_Quadratic_Private(xi, eta, coords, &new_x, &new_y, &dxxi, &dxet, &dyxi, &dyet);
    CHKERRQ(ierr);

    /* Check for convergence -- I should maybe make the tolerance variable */
    residual_x = new_x - x;
    residual_y = new_y - y;
    if (PetscAbsReal(residual_x) + PetscAbsReal(residual_y) < 1.0e-6) break;

    /* Solve the system */
    jac = PetscAbsReal(dxxi*dyet - dxet*dyxi);
    if (jac < 1.0e-14) {
      iter = maxIters;
      break;
    }

    /* These are the elements of the inverse matrix */
    invjac = 1/jac;
    dxix   =  dyet*invjac;
    dxiy   = -dxet*invjac;
    detx   = -dyxi*invjac;
    dety   =  dxxi*invjac;
    xi    -= dxix*residual_x + dxiy*residual_y;
    eta   -= detx*residual_x + dety*residual_y;
  }
  if (iter == maxIters) {
    PetscLogInfo(PETSC_NULL, "DiscTransformCoords_Triangular_1D_Quadratic: Newton iteration did not converge\n");
    PetscLogInfo(PETSC_NULL, "x: %g y: %g maxIters: %d\n", x, y, maxIters);
    for(iter = 0; iter < 6; iter++) {
      PetscLogInfo(PETSC_NULL, "  x%d: %g y%d: %g\n", iter, coords[iter*2+0], iter, coords[iter*2+1]);
    }
    /* Use linear interpolation */
    xi  = dxix*(x - coords[0*2+0]) + dxiy*(y - coords[0*2+1]);
    eta = detx*(x - coords[0*2+0]) + dety*(y - coords[0*2+1]);
  }

  *newXi  = xi;
  *newEta = eta;
  PetscLogFlops(7+15+19*iter);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateFunctionGalerkin_Triangular_1D_Quadratic"
static int DiscEvaluateFunctionGalerkin_Triangular_1D_Quadratic(Discretization disc, Mesh mesh, PointFunction f,
                                                                PetscScalar alpha, int elem, PetscScalar *array, void *ctx)
{
  Mesh_Triangular *tri               = (Mesh_Triangular *) mesh->data;
  double          *nodes             = tri->nodes;
  int             *elements          = tri->faces;
  int              numCorners        = mesh->numCorners;
  int              comp              = disc->comp;              /* The number of components in this field */
  int              funcs             = disc->funcs;             /* The number of shape functions per component */
  PetscScalar     *funcVal           = disc->funcVal;           /* Function value at a quadrature point */
  int              numQuadPoints     = disc->numQuadPoints;     /* Number of points used for Gaussian quadrature */
  double          *quadWeights       = disc->quadWeights;       /* Weights in the standard element for Gaussian quadrature */
  double          *quadShapeFuncs    = disc->quadShapeFuncs;    /* Shape functions evaluated at quadrature points */
  double          *quadShapeFuncDers = disc->quadShapeFuncDers; /* Shape function derivatives at quadrature points */
  double           jac;                                         /* |J| for map to standard element */
  double           x, y;                                        /* The integration point */
  double           dxxi;                                        /* \PartDer{x}{\xi}  */
  double           dxet;                                        /* \PartDer{x}{\xi}  */
  double           dyxi;                                        /* \PartDer{y}{\eta} */
  double           dyet;                                        /* \PartDer{y}{\eta} */
  double           coords[MAX_CORNERS*2];
  int              rank = -1;
  int              i, j, k, func, p;
#ifdef PETSC_USE_BOPT_g
  PetscTruth       opt;
#endif
  int              ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);

  /* For dummy collective calls */
  if (array == PETSC_NULL) {
    for(p = 0; p < numQuadPoints; p++) {
      ierr = (*f)(0, 0, PETSC_NULL, PETSC_NULL, PETSC_NULL, PETSC_NULL, ctx);                             CHKERRQ(ierr);
    }
    PetscFunctionReturn(0);
  }

#ifdef PETSC_USE_BOPT_g
  if ((elem < 0) || (elem >= mesh->part->numOverlapElements)) {
    SETERRQ2(PETSC_ERR_ARG_OUTOFRANGE, "Invalid element %d should be in [0,%d)", elem, mesh->part->numOverlapElements);
  }
#endif
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element */
  for(i = 0; i < numCorners; i++) {
    coords[i*2]   = nodes[elements[elem*numCorners+i]*2];
    coords[i*2+1] = nodes[elements[elem*numCorners+i]*2+1];
  }

  /* Check for constant jacobian here */
  if (PETSC_FALSE) {
    jac = PetscAbsReal((coords[2] - coords[0])*(coords[5] - coords[1]) - (coords[4] - coords[0])*(coords[3] - coords[1]));
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "[%d], elem: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                  rank, elem, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
  }
#ifdef PETSC_USE_BOPT_g
  ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                        CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                rank, elem, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
    PetscPrintf(PETSC_COMM_SELF, "  x4: %g y4: %g x5: %g y5: %g x6: %g y6: %g\n",
                coords[6], coords[7], coords[8], coords[9], coords[10], coords[11]);
  }
#endif

  /* Calculate element vector entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++)
  {
    /* x                    = \sum^{funcs}_{f=1} x_f \phi^f(p)
       \PartDer{x}{\xi}(p)  = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{x}{\eta}(p) = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\eta}
       y                    = \sum^{funcs}_{f=1} y_f \phi^f(p)
       \PartDer{y}{\xi}(p)  = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{y}{\eta}(p) = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\eta} */
    x    = 0.0; y    = 0.0;
    dxxi = 0.0; dxet = 0.0;
    dyxi = 0.0; dyet = 0.0;
    if (mesh->isPeriodic == PETSC_TRUE) {
      for(func = 0; func < funcs; func++)
      {
        x    += MeshPeriodicRelativeX(mesh, coords[func*2],   coords[0])*quadShapeFuncs[p*funcs+func];
        dxxi += MeshPeriodicRelativeX(mesh, coords[func*2],   coords[0])*quadShapeFuncDers[p*funcs*2+func*2];
        dxet += MeshPeriodicRelativeX(mesh, coords[func*2],   coords[0])*quadShapeFuncDers[p*funcs*2+func*2+1];
        y    += MeshPeriodicRelativeY(mesh, coords[func*2+1], coords[1])*quadShapeFuncs[p*funcs+func];
        dyxi += MeshPeriodicRelativeY(mesh, coords[func*2+1], coords[1])*quadShapeFuncDers[p*funcs*2+func*2];
        dyet += MeshPeriodicRelativeY(mesh, coords[func*2+1], coords[1])*quadShapeFuncDers[p*funcs*2+func*2+1];
      }
      x = MeshPeriodicX(mesh, x);
      y = MeshPeriodicY(mesh, y);
    } else {
      for(func = 0; func < funcs; func++)
      {
        x    += coords[func*2]  *quadShapeFuncs[p*funcs+func];
        dxxi += coords[func*2]  *quadShapeFuncDers[p*funcs*2+func*2];
        dxet += coords[func*2]  *quadShapeFuncDers[p*funcs*2+func*2+1];
        y    += coords[func*2+1]*quadShapeFuncs[p*funcs+func];
        dyxi += coords[func*2+1]*quadShapeFuncDers[p*funcs*2+func*2];
        dyet += coords[func*2+1]*quadShapeFuncDers[p*funcs*2+func*2+1];
      }
    }
    jac = PetscAbsReal(dxxi*dyet - dxet*dyxi);
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
		  rank, p, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
    ierr = (*f)(1, comp, &x, &y, PETSC_NULL, funcVal, ctx);                                               CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
    ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                      CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p: %d jac: %g", rank, p, jac);
     for(j = 0; j < comp; j++)
        PetscPrintf(PETSC_COMM_SELF, " func[%d]: %g", j, PetscRealPart(funcVal[j]));
      PetscPrintf(PETSC_COMM_SELF, "\n");
    }
#endif

    for(i = 0, k = 0; i < funcs; i++) {
      for(j = 0; j < comp; j++, k++) {
        array[k] += alpha*funcVal[j]*quadShapeFuncs[p*funcs+i]*jac*quadWeights[p];
#ifdef PETSC_USE_BOPT_g
        ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                  CHKERRQ(ierr);
        if (opt == PETSC_TRUE) {
          PetscPrintf(PETSC_COMM_SELF, "[%d]  array[%d]: %g\n", rank, k, PetscRealPart(array[k]));
        }
#endif
      }
    }
  }
  PetscLogFlops((3 + 12*funcs + 5*funcs*comp) * numQuadPoints);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateOperatorGalerkin_Triangular_1D_Quadratic"
static int DiscEvaluateOperatorGalerkin_Triangular_1D_Quadratic(Discretization disc, Mesh mesh, int elemSize,
                                                                int rowStart, int colStart, int op, PetscScalar alpha,
                                                                int elem, PetscScalar *field, PetscScalar *array, void *ctx)
{
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  double          *nodes      = tri->nodes;          /* The node coordinates */
  int             *elements   = tri->faces;          /* The element corners */
  int              numCorners = mesh->numCorners;    /* The number of corners per element */
  Operator         oper       = disc->operators[op]; /* The operator to discretize */
  Discretization   test       = oper->test;          /* The space of test functions */
  OperatorFunction opFunc     = oper->opFunc;        /* Integrals of operators which depend on J */
  PetscScalar     *precompInt = oper->precompInt;    /* Precomputed integrals of the operator on shape functions */
  int              rowSize    = test->size;          /* The number of shape functions per element */
  int              colSize    = disc->size;          /* The number of test  functions per element */
  double           x21, x31, y21, y31;               /* Coordinates of the element, with point 1 at the origin */
  double           jac;                              /* |J| for map to standard element */
  double           coords[MAX_CORNERS*2];            /* Coordinates of the element */
  int              rank;
  int              i, j, f;
  int              ierr; 

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
  /* Check for valid operator */
  if ((op < 0) || (op >= disc->numOps) || (!disc->operators[op])) SETERRQ(PETSC_ERR_ARG_WRONG, "Invalid operator");
#endif

  if (precompInt != PETSC_NULL)
  {
    /* Calculate the determinant of the inverse Jacobian of the map to the standard element
       which has been specified as constant here - 1/|x_{21} y_{31} - x_{31} y_{21}| */
    if (mesh->isPeriodic == PETSC_TRUE) {
      x21 = MeshPeriodicDiffX(mesh, nodes[elements[elem*numCorners+1]*2]   - nodes[elements[elem*numCorners]*2]);
      x31 = MeshPeriodicDiffX(mesh, nodes[elements[elem*numCorners+2]*2]   - nodes[elements[elem*numCorners]*2]);
      y21 = MeshPeriodicDiffY(mesh, nodes[elements[elem*numCorners+1]*2+1] - nodes[elements[elem*numCorners]*2+1]);
      y31 = MeshPeriodicDiffY(mesh, nodes[elements[elem*numCorners+2]*2+1] - nodes[elements[elem*numCorners]*2+1]);
    } else {
      x21 = nodes[elements[elem*numCorners+1]*2]   - nodes[elements[elem*numCorners]*2];
      x31 = nodes[elements[elem*numCorners+2]*2]   - nodes[elements[elem*numCorners]*2];
      y21 = nodes[elements[elem*numCorners+1]*2+1] - nodes[elements[elem*numCorners]*2+1];
      y31 = nodes[elements[elem*numCorners+2]*2+1] - nodes[elements[elem*numCorners]*2+1];
    }
    jac = PetscAbsReal(x21*y31 - x31*y21);
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]x21: %g y21: %g x31: %g y31: %g jac: %g\n", rank, x21, y21, x31, y31, jac);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
    /* PetscPrintf(PETSC_COMM_SELF, "x21: %g y21: %g x31: %g y31: %g\n", x21, y21, x31, y31, jac); */

    /* Calculate element matrix entries which are all precomputed */
    for(i = 0; i < rowSize; i++)
      for(j = 0; j < colSize; j++)
        array[(i+rowStart)*elemSize + j+colStart] += alpha*precompInt[i*colSize + j]*jac;
    PetscLogFlops(7 + 2*rowSize*colSize);
  }
  else
  {
    if (opFunc == PETSC_NULL) SETERRQ(PETSC_ERR_ARG_CORRUPT, "Invalid function");
    if (mesh->isPeriodic == PETSC_TRUE) {
      coords[0*2+0] = nodes[elements[elem*numCorners+0]*2+0];
      coords[0*2+1] = nodes[elements[elem*numCorners+0]*2+1];
      for(f = 1; f < PetscMax(disc->funcs, test->funcs); f++) {
        coords[f*2+0] = MeshPeriodicRelativeX(mesh, nodes[elements[elem*numCorners+f]*2+0], coords[0*2+0]);
        coords[f*2+1] = MeshPeriodicRelativeY(mesh, nodes[elements[elem*numCorners+f]*2+1], coords[0*2+1]);
      }
    } else {
      for(f = 0; f < PetscMax(disc->funcs, test->funcs); f++) {
        coords[f*2+0] = nodes[elements[elem*numCorners+f]*2+0];
        coords[f*2+1] = nodes[elements[elem*numCorners+f]*2+1];
      }
    }

    ierr = (*opFunc)(disc, test, rowSize, colSize, rowStart, colStart, elemSize, coords, alpha, field, array, ctx);
    CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateNonlinearOperatorGalerkin_Triangular_1D_Quadratic"
static int DiscEvaluateNonlinearOperatorGalerkin_Triangular_1D_Quadratic(Discretization disc, Mesh mesh, NonlinearOperator f,
                                                                         PetscScalar alpha, int elem, int numArgs, PetscScalar **field,
                                                                         PetscScalar *vec, void *ctx)
{
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  double          *nodes      = tri->nodes;
  int             *elements   = tri->faces;
  int              numCorners = mesh->numCorners;
  int              comp       = disc->comp;     /* The number of components in this field */
  int              funcs      = disc->funcs;    /* The number of shape functions per component */
  PetscScalar     *funcVal    = disc->funcVal;  /* Function value at a quadrature point */
  PetscScalar    **fieldVal   = disc->fieldVal; /* Field value and derivatives at a quadrature point */
  double           jac;                         /* |J| for map to standard element */
  double           invjac;                      /* |J^{-1}| for map from standard element */
  int              numQuadPoints;               /* Number of points used for Gaussian quadrature */
  double          *quadWeights;                 /* Weights in the standard element for Gaussian quadrature */
  double          *quadShapeFuncs;              /* Shape functions evaluated at quadrature points */
  double          *quadShapeFuncDers;           /* Shape function derivatives evaluated at quadrature points */
  double           x, y;                        /* The integration point */
  double           dxxi;                        /* \PartDer{x}{\xi}  */
  double           dxet;                        /* \PartDer{x}{\eta}  */
  double           dyxi;                        /* \PartDer{y}{\xi} */
  double           dyet;                        /* \PartDer{y}{\eta} */
  double           dxix;                        /* \PartDer{\xi}{x}  */
  double           detx;                        /* \PartDer{\eta}{x} */
  double           dxiy;                        /* \PartDer{\xi}{y}  */
  double           dety;                        /* \PartDer{\eta}{y} */
  PetscScalar      dfxi;                        /* \PartDer{field}{\xi}  */
  PetscScalar      dfet;                        /* \PartDer{field}{\eta} */
  double           coords[12];                  /* Coordinates of the element */
  int              rank = -1;
  int              i, j, k, func, p, arg;
#ifdef PETSC_USE_BOPT_g
  PetscTruth       opt;
#endif
  int              ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);
  numQuadPoints     = disc->numQuadPoints;
  quadWeights       = disc->quadWeights;
  quadShapeFuncs    = disc->quadShapeFuncs;
  quadShapeFuncDers = disc->quadShapeFuncDers;

  /* Calculate the determinant of the inverse Jacobian of the map to the standard element */
  if (mesh->isPeriodic == PETSC_TRUE) {
    coords[0*2+0] = nodes[elements[elem*numCorners+0]*2+0];
    coords[0*2+1] = nodes[elements[elem*numCorners+0]*2+1];
    for(func = 1; func < funcs; func++) {
      coords[func*2+0] = MeshPeriodicRelativeX(mesh, nodes[elements[elem*numCorners+func]*2+0], coords[0*2+0]);
      coords[func*2+1] = MeshPeriodicRelativeY(mesh, nodes[elements[elem*numCorners+func]*2+1], coords[0*2+1]);
    }
  } else {
    for(func = 0; func < funcs; func++) {
      coords[func*2+0] = nodes[elements[elem*numCorners+func]*2+0];
      coords[func*2+1] = nodes[elements[elem*numCorners+func]*2+1];
    }
  }
  /* Check for constant jacobian here */
  if (PETSC_FALSE) {
    jac = PetscAbsReal((coords[2] - coords[0])*(coords[5] - coords[1]) - (coords[4] - coords[0])*(coords[3] - coords[1]));
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "[%d], elem: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
		  rank, elem, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
  }
#ifdef PETSC_USE_BOPT_g
  ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                        CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                rank, elem, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
    PetscPrintf(PETSC_COMM_SELF, "  x4: %g y4: %g x5: %g y5: %g x6: %g y6: %g\n",
                coords[6], coords[7], coords[8], coords[9], coords[10], coords[11]);
  }
#endif

  /* Calculate element vector entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++) {
    /* x                      = \sum^{funcs}_{f=1} x_f   \phi^f(p)
       \PartDer{x}{\xi}(p)    = \sum^{funcs}_{f=1} x_f   \PartDer{\phi^f(p)}{\xi}
       \PartDer{x}{\eta}(p)   = \sum^{funcs}_{f=1} x_f   \PartDer{\phi^f(p)}{\eta}
       y                      = \sum^{funcs}_{f=1} y_f   \phi^f(p)
       \PartDer{y}{\xi}(p)    = \sum^{funcs}_{f=1} y_f   \PartDer{\phi^f(p)}{\xi}
       \PartDer{y}{\eta}(p)   = \sum^{funcs}_{f=1} y_f   \PartDer{\phi^f(p)}{\eta}
       u^i                    = \sum^{funcs}_{f=1} u^i_f \phi^f(p)
       \PartDer{u^i}{\xi}(p)  = \sum^{funcs}_{f=1} u^i_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{u^i}{\eta}(p) = \sum^{funcs}_{f=1} u^i_f \PartDer{\phi^f(p)}{\eta} */
    x    = 0.0; y    = 0.0;
    dxxi = 0.0; dyxi = 0.0;
    dxet = 0.0; dyet = 0.0;
    for(arg = 0; arg < numArgs; arg++)
      for(j = 0; j < comp*3; j++)
        fieldVal[arg][j] = 0.0;
    for(func = 0; func < funcs; func++) {
      x    += coords[func*2]  *quadShapeFuncs[p*funcs+func];
      dxxi += coords[func*2]  *quadShapeFuncDers[p*funcs*2+func*2];
      dxet += coords[func*2]  *quadShapeFuncDers[p*funcs*2+func*2+1];
      y    += coords[func*2+1]*quadShapeFuncs[p*funcs+func];
      dyxi += coords[func*2+1]*quadShapeFuncDers[p*funcs*2+func*2];
      dyet += coords[func*2+1]*quadShapeFuncDers[p*funcs*2+func*2+1];
      for(arg = 0; arg < numArgs; arg++) {
        for(j = 0; j < comp; j++) {
          fieldVal[arg][j*3]   += field[arg][func*comp+j]*quadShapeFuncs[p*funcs+func];
          fieldVal[arg][j*3+1] += field[arg][func*comp+j]*quadShapeFuncDers[p*funcs*2+func*2];
          fieldVal[arg][j*3+2] += field[arg][func*comp+j]*quadShapeFuncDers[p*funcs*2+func*2+1];
        }
      }
    }
    if (mesh->isPeriodic == PETSC_TRUE) {
      x = MeshPeriodicX(mesh, x);
      y = MeshPeriodicY(mesh, y);
    }
    jac = PetscAbsReal(dxxi*dyet - dxet*dyxi);
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                  rank, p, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
    /* These are the elements of the inverse matrix */
    invjac = 1/jac;
    dxix   =  dyet*invjac;
    dxiy   = -dxet*invjac;
    detx   = -dyxi*invjac;
    dety   =  dxxi*invjac;

    /* Convert the field derivatives to old coordinates */
    for(arg = 0; arg < numArgs; arg++)
      for(j = 0; j < comp; j++) {
        dfxi                 = fieldVal[arg][j*3+1];
        dfet                 = fieldVal[arg][j*3+2];
        fieldVal[arg][j*3+1] = dfxi*dxix + dfet*detx;
        fieldVal[arg][j*3+2] = dfxi*dxiy + dfet*dety;
      }

    ierr = (*f)(1, comp, &x, &y, PETSC_NULL, numArgs, fieldVal, funcVal, ctx);                            CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
    ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                      CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p: %d jac: %g", rank, p, jac);
      for(j = 0; j < comp; j++)
        PetscPrintf(PETSC_COMM_SELF, " func[%d]: %g", j, PetscRealPart(funcVal[j]));
      PetscPrintf(PETSC_COMM_SELF, "\n");
    }
#endif

    for(i = 0, k = 0; i < funcs; i++) {
      for(j = 0; j < comp; j++, k++) {
        vec[k] += alpha*funcVal[j]*quadShapeFuncs[p*funcs+i]*jac*quadWeights[p];
#ifdef PETSC_USE_BOPT_g
        ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                  CHKERRQ(ierr);
        if (opt == PETSC_TRUE) {
          PetscPrintf(PETSC_COMM_SELF, "[%d]  vec[%d]: %g\n", rank, k, PetscRealPart(vec[k]));
        }
#endif
      }
    }
  }
  PetscLogFlops(((12 + (6*numArgs + 5)*comp)*funcs + 8 + 6*numArgs*comp) * numQuadPoints);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateALEOperatorGalerkin_Triangular_1D_Quadratic"
static int DiscEvaluateALEOperatorGalerkin_Triangular_1D_Quadratic(Discretization disc, Mesh mesh, int elemSize,
                                                                   int rowStart, int colStart, int op, PetscScalar alpha,
                                                                   int elem, PetscScalar *field, PetscScalar *ALEfield, PetscScalar *array,
                                                                   void *ctx)
{
  Mesh_Triangular    *tri        = (Mesh_Triangular *) mesh->data;
  double             *nodes      = tri->nodes;
  int                *elements   = tri->faces;
  int                 numCorners = mesh->numCorners;
  Discretization      test;                  /* The space of test functions */
  Operator            oper;                  /* The operator to discretize */
  int                 rowSize;               /* The number of shape functions per element */
  int                 colSize;               /* The number of test  functions per element */
  ALEOperatorFunction opFunc;                /* Integrals of operators which depend on J */
  double              coords[MAX_CORNERS*2]; /* Coordinates of the element */
  int                 f;
  int                 ierr; 

  PetscFunctionBegin;
#ifdef PETSC_USE_BOPT_g
  /* Check for valid operator */
  if ((op < 0) || (op >= disc->numOps) || (!disc->operators[op])) SETERRQ(PETSC_ERR_ARG_WRONG, "Invalid operator");
#endif
  /* Get discretization info */
  oper    = disc->operators[op];
  opFunc  = oper->ALEOpFunc;
  test    = oper->test;
  rowSize = test->size;
  colSize = disc->size;

  if (opFunc == PETSC_NULL) SETERRQ(PETSC_ERR_ARG_CORRUPT, "Invalid function");
  if (mesh->isPeriodic == PETSC_TRUE) {
    coords[0*2+0] = nodes[elements[elem*numCorners+0]*2+0];
    coords[0*2+1] = nodes[elements[elem*numCorners+0]*2+1];
    for(f = 1; f < PetscMax(disc->funcs, test->funcs); f++) {
      coords[f*2+0] = MeshPeriodicRelativeX(mesh, nodes[elements[elem*numCorners+f]*2+0], coords[0*2+0]);
      coords[f*2+1] = MeshPeriodicRelativeY(mesh, nodes[elements[elem*numCorners+f]*2+1], coords[0*2+1]);
    }
  } else {
    for(f = 0; f < PetscMax(disc->funcs, test->funcs); f++) {
      coords[f*2+0] = nodes[elements[elem*numCorners+f]*2+0];
      coords[f*2+1] = nodes[elements[elem*numCorners+f]*2+1];
    }
  }

  ierr = (*opFunc)(disc, test, rowSize, colSize, rowStart, colStart, elemSize, coords, alpha, field, ALEfield, array, ctx);
  CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateNonlinearALEOperatorGalerkin_Triangular_1D_Quadratic"
static int DiscEvaluateNonlinearALEOperatorGalerkin_Triangular_1D_Quadratic(Discretization disc, Mesh mesh, NonlinearOperator f,
                                                                            PetscScalar alpha, int elem, int numArgs, PetscScalar **field,
                                                                            PetscScalar *ALEfield, PetscScalar *vec, void *ctx)
{
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  double          *nodes      = tri->nodes;
  int             *elements   = tri->faces;
  int              numCorners = mesh->numCorners;
  int              comp       = disc->comp;     /* The number of components in this field */
  int              funcs      = disc->funcs;    /* The number of shape functions per component */
  PetscScalar     *funcVal    = disc->funcVal;  /* Function value at a quadrature point */
  PetscScalar    **fieldVal   = disc->fieldVal; /* Field value and derivatives at a quadrature point */
  double           jac;                         /* |J| for map to standard element */
  double           invjac;                      /* |J^{-1}| for map from standard element */
  int              numQuadPoints;               /* Number of points used for Gaussian quadrature */
  double          *quadWeights;                 /* Weights in the standard element for Gaussian quadrature */
  double          *quadShapeFuncs;              /* Shape functions evaluated at quadrature points */
  double          *quadShapeFuncDers;           /* Shape function derivatives evaluated at quadrature points */
  double           x, y;                        /* The integration point */
  double           dxxi;                        /* \PartDer{x}{\xi}  */
  double           dxet;                        /* \PartDer{x}{\eta}  */
  double           dyxi;                        /* \PartDer{y}{\xi} */
  double           dyet;                        /* \PartDer{y}{\eta} */
  double           dxix;                        /* \PartDer{\xi}{x}  */
  double           detx;                        /* \PartDer{\eta}{x} */
  double           dxiy;                        /* \PartDer{\xi}{y}  */
  double           dety;                        /* \PartDer{\eta}{y} */
  PetscScalar      dfxi;                        /* \PartDer{field}{\xi}  */
  PetscScalar      dfet;                        /* \PartDer{field}{\eta} */
  double           coords[12];                  /* Coordinates of the element */
  int              rank;
  int              i, j, k, func, p, arg;
#ifdef PETSC_USE_BOPT_g
  PetscTruth       opt;
#endif
  int              ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);
  numQuadPoints     = disc->numQuadPoints;
  quadWeights       = disc->quadWeights;
  quadShapeFuncs    = disc->quadShapeFuncs;
  quadShapeFuncDers = disc->quadShapeFuncDers;

  /* Calculate the determinant of the inverse Jacobian of the map to the standard element */
  if (mesh->isPeriodic == PETSC_TRUE) {
    coords[0*2+0] = nodes[elements[elem*numCorners+0]*2+0];
    coords[0*2+1] = nodes[elements[elem*numCorners+0]*2+1];
    for(func = 1; func < funcs; func++) {
      coords[func*2+0] = MeshPeriodicRelativeX(mesh, nodes[elements[elem*numCorners+func]*2+0], coords[0*2+0]);
      coords[func*2+1] = MeshPeriodicRelativeY(mesh, nodes[elements[elem*numCorners+func]*2+1], coords[0*2+1]);
    }
  } else {
    for(func = 0; func < funcs; func++) {
      coords[func*2+0] = nodes[elements[elem*numCorners+func]*2+0];
      coords[func*2+1] = nodes[elements[elem*numCorners+func]*2+1];
    }
  }
  /* Check for constant jacobian here */
  if (PETSC_FALSE) {
    jac = PetscAbsReal((coords[2] - coords[0])*(coords[5] - coords[1]) - (coords[4] - coords[0])*(coords[3] - coords[1]));
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "[%d], elem: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                  rank, elem, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
  }
#ifdef PETSC_USE_BOPT_g
  ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                        CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                rank, elem, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
    PetscPrintf(PETSC_COMM_SELF, "  x4: %g y4: %g x5: %g y5: %g x6: %g y6: %g\n",
                coords[6], coords[7], coords[8], coords[9], coords[10], coords[11]);
  }
#endif

  /* Calculate element vector entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++) {
    /* x                      = \sum^{funcs}_{f=1} x_f   \phi^f(p)
       \PartDer{x}{\xi}(p)    = \sum^{funcs}_{f=1} x_f   \PartDer{\phi^f(p)}{\xi}
       \PartDer{x}{\eta}(p)   = \sum^{funcs}_{f=1} x_f   \PartDer{\phi^f(p)}{\eta}
       y                      = \sum^{funcs}_{f=1} y_f   \phi^f(p)
       \PartDer{y}{\xi}(p)    = \sum^{funcs}_{f=1} y_f   \PartDer{\phi^f(p)}{\xi}
       \PartDer{y}{\eta}(p)   = \sum^{funcs}_{f=1} y_f   \PartDer{\phi^f(p)}{\eta}
       u^i                    = \sum^{funcs}_{f=1} u^i_f \phi^f(p)
       \PartDer{u^i}{\xi}(p)  = \sum^{funcs}_{f=1} u^i_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{u^i}{\eta}(p) = \sum^{funcs}_{f=1} u^i_f \PartDer{\phi^f(p)}{\eta} */
    x    = 0.0; y    = 0.0;
    dxxi = 0.0; dyxi = 0.0;
    dxet = 0.0; dyet = 0.0;
    for(arg = 0; arg < numArgs; arg++)
      for(j = 0; j < comp*3; j++)
        fieldVal[arg][j] = 0.0;
    for(func = 0; func < funcs; func++)
    {
      x    += coords[func*2]  *quadShapeFuncs[p*funcs+func];
      dxxi += coords[func*2]  *quadShapeFuncDers[p*funcs*2+func*2];
      dxet += coords[func*2]  *quadShapeFuncDers[p*funcs*2+func*2+1];
      y    += coords[func*2+1]*quadShapeFuncs[p*funcs+func];
      dyxi += coords[func*2+1]*quadShapeFuncDers[p*funcs*2+func*2];
      dyet += coords[func*2+1]*quadShapeFuncDers[p*funcs*2+func*2+1];
      for(arg = 0; arg < numArgs; arg++) {
        for(j = 0; j < comp; j++) {
          fieldVal[arg][j*3]   += (field[arg][func*comp+j] - ALEfield[func*comp+j])*quadShapeFuncs[p*funcs+func];
          fieldVal[arg][j*3+1] += field[arg][func*comp+j]*quadShapeFuncDers[p*funcs*2+func*2];
          fieldVal[arg][j*3+2] += field[arg][func*comp+j]*quadShapeFuncDers[p*funcs*2+func*2+1];
        }
      }
    }
    if (mesh->isPeriodic == PETSC_TRUE) {
      x = MeshPeriodicX(mesh, x);
      y = MeshPeriodicY(mesh, y);
    }
    jac = PetscAbsReal(dxxi*dyet - dxet*dyxi);
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                  rank, p, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
    /* These are the elements of the inverse matrix */
    invjac = 1/jac;
    dxix   =  dyet*invjac;
    dxiy   = -dxet*invjac;
    detx   = -dyxi*invjac;
    dety   =  dxxi*invjac;

    /* Convert the field derivatives to old coordinates */
    for(arg = 0; arg < numArgs; arg++) {
      for(j = 0; j < comp; j++) {
        dfxi                 = fieldVal[arg][j*3+1];
        dfet                 = fieldVal[arg][j*3+2];
        fieldVal[arg][j*3+1] = dfxi*dxix + dfet*detx;
        fieldVal[arg][j*3+2] = dfxi*dxiy + dfet*dety;
      }
    }

    ierr = (*f)(1, comp, &x, &y, PETSC_NULL, numArgs, fieldVal, funcVal, ctx);                            CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
    ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                      CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p: %d jac: %g", rank, p, jac);
      for(j = 0; j < comp; j++)
        PetscPrintf(PETSC_COMM_SELF, " func[%d]: %g", j, PetscRealPart(funcVal[j]));
      PetscPrintf(PETSC_COMM_SELF, "\n");
    }
#endif

    for(i = 0, k = 0; i < funcs; i++) {
      for(j = 0; j < comp; j++, k++) {
        vec[k] += alpha*funcVal[j]*quadShapeFuncs[p*funcs+i]*jac*quadWeights[p];
#ifdef PETSC_USE_BOPT_g
        ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                  CHKERRQ(ierr);
        if (opt == PETSC_TRUE) {
          PetscPrintf(PETSC_COMM_SELF, "[%d]  vec[%d]: %g\n", rank, k, PetscRealPart(vec[k]));
        }
#endif
      }
    }
  }
  PetscLogFlops(((12 + (7*numArgs + 5)*comp)*funcs + 8 + 6*numArgs*comp) * numQuadPoints);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Identity_Triangular_1D_Quadratic"
int Identity_Triangular_1D_Quadratic(Discretization disc, Discretization test, int rowSize, int colSize,
                                     int globalRowStart, int globalColStart, int globalSize, double *coords,
                                     PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
  int     numQuadPoints;     /* Number of points used for Gaussian quadrature */
  double *quadWeights;       /* Weights in the standard element for Gaussian quadrature */
  double *quadShapeFuncs;    /* Shape functions evaluated at quadrature points */
  double *quadTestFuncs;     /* Test  functions evaluated at quadrature points */
  double *quadShapeFuncDers; /* Shape function derivatives evaluated at quadrature points */
  double  dxxi;              /* \PartDer{x}{\xi}  */
  double  dxet;              /* \PartDer{x}{\eta} */
  double  dyxi;              /* \PartDer{y}{\xi}  */
  double  dyet;              /* \PartDer{y}{\eta} */
  double  jac;               /* |J| for map to standard element */
  int     comp;              /* The number of field components */
  int     funcs;             /* The number of shape functions */
  int     i, j, c, f, p;

  PetscFunctionBegin;
  /* Calculate element matrix entries by Gaussian quadrature */
  comp              = disc->comp;
  funcs             = disc->funcs;
  numQuadPoints     = disc->numQuadPoints;
  quadWeights       = disc->quadWeights;
  quadShapeFuncs    = disc->quadShapeFuncs;
  quadTestFuncs     = test->quadShapeFuncs;
  quadShapeFuncDers = disc->quadShapeFuncDers;
  for(p = 0; p < numQuadPoints; p++)
  {
    /* \PartDer{x}{\xi}(p)  = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{x}{\eta}(p) = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\eta}
       \PartDer{y}{\xi}(p)  = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{y}{\eta}(p) = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\eta} */
    dxxi = 0.0; dxet = 0.0;
    dyxi = 0.0; dyet = 0.0;
    for(f = 0; f < funcs; f++)
    {
      dxxi += coords[f*2]  *quadShapeFuncDers[p*funcs*2+f*2];
      dxet += coords[f*2]  *quadShapeFuncDers[p*funcs*2+f*2+1];
      dyxi += coords[f*2+1]*quadShapeFuncDers[p*funcs*2+f*2];
      dyet += coords[f*2+1]*quadShapeFuncDers[p*funcs*2+f*2+1];
    }
    jac = PetscAbsReal(dxxi*dyet - dxet*dyxi);
#ifdef PETSC_USE_BOPT_g
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "p: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                  p, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
#endif

    for(i = 0; i < funcs; i++)
    {
      for(j = 0; j < funcs; j++)
      {
				for(c = 0; c < comp; c++)
				{
					array[(i*comp+c+globalRowStart)*globalSize + j*comp+c+globalColStart] +=
						alpha*quadTestFuncs[p*funcs+i]*quadShapeFuncs[p*funcs+j]*jac*quadWeights[p];
					/* PetscPrintf(PETSC_COMM_SELF, "  array[%d,%d]: %g\n", i*comp+c+globalRowStart, j*comp+c+globalColStart,
											array[(i*comp+c+globalRowStart)*globalSize + j*comp+c+globalColStart]); */
				}
      }
    }
  }
  PetscLogFlops((8*funcs + 3 + 5*funcs*funcs*comp) * numQuadPoints);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Laplacian_Triangular_1D_Quadratic"
int Laplacian_Triangular_1D_Quadratic(Discretization disc, Discretization test, int rowSize, int colSize,
                                      int globalRowStart, int globalColStart, int globalSize, double *coords,
                                      PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
  int     numQuadPoints;     /* Number of points used for Gaussian quadrature */
  double *quadWeights;       /* Weights in the standard element for Gaussian quadrature */
  double *quadShapeFuncDers; /* Shape function derivatives evaluated at quadrature points */
  double *quadTestFuncDers;  /* Test  function derivatives evaluated at quadrature points */
  double  dxxi;              /* \PartDer{x}{\xi}  */
  double  dxet;              /* \PartDer{x}{\eta} */
  double  dyxi;              /* \PartDer{y}{\xi}  */
  double  dyet;              /* \PartDer{y}{\eta} */
  double  dxix;              /* \PartDer{\xi}{x}  */
  double  detx;              /* \PartDer{\eta}{x} */
  double  dxiy;              /* \PartDer{\xi}{y}  */
  double  dety;              /* \PartDer{\eta}{y} */
  double  dphix;             /* \PartDer{\phi_i}{x} \times \PartDer{\phi_j}{x} */
  double  dphiy;             /* \PartDer{\phi_i}{y} \times \PartDer{\phi_j}{y} */
  double  jac;               /* |J| for map to standard element */
  double  invjac;            /* |J^{-1}| for map from standard element */
	int     comp;              /* The number of field components */
	int     funcs;             /* The number of shape functions */
  int     i, j, c, f, p;

  PetscFunctionBegin;
  /* Calculate element matrix entries by Gaussian quadrature */
  comp              = disc->comp;
  funcs             = disc->funcs;
  numQuadPoints     = disc->numQuadPoints;
  quadWeights       = disc->quadWeights;
  quadShapeFuncDers = disc->quadShapeFuncDers;
  quadTestFuncDers  = test->quadShapeFuncDers;
  for(p = 0; p < numQuadPoints; p++)
  {
    /* \PartDer{x}{\xi}(p)  = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{x}{\eta}(p) = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\eta}
       \PartDer{y}{\xi}(p)  = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{y}{\eta}(p) = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\eta} */
    dxxi = 0.0; dxet = 0.0;
    dyxi = 0.0; dyet = 0.0;
    for(f = 0; f < funcs; f++)
    {
      dxxi += coords[f*2]  *quadShapeFuncDers[p*funcs*2+f*2];
      dxet += coords[f*2]  *quadShapeFuncDers[p*funcs*2+f*2+1];
      dyxi += coords[f*2+1]*quadShapeFuncDers[p*funcs*2+f*2];
      dyet += coords[f*2+1]*quadShapeFuncDers[p*funcs*2+f*2+1];
    }
    jac  = PetscAbsReal(dxxi*dyet - dxet*dyxi);
#ifdef PETSC_USE_BOPT_g
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "p: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                  p, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
#endif
		/* These are the elements of the inverse matrix */
    invjac =  1.0/jac;
		dxix   =  dyet*invjac;
		dxiy   = -dxet*invjac;
		detx   = -dyxi*invjac;
		dety   =  dxxi*invjac;

    for(i = 0; i < funcs; i++)
    {
      for(j = 0; j < funcs; j++)
      {
        dphix = (quadTestFuncDers[p*funcs*2+i*2]*dxix + quadTestFuncDers[p*funcs*2+i*2+1]*detx)*
          (quadShapeFuncDers[p*funcs*2+j*2]*dxix + quadShapeFuncDers[p*funcs*2+j*2+1]*detx);
        dphiy = (quadTestFuncDers[p*funcs*2+i*2]*dxiy + quadTestFuncDers[p*funcs*2+i*2+1]*dety)*
          (quadShapeFuncDers[p*funcs*2+j*2]*dxiy + quadShapeFuncDers[p*funcs*2+j*2+1]*dety);
				for(c = 0; c < comp; c++)
				{
					array[(i*comp+c+globalRowStart)*globalSize + j*comp+c+globalColStart] +=
						-alpha*(dphix + dphiy)*jac*quadWeights[p];
					/* PetscPrintf(PETSC_COMM_SELF, "  array[%d,%d]: %g\n", i*comp+c+globalRowStart, j*comp+c+globalColStart,
											array[(i*comp+c+globalRowStart)*globalSize + j*comp+c+globalColStart]); */
				}
      }
    }
  }
  PetscLogFlops((8*funcs + 8 + 19*funcs*funcs*comp) * numQuadPoints);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Weighted_Laplacian_Triangular_1D_Quadratic"
int Weighted_Laplacian_Triangular_1D_Quadratic(Discretization disc, Discretization test, int rowSize, int colSize,
                                               int globalRowStart, int globalColStart, int globalSize, double *coords,
                                               PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
  int     numQuadPoints;     /* Number of points used for Gaussian quadrature */
  double *quadWeights;       /* Weights in the standard element for Gaussian quadrature */
  double *quadShapeFuncDers; /* Shape function derivatives evaluated at quadrature points */
  double *quadTestFuncDers;  /* Test  function derivatives evaluated at quadrature points */
  double  dxxi;              /* \PartDer{x}{\xi}  */
  double  dxet;              /* \PartDer{x}{\eta} */
  double  dyxi;              /* \PartDer{y}{\xi}  */
  double  dyet;              /* \PartDer{y}{\eta} */
  double  dxix;              /* \PartDer{\xi}{x}  */
  double  detx;              /* \PartDer{\eta}{x} */
  double  dxiy;              /* \PartDer{\xi}{y}  */
  double  dety;              /* \PartDer{\eta}{y} */
  double  dphix;             /* \PartDer{\phi_i}{x} \times \PartDer{\phi_j}{x} */
  double  dphiy;             /* \PartDer{\phi_i}{y} \times \PartDer{\phi_j}{y} */
  double  jac;               /* |J| for map to standard element */
  double  invjac;            /* |J^{-1}| for map from standard element */
	int     comp;              /* The number of field components */
	int     funcs;             /* The number of shape functions */
  int     i, j, c, f, p;

  /* Each element is weighted by its Jacobian. This is supposed to make smaller elements "stiffer". */
  PetscFunctionBegin;
  /* Calculate element matrix entries by Gaussian quadrature */
	comp              = disc->comp;
	funcs             = disc->funcs;
  numQuadPoints     = disc->numQuadPoints;
  quadWeights       = disc->quadWeights;
  quadShapeFuncDers = disc->quadShapeFuncDers;
  quadTestFuncDers  = test->quadShapeFuncDers;
  for(p = 0; p < numQuadPoints; p++)
  {
    /* \PartDer{x}{\xi}(p)  = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{x}{\eta}(p) = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\eta}
       \PartDer{y}{\xi}(p)  = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{y}{\eta}(p) = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\eta} */
    dxxi = 0.0; dxet = 0.0;
    dyxi = 0.0; dyet = 0.0;
    for(f = 0; f < funcs; f++)
    {
      dxxi += coords[f*2]  *quadShapeFuncDers[p*funcs*2+f*2];
      dxet += coords[f*2]  *quadShapeFuncDers[p*funcs*2+f*2+1];
      dyxi += coords[f*2+1]*quadShapeFuncDers[p*funcs*2+f*2];
      dyet += coords[f*2+1]*quadShapeFuncDers[p*funcs*2+f*2+1];
    }
    jac  = PetscAbsReal(dxxi*dyet - dxet*dyxi);
#ifdef PETSC_USE_BOPT_g
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "p: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                  p, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
#endif
		/* These are the elements of the inverse matrix */
    invjac =  1.0/jac;
		dxix   =  dyet*invjac;
		dxiy   = -dxet*invjac;
		detx   = -dyxi*invjac;
		dety   =  dxxi*invjac;

    for(i = 0; i < funcs; i++)
    {
      for(j = 0; j < funcs; j++)
      {
        dphix = (quadTestFuncDers[p*funcs*2+i*2]*dxix + quadTestFuncDers[p*funcs*2+i*2+1]*detx)*
          (quadShapeFuncDers[p*funcs*2+j*2]*dxix + quadShapeFuncDers[p*funcs*2+j*2+1]*detx);
        dphiy = (quadTestFuncDers[p*funcs*2+i*2]*dxiy + quadTestFuncDers[p*funcs*2+i*2+1]*dety)*
          (quadShapeFuncDers[p*funcs*2+j*2]*dxiy + quadShapeFuncDers[p*funcs*2+j*2+1]*dety);
				for(c = 0; c < comp; c++)
				{
					array[(i*comp+c+globalRowStart)*globalSize + j*comp+c+globalColStart] +=
						-alpha*(dphix + dphiy)*quadWeights[p];
				}
      }
    }
  }
  PetscLogFlops((8*funcs + 8 + 18*funcs*funcs*comp) * numQuadPoints);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Divergence_Triangular_1D_Quadratic"
int Divergence_Triangular_1D_Quadratic(Discretization disc, Discretization test, int rowSize, int colSize,
                                       int globalRowStart, int globalColStart, int globalSize, double *coords,
                                       PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
	/* We are using the convention that

		   \nabla \matrix{v_1 \cr v_2 \cr \vdots \cr v_n} =
			   \matrix{v^{(1)}_1 \cr \vdots \cr v^{(d)}_1 \cr v^{(1)}_2 \cr \vdots \cr v^{(d)}_n}

		 and

		   \nabla \cdot \matrix{v^{(1)}_1 \cr \vdots \cr v^{(d)}_1 \cr v^{(1)}_2 \cr \vdots \cr v^{(d)}_n} =
			   \matrix{v_1 \cr v_2 \cr \vdots \cr v_n}

		 where $d$ is the number of space dimensions. This agrees with the convention which allows
		 $\Delta \matrix{u_1 \cr u_2} = 0$ to denote a set of scalar equations	This also requires that
     the dimension of a vector must be divisible by the space dimension in order to be acted upon by
     the divergence operator */
  int     numQuadPoints;     /* Number of points used for Gaussian quadrature */
  double *quadWeights;       /* Weights in the standard element for Gaussian quadrature */
  double *quadTestFuncs;     /* Test  functions evaluated at quadrature points */
  double *quadShapeFuncDers; /* Shape function derivatives evaluated at quadrature points */
  double  dxxi;              /* \PartDer{x}{\xi}  */
  double  dxet;              /* \PartDer{x}{\eta} */
  double  dyxi;              /* \PartDer{y}{\xi}  */
  double  dyet;              /* \PartDer{y}{\eta} */
  double  dxix;              /* \PartDer{\xi}{x}  */
  double  detx;              /* \PartDer{\eta}{x} */
  double  dxiy;              /* \PartDer{\xi}{y}  */
  double  dety;              /* \PartDer{\eta}{y} */
  double  dphix;             /* \PartDer{\phi_i}{x} \times \PartDer{\phi_j}{x} */
  double  dphiy;             /* \PartDer{\phi_i}{y} \times \PartDer{\phi_j}{y} */
  double  jac;               /* |J| for map to standard element */
  double  invjac;            /* |J^{-1}| for map from standard element */
  int     comp;              /* The number of field components */
  int     tcomp;             /* The number of field components for the test field */
  int     funcs;             /* The number of shape functions */
  int     tfuncs;            /* The number of test functions */
  int     i, j, c, tc, f, p;

  PetscFunctionBegin;
  /* Calculate element matrix entries by Gaussian quadrature */
  comp              = disc->comp;
  tcomp             = test->comp;
  funcs             = disc->funcs;
  tfuncs            = test->funcs;
  numQuadPoints     = disc->numQuadPoints;
  quadWeights       = disc->quadWeights;
  quadTestFuncs     = test->quadShapeFuncs;
  quadShapeFuncDers = disc->quadShapeFuncDers;
  for(p = 0; p < numQuadPoints; p++) {
    /* \PartDer{x}{\xi}(p)  = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{x}{\eta}(p) = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\eta}
       \PartDer{y}{\xi}(p)  = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{y}{\eta}(p) = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\eta} */
    dxxi = 0.0; dxet = 0.0;
    dyxi = 0.0; dyet = 0.0;
    for(f = 0; f < funcs; f++)
    {
      dxxi += coords[f*2]  *quadShapeFuncDers[p*funcs*2+f*2];
      dxet += coords[f*2]  *quadShapeFuncDers[p*funcs*2+f*2+1];
      dyxi += coords[f*2+1]*quadShapeFuncDers[p*funcs*2+f*2];
      dyet += coords[f*2+1]*quadShapeFuncDers[p*funcs*2+f*2+1];
    }
    jac  = PetscAbsReal(dxxi*dyet - dxet*dyxi);
#ifdef PETSC_USE_BOPT_g
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "p: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
									p, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
#endif
		/* PetscPrintf(PETSC_COMM_SELF, "p: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
								p, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]); */
		/* These are the elements of the inverse matrix */
    invjac =  1.0/jac;
    dxix   =  dyet*invjac;
    dxiy   = -dxet*invjac;
    detx   = -dyxi*invjac;
    dety   =  dxxi*invjac;

		/* The rows are test functions */
    for(i = 0; i < tfuncs; i++)
    {
      for(tc = 0; tc < tcomp; tc++)
      {
        /* The columns are shape functions */
        for(j = 0; j < funcs; j++)
        {
          dphix = quadShapeFuncDers[p*funcs*2+j*2]*dxix + quadShapeFuncDers[p*funcs*2+j*2+1]*detx;
          dphiy = quadShapeFuncDers[p*funcs*2+j*2]*dxiy + quadShapeFuncDers[p*funcs*2+j*2+1]*dety;
          /* We divide by the number of space dimensions */
          for(c = 0; c < comp/2; c++)
          {
            array[(i*tcomp+tc+globalRowStart)*globalSize + j*comp+c*2+globalColStart] +=
              alpha*dphix*quadTestFuncs[p*tfuncs+i]*jac*quadWeights[p];
							/* PetscPrintf(PETSC_COMM_SELF, "  array[%d,%d]: %g\n", i*tcomp+tc+globalRowStart, j*comp+c*2+globalColStart,
													array[(i*tcomp+tc+globalRowStart)*globalSize + j*comp+c*2+globalColStart]); */
            array[(i*tcomp+tc+globalRowStart)*globalSize + j*comp+c*2+1+globalColStart] +=
              alpha*dphiy*quadTestFuncs[p*tfuncs+i]*jac*quadWeights[p];
							/* PetscPrintf(PETSC_COMM_SELF, "  array[%d,%d]: %g\n", i*tcomp+tc+globalRowStart, j*comp+c*2+1+globalColStart,
													array[(i*tcomp+tc+globalRowStart)*globalSize + j*comp+c*2+1+globalColStart]); */
					}
				}
			}
    }
  }
  PetscLogFlops((8*funcs + 8 + 8*tfuncs*tcomp*funcs*comp) * numQuadPoints);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscInterpolateField_Triangular_1D_Quadratic"
int DiscInterpolateField_Triangular_1D_Quadratic(Discretization disc, Mesh oldMesh, int elem, double x, double y, double z,
                                                 PetscScalar *oldFieldVal, PetscScalar *newFieldVal, InterpolationType type)
{
  Mesh_Triangular *tri        = (Mesh_Triangular *) oldMesh->data;
  int              numCorners = oldMesh->numCorners;
  int             *elements   = tri->faces;
  int             *neighbors  = tri->neighbors;
  double          *nodes      = tri->nodes;
  double           coords[24];  /* Coordinates of our "big element" */
  double           xi, eta;     /* Canonical coordinates of the point */
  double           x21, x31, y21, y31, jac, invjac, dx, dy, dxix, dxiy, detx, dety, xiOld, etaOld;
  int              comp = disc->comp;
  int              neighbor, corner, nelem, node, c;
  int              ierr;

  PetscFunctionBegin;
  /* No scheme in place for boundary elements */
  for(neighbor = 0; neighbor < 3; neighbor++)
    if (neighbors[elem*3+neighbor] < 0)
      type = INTERPOLATION_LOCAL;

  switch (type)
  {
  case INTERPOLATION_LOCAL:
    if (oldMesh->isPeriodic == PETSC_TRUE) {
      coords[0*2+0] = MeshPeriodicRelativeX(oldMesh, nodes[elements[elem*numCorners+0]*2+0], x);
      coords[0*2+1] = MeshPeriodicRelativeY(oldMesh, nodes[elements[elem*numCorners+0]*2+1], y);
      coords[1*2+0] = MeshPeriodicRelativeX(oldMesh, nodes[elements[elem*numCorners+1]*2+0], x);
      coords[1*2+1] = MeshPeriodicRelativeY(oldMesh, nodes[elements[elem*numCorners+1]*2+1], y);
      coords[2*2+0] = MeshPeriodicRelativeX(oldMesh, nodes[elements[elem*numCorners+2]*2+0], x);
      coords[2*2+1] = MeshPeriodicRelativeY(oldMesh, nodes[elements[elem*numCorners+2]*2+1], y);
      coords[3*2+0] = MeshPeriodicRelativeX(oldMesh, nodes[elements[elem*numCorners+3]*2+0], x);
      coords[3*2+1] = MeshPeriodicRelativeY(oldMesh, nodes[elements[elem*numCorners+3]*2+1], y);
      coords[4*2+0] = MeshPeriodicRelativeX(oldMesh, nodes[elements[elem*numCorners+4]*2+0], x);
      coords[4*2+1] = MeshPeriodicRelativeY(oldMesh, nodes[elements[elem*numCorners+4]*2+1], y);
      coords[5*2+0] = MeshPeriodicRelativeX(oldMesh, nodes[elements[elem*numCorners+5]*2+0], x);
      coords[5*2+1] = MeshPeriodicRelativeY(oldMesh, nodes[elements[elem*numCorners+5]*2+1], y);
    } else {
      coords[0*2+0] = nodes[elements[elem*numCorners+0]*2+0];
      coords[0*2+1] = nodes[elements[elem*numCorners+0]*2+1];
      coords[1*2+0] = nodes[elements[elem*numCorners+1]*2+0];
      coords[1*2+1] = nodes[elements[elem*numCorners+1]*2+1];
      coords[2*2+0] = nodes[elements[elem*numCorners+2]*2+0];
      coords[2*2+1] = nodes[elements[elem*numCorners+2]*2+1];
      coords[3*2+0] = nodes[elements[elem*numCorners+3]*2+0];
      coords[3*2+1] = nodes[elements[elem*numCorners+3]*2+1];
      coords[4*2+0] = nodes[elements[elem*numCorners+4]*2+0];
      coords[4*2+1] = nodes[elements[elem*numCorners+4]*2+1];
      coords[5*2+0] = nodes[elements[elem*numCorners+5]*2+0];
      coords[5*2+1] = nodes[elements[elem*numCorners+5]*2+1];
    }
    /* Get the (\xi,\eta) coordinates of the point */
    ierr = DiscTransformCoords_Triangular_1D_Quadratic(x, y, coords, &xi, &eta);                         CHKERRQ(ierr);
    if ((xi < -1.0e-02) || (eta < -1.0e-02) || (xi > 1.01) || (eta > 1.01)) {
      xiOld  = xi;
      etaOld = eta;
      /* Use linear approximation */
      x21    = coords[1*2+0] - coords[0*2+0];
      x31    = coords[2*2+0] - coords[0*2+0];
      y21    = coords[1*2+1] - coords[0*2+1];
      y31    = coords[2*2+1] - coords[0*2+1];
      dx     = x - coords[0*2+0];
      dy     = y - coords[0*2+1];
      jac    = PetscAbsReal(x21*y31 - x31*y21);
      invjac = 1/jac;
      dxix   =  y31*invjac;
      dxiy   = -x31*invjac;
      detx   = -y21*invjac;
      dety   =  x21*invjac;

      xi     = dxix*dx + dxiy*dy;
      eta    = detx*dx + dety*dy;
      PetscPrintf(PETSC_COMM_SELF, "elem: %d x: %g y: %g xiOld: %g etaOld: %g xi: %g eta: %g\n", elem, x, y, xiOld, etaOld, xi, eta);
    }
    for(c = 0; c < comp; c++) {
      newFieldVal[c] = oldFieldVal[0*comp+c]*(1.0 - xi - eta)*(1.0 - 2.0*xi - 2.0*eta) +
        oldFieldVal[1*comp+c]*xi *(2.0*xi  - 1.0)      +
        oldFieldVal[2*comp+c]*eta*(2.0*eta - 1.0)      +
        oldFieldVal[3*comp+c]*4.0*xi*eta               +
        oldFieldVal[4*comp+c]*4.0*eta*(1.0 - xi - eta) +
        oldFieldVal[5*comp+c]*4.0*xi *(1.0 - xi - eta);
    }
    PetscLogFlops(34*comp);
    break;
  case INTERPOLATION_HALO:
    /* Here is our "big element" where numbers in parantheses represent
       the numbering on the old little element:

           2
           |\
           | \
           |  \
           6   5
           |    \
           |     \
           |      \
       (1) 7---*---4 (0)
           |\      |\
           | \     | \
           |  \    |  \
           8   *   *   3
           |    \  |    \
           |     \ |     \
           |      \|      \
           0---9--10--11---1
                  (2)

       We search for the neighbor node by looking for the vertex not a member of the original element.
    */
    for(neighbor = 0; neighbor < 3; neighbor++)
    {
      nelem = neighbors[elem*3+neighbor];
      for(corner = 0; corner < 3; corner++)
      {
        node = elements[nelem*numCorners+corner];
        if ((node != elements[elem*numCorners+((neighbor+1)%3)]) && (node != elements[elem*numCorners+((neighbor+2)%3)]))
        {
          /* The neighboring elements give the vertices */
          coords[neighbor*2]   = nodes[node*2];
          coords[neighbor*2+1] = nodes[node*2+1];
          break;
        }
      }
    }
    /* Element vertices form midnodes */
    coords[3*2]   = nodes[elements[elem*numCorners]*2];
    coords[3*2+1] = nodes[elements[elem*numCorners]*2+1];
    coords[4*2]   = nodes[elements[elem*numCorners+1]*2];
    coords[4*2+1] = nodes[elements[elem*numCorners+1]*2+1];
    coords[5*2]   = nodes[elements[elem*numCorners+2]*2];
    coords[5*2+1] = nodes[elements[elem*numCorners+2]*2+1];
    /* Treat 4 triangles as one big element with quadratic shape functions */
    SETERRQ(PETSC_ERR_SUP, "Unsupported interpolation type");
  default:
    SETERRQ(PETSC_ERR_ARG_WRONG, "Unknown interpolation type");
  }
  
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscInterpolateElementVec_Triangular_1D_Quadratic"
int DiscInterpolateElementVec_Triangular_1D_Quadratic(Discretization disc, ElementVec vec, Discretization newDisc, ElementVec newVec)
{
  int          comp  = disc->comp;
  int          size  = disc->size;
  PetscScalar *array, *newArray;
  PetscTruth   islin, isquad;
  int          f, c;
  int          ierr;

  PetscFunctionBegin;
  ierr = ElementVecGetArray(vec,    &array);                                                              CHKERRQ(ierr);
  ierr = ElementVecGetArray(newVec, &newArray);                                                           CHKERRQ(ierr);
  ierr = PetscTypeCompare((PetscObject) newDisc, DISCRETIZATION_TRIANGULAR_1D_LINEAR,    &islin);         CHKERRQ(ierr);
  ierr = PetscTypeCompare((PetscObject) newDisc, DISCRETIZATION_TRIANGULAR_1D_QUADRATIC, &isquad);        CHKERRQ(ierr);
  if (isquad == PETSC_TRUE) {
    ierr = PetscMemcpy(newArray, array, size * sizeof(PetscScalar));                                      CHKERRQ(ierr);
  } else if (islin == PETSC_TRUE) {
    for(f = 0; f < newDisc->funcs; f++) {
      for(c = 0; c < comp; c++) {
        newArray[f*comp+c] = array[f*comp+c];
      }
    }
  } else {
    SETERRQ(PETSC_ERR_SUP, "Discretization not supported");
  }
  ierr = ElementVecRestoreArray(vec,    &array);                                                          CHKERRQ(ierr);
  ierr = ElementVecRestoreArray(newVec, &newArray);                                                       CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscSetupQuadrature_Triangular_1D_Quadratic"
/*
  DiscSetupQuadrature_Triangular_1D_Quadratic - Setup Gaussian quadrature with a 7 point integration rule

  Input Parameter:
. disc - The Discretization
*/
int DiscSetupQuadrature_Triangular_1D_Quadratic(Discretization disc) {
  int    dim   = disc->dim;
  int    funcs = disc->funcs;
  double xi, eta;
  int    p;
  int    ierr;

  PetscFunctionBegin;
  disc->numQuadPoints = 7;
  ierr = PetscMalloc(disc->numQuadPoints*dim       * sizeof(double), &disc->quadPoints);                  CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints           * sizeof(double), &disc->quadWeights);                 CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints*funcs     * sizeof(double), &disc->quadShapeFuncs);              CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints*funcs*dim * sizeof(double), &disc->quadShapeFuncDers);           CHKERRQ(ierr);
  PetscLogObjectMemory(disc, (disc->numQuadPoints*(funcs*(dim+1) + dim+1)) * sizeof(double));
  disc->quadPoints[0]  = 1.0/3.0;
  disc->quadPoints[1]  = disc->quadPoints[0];
  disc->quadWeights[0] = 0.11250000000000;
  disc->quadPoints[2]  = 0.797426985353087;
  disc->quadPoints[3]  = 0.101286507323456;
  disc->quadWeights[1] = 0.0629695902724135;
  disc->quadPoints[4]  = disc->quadPoints[3];
  disc->quadPoints[5]  = disc->quadPoints[2];
  disc->quadWeights[2] = disc->quadWeights[1];
  disc->quadPoints[6]  = disc->quadPoints[4];
  disc->quadPoints[7]  = disc->quadPoints[3];
  disc->quadWeights[3] = disc->quadWeights[1];
  disc->quadPoints[8]  = 0.470142064105115;
  disc->quadPoints[9]  = 0.059715871789770;
  disc->quadWeights[4] = 0.066197076394253;
  disc->quadPoints[10] = disc->quadPoints[8];
  disc->quadPoints[11] = disc->quadPoints[8];
  disc->quadWeights[5] = disc->quadWeights[4];
  disc->quadPoints[12] = disc->quadPoints[9];
  disc->quadPoints[13] = disc->quadPoints[8];
  disc->quadWeights[6] = disc->quadWeights[4];
  for(p = 0; p < disc->numQuadPoints; p++) {
    xi  = disc->quadPoints[p*2];
    eta = disc->quadPoints[p*2+1];
    /* \phi^0: 1 - 3 (\xi + \eta) + 2 (\xi + \eta)^2 */
    disc->quadShapeFuncs[p*funcs]            =  1.0 - 3.0*(xi + eta) + 2.0*(xi + eta)*(xi + eta);
    disc->quadShapeFuncDers[p*funcs*2+0*2]   = -3.0 + 4.0*(xi + eta);
    disc->quadShapeFuncDers[p*funcs*2+0*2+1] = -3.0 + 4.0*(xi + eta);
    /* \phi^1: \xi (2\xi - 1) */
    disc->quadShapeFuncs[p*funcs+1]          =  xi*(2.0*xi - 1.0);
    disc->quadShapeFuncDers[p*funcs*2+1*2]   =  4.0*xi - 1.0;
    disc->quadShapeFuncDers[p*funcs*2+1*2+1] =  0.0;
    /* \phi^2: \eta (2\eta - 1) */
    disc->quadShapeFuncs[p*funcs+2]          =  eta*(2.0*eta - 1.0);
    disc->quadShapeFuncDers[p*funcs*2+2*2]   =  0.0;
    disc->quadShapeFuncDers[p*funcs*2+2*2+1] =  4.0*eta - 1.0;
    /* \phi^3: 4 \xi \eta */
    disc->quadShapeFuncs[p*funcs+3]          =  4.0*xi*eta;
    disc->quadShapeFuncDers[p*funcs*2+3*2]   =  4.0*eta;
    disc->quadShapeFuncDers[p*funcs*2+3*2+1] =  4.0*xi;
    /* \phi^4: 4 \eta (1 - \xi - \eta) */
    disc->quadShapeFuncs[p*funcs+4]          =  4.0*eta*(1.0 - xi - eta);
    disc->quadShapeFuncDers[p*funcs*2+4*2]   = -4.0*eta;
    disc->quadShapeFuncDers[p*funcs*2+4*2+1] = -8.0*eta + 4.0*(1.0 - xi);
    /* \phi^5: 4 \xi (1 - \xi - \eta) */
    disc->quadShapeFuncs[p*funcs+5]          =  4.0*xi*(1.0 - xi - eta);
    disc->quadShapeFuncDers[p*funcs*2+5*2]   = -8.0*xi + 4.0*(1.0 - eta);
    disc->quadShapeFuncDers[p*funcs*2+5*2+1] = -4.0*xi;
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscSetupOperators_Triangular_1D_Quadratic"
/*
  DiscSetupOperators_Triangular_1D_Quadratic - Setup the default operators

  Input Parameter:
. disc - The Discretization
*/
int DiscSetupOperators_Triangular_1D_Quadratic(Discretization disc) {
  int newOp;
  int ierr;

  PetscFunctionBegin;
  /* The Identity operator I -- the matrix is symmetric */
  ierr = DiscretizationRegisterOperator(disc, Identity_Triangular_1D_Quadratic, &newOp);                  CHKERRQ(ierr);
  if (newOp != IDENTITY) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", IDENTITY);
  /* The Laplacian operator \Delta -- the matrix is symmetric */
  ierr = DiscretizationRegisterOperator(disc, Laplacian_Triangular_1D_Quadratic, &newOp);                 CHKERRQ(ierr);
  if (newOp != LAPLACIAN) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", LAPLACIAN);
  /* The Gradient operator \nabla -- the matrix is rectangular */
  ierr = DiscretizationRegisterOperator(disc, PETSC_NULL, &newOp);                                        CHKERRQ(ierr);
  if (newOp != GRADIENT) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", GRADIENT);
  /* The Divergence operator \nabla\cdot -- the matrix is rectangular */
  ierr = DiscretizationRegisterOperator(disc, Divergence_Triangular_1D_Quadratic, &newOp);                CHKERRQ(ierr);
  if (newOp != DIVERGENCE) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", DIVERGENCE);
  /* The weighted Laplacian operator -- the matrix is symmetric */
  ierr = DiscretizationRegisterOperator(disc, Weighted_Laplacian_Triangular_1D_Quadratic, &newOp);        CHKERRQ(ierr);
  if (newOp != WEIGHTED_LAP) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", WEIGHTED_LAP);
  PetscFunctionReturn(0);
}

static struct _DiscretizationOps DOps = {PETSC_NULL/* DiscretizationSetup */,
                                         DiscSetupOperators_Triangular_1D_Quadratic,
                                         PETSC_NULL/* DiscretizationSetFromOptions */,
                                         DiscView_Triangular_1D_Quadratic,
                                         DiscDestroy_Triangular_1D_Quadratic,
                                         DiscEvaluateFunctionGalerkin_Triangular_1D_Quadratic,
                                         DiscEvaluateOperatorGalerkin_Triangular_1D_Quadratic,
                                         DiscEvaluateALEOperatorGalerkin_Triangular_1D_Quadratic,
                                         DiscEvaluateNonlinearOperatorGalerkin_Triangular_1D_Quadratic,
                                         DiscEvaluateNonlinearALEOperatorGalerkin_Triangular_1D_Quadratic,
                                         DiscInterpolateField_Triangular_1D_Quadratic,
                                         DiscInterpolateElementVec_Triangular_1D_Quadratic};

EXTERN_C_BEGIN
#undef  __FUNCT__
#define __FUNCT__ "DiscCreate_Triangular_1D_Quadratic"
int DiscCreate_Triangular_1D_Quadratic(Discretization disc) {
  int    arg;
  int    ierr;

  PetscFunctionBegin;
  if (disc->comp <= 0) {
    SETERRQ(PETSC_ERR_ARG_WRONG, "Discretization must have at least 1 component. Call DiscretizationSetNumComponents() to set this.");
  }
  ierr = PetscMemcpy(disc->ops, &DOps, sizeof(struct _DiscretizationOps));                                CHKERRQ(ierr);
  disc->dim   = 2;
  disc->funcs = 6;
  disc->size  = disc->funcs*disc->comp;

  ierr = DiscretizationSetupDefaultOperators(disc);                                                       CHKERRQ(ierr);
  ierr = DiscSetupQuadrature_Triangular_1D_Quadratic(disc);                                               CHKERRQ(ierr);

  /* Storage */
  ierr = PetscMalloc(disc->comp * sizeof(PetscScalar),   &disc->funcVal);                                 CHKERRQ(ierr);
  ierr = PetscMalloc(2          * sizeof(PetscScalar *), &disc->fieldVal);                                CHKERRQ(ierr);
  for(arg = 0; arg < 2; arg++) {
    ierr = PetscMalloc(disc->comp*(disc->dim+1) * sizeof(PetscScalar), &disc->fieldVal[arg]);             CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}
EXTERN_C_END
