/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     tifload.c                                                      *
 *            Load TIFF image taking care for specific TAGS                  *
 * Uses:      LibTIFF                                                        *
 * Author:    Andreas Tille                                                  *
 * Date:      18.06.1998                                                     *
 * Copyright: Andreas Tille, 1999; GNU Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <tiffio.h>

#include "paul.h"

int PaulLoadTIFF(PICTURE *bild, FILE *fp, char *file, long flag)
/* load TIFF files and set spezific tags in the PICTURE structure
 * if only info is interesting and quiet operation is set the image data will be ignored
 * --- Parameter: ---
 * PICTURE *bild          : image structure which should store the data in the TIFF file
 * FILE    *fp            : file pointer which points to open TIFF file
 * char    *file          : file name (TIFFlib curiousely enough needs this)
 * long     flag          : may be only image info is whished -> ask this flag
 * --- Return: ---
 * PICTURE *bild          : image structure with data
 * int     PaulLoadTIFF() : RET_ERR or RET_OK
 */
{
  TIFF          *tif;
  unsigned char *ptr, r, g, b, a;
  int            fd, x, y;
  uint32        *rast, *tptr;
  CHUNK         *cp;
  struct tm      tmp[1];
  float          res;
  short          sval;
  char          *s;
  C_VALUE        cv;
   
  bild->trans=0;
  g_return_val_if_fail ( fp, RET_ERR );
  g_return_val_if_fail ( file, RET_ERR ) ;
  g_return_val_if_fail ( (fd = fileno(fp)) >= 0, RET_ERR );
  /* Apparently rewind(f) isn't sufficient */
  lseek(fd, (long) 0, 0);  
  /* So why does libtif need a filename here ??? */
  g_return_val_if_fail ( (tif = TIFFFdOpen(fd, file, "r")), RET_ERR );

  if ( TIFFGetField(tif,TIFFTAG_IMAGEWIDTH,      &(bild->W)) != 1 ||
       TIFFGetField(tif,TIFFTAG_IMAGELENGTH,     &(bild->H)) != 1 ) {
    g_warning(_("Error while reading dimensions of %s"), bild->file);
    TIFFClose(tif);
    return RET_ERR;
  }

  if ( TIFFGetField(tif,TIFFTAG_SAMPLESPERPIXEL, &x) != 1 ) {
    g_warning(_("Error while reading \"Samples per Pixel\"-Tag in %s"), bild->file);
    bild->spp = 3;
  } else 
    bild->spp = x;
  
  if ( bild->spec ) 
    for ( cp = bild->spec; cp->key != NULL; cp++ ) {
      y = GetTag(cp->key);
      if ( TIFFTAG_PAGENUMBER == y || TIFFTAG_HALFTONEHINTS == y ) {
	if ( TIFFGetField(tif, y, &sval) == 1 ) {
	  cv.s = g_strdup_printf("%i", sval);
          CopySpec(bild->spec, cp->key, cv);
          FREE(cv.s);
	}
        continue;
      }
      if ( (y >= 0) && (TIFFGetField(tif, y, &s) == 1) ) {
        cv.s = s;
        CopySpec(bild->spec, cp->key, cv);
      }
    }

  if ( TIFFGetField(tif, TIFFTAG_DATETIME, &s) == 1 ) {
    if ( 6 == (y = sscanf(s, "%4u:%2u:%2u %2u:%2u:%2u",
                          &(tmp->tm_year), &(tmp->tm_mon), &(tmp->tm_mday), 
                          &(tmp->tm_hour), &(tmp->tm_min), &(tmp->tm_sec))) ) {
      tmp->tm_year -= 1900;
      tmp->tm_mon  -= 1;
      bild->zeit = mktime(tmp);
    }
  }
  if ( TIFFGetField(tif, TIFFTAG_XRESOLUTION, &res) == 1 ) bild->res      = (int)res;
  if ( TIFFGetField(tif, TIFFTAG_XPOSITION, &res)   == 1 ) bild->x_offset = (int)res;
  if ( TIFFGetField(tif, TIFFTAG_YPOSITION, &res)   == 1 ) bild->y_offset = (int)res;
   
  GetPictureSpecs(bild, file);
  bild->size = bild->W * bild->H;

  if ( OnlyInfo(flag) ) {
    TIFFClose(tif);
    bild->DATA = NULL;
    return RET_OK;
  }
  rast = g_new(uint32, bild->size);
  if ( TIFFReadRGBAImage(tif, bild->W, bild->H, rast, 0)) {
    ptr = bild->DATA = g_malloc(3*bild->size);
    for (y = 0; y < bild->H; y++) {
      tptr  = rast;
      tptr += ((bild->H - y-1) * bild->W);
      for (x = 0; x < bild->W; x++) {
        if ( IsMonochrom(bild) ) {
          *ptr++ = 0; 
          *ptr++ = TIFFGetG(*tptr++); 
          *ptr++ = 0;
        } else {
          a = TIFFGetA(*tptr);
          b = TIFFGetB(*tptr);
          g = TIFFGetG(*tptr);
          r = TIFFGetR(*tptr);
          tptr++;
          if (a<128) {
            *ptr++ = 255;
            *ptr++ = 0;
            *ptr++ = 255;
            bild->trans=1;
          } else {
            if ((r==255)&&(g==0)&&(b==255)) r=254;  /*???????????????????*/
              *ptr++ = r;
              *ptr++ = g;
              *ptr++ = b;
          }
        }
      }
    }
  }
  FREE(rast);
  TIFFClose(tif);
  return RET_OK;
}

