/******************************************************************************
 *                                                                            *
 * Program:   paul                                                            *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                 *
 *            (L)aserbildern                                                  *
 * Modul:     paulinfo.c                                                      *
 *            Output of all available information contained in the image file *
 * Author:    Andreas Tille                                                   *
 * Date:      07.05.1998                                                      *
 * Copyright: Andreas Tille, 1999; GNU Public License                         *
 *                                                                            *
 ******************************************************************************/

#include <ctype.h>
#include <stdio.h>

#include "paul.h"
#include "names.h"

void PaulInfo(GList *piclist, long flag)
/* print image info to stdout
 * --- Parameter: ---
 * GList *piclist: picture list, which should be processed
 * long   flag   : print Gamma curve or not
 */
{
  CHUNK     *cp;
  png_color *gamma, *fip;
  PICTURE   *bild;
  GList     *pl;
  const char *GAMMA;
  GAMMA = _("Gamma Table");
   
  if ( !NBILDER(piclist) ) return;

  for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) {
    printf("%s: %s\n", File, bild->file);
    printf(" %s: %i x %i = %i\n", Imagesize, bild->W, bild->H, bild->size);
    printf(" %s: %i\n", Samples_p_P, bild->spp);
    if ( bild->zeit > 0 )
      printf(" %s: %s\n", Source_date, time_t2string(bild->zeit));
    for ( cp = bild->spec; cp->key != NULL; cp++ ) {
      char *buf;
       
      if ( (buf = Chunk2Char(cp)) ) {
        printf(" %s: %s\n", cp->key, buf);
        FREE(buf);
      }
    }
      
    if ( bild->res ) printf(_(" Scanner resolution: %i DPI\n"), bild->res);
    if ( bild->x_offset != (unsigned long)-1 ) 
      printf(_(" Scanner offset: (%li,%li)\n"), bild->x_offset, bild->y_offset);
    if ( (flag & PRINTGAMMA) && bild->n_gamma && bild->gamma ){
      printf(_(" %s: %i Values\n"), GAMMA, bild->n_gamma);
      for ( fip = (gamma = bild->gamma) + bild->n_gamma; gamma < fip; gamma++ )
        printf(" %*c%5i%5i%5i\n", strlen(GAMMA), ' ', gamma->red, gamma->green, gamma->blue); 
    }
  }
}

static void TabLabel(GtkWidget *table, const char *desc, int i)
/* Label in info table
 * --- Parameter: ---
 * GtkWidget  *table : table to draw in
 * char       *desc  : label to mark the entry
 * int         i     : row number to draw in
 */
{
  GtkWidget      *label;

  label = gtk_label_new(desc);
  gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, i, i+1, GTK_EXPAND|GTK_FILL, 0, 2, 0);
}

static void FocusInEntry(GtkWidget *entry)
/* Trick to get the string in entry marked, even if it was clicked with the mouse
 * --- Parameter: ---
 * GtkWidget *entry: entry with value of chunk
 */
{
  g_return_if_fail ( entry ) ;
  g_return_if_fail ( GTK_IS_ENTRY(entry) ) ; 
/*  gtk_editable_select_region(GTK_EDITABLE(entry), 0, GTK_ENTRY(entry)->text_length); */
  gtk_editable_select_region(GTK_EDITABLE(entry), 0, -1);

  gtk_widget_show(entry);
}

static void EntryChanged(GtkWidget *entry, CHUNK *chunk)
/* Set edited chunk and flag to enable saving of image because of changes
 * --- Parameter: ---
 * GtkWidget *entry: entry with value of chunk, PICTURE or PAUL set as user data
 * CHUNK     *chunk: chunk to edit
 * --- Return: ---
 * CHUNK     *chunk: ->val.s with new allocated memory
 */
{
  PICTURE *bild = NULL;
  char    *buf;
   
  g_return_if_fail ( entry && chunk ) ;
  g_return_if_fail ( GTK_IS_ENTRY(entry) ) ; 
  if ( (bild = gtk_object_get_user_data(GTK_OBJECT(entry))) ) 
    g_return_if_fail ( IS_PICTURE(bild) );
  
  g_return_if_fail ( (buf = gtk_entry_get_text(GTK_ENTRY(entry))) );

  g_return_if_fail ( Char2Chunk(chunk, buf) != RET_ERR ) ;
 
  chunk->flag |= VALID;
  if ( bild ) bild->flag |= SAFEFLAG;
}

static void EntryActivated(GtkWidget *entry, CHUNK *chunk)
/* Set flags to notificate that the chunk was changed and is valid now
 * --- Parameter: ---
 * GtkWidget *entry: entry with value of chunk, PICTURE or PAUL set as user data
 * CHUNK     *chunk: chunk to edit
 */
{
  PICTURE *bild = NULL;
   
  g_return_if_fail ( entry && chunk ) ;
  g_return_if_fail ( GTK_IS_ENTRY(entry) ) ; 
  if ( (bild = gtk_object_get_user_data(GTK_OBJECT(entry))) ) 
    g_return_if_fail ( IS_PICTURE(bild) );
  
  chunk->flag |= VALID;
  if ( bild ) bild->flag |= SAFEFLAG;
}

static void TextChunkCell(GtkWidget *table, const char *value, int i, CHUNK *chunk)
/* Entrys in info table for text entrys
 * --- Parameter: ---
 * GtkWidget  *table : table to draw in
 * char       *value : text to print
 * int         i     : row number to draw in
 * CHUNK       chunk : chunk to edit if editable, ales NULL
 */
{
  GtkWidget      *entry;

  g_return_if_fail ( table ) ;
  g_return_if_fail ( GTK_IS_TABLE(table) ) ;
  
  entry = gtk_entry_new();
  if ( value ) gtk_entry_set_text(GTK_ENTRY(entry), value);
  if ( chunk ) {
    PICTURE *bild;
     
    if ( (bild = gtk_object_get_user_data(GTK_OBJECT(table))) ) {
      g_return_if_fail ( IS_PICTURE(bild) );
      gtk_object_set_user_data(GTK_OBJECT(entry), bild);
    }
    
    gtk_signal_connect(GTK_OBJECT(entry), "changed", GTK_SIGNAL_FUNC(EntryChanged), chunk);
    gtk_signal_connect(GTK_OBJECT(entry), "activate", GTK_SIGNAL_FUNC(EntryActivated), chunk);
    gtk_signal_connect(GTK_OBJECT(entry), "focus_in_event", GTK_SIGNAL_FUNC(FocusInEntry), entry);
    gtk_entry_set_editable(GTK_ENTRY(entry), TRUE);
    gtk_editable_select_region(GTK_EDITABLE(entry), 0, -1);
  } else {
    gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
  }

  gtk_table_attach(GTK_TABLE(table), entry, 1, 2, i, i+1, GTK_EXPAND|GTK_FILL, 0, 2, 0);
  gtk_widget_show(entry);
}

static void LongTextChunkCell(GtkWidget *table, const char *value, int i)
/* Entrys in info table for long text entrys
 * --- Parameter: ---
 * GtkWidget  *table : table to draw in
 * char       *value : text to print
 * int         i     : row number to draw in
 */
{
#define LINE_HEIGHT  17
#define LINE_LETTERS 65
  GtkWidget      *text;
  register  char *ap, *nlp;
  int             nl = 1;

  g_return_if_fail ( table ) ;
  g_return_if_fail ( GTK_IS_TABLE(table) ) ;
  
  text = gtk_text_new(NULL, NULL);
  gtk_text_set_editable(GTK_TEXT(text), FALSE);
  gtk_widget_show(text);
  gtk_table_attach(GTK_TABLE(table), text, 1, 2, i, i+1, GTK_EXPAND|GTK_FILL, 0, 2, 0);
  gtk_widget_realize(text); 

  gtk_text_set_word_wrap(GTK_TEXT(text), TRUE);
  gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL, value, -1);

  ap = (char *)value;
  while ( (nlp = strchr(ap, '\n')) ) {
    nl++;
    ap = ++nlp;
  }
  if ( nl == 1 && strlen(value) > LINE_LETTERS ) nl += strlen(value) / LINE_LETTERS;

  gtk_widget_set_usize(text, 400, nl * LINE_HEIGHT + 3); 
#undef LINE_HEIGHT
#undef LINE_LETTERS
}


static void InfoTableEntry(GtkWidget *table, CHUNK *chunk, int i)
/* Entrys in info table for different types
 * --- Parameter: ---
 * GtkWidget  *table : table to draw in
 * CHUNK      *chunk : data to print
 * int         i     : row number to draw in
 */
{
  char      *buf = NULL;

  TabLabel(table, chunk->key, i);

  switch ( chunk->typ ) {
  case CT_LTEXT: 
    LongTextChunkCell(table, chunk->val.s, i);
    break;
  case CT_INT:
    buf = g_strdup_printf("%i", chunk->val.i);
    TextChunkCell(table, buf, i, chunk);
    FREE(buf);
    break;
  case CT_DOUBLE:
    buf = g_strdup_printf("%g", chunk->val.d);
    TextChunkCell(table, buf, i, chunk);
    FREE(buf);
    break;
  case CT_STRING: 
    TextChunkCell(table, chunk->val.s, i, IsEditable(chunk) ? chunk : NULL );
    break;
  }
}

void InfoFrame(GtkWidget *child, PICTURE *bild)
/* Create frame with infos to a certain picture
 * --- Parameter: ---
 * PICTURE *bild         : structure with image data
 * --- Return: ---
 * GtkWidget *InfoFrame(): frame with image infos 
 */
{
  GtkWidget *table;
  char      *buf;
  CHUNK     *cp;
  int        i;

  g_return_if_fail ( child );
  g_return_if_fail ( GTK_IS_FRAME(child) );
  g_return_if_fail ( IS_PICTURE(bild) );
  
  gtk_container_set_border_width(GTK_CONTAINER(child), 10);
  gtk_widget_show(child);

  i = 2;
  if ( bild->zeit > 0 ) i++;
  i += NumValidChunks(bild->spec);

  table = gtk_table_new(i, 2, FALSE);
  gtk_container_add(GTK_CONTAINER(child), table);
  gtk_object_set_user_data(GTK_OBJECT(table), bild);

  i = 0;
  buf = g_strdup_printf("%i x %i = %i", bild->W, bild->H, bild->size);
  TabLabel(table, Imagesize, i);
  TextChunkCell(table, buf, i++, NULL);
  sprintf(buf, "%i", bild->spp);
  TabLabel(table, _("Samples per Pixel"), i);
  TextChunkCell(table, buf, i++, NULL);
  FREE(buf);
  if ( bild->zeit > 0 ) {
    TabLabel(table, _("Source date"), i);
    TextChunkCell(table, time_t2string(bild->zeit), i++, NULL);
  }

  for ( cp = bild->spec; cp->key != NULL; cp++ ) 
    if ( IsValid(cp) || IsEditable(cp) ) InfoTableEntry(table, cp, i++);
}

#ifdef PHYSICAL_PAUL
static void PhysicalInitShow(GtkWidget *button, PHYS_GLOBAL **pg)
/* Initialisation of global physical data after user request
 * --- Parameter: ---
 * GtkWidget    *button : request for editing global physical data button
 * PHYS_GLOBAL **pg     : pointer to global physical data structure which has to
 *                        be empty (*pg = NULL) at this state
 */
{
  GtkWidget *box;
  CHUNK     *cp;
   
  g_return_if_fail ( button ) ;
  g_return_if_fail ( GTK_IS_BUTTON(button) ) ;
  g_return_if_fail ( (box = gtk_object_get_user_data(GTK_OBJECT(button))) ) ;
  g_return_if_fail ( GTK_IS_BOX(box) ) ;
  g_return_if_fail ( !*pg ) ;
  
  gtk_widget_destroy(button);
  button = NULL;

  *pg = PhysicalInit(NULL);
  for ( cp = PG->spec; cp->key != NULL; cp++ ) 
    cp->flag |= VALID;
  PhysicalInfoFrame(box, pg);
}


void PhysicalInfoFrame(GtkWidget *box, PHYS_GLOBAL **pg)
/* Add global physical informations to the info frame
 * --- Parameter: ---
 * GtkWidget    *box : box in which the notebook for each image is placed
 * PHYS_GLOBAL **pg  : pointer to the global physical info structure
 *                     If it points to NULL a Button will be created.. The callback
 *                     then creates an initial global physical data structure.
 */
{
  GtkWidget *frame, *table, *button, *pbox;
  int        i;
  CHUNK     *cp;
  char      *buf;

  g_return_if_fail( box );
  g_return_if_fail( GTK_IS_BOX(box) );

  if ( !*pg ) {
    button = gtk_button_new_with_label(_("Edit physical information"));
    gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(PhysicalInitShow), pg);
    gtk_signal_connect(GTK_OBJECT(button), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), 
                      &button);
    gtk_object_set_user_data(GTK_OBJECT(button), box);
    gtk_box_pack_start(GTK_BOX(box), button, FALSE, FALSE, 5);
    GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
    gtk_widget_show(button);
    
    return;
  }
  
  pbox  = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box), pbox, FALSE, FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(pbox), 5);
  
  frame = gtk_frame_new(NULL);
  gtk_container_add(GTK_CONTAINER(pbox), frame);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);

  pbox  = gtk_hbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(frame), pbox);
  gtk_container_set_border_width(GTK_CONTAINER(pbox), 10);

  if ( (*pg)->file ) 
    buf = g_strdup_printf("%s of %s", Physical, (*pg)->file);
  else
    buf = g_strdup(Physical);
  frame = gtk_frame_new(buf);
  FREE(buf);
  
  gtk_container_add(GTK_CONTAINER(pbox), frame);

  pbox  = gtk_hbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(frame), pbox);
  gtk_container_set_border_width(GTK_CONTAINER(pbox), 10);

  i = 5 + SpecIsValid((*pg)->spec, PH_probe) + SpecIsValid((*pg)->spec, PH_date);
  table = gtk_table_new(i, 2, FALSE);
  gtk_container_add(GTK_CONTAINER(pbox), table);

  i = 0;
  for ( cp = PG->spec; cp->key != NULL; cp++ ) 
    if ( IsValid(cp) ) InfoTableEntry(table, cp, i++);

  gtk_widget_show_all(box);
  
}
#endif

