/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef PATCHBAYAREA_H
#define PATCHBAYAREA_H

#include <string>
#include <libgnomecanvasmm.h>
#include "Connection.h"
#include "Module.h"

using std::string;


/** Graphical patch bay widget with modules, ports, connections, etc.
 *
 * \ingroup Canvas
 */
namespace PatchBay {
	
class Port;
class Module;


/** \defgroup Canvas Modular patch bay canvas widget.
 *
 * A generic patch bay widget using libgnomecanvas.  It's used by Om and
 * Patchage, but could be used by anyone.
 */


/** The Patch Bay canvas widget.
 *
 * Applications must override some virtual methods to make the widget actually
 * do anything (ie connect).
 *
 * \ingroup Canvas
 */
class PatchBayArea : public Gnome::Canvas::Canvas
{
public:
	PatchBayArea(double width, double height);
	virtual	~PatchBayArea();

	void destroy();
	
	void add_module(Module* m);
	void remove_module(const string& name);
		
	void add_connection(const string& mod1_name, const string& port1_name, const string& mod2_name, const string& port2_name);
	bool remove_connection(const string& mod1_name, const string& port1_name, const string& mod2_name, const string& port2_name);
	
	Connection* add_connection(Port* port1, Port* port2);

	ModuleMap& modules() { return m_modules; }

	Module* find_module(const string& name);
	Port*   find_port(const string& module_name, const string& port_name);
	
	void set_default_placement(Module* m);
	
	float zoom() { return m_zoom; }
	void  zoom(float pix_per_unit);
	
	double width() const  { return m_width; }
	double height() const { return m_height; }

	bool on_expose_event(GdkEventExpose* ev);

	// Event handlers
	virtual bool port_event(GdkEvent* event, Port* port);
	//bool canvas_event(GdkEvent* event);
	virtual bool connection_drag_handler(GdkEvent* event);
	//bool scroll_drag_handler(GdkEvent* event);
	virtual void port_menu_disconnect_all();
	
	// Don't call this anywhere except in the gtk thread!
	void        do_zoom();

protected:
	
	/** Make a connection.  Should be overridden by an implementation to do something. */
	virtual void connect(const Port* const port1, const Port* const port2) = 0;
	
	/** Disconnect two ports.  Should be overridden by an implementation to do something */
	virtual void disconnect(const Port* const port1, const Port* const port2) = 0;

//private:
	Connection* get_connection(const Port* port1, const Port* port2);
	void        remove_connection(Connection* c);
	bool        remove_connection(Port* port1, Port* port2);
	void        active_port(Port* p);
	Port*       active_port() { return m_active_port; }
	bool        are_connected(const Port* port1, const Port* port2);
	Port*       get_port_at(double x, double y);

	void ports_joined(Port* port1, Port* port2);
	
	ModuleMap		m_modules;
	ConnectionList	m_connections;
	Port*			m_active_port;
	Port*           m_clicked_port;

	//Gdk::Cursor*      m_invisible_cursor;

	Gnome::Canvas::Rect m_base_rect;

	bool            m_connection_dragging;
	bool            m_scroll_dragging;
	Connection*     m_drag_connection;

	float  m_current_zoom; // actual current zoom level
	float  m_zoom;         // zoom slider zoom level, could be different if not updated yet
	double m_width;
	double m_height;

/* FIXME
	// Scroll offsets at beginning of scroll drag event
	int    m_scroll_offset_x;
	int    m_scroll_offset_y;

	// Origin where scroll drag event started
	double m_scroll_origin_x;
	double m_scroll_origin_y;
*/
};


} // namespace PatchBay

#endif // PATCHBAYAREA_H
