/*
**  POP3Store.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#import <Pantomime/POP3Store.h>

#import <Pantomime/Constants.h>
#import <Pantomime/GSMD5.h>
#import <Pantomime/POP3Folder.h>
#import <Pantomime/TCPConnection.h>
#import <Pantomime/URLName.h>

@implementation POP3Store

//
// This method implements a part of the Service Protocol.
//
- (id) initWithName: (NSString *) theName
               port: (int) thePort
{
  NSString *aString;

  self = [super init];
  
  // We set the initial class properties
  [self setName: theName];
  [self setPort: thePort];
  [self setUseAPOP: NO];
  messagesHaveBeenPrefetched = NO;
  
  tcpConnection = [[TCPConnection alloc] initWithName: theName
					   port: thePort];

  if ( !tcpConnection )
    {
      AUTORELEASE(self);
      return nil;
    }
  
  pop3Folder = [[POP3Folder alloc] initWithName: @"Inbox"];
  [pop3Folder setStore: (Store*)self];

  if ( [self responseFromServerIsValid: &aString] )
    {
      NSRange range1, range2;
      NSLog(@"POP3Store: Connected!");

      range1 = [aString rangeOfString: @"<"];
      range2 = [aString rangeOfString: @">"];

      if (range1.length && range2.length)
	{
	  [self setTimestamp: [aString substringWithRange: NSMakeRange(range1.location, 
								       range2.location - range1.location + 1)] ];
	}
      else
	{
	  [self setTimestamp: nil];
	}
    }
  else
    {
      AUTORELEASE(self);
      NSLog(@"POP3Store: Not connected!");
      return nil;      
    }

  return self;
}

//
// This method provide a default init method using
// the default POP3 port - 110.
// 
- (id) initWithName: (NSString *) theName
{
  return [self initWithName: theName
	       port: 110];
}


//
//
//
- (id) initWithURL: (NSString *) theURL
{
  URLName *urlName;
  
  urlName = [[URLName alloc] initWithString: theURL];
  
  self = [self initWithName: [urlName host]
	       port: 110];

  RELEASE(urlName);
  
  return self;
}


//
//
//
- (void) dealloc
{
  RELEASE(pop3Folder);
  RELEASE(name);
  
  TEST_RELEASE(timestamp);
  TEST_RELEASE(tcpConnection);
  
  [super dealloc];
}

//
// This method authenticates the Store to the POP3 server.
// In case of an error, it returns NO.
//
- (BOOL) authenticateWithUsername: (NSString*) username
			 password: (NSString*) password
{
  NS_DURING
    {
      // If we MUST use APOP
      if ( [self useAPOP] )
	{
	  if ( [self apopAuthenticateWithUsername: username
		     password: password] )
	    {
	      NS_VALUERETURN(YES, BOOL);
	    }
	}
      
      //
      // We aren't using APOP *OR* APOP authentication failed (since our server
      // migth not support APOP. 
      // Let's try the standard authentication mecanism.
      //
      [[self tcpConnection] writeLine: [NSString stringWithFormat: @"USER %@", username]];
      
      if ( ![self responseFromServerIsValid: NULL] )
	{  
	  // We MUST use APOP since the standard authentication doesn't work.
	  NS_VALUERETURN([self apopAuthenticateWithUsername: username
			       password: password], BOOL);
	}
      
      [[self tcpConnection] writeLine: [NSString stringWithFormat: @"PASS %@", password]];
      
      if ( ![self responseFromServerIsValid: NULL] )
	{
	  NSLog( [NSString stringWithFormat: @"POP3 password is invalid on %@.", [self name]] );
	  NS_VALUERETURN(NO,BOOL);
	}
      
      NS_VALUERETURN(YES,BOOL);
    }
  NS_HANDLER
    {
      NSLog(@"POP3Store: A timeout occured while authenticating with the POP3 server.");
    }
  NS_ENDHANDLER

  return NO;
}

- (BOOL) apopAuthenticateWithUsername: (NSString*) username
                             password: (NSString*) password
{
  NSString *aString;
  GSMD5 *md5;
  
  md5 = [[GSMD5 alloc] init];
  
  [md5 updateWithString: [NSString stringWithFormat: @"%@%@", [self timestamp], password]
       usingEncoding: NSASCIIStringEncoding];
  [md5 calculateDigest];
  
  aString = DataToHexString( [md5 digest] );

  [[self tcpConnection] writeLine: [NSString stringWithFormat: @"APOP %@ %@", username, aString]];
  
  RELEASE(md5);
  
  if ( [self responseFromServerIsValid: NULL] )
    {
      return YES;
    }
  
  NSLog(@"POP3Store: APOP authentication failed.");
  return NO;
}


- (BOOL) responseFromServerIsValid: (NSString **) theResponse;
{
  NS_DURING
    {
      NSString *aString;
      
      aString = [[self tcpConnection] readLine];
      
      if (theResponse != NULL)
	{
	  *theResponse = aString;
	}
      
      if ( aString && [[aString substringToIndex: 3] isEqualToString: @"+OK"] )
	{
	  NS_VALUERETURN(YES, BOOL);
	}
      else
	{
	  NS_VALUERETURN(NO, BOOL);
	}
    }
  NS_HANDLER
    {
      NSLog(@"POP3Store: A timeout occured while verifying the response code with the POP3 server.");
    }
  NS_ENDHANDLER

  return NO;
}

- (NSString *) name
{
  return name;
}

- (void) setName: (NSString *) theName
{
  RETAIN(theName);
  RELEASE(name);
  name = theName;
}

- (int) port
{
  return port;
}

- (void) setPort: (int) thePort
{
  port = thePort;
}

- (TCPConnection *) tcpConnection
{
  return tcpConnection;
}

//
// The default folder in POP3 is always Inbox. This method will prefetch
// the messages of a POP3 folder if they haven't been prefetched before.
//
- (id) defaultFolder
{
  if ( !messagesHaveBeenPrefetched )
    {
      messagesHaveBeenPrefetched = YES;
      
      return [self folderForName: @"Inbox"
		   prefetch: YES];
    }
  
  return nil;
}

//
// This method will always return nil if theName is not
// equal to Inbox (case-insensitive) since you cannot
// access an other mailbox (other than Inbox) using the
// POP3 protocol.
//
- (id) folderForName: (NSString *) theName
{

  if ( [theName caseInsensitiveCompare: @"Inbox"] == NSOrderedSame )
    {
      return [self defaultFolder];
    }
  
  return nil;
}


//
//
//
- (POP3Folder *) folderForName: (NSString *) theName
		      prefetch: (BOOL) aBOOL
{
  if ( aBOOL )
    {
      [pop3Folder prefetch];
    }
  
  return pop3Folder;
}

//
//
//
- (id) folderForURL: (NSString *) theURL
{
  return [self defaultFolder];
}


//
//
//
- (void) close
{
  NS_DURING
    {
      [[self tcpConnection] writeLine: @"QUIT"];
      [[self tcpConnection] close];
    }
  NS_HANDLER
    {
      NSLog(@"POP3Store: A timeout occured while closing the connection with the POP3 server.");
    }
  NS_ENDHANDLER
}

//
// In POP3, you're not allowed to create folders.
//
- (BOOL) createFolderWithName: (NSString *) theName
{
  return NO;
}


//
// In POP3, you're not allowed to delete a folder
//
- (BOOL) deleteFolderWithName: (NSString *) theName
{
  return NO;
}


//
//
//
- (NSString *) timestamp
{
  return timestamp;
}

- (void) setTimestamp: (NSString *) theTimestamp
{
  if ( theTimestamp )
    {
      RETAIN(theTimestamp);
      RELEASE(timestamp);
      timestamp = theTimestamp;
    }
  else
    {
      RELEASE(timestamp);
      timestamp = nil;
    }
}


//
//
//
- (BOOL) useAPOP
{
  return useAPOP;
}

- (void) setUseAPOP: (BOOL) aBOOL
{
  useAPOP = aBOOL;
}

@end
