/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <libgnomeui/libgnomeui.h>

#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-file.h>

#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/util.h>
#include <pan/dialogs/dialogs.h>
#include <pan/xpm/caution.xpm>

typedef struct
{
	GnomePixmap * error_pixmap;
	GtkWidget * window;        /* main window */
	GtkWidget * log;           /* GtkCList of log contents */
	GtkWidget * appbar;
	GtkWidget * file_entry;
}
LogViewer;

static void
log_close_button_cb (GtkWidget * widget, LogViewer * view)
{
	debug_enter ("log_close_button_cb");

	g_return_if_fail (view != NULL);
	gtk_widget_destroy (GTK_WIDGET(view->window));

	debug_exit ("log_close_button_cb");
}

static void
log_clear_cb (GtkWidget * widget, LogViewer * view)
{
	log_clear ();
}

static void log_save_cb (GtkWidget *widget, LogViewer *lv);
static void log_save_ok_clicked (GtkWidget *widget, LogViewer *lv);


static GnomeUIInfo log_file_menu[] =
{
	GNOMEUIINFO_MENU_SAVE_AS_ITEM (log_save_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_CLOSE_ITEM (log_close_button_cb, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo log_main_menu[] =
{
	GNOMEUIINFO_MENU_FILE_TREE (log_file_menu),
	GNOMEUIINFO_END
};

static GnomeUIInfo log_toolbar[] =
{
	GNOMEUIINFO_ITEM_STOCK (N_("Save"), N_("Save this log to a file."), 
				log_save_cb, GNOME_STOCK_PIXMAP_SAVE),
	GNOMEUIINFO_ITEM_STOCK (N_("Clear"), N_("Clear this log."),
				log_clear_cb, GNOME_STOCK_PIXMAP_CLEAR),
	GNOMEUIINFO_ITEM_STOCK (N_("Close"), N_("Close the Log Viewer."),
				log_close_button_cb, GNOME_STOCK_PIXMAP_CLOSE),
	GNOMEUIINFO_END

};


static void
log_save_cb (GtkWidget * widget, LogViewer * view)
{
	view->file_entry = gtk_file_selection_new (_("Save the log to file"));

	gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (view->file_entry));

	gtk_signal_connect_object (GTK_OBJECT (view->file_entry), "destroy",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (view->file_entry));
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (view->file_entry)->cancel_button),
				   "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (view->file_entry));
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (view->file_entry)->ok_button),
			    "clicked",
			    GTK_SIGNAL_FUNC (log_save_ok_clicked),
			    view);

	gtk_widget_show_all (view->file_entry);
}

static void
log_save_ok_clicked (GtkWidget * widget, LogViewer * view)
{
	const gchar *filename;
	FILE * fp = NULL;

	filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (view->file_entry));
	if (is_nonempty_string (filename))
		fp = fopen (filename, "w");

	if (fp != NULL)
	{
		guint i;
		GPtrArray * entries = log_get_entries ();
		for (i=0; i!=entries->len; ++i) {
			const LogEntry * entry = (const LogEntry*) g_ptr_array_index (entries, i);
			gchar date[256] = { '\0' };
			get_date_display_string (entry->date, body_date_format, date, sizeof(date));
			fprintf (fp, "%s - %s\n", date, entry->message);
		}
		fclose (fp);
	}

	gtk_widget_destroy (GTK_WIDGET(view->file_entry));
	view->file_entry = NULL;
}




static void
log_viewer_add_entry_nolock (LogViewer * view, LogEntry * entry)
{
	int row;
	gchar buf[256];
	gchar * data [3];

	g_return_if_fail (entry != NULL);
	g_return_if_fail (view != NULL);

	get_date_display_string (entry->date, body_date_format, buf, sizeof(buf));

	data[0] = "";
	data[1] = buf;
	data[2] = entry->message;
	row = gtk_clist_prepend (GTK_CLIST(view->log), data);
	if (entry->severity == LOG_ERROR)
		gtk_clist_set_pixmap (GTK_CLIST(view->log), row, 0,
		view->error_pixmap->pixmap,
		view->error_pixmap->mask);
}

static void
log_viewer_add_entry (LogViewer * view, LogEntry * entry)
{
	g_return_if_fail (entry != NULL);
	g_return_if_fail (view != NULL);

	pan_lock ();
	log_viewer_add_entry_nolock (view, entry);
	pan_unlock ();
}

static void
log_list_appended_cb (gpointer obj, gpointer arg, gpointer data)
{
	LogEntry * entry = (LogEntry*) obj;
	LogViewer * view = (LogViewer*) data;
	debug_enter ("log_list_appended_cb");

	log_viewer_add_entry (view, entry);

	debug_exit ("log_list_appended_cb");
}

static void
log_viewer_rebuild_ui (LogViewer * view)
{
	gint i;
	GPtrArray * entries = log_get_entries ();
	GtkCList * clist = GTK_CLIST(view->log);
	debug_enter ("log_viewer_rebuild_ui");

	pan_lock ();
	gtk_clist_freeze (clist);
	gtk_clist_clear (clist);
	for (i=0; i!=entries->len; ++i)
		log_viewer_add_entry_nolock (view, (LogEntry*)g_ptr_array_index(entries,i));
	gtk_clist_thaw (clist);
	pan_unlock ();

	debug_exit ("log_viewer_rebuild_ui");
}

static void
log_list_changed_cb (gpointer obj, gpointer arg, gpointer data)
{
	LogViewer * view = (LogViewer*) data;
	log_viewer_rebuild_ui (view);
}

static gint
log_delete_cb (GtkWidget * w, GdkEvent * e, gpointer data)
{
	LogViewer * view = (LogViewer*) data;
	gui_save_window_size (view->window, "log_dialog_3");
	gui_save_column_widths (view->log, "log_dialog_3");
	return FALSE;
}    
static void
log_destroy_cb (GtkWidget * widget, LogViewer * view)
{
	debug_enter ("log_destroy_cb");
	g_return_if_fail (view != NULL);

	pan_callback_remove (log_get_entry_list_changed_callback(),
	                     log_list_changed_cb,
	                     view);
	pan_callback_remove (log_get_entry_added_callback(),
	                     log_list_appended_cb,
	                     view);
	gtk_widget_destroy (GTK_WIDGET(view->error_pixmap));
	g_free (view);

	debug_exit ("log_destroy_cb");
}


static void
dialog_log_viewer_create (LogViewer * view)
{
	GnomeApp * app;
	GtkWidget * scrolled_window;

	view->window = gnome_app_new ("Log", _("Pan - Log Viewer"));
	app = GNOME_APP(view->window);

	view->error_pixmap = (GnomePixmap*) gnome_pixmap_new_from_xpm_d (caution_xpm);

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);

	view->appbar = gnome_appbar_new (FALSE, TRUE, GNOME_PREFERENCES_USER);
	gnome_app_set_statusbar (app, GTK_WIDGET (view->appbar));

	gnome_app_set_contents (app, scrolled_window);
	gnome_app_create_menus_with_data (app, log_main_menu, view);

	gnome_app_create_toolbar_with_data (app, log_toolbar, view);
	gnome_app_install_menu_hints (app, log_main_menu);

	gtk_signal_connect (GTK_OBJECT(view->window), "delete_event",
	                    GTK_SIGNAL_FUNC(log_delete_cb), view);                                               
	gtk_signal_connect (GTK_OBJECT (view->window), "destroy",
	                    GTK_SIGNAL_FUNC (log_destroy_cb), view);

	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);

	gtk_container_add (GTK_CONTAINER (scrolled_window), view->log);
}


void
dialog_log_viewer (void)
{
	GtkCList * clist;
	LogViewer * view;

	gchar * titles[3];
	titles[0] = "";
	titles[1] = _("Time");
	titles[2] = _("Log Entry");

	view = g_new0 (LogViewer, 1);
	view->log = gtk_clist_new_with_titles (3, titles);
	clist = GTK_CLIST(view->log);
	gtk_clist_set_column_width (clist, 0, 20);
	gtk_clist_set_column_width (clist, 1, 150);
	gtk_clist_set_column_width (clist, 2, 1500);
	gui_restore_column_widths (view->log, "log_dialog_3");

	dialog_log_viewer_create (view);
	pan_callback_add (log_get_entry_list_changed_callback(), log_list_changed_cb, view);
	pan_callback_add (log_get_entry_added_callback(), log_list_appended_cb, view);
	log_viewer_rebuild_ui (view);

	gtk_window_set_default_size (GTK_WINDOW (view->window), 500, 336);
	gui_restore_window_size (view->window, "log_dialog_3");
	gtk_widget_show_all (view->window);
}
