/*
 * Copyright (c) 2001, 2002 SuSE GmbH Nuernberg, Germany.
 * Author: Thorsten Kukuk <kukuk@suse.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#define _GNU_SOURCE

#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif

#include <grp.h>
#include <pwd.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <dirent.h>
#include <sys/stat.h>

#include "common.h"

int
login_protect (const char *path, int mask, uid_t uid, gid_t gid, FILE *fp)
{
  size_t pathlen = strlen (path);

  if (path[pathlen - 2] == '/' && path[pathlen - 1] == '*')
    {
      char *new_path = alloca (pathlen);
      DIR *dir;

      strcpy (new_path, path);
      new_path[pathlen - 1] = 0;

      if ((dir = opendir (new_path)) == 0)
	{
	  __log_err (LOG_ERR, "opendir(%s): %s", path, strerror (errno));
        }
      else
	{
	  struct dirent *ent;

	  while ((ent = readdir (dir)) != 0)
	    {
	      if (strcmp(ent->d_name, ".") != 0
		  && strcmp(ent->d_name, "..") != 0)
		{
		  char *cp;
		  char *buf = alloca (pathlen +
				      strlen (ent->d_name) + 1);

		  cp = stpcpy (buf, new_path);
		  strcpy (cp, ent->d_name);

		  login_protect (buf, mask, uid, gid, fp);
                }
            }
	  closedir (dir);
        }
    }
  else
    {
      if (fp != NULL)
	{
	  struct stat sbuf;

	  if (stat (path, &sbuf) == 0)
	    fprintf (fp, "%s %#o %lu %lu\n", path, sbuf.st_mode & 07777,
		     (unsigned long)sbuf.st_uid, (unsigned long)sbuf.st_gid);
	}
      if (chmod (path, mask) && errno != ENOENT)
	__log_err (LOG_ERR, "chmod(%s): %s", path, strerror (errno));
#if 0
      if (chown (path, uid, gid) && errno != ENOENT)
	__log_err (LOG_ERR, "chown(%s): %s", path, strerror (errno));
#else
      /* Don't change the group id, so that the sysadmin can use the
	 group to allow access to more user.  */
      if (chown (path, uid, -1) && errno != ENOENT)
	__log_err (LOG_ERR, "chown(%s): %s", path, strerror (errno));
#endif
    }
  return 0;
}
