//
// C++ Interface: simplescorecalculationstrategy
//
// Description: 
//
//
// Author: Benjamin Mesing <bensmail@gmx.net>, (C) 2005
//
// Copyright: See COPYING file that comes with this distribution
//
//
#ifndef __NAPT_SIMPLESCORECALCULATIONSTRATEGY_H_2005_07_31
#define __NAPT_SIMPLESCORECALCULATIONSTRATEGY_H_2005_07_31

#include "aptsearchscorecalculationstrategy.h"

namespace NApt 
{

class IPackageDB;

/** The scores are calculated for each package by:
	* \f[ 0.5 \cdot \frac{incPatNameFound}{incPatNum} + 0.5 \cdot \frac{incPatDescFound}{maxIncPatFound} \f] 
	* where 
	* <ul>
	* 	<li><em>incPatNameFound</em> is the number of include patterns that were found in the 
	*	package name (no duplicates counted)</li>
	* 	<li><em>incPatNum</em> is #_includePatterns .size()</li>
	* 	<li><em>incPatDescFound</em> is the number of include patterns found in the package
	*	description (duplicates counted)</li>
	* 	<li><em>maxIncPatFound</em> is the the maximum of incPatDescFound from all packages in 
	*	<i>packages</i>.</li>
	* </ul>
	* 
	* @author Benjamin Mesing
	*/
class SimpleScoreCalculationStrategy : public AptSearchScoreCalculationStrategy
{
	/** @brief Holds the interface used to access the package database directly. */
	IPackageDB* _pPackageDb;
public:
	SimpleScoreCalculationStrategy(IPackageDB* pPackageDb);
	~SimpleScoreCalculationStrategy();

	/** @name IScoreCalculationStrategy interface
	  * 
	  * Implementation of the IScoreCalculationStrategy interface
	  */
	//@{
	virtual void calculateScore(const set<string>& packages);
	//@}
protected:
	/** Information used to calculate the scores for a package. */
	struct ScoreInformation
	{
	public:
		ScoreInformation(const string& package)	
		{
			_package = package;
			_descriptionCount = 0;
			_nameCount = 0;
		}
		
		const string& package() const	{ return _package; }
		void incrementNameCount()	{ ++ _nameCount; };
		void addDescriptionCount(int count)
		{
			_descriptionCount += count; 
			if ( _descriptionCount > _maximumDescriptionCount)
			{
				_maximumDescriptionCount = _descriptionCount;
			}
		}
		void incrementDescriptionCount()	
		{ 
			++_descriptionCount; 
			if ( _descriptionCount > _maximumDescriptionCount)
			{
				_maximumDescriptionCount = _descriptionCount;
			}
		};
		int getNameCount() const	{ return _nameCount; }
		int getDescriptionCount() const	{ return _descriptionCount; }
		static void clearMaximumDescriptionCount()	{ _maximumDescriptionCount = 0; }
		static int maximumDescriptionCount()	{ return _maximumDescriptionCount; }
	private:
		/** @brief Holds the maximum description count that was reached. */
		static int _maximumDescriptionCount;
		string _package;
		/** @brief Used to count the occurences of the search patterns in the description */
		int _descriptionCount;
		/** @brief Used to count the number of search pattern occuring in the name. */
		int _nameCount;
	};

	
	/** @brief Returns the score information for the handed package. 
	  *
	  * @throws NPlugin::PackageNotFoundException if no data was available for the handed package.
	  */
	ScoreInformation getScoreInformation(const string& package, bool cs) const;

};

}

#endif	// __NAPT_SIMPLESCORECALCULATIONSTRATEGY_H_2005_07_31
