#include <sstream>
#include <iostream>

#include <qlistbox.h>
#include <qpoint.h>
#include <qpopupmenu.h>
#include <qlayout.h>
#include <qmessagebox.h>
#include <qcheckbox.h>
#include <qlabel.h>

#include <ExpressionFilter.h>
#include <FilterChain.h>
#include <TagcollBuilder.h>
#include <Exception.h>

#include <helpers.h>
#include <exception.h>


#include "debtagsplugin.h"

#include "stringtagcollbuilder.h"
#include "iprovider.h"

#include "selectioninputanddisplay.h"
#include "tagselectionwidget.h"
#include "choosentagsdisplay.h"


namespace NPlugin
{

/////////////////////////////////////////////////////
// Constructors/ Destructors
/////////////////////////////////////////////////////

DebtagsPlugin::DebtagsPlugin(const DebtagsPluginContainer& container) :
	_container(container)
{ 
	_pChooserWidget = 0;
	_pTagsDisplay = 0;
	_pHandleMaker = 0;
	_pProvider = 0;
	_pIncludeSelection = 0;
	_pExcludeSelection = 0;
	_isInactive = true;
}

DebtagsPlugin::~DebtagsPlugin() 
{ 
	delete _pChooserWidget;
	delete _pTagsDisplay;
	delete _pIncludeSelection;
	delete _pExcludeSelection;
}

/////////////////////////////////////////////////////
// Plugin Interface
/////////////////////////////////////////////////////

void DebtagsPlugin::init(IProvider* pProvider)
{
	QMainWindow* pWindow = pProvider->mainWindow();
	_pProvider = pProvider;
	_pHandleMaker = &pProvider->handleMaker();

	_pChooserWidget = new TagChooserWidget(pWindow, "TagChooserWidget");
	_pTagsDisplay = new ChoosenTagsDisplay(pWindow, "ChoosenTagsDisplay");
	
	// this will be garbage collected if the parent _pChooserWidget will be deleted
	NWidgets::TagSelectionWidget* pIncludeSelectionWidget = 
		new NWidgets::TagSelectionWidget(_pChooserWidget, "IncludeSelection");
	NWidgets::TagSelectionWidget* pExcludeSelectionWidget = 
		new NWidgets::TagSelectionWidget(_pChooserWidget, "ExcludeSelection");
	
	_pTagsDisplay->show();
	// currently disabled as the exclude tag option is not too useful and wastes some space
	_pChooserWidget->_pExcludeTagsCheck->setShown(false);
	// assume that the vocabulary is not accessible if one of the pointers is not set
	bool debtagsEnabled = (_container.collection() != 0);
	_pIncludeSelection = new NWidgets::SelectionInputAndDisplay
	(
		pWindow,
		"IncludeSelection",
		pIncludeSelectionWidget,
		_pTagsDisplay->_pIncludeTagsView,
		_pTagsDisplay->_pIncludeViewLabel
	);
	_pIncludeSelection->loadVocabulary(_container.facets());
	_pExcludeSelection = new NWidgets::SelectionInputAndDisplay
	(
		pWindow,
		"ExcludeSelection",
		pExcludeSelectionWidget,
		_pTagsDisplay->_pExcludeTagsView,
		_pTagsDisplay->_pExcludeViewLabel
	);
	_pExcludeSelection->loadVocabulary(_container.facets());
	static_cast<QVBoxLayout*>
		(_pChooserWidget->layout())->insertWidget(1, _pIncludeSelection->tagSelectionWidget());
	static_cast<QVBoxLayout*>
		(_pChooserWidget->layout())->insertWidget(4, _pExcludeSelection->tagSelectionWidget());
	connect(
		_pIncludeSelection->tagSelectionListView(), SIGNAL(tagItemsSelected(const set<TagItem*>&)),
		this, SLOT(evaluateSearch()) 
	);
	connect(
		_pExcludeSelection->tagSelectionListView(), SIGNAL(tagItemsSelected(const set<TagItem*>&)),
		this, SLOT(evaluateSearch()) 
	);
	

	_pIncludeSelection->tagSelectionListView()->setColl(_container.collection());
	showExcludeWidgets(false);	// hide the exclude selection part by default
	connect(_pChooserWidget->_pExcludeTagsCheck, SIGNAL(toggled(bool)), this, SLOT(showExcludeWidgets(bool)));
	
	if ( _container.collection()==0 )
		setWidgetsEnabled(false);
}


QString DebtagsPlugin::title() const 
{ 
	return QString("Debtags Plugin"); 
}

QString DebtagsPlugin::briefDescription() const 
{ 
	return QString("Offers information and search using the debtags system"); 
}

QString DebtagsPlugin::description() const 
{ 
	return QString("This plugin shows the tags for a program in the detailed view.\n"
		"It also offers searching by tags that can be selected from a list."); 
}

//
// Search Plugin Interface 
//

QWidget* DebtagsPlugin::shortInputAndFeedbackWidget() const
{
	return _pTagsDisplay;
}

const Tagcoll::OpSet<int>& DebtagsPlugin::searchResult() const
{ 
	return _searchResult; 
};

void DebtagsPlugin::clearSearch()
{
	_pIncludeSelection->tagSelectionListView()->deselectAll();
	_pExcludeSelection->tagSelectionListView()->deselectAll();
}

//
// Information Plugin Interface 
//

QString DebtagsPlugin::informationText(int packageID)
{
	{	// add the tags of the package to the description
		Tagcoll::OpSet<string> tagset = _container.collection()->getTagsetForItem(packageID);
		if (tagset.empty()) return _emptyString;
		QString detailsString = "<b>Tags:</b> ";
		for (Tagcoll::OpSet<string>::iterator it = tagset.begin(); ; )
		{
			detailsString += toQString( *it );
			if ( ++it == tagset.end() )
			{
				detailsString.append("\n");
				break;
			}
			detailsString += ", ";
		}
		return detailsString;
	}
}


/////////////////////////////////////////////////////
// Helper Methods
/////////////////////////////////////////////////////

void DebtagsPlugin::showExcludeWidgets( bool display )
{
	_pExcludeSelection->setShown(display);
	_pChooserWidget->_pExcludeInputLabel->setShown(display);
}


/** This creates a search expression which can be used as input for an ExpressionFiler.
  * @returns the filter string
  */
std::string DebtagsPlugin::createSearchExpression()
{
	ostringstream oexpr;
	// the list view that displays the selected tags
	QListBox* pTagsView = _pTagsDisplay->_pIncludeTagsView;
	for (uint i = 0; i < pTagsView->count(); ++i)
	{
		oexpr << pTagsView->text(i);
		// if we do not have the last item or we have tags to exclude
		if ( i+1< pTagsView->count() || _pTagsDisplay->_pExcludeTagsView->count()!=0)	
			oexpr << " && ";
	}
	// the list view that displays the selected tags
	pTagsView = _pTagsDisplay->_pExcludeTagsView;
	for (uint i = 0; i < pTagsView->count(); ++i)
	{
		oexpr << "!" << pTagsView->text(i);
		if ( i+1<pTagsView->count())	// if we do not have the last item 
			oexpr << " && ";
	}
	return oexpr.str();	
}


void DebtagsPlugin::evaluateSearch()
{
	_pProvider->reportBusy(this, tr("Performing full text search on Package Database"));
//	statusBar()->message(tr("Searching Package Database for tags"));
	_searchResult.clear();
	string expression = createSearchExpression();
	if (expression.empty())	// if nothing is selected, do not show anything
	{
		_isInactive = true;
	}
	else
	{
		_isInactive = false;
		if (_pTagsDisplay->_pExcludeTagsView->count()!=0)	// if we want to exclude some tags
		{
			// send the package collection through the filter into the result
			// and than copy the result to the _searchResult
			Tagcoll::FilterChain<int, string> filters;
			#warning this must be fixed!!
			#warning this must be fixed!!
			#warning this must be fixed!!
			/// @todo fix the Expression Filter (hopefully Enrico will make it a template)
/*			Tagcoll::ExpressionFilter filter;
			if ( !filter.setExpression(expression) )	// no error should occur!
				cerr << "Programmers error please report to bensmail@gmx.net: Error in tag expression "
					<< endl;
			filters.appendFilter(&filter);
			Tagcoll::TagCollection result;
			filters.setConsumer(&result);
			_container.collection()->output(filters);
			_searchResult = result.getAllItems();*/
		}
		else	// do not use TagcollFilter
		{
			QListBox* pIncludeTagsView = _pTagsDisplay->_pIncludeTagsView;
			Tagcoll::OpSet<string> ts;
			for (uint i = 0; i < pIncludeTagsView->count(); ++i)
			{
				ts.insert( toString(pIncludeTagsView->text(i)) );
			}
			_searchResult = _container.collection()->getCompanionItems(ts);
		}
	}
	_pProvider->reportReady(this);
	emit searchChanged(this);
}


void DebtagsPlugin::debtagsDataChanged()
{
	clearSearch();
	if (_container.collection()==0)
		setWidgetsEnabled(false);
	else
	{
		_pIncludeSelection->loadVocabulary(_container.facets());
		_pExcludeSelection->loadVocabulary(_container.facets());
		setWidgetsEnabled(true);
	}
//	clearSearch();
	_pIncludeSelection->tagSelectionListView()->setColl(_container.collection());
}

void DebtagsPlugin::setWidgetsEnabled(bool enabled)
{
	_pChooserWidget->setEnabled(enabled);
	_pTagsDisplay->setEnabled(enabled);
}



}	// namespace NPlugin

