
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: scheduler.c,v 1.7 2006/01/21 17:53:37 mschwerin Exp $
 *
 */
#include "config.h"

#include <errno.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>

#include "i18n.h"
#include "heap.h"
#include "list.h"
#include "logger.h"
#include "scheduler.h"

typedef struct {
    time_t start_time;
    l_list_t *jobs;
    pthread_t scheduler_thread;
    pthread_mutex_t jobs_mutex;
    pthread_mutex_t job_execution_mutex;
    pthread_mutex_t wait_mutex;
    pthread_cond_t jobs_reorganize;
    int job_id;
    int keep_going;
} ox_scheduler_t;

typedef struct {
    struct timespec ts;         /* when to do */
    void (*cb) (void *data);
    void *data;
    int id;
    int priority;
} job_t;

static ox_scheduler_t *ox_scheduler = NULL;

static void *
scheduler_thread (void *data)
{
    job_t *job;
    int ret;
    void (*cb) (void *data);
    void *cb_data;

    if (!ox_scheduler) {
        pthread_exit (NULL);
        return NULL;
    }
    pthread_mutex_lock (&ox_scheduler->wait_mutex);

    while (ox_scheduler->keep_going) {

        pthread_mutex_lock (&ox_scheduler->jobs_mutex);
        job = l_list_last (ox_scheduler->jobs);

        if (!job) {             /* no jobs for me */
            pthread_mutex_unlock (&ox_scheduler->jobs_mutex);
            pthread_cond_wait (&ox_scheduler->jobs_reorganize,
                               &ox_scheduler->wait_mutex);
            continue;
        } else {
            pthread_mutex_unlock (&ox_scheduler->jobs_mutex);
        }

        ret = pthread_cond_timedwait (&ox_scheduler->jobs_reorganize,
                                      &ox_scheduler->wait_mutex, &job->ts);

        if (ret == ETIMEDOUT) {
            /* lets do a job */
            pthread_mutex_lock (&ox_scheduler->job_execution_mutex);
            pthread_mutex_lock (&ox_scheduler->jobs_mutex);
            job = l_list_last (ox_scheduler->jobs);
            cb = NULL;
            cb_data = NULL;
            if (job) {
                cb = job->cb;
                cb_data = job->data;
            }
            l_list_remove (ox_scheduler->jobs, job);
            ho_free (job);
            pthread_mutex_unlock (&ox_scheduler->jobs_mutex);

            if (cb)
                cb (cb_data);

            pthread_mutex_unlock (&ox_scheduler->job_execution_mutex);
        }
    }
    pthread_mutex_unlock (&ox_scheduler->wait_mutex);
    pthread_exit (NULL);
}

void
start_scheduler (void)
{
    struct timeval tv;
    struct timezone tz;

    ox_scheduler = ho_new (ox_scheduler_t);

    gettimeofday (&tv, &tz);
    ox_scheduler->start_time = tv.tv_sec;

    ox_scheduler->jobs = l_list_new ();
    pthread_mutex_init (&ox_scheduler->wait_mutex, NULL);
    pthread_mutex_init (&ox_scheduler->jobs_mutex, NULL);
    pthread_mutex_init (&ox_scheduler->job_execution_mutex, NULL);
    pthread_cond_init (&ox_scheduler->jobs_reorganize, NULL);
    ox_scheduler->job_id = 0;
    ox_scheduler->keep_going = 1;

    if (pthread_create (&ox_scheduler->scheduler_thread,
                        NULL, scheduler_thread, ox_scheduler) != 0) {
        fatal (_("Could not create scheduler thread: %s!"), strerror (errno));
        abort ();
    }
}

static void
job_free_cb (void *j1)
{
    ho_free (j1);
}

void
stop_scheduler (void)
{
    void *ret = NULL;

    if (!ox_scheduler)
        return;

    ox_scheduler->keep_going = 0;
    pthread_cond_signal (&ox_scheduler->jobs_reorganize);

    pthread_join (ox_scheduler->scheduler_thread, ret);

    l_list_free (ox_scheduler->jobs, job_free_cb);

    pthread_mutex_destroy (&ox_scheduler->wait_mutex);
    pthread_mutex_destroy (&ox_scheduler->jobs_mutex);
    pthread_cond_destroy (&ox_scheduler->jobs_reorganize);

    ho_free (ox_scheduler);
    ox_scheduler = NULL;

    debug ("Successfully stopped scheduler thread.");
}

static int
job_swap_cb (void *j1, void *j2)
{
    return ((job_t *) j1)->priority < ((job_t *) j2)->priority;
}

int
schedule_job (int delay, void (*cb) (void *data), void *data)
{

    struct timeval tv;
    struct timezone tz;
    job_t *job = ho_new (job_t);
    int msec;

    if (!ox_scheduler)
        return -1;

    gettimeofday (&tv, &tz);

    job->ts.tv_sec = (delay / 1000) + tv.tv_sec;
    msec = delay % 1000;
    if ((msec + tv.tv_usec / 1000) >= 1000)
        job->ts.tv_sec++;
    msec = (msec + tv.tv_usec / 1000) % 1000;
    job->ts.tv_nsec = msec * 1000000;

    job->cb = cb;
    job->data = data;
    job->id = ++ox_scheduler->job_id;
    job->priority = (job->ts.tv_sec - ox_scheduler->start_time) * 1000 + msec;

    pthread_mutex_lock (&ox_scheduler->jobs_mutex);
    l_list_append (ox_scheduler->jobs, job);
    l_list_sort (ox_scheduler->jobs, job_swap_cb);
    pthread_mutex_unlock (&ox_scheduler->jobs_mutex);

    pthread_cond_signal (&ox_scheduler->jobs_reorganize);

    return ox_scheduler->job_id;
}

void
cancel_job (int job_id)
{
    job_t *job;

    if (!ox_scheduler)
        return;

    pthread_mutex_lock (&ox_scheduler->jobs_mutex);
    job = l_list_first (ox_scheduler->jobs);
    while (job) {
        if (job->id == job_id) {
            l_list_remove (ox_scheduler->jobs, job);
            ho_free (job);
            break;
        }
        job = l_list_next (ox_scheduler->jobs, job);
    }
    pthread_mutex_unlock (&ox_scheduler->jobs_mutex);
    pthread_cond_signal (&ox_scheduler->jobs_reorganize);
}

void
lock_job_mutex (void)
{
    if (!ox_scheduler)
        return;
    pthread_mutex_lock (&ox_scheduler->job_execution_mutex);
}

void
unlock_job_mutex (void)
{
    if (!ox_scheduler)
        return;
    pthread_mutex_unlock (&ox_scheduler->job_execution_mutex);
}
