
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: playing_menu.c,v 1.54 2006/01/17 13:55:53 mschwerin Exp $
 *
 */
#include "config.h"

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>

#include "codeset.h"
#include "dvd_menu.h"
#include "environment.h"
#include "gui_utils.h"
#include "heap.h"
#include "i18n.h"
#include "lang.h"
#include "logger.h"
#include "main_menu.h"
#include "meta_info.h"
#include "oxine.h"
#include "playing_menu.h"
#include "playlist_menu.h"

static int playing_menu_show_file_info = 1;


/*
 * **************************************************************************
 * Callbacks of the stream parameters menu
 * **************************************************************************
 */
static void
update_param_label (void *oxine_p, otk_widget_t * label,
                    odk_stream_param_t param)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    char *param_string = odk_get_stream_param_as_string (oxine->odk, param);
    otk_label_set_text (label, param_string);
    ho_free (param_string);
}

static void
volume_mute (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk, ODK_PARAM_AUDIO_MUTE, 1, 0, 1);
}

static void
volume_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk, ODK_PARAM_AUDIO_VOLUME, 5, 0, 100);
}

static void
volume_upcall (void *oxine_p, otk_widget_t * label)
{
    update_param_label (oxine_p, label, ODK_PARAM_AUDIO_VOLUME);
}

static void
volume_minus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk, ODK_PARAM_AUDIO_VOLUME, -5, 0, 100);
}

static void
audio_channel_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    int max_channels = odk_get_stream_info (oxine->odk,
                                            ODK_STREAM_INFO_MAX_AUDIO_CHANNEL);
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_AUDIO_CHANNEL,
                             1, -2, max_channels - 1);
}

static void
audio_channel_minus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    int max_channels = odk_get_stream_info (oxine->odk,
                                            ODK_STREAM_INFO_MAX_AUDIO_CHANNEL);
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_AUDIO_CHANNEL,
                             -1, -2, max_channels - 1);
}

static void
audio_channel_upcall (void *oxine_p, otk_widget_t * label)
{
    update_param_label (oxine_p, label, ODK_PARAM_AUDIO_CHANNEL);
}

static void
spu_channel_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    int max_channels = odk_get_stream_info (oxine->odk,
                                            ODK_STREAM_INFO_MAX_SPU_CHANNEL);
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_SPU_CHANNEL, 1, -2, max_channels - 1);
}

static void
spu_channel_minus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    int max_channels = odk_get_stream_info (oxine->odk,
                                            ODK_STREAM_INFO_MAX_SPU_CHANNEL);
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_SPU_CHANNEL, -1, -2, max_channels - 1);
}

static void
spu_channel_upcall (void *oxine_p, otk_widget_t * label)
{
    update_param_label (oxine_p, label, ODK_PARAM_SPU_CHANNEL);
}

static void
av_offset_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_AUDIO_OFFSET, 9, -900000, 900000);
}

static void
av_offset_minus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_AUDIO_OFFSET, -9, -900000, 900000);
}

static void
av_offset_upcall (void *oxine_p, otk_widget_t * label)
{
    update_param_label (oxine_p, label, ODK_PARAM_AUDIO_OFFSET);
}

static void
spu_offset_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_SPU_OFFSET, 9, -900000, 900000);
}

static void
spu_offset_minus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_SPU_OFFSET, -9, -900000, 900000);
}

static void
spu_offset_upcall (void *oxine_p, otk_widget_t * label)
{
    update_param_label (oxine_p, label, ODK_PARAM_SPU_OFFSET);
}

static void
deinterlace_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk, ODK_PARAM_VO_DEINTERLACE, 1, 0, 1);
}

static void
deinterlace_upcall (void *oxine_p, otk_widget_t * label)
{
    update_param_label (oxine_p, label, ODK_PARAM_VO_DEINTERLACE);
}

static void
aspect_ratio_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_VO_ASPECT_RATIO, 1, ODK_VO_ASPECT_AUTO,
                             ODK_VO_ASPECT_DVB);
}

static void
aspect_ratio_minus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_VO_ASPECT_RATIO, -1,
                             ODK_VO_ASPECT_AUTO, ODK_VO_ASPECT_DVB);
}

static void
aspect_ratio_upcall (void *oxine_p, otk_widget_t * label)
{
    update_param_label (oxine_p, label, ODK_PARAM_VO_ASPECT_RATIO);
}

#ifdef HUE_AND_SATURATION_WORK
static void
saturation_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_VO_SATURATION, 500, 0, 65535);
}

static void
saturation_minus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_VO_SATURATION, -500, 0, 65535);
}

static void
hue_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk, ODK_PARAM_VO_HUE, 500, 0, 65535);
}

static void
hue_minus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk, ODK_PARAM_VO_HUE, -500, 0, 65535);
}
#endif

static void
contrast_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_VO_CONTRAST, 500, 0, 65535);
}

static void
contrast_minus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_VO_CONTRAST, -500, 0, 65535);
}

static void
contrast_upcall (void *oxine_p, otk_widget_t * label)
{
    update_param_label (oxine_p, label, ODK_PARAM_VO_CONTRAST);
}

static void
brightness_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_VO_BRIGHTNESS, 500, 0, 65535);
}

static void
brightness_minus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_VO_BRIGHTNESS, -500, 0, 65535);
}

static void
brightness_upcall (void *oxine_p, otk_widget_t * label)
{
    update_param_label (oxine_p, label, ODK_PARAM_VO_BRIGHTNESS);
}

/*
 * **************************************************************************
 * GUI of the stream parameter menu
 * **************************************************************************
 */
static void
show_stream_param (oxine_t * oxine, odk_stream_param_t param,
                   otk_label_uc_t upcall, int x1, int x2, int y)
{
    otk_widget_t *label;
    int alignleft = OTK_ALIGN_LEFT | OTK_ALIGN_TOP;

    char *param_name = odk_get_stream_param_name (oxine->odk, param);
    char *param_string = odk_get_stream_param_as_string (oxine->odk, param);

    label = otk_label_new (oxine->otk, x1, y, alignleft, param_name);
    otk_label_set_max_width (label, x2 - x1 - 20);

    label = otk_label_new (oxine->otk, x2, y, alignleft, param_string);
    otk_label_set_max_width (label, 650 - x2);
    otk_label_set_upcall (label, upcall, oxine);

    ho_free (param_name);
    ho_free (param_string);
}


void
show_stream_parameter_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    clean_otk_stuff (oxine);

    odk_forward_events_to_xine (oxine->odk, 0);

    oxine->repaint_menu = show_stream_parameter_menu_cb;
    oxine->user_interface_is_visible = 1;

    otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);
    show_clock (oxine);

    int x1 = 20;
    int x2 = 400;
    int y = 100;

    // Volume
    show_stream_param (oxine, ODK_PARAM_AUDIO_VOLUME,
                       volume_upcall, x1, x2, y);

    otk_widget_t *b;
    b = otk_button_new (oxine->otk, 670, y, 30, 30, "m", volume_mute, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    otk_widget_set_focus (b);
    b = otk_button_new (oxine->otk, 710, y, 30, 30, "-", volume_minus, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    b = otk_button_new (oxine->otk, 750, y, 30, 30, "+", volume_plus, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    y += 50;

    if (odk_get_stream_info (oxine->odk, ODK_STREAM_INFO_HAS_VIDEO)) {
        // Audio Channel
        show_stream_param (oxine, ODK_PARAM_AUDIO_CHANNEL,
                           audio_channel_upcall, x1, x2, y);

        b = otk_button_new (oxine->otk, 710, y, 30, 30, "-",
                            audio_channel_minus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        b = otk_button_new (oxine->otk, 750, y, 30, 30, "+",
                            audio_channel_plus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 40;

        // Audio-Video Offset
        show_stream_param (oxine, ODK_PARAM_AUDIO_OFFSET,
                           av_offset_upcall, x1, x2, y);

        b = otk_button_new (oxine->otk, 710, y, 30, 30, "-",
                            av_offset_minus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        b = otk_button_new (oxine->otk, 750, y, 30, 30, "+",
                            av_offset_plus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 50;

        // Subtitle Channel
        show_stream_param (oxine, ODK_PARAM_SPU_CHANNEL, spu_channel_upcall,
                           x1, x2, y);

        b = otk_button_new (oxine->otk, 710, y, 30, 30, "-",
                            spu_channel_minus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        b = otk_button_new (oxine->otk, 750, y, 30, 30, "+",
                            spu_channel_plus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 40;

        // Subtitle Offset
        show_stream_param (oxine, ODK_PARAM_SPU_OFFSET, spu_offset_upcall, x1,
                           x2, y);

        b = otk_button_new (oxine->otk, 710, y, 30, 30, "-",
                            spu_offset_minus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        b = otk_button_new (oxine->otk, 750, y, 30, 30, "+",
                            spu_offset_plus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 50;

        // Aspect Ratio
        show_stream_param (oxine, ODK_PARAM_VO_ASPECT_RATIO,
                           aspect_ratio_upcall, x1, x2, y);

        b = otk_button_new (oxine->otk, 710, y, 30, 30, "-",
                            aspect_ratio_minus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        b = otk_button_new (oxine->otk, 750, y, 30, 30, "+",
                            aspect_ratio_plus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 40;

        // Deinterlace
        show_stream_param (oxine, ODK_PARAM_VO_DEINTERLACE,
                           deinterlace_upcall, x1, x2, y);

        b = otk_button_new (oxine->otk, 710, y, 30, 30, "-",
                            deinterlace_plus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        b = otk_button_new (oxine->otk, 750, y, 30, 30, "+",
                            deinterlace_plus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 50;

        // Brightness
        show_stream_param (oxine, ODK_PARAM_VO_BRIGHTNESS, brightness_upcall,
                           x1, x2, y);

        b = otk_button_new (oxine->otk, 710, y, 30, 30, "-",
                            brightness_minus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        b = otk_button_new (oxine->otk, 750, y, 30, 30, "+",
                            brightness_plus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 40;

        // Contrast
        show_stream_param (oxine, ODK_PARAM_VO_CONTRAST, contrast_upcall, x1,
                           x2, y);

        b = otk_button_new (oxine->otk, 710, y, 30, 30, "-",
                            contrast_minus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        b = otk_button_new (oxine->otk, 750, y, 30, 30, "+",
                            contrast_plus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 40;

#ifdef HUE_AND_SATURATION_WORK
        // TODO: what exactly is the problem here?
        // Hue
        show_stream_param (oxine, ODK_PARAM_VO_HUE, hue_upcall, x1, x2, y);

        b = otk_button_new (oxine->otk, 710, y, 30, 30, "-",
                            hue_minus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        b = otk_button_new (oxine->otk, 750, y, 30, 30, "+", hue_plus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 40;

        // Saturation
        show_stream_param (oxine, ODK_PARAM_VO_SATURATION, saturation_upcall,
                           x1, x2, y);

        b = otk_button_new (oxine->otk, 710, y, 30, 30, "-",
                            saturation_minus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        b = otk_button_new (oxine->otk, 750, y, 30, 30, "+",
                            saturation_plus, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 40;
#endif
    }

    oxine->backto_menu = show_playing_menu_cb;
    b = otk_button_new (oxine->otk, 20, 540, 180, 40, _("Back"),
                        oxine->backto_menu, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);

    otk_draw (oxine->otk);
}

/*
 * **************************************************************************
 * Callbacks of the playing menu
 * **************************************************************************
 */
static void
show_list_cb (void *oxine_p)
{
    playing_menu_show_file_info = 0;
    show_playing_menu_cb (oxine_p);
}

static void
show_info_cb (void *oxine_p)
{
    playing_menu_show_file_info = 1;
    show_playing_menu_cb (oxine_p);
}

static void
change_playlist_mode_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    switch (oxine->playlist->playmode) {
        case PLAYLIST_MODE_NORMAL:
            oxine->playlist->playmode = PLAYLIST_MODE_REPEAT;

            break;
        case PLAYLIST_MODE_REPEAT:
            oxine->playlist->playmode = PLAYLIST_MODE_RANDOM;

            break;
        case PLAYLIST_MODE_RANDOM:
            oxine->playlist->playmode = PLAYLIST_MODE_NORMAL;

            break;
    }
    show_playing_menu_cb (oxine_p);
}

static void
track_select_cb (void *list_cb_data, void *entry_cb_data)
{
    oxine_t *oxine = (oxine_t *) list_cb_data;

    playlist_play_item (oxine, (playitem_t *) entry_cb_data);

    /* Save current channel to config file. */
    if (odk_current_is_dvb (oxine->odk))
        xine_config_save (oxine->xine, get_file_config ());
}

static void
playlist_prev_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_PREV;

    otk_send_event (oxine->otk, &event);
}

static void
playlist_next_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_NEXT;

    otk_send_event (oxine->otk, &event);
}

static void
media_play_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_PLAY;

    otk_send_event (oxine->otk, &event);

    assert (oxine->repaint_menu);
    oxine->repaint_menu (oxine);
}

static void
media_pause_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_PAUSE;

    otk_send_event (oxine->otk, &event);

    assert (oxine->repaint_menu);
    oxine->repaint_menu (oxine);
}

static void
media_stop_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_STOP;

    otk_send_event (oxine->otk, &event);
}

static void
progressbar_selected (void *oxine_p, int position)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_SEEK;
    event.data.how = position;

    otk_send_event (oxine->otk, &event);
}

static int
get_stream_progress (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    return odk_get_stream_param (oxine->odk, ODK_PARAM_POSITION);
}

/* 
 * **************************************************************************
 * GUI-Layout of the playing menu
 * **************************************************************************
 */
static void
set_time_gone_string (void *oxine_p, otk_widget_t * widget)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    char time_gone_str[64];

    int pos_time;
    int length;

    if (odk_get_pos_length (oxine->odk, NULL, &pos_time, &length)) {
        pos_time /= 1000;
        sprintf (time_gone_str, "%d:%02d:%02d",
                 pos_time / 3600, (pos_time % 3600) / 60,
                 ((pos_time % 3600) % 60));
        otk_label_set_text (widget, time_gone_str);
    }
}


static void
set_time_total_string (void *oxine_p, otk_widget_t * widget)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    char time_total_str[64];

    int pos_time;
    int length;

    if (odk_get_pos_length (oxine->odk, NULL, &pos_time, &length)) {
        if (length == 0)
            length = pos_time;
        length /= 1000;
        sprintf (time_total_str, "%d:%02d:%02d",
                 length / 3600, (length % 3600) / 60, ((length % 3600) % 60));
        otk_label_set_text (widget, time_total_str);
    }
}


static void
show_playing_menu_file_info (oxine_t * oxine)
{
    int alignleft = OTK_ALIGN_LEFT | OTK_ALIGN_TOP;
    int alignright = OTK_ALIGN_RIGHT | OTK_ALIGN_TOP;

    static int current_year = 0;
    if (!current_year) {
        time_t current = time (NULL);
        struct tm *brokentime = localtime (&current);
        current_year = brokentime->tm_year + 1900;
    }

    char *title = odk_get_meta_info (oxine->odk, META_INFO_TITLE);
    char *artist = odk_get_meta_info (oxine->odk, META_INFO_ARTIST);
    char *album = odk_get_meta_info (oxine->odk, META_INFO_ALBUM);
    char *year = odk_get_meta_info (oxine->odk, META_INFO_YEAR);
    char *genre = odk_get_meta_info (oxine->odk, META_INFO_GENRE);
    char *track = odk_get_meta_info (oxine->odk, XINE_META_INFO_TRACK_NUMBER);

    if (!title)
        title = create_title (odk_get_current_mrl (oxine->odk));

    int x1 = 220;
    int x2 = 400;
    int y = 95;

    otk_widget_t *label;
    label = otk_label_new (oxine->otk, x1, y, alignleft, title);
    otk_label_set_max_width (label, 560);
    otk_widget_set_font (label, "sans", 40);
    y += 50;

    if (album && strlen (album) > 0) {
        otk_label_new (oxine->otk, x2 - 20, y, alignright, _("Album:"));
        label = otk_label_new (oxine->otk, x2, y, alignleft, album);
        otk_label_set_max_width (label, 780 - x2);
        y += 40;
    }

    if (artist && strlen (artist) > 0) {
        otk_label_new (oxine->otk, x2 - 20, y, alignright, _("Artist:"));
        label = otk_label_new (oxine->otk, x2, y, alignleft, artist);
        otk_label_set_max_width (label, 780 - x2);
        y += 40;
    }

    if (year && atoi (year) > 0 && atoi (year) <= current_year) {
        otk_label_new (oxine->otk, x2 - 20, y, alignright, _("Year:"));
        label = otk_label_new (oxine->otk, x2, y, alignleft, year);
        otk_label_set_max_width (label, 780 - x2);
        y += 40;
    }

    if (track && strlen (track) > 0) {
        otk_label_new (oxine->otk, x2 - 20, y, alignright, _("Track:"));
        label = otk_label_new (oxine->otk, x2, y, alignleft, track);
        otk_label_set_max_width (label, 780 - x2);
        y += 40;
    }

    if (genre && strlen (genre) > 0) {
        genre[0] = toupper (genre[0]);
        otk_label_new (oxine->otk, x2 - 20, y, alignright, _("Genre:"));
        label = otk_label_new (oxine->otk, x2, y, alignleft, genre);
        otk_label_set_max_width (label, 780 - x2);
        y += 40;
    }

    y += 10;
    otk_label_new (oxine->otk, x2 - 20, y, alignright, _("Length:"));
    label = otk_label_new (oxine->otk, x2, y, alignleft, "0:00:00");
    set_time_total_string (oxine, label);
    otk_label_set_upcall (label, set_time_total_string, oxine);
    otk_widget_set_update (label, 1);

    y += 40;
    otk_label_new (oxine->otk, x2 - 20, y, alignright, _("Elapsed:"));
    label = otk_label_new (oxine->otk, x2, y, alignleft, "0:00:00");
    set_time_gone_string (oxine, label);
    otk_label_set_upcall (label, set_time_gone_string, oxine);
    otk_widget_set_update (label, 1);

    if (title)
        ho_free (title);
    if (artist)
        ho_free (artist);
    if (album)
        ho_free (album);
    if (year)
        ho_free (year);
    if (genre)
        ho_free (genre);
    if (track)
        ho_free (track);
}


/*
 * This menu shows a list of all the files in the playlist.
 */
static void
show_playing_menu_list (oxine_t * oxine)
{
    otk_widget_t *l;

    if (odk_current_is_dvb (oxine->odk)) {
        l = otk_list_new (oxine->otk, 220, 100, 560, 480,
                          OTK_LIST_NO_SELECTION, 0, oxine);
    }

    else {
        l = otk_list_new (oxine->otk, 220, 100, 560, 400,
                          OTK_LIST_NO_SELECTION, 0, oxine);
    }

    playitem_t *cur = playlist_first (oxine->playlist);
    while (cur) {
        otk_listentry_new (l, cur->title, track_select_cb, cur, NULL, NULL);
        cur = playlist_next (oxine->playlist, cur);
    }

    int currentpos = playlist_current_pos (oxine->playlist);
    otk_list_set_selected (l, currentpos, 1);
    otk_list_set_pos (l, currentpos);
}


static void
show_controls (oxine_t * oxine)
{
    if (odk_current_is_dvb (oxine->odk))
        return;

    /* If we're playing a playlist and the playlist has more than one entry, a
     * lot of buttons have to be shown. */
    int playlist_with_more_than_one_entry =
        (playlist_length (oxine->playlist) > 1);

    int x = 20;
    int y = 530;
    otk_widget_t *b;

    if (odk_get_stream_param (oxine->odk, ODK_PARAM_SPEED) == ODK_SPEED_PAUSE) {
        b = otk_button_new (oxine->otk, x, y, 50, 50, ">",
                            media_play_cb, oxine);
    } else {
        b = otk_button_new (oxine->otk, x, y, 50, 50, "<",
                            media_pause_cb, oxine);
    }
    otk_widget_set_focus (b);
    otk_widget_set_font (b, "cetus", 20);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    x += 65;

    y = 535;
    b = otk_button_new (oxine->otk, x, y, 40, 40, "}", media_stop_cb, oxine);
    otk_widget_set_font (b, "cetus", 20);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    x += 50;

    if (playlist_with_more_than_one_entry) {
        b = otk_button_new (oxine->otk, x, y, 40, 40, "[",
                            playlist_prev_cb, oxine);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        x += 50;

        b = otk_button_new (oxine->otk, x, y, 40, 40, "]",
                            playlist_next_cb, oxine);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        x += 50;
    }

    x += 5;
    b = otk_slider_new (oxine->otk, x, y, 780 - x, 40, OTK_SLIDER_HORIZONTAL,
                        get_stream_progress, oxine, progressbar_selected,
                        oxine);
    otk_widget_set_update (b, 1);
}


/*
 * This menu is shown while playing one or more titles.
 */
void
show_playing_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    clean_otk_stuff (oxine);

    odk_forward_events_to_xine (oxine->odk, 0);

    oxine->repaint_menu = show_playing_menu_cb;
    oxine->backto_menu = show_playing_menu_cb;
    oxine->user_interface_is_visible = 1;

    /* If we're playing a playlist and the playlist has more than one entry, a
     * lot of buttons have to be shown. */
    int playlist_with_more_than_one_entry =
        (playlist_length (oxine->playlist) > 1);

    otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);

    show_clock (oxine);

    int x = 20;
    int y = 100;

    otk_widget_t *b = NULL;
    if (odk_current_is_dvd (oxine->odk)) {
        b = otk_button_new (oxine->otk, x, y, 180, 40,
                            DVD_MENU_TITLE_1, dvd_menu1_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 50;

        b = otk_button_new (oxine->otk, x, y, 180, 40,
                            DVD_MENU_TITLE_4, dvd_menu4_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 50;

        b = otk_button_new (oxine->otk, x, y, 180, 40,
                            DVD_MENU_TITLE_6, dvd_menu6_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 50;
    }
    if (!odk_current_is_dvd (oxine->odk)
        && !odk_current_is_dvb (oxine->odk)) {
        if (playlist_with_more_than_one_entry) {
            if (playing_menu_show_file_info) {
                b = otk_button_new (oxine->otk, x, y, 180, 40,
                                    _("Show List"), show_list_cb, oxine);
            } else {
                b = otk_button_new (oxine->otk, x, y, 180, 40,
                                    _("Show Info"), show_info_cb, oxine);
            }
            otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
            y += 50;

            if (oxine->playlist->playmode == PLAYLIST_MODE_NORMAL) {
                b = otk_button_new (oxine->otk, x, y, 180, 40,
                                    _("Normal Mode"),
                                    change_playlist_mode_cb, oxine);
            } else if (oxine->playlist->playmode == PLAYLIST_MODE_REPEAT) {
                b = otk_button_new (oxine->otk, x, y, 180, 40,
                                    _("Loop Mode"),
                                    change_playlist_mode_cb, oxine);
            } else if (oxine->playlist->playmode == PLAYLIST_MODE_RANDOM) {
                b = otk_button_new (oxine->otk, x, y, 180, 40,
                                    _("Shuffle Mode"),
                                    change_playlist_mode_cb, oxine);
            }
            otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
            y += 50;
        }
    }
    {
        b = otk_button_new (oxine->otk, x, y, 180, 40,
                            _("Settings"),
                            show_stream_parameter_menu_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 50;
    }
    {
        b = otk_button_new (oxine->otk, x, y, 180, 40,
                            _("Eject"), eject_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        y += 50;
    }
    if (odk_current_is_dvb (oxine->odk)) {
        b = otk_button_new (oxine->otk, x, y, 180, 40, _("Mainmenu"),
                            show_main_menu_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    }

    if ((playing_menu_show_file_info || odk_current_is_dvd (oxine->odk))
        && !odk_current_is_dvb (oxine->odk)) {
        show_playing_menu_file_info (oxine);
    } else {
        show_playing_menu_list (oxine);
    }

    show_controls (oxine);

    otk_draw (oxine->otk);
}


void
show_control_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    odk_forward_events_to_xine (oxine->odk, 0);

    oxine->repaint_menu = show_control_menu_cb;
    oxine->control_menu_is_visible = 1;

    /* This is necessary, so we can add to a already existing window (e.g.
     * when we're already displaying some stream information.)*/
    if (!odk_is_osd_visible (oxine->odk)) {
        otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);
    }

    show_controls (oxine);

    otk_draw (oxine->otk);
}


/*
 * **************************************************************************
 * GUI-Layout of the info menu
 * **************************************************************************
 */
void
show_stream_info_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    clean_otk_stuff (oxine);

    int alignleft = OTK_ALIGN_LEFT | OTK_ALIGN_TOP;
    int alignright = OTK_ALIGN_RIGHT | OTK_ALIGN_TOP;
    int aligncenter = OTK_ALIGN_CENTER | OTK_ALIGN_TOP;

    otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);

    static int current_year = 0;
    if (!current_year) {
        time_t current = time (NULL);
        struct tm *brokentime = localtime (&current);
        current_year = brokentime->tm_year + 1900;
    }

    char *title = odk_get_meta_info (oxine->odk, META_INFO_TITLE);
    char *artist = odk_get_meta_info (oxine->odk, META_INFO_ARTIST);
    char *album = odk_get_meta_info (oxine->odk, META_INFO_ALBUM);
    char *year = odk_get_meta_info (oxine->odk, META_INFO_YEAR);
    char *genre = odk_get_meta_info (oxine->odk, META_INFO_GENRE);
    char *track = odk_get_meta_info (oxine->odk, XINE_META_INFO_TRACK_NUMBER);

    if (!title)
        title = create_title (odk_get_current_mrl (oxine->odk));

    int x2 = 400;
    int y = 180;

    otk_widget_t *label;
    label = otk_label_new (oxine->otk, x2, y, aligncenter, title);
    otk_label_set_max_width (label, 760);
    otk_widget_set_font (label, "sans", 40);
    y += 50;

    if (album && strlen (album) > 0) {
        otk_label_new (oxine->otk, x2 - 20, y, alignright, _("Album:"));
        label = otk_label_new (oxine->otk, x2, y, alignleft, album);
        otk_label_set_max_width (label, 780 - x2);
        y += 40;
    }

    if (artist && strlen (artist) > 0) {
        otk_label_new (oxine->otk, x2 - 20, y, alignright, _("Artist:"));
        label = otk_label_new (oxine->otk, x2, y, alignleft, artist);
        otk_label_set_max_width (label, 780 - x2);
        y += 40;
    }

    if (year && atoi (year) > 0 && atoi (year) <= current_year) {
        otk_label_new (oxine->otk, x2 - 20, y, alignright, _("Year:"));
        label = otk_label_new (oxine->otk, x2, y, alignleft, year);
        otk_label_set_max_width (label, 780 - x2);
        y += 40;
    }

    if (track && strlen (track) > 0) {
        otk_label_new (oxine->otk, x2 - 20, y, alignright, _("Track:"));
        label = otk_label_new (oxine->otk, x2, y, alignleft, track);
        otk_label_set_max_width (label, 780 - x2);
        y += 40;
    }

    if (genre && strlen (genre) > 0) {
        genre[0] = toupper (genre[0]);
        otk_label_new (oxine->otk, x2 - 20, y, alignright, _("Genre:"));
        label = otk_label_new (oxine->otk, x2, y, alignleft, genre);
        otk_label_set_max_width (label, 780 - x2);
        y += 40;
    }

    if (title)
        ho_free (title);
    if (artist)
        ho_free (artist);
    if (album)
        ho_free (album);
    if (year)
        ho_free (year);
    if (genre)
        ho_free (genre);
    if (track)
        ho_free (track);
}
