
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: list.c,v 1.12 2006/01/19 09:31:32 mschwerin Exp $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>

#include "heap.h"
#include "list.h"

#ifdef L_LIST_TEST

/* 
 * ***************************************************************************
 * Name:            l_list_test
 * Access:          private
 *
 * Description:     Tests the list for errors.
 * ***************************************************************************
 */
static void
l_list_test (l_list_t * list)
{
    assert (list);

    /* Check if we find the correct number of entries when walking forwards. */
    {
        int i = 0;
        l_item_t *item = list->first;
        while (item) {
            i++;
            item = item->next;
        }
        assert (i == list->length);
    }
    /* Check if we find the correct number of entries when walking backwards. */
    {
        int i = 0;
        l_item_t *item = list->last;
        while (item) {
            i++;
            item = item->prev;
        }
        assert (i == list->length);
    }
    /* Check that we get the same items at the same position when walking
     * forwards and backwards. */
    {
        l_item_t *items[list->length];
        int i = 0;
        l_item_t *item = list->first;
        while (item) {
            items[i] = item;
            i++;
            item = item->next;
        }

        i = list->length - 1;
        item = list->last;
        while (item) {
            assert (items[i] == item);
            i--;
            item = item->prev;
        }
    }
}
#endif


/* 
 * ***************************************************************************
 * Name:            l_list_new
 * Access:          public
 *
 * Description:     Creates a new empty list.
 * ***************************************************************************
 */
l_list_t *
l_list_new (void)
{
    l_list_t *list = ho_new (l_list_t);

    list->length = 0;

    list->first = NULL;
    list->last = NULL;
    list->current = NULL;

    return list;
}


/* 
 * ***************************************************************************
 * Name:            l_list_free
 * Access:          public
 *
 * Description:     Clears and frees the list. free_data_cb is a callback that
 *                  can be used to free the memory used by data.
 * ***************************************************************************
 */
void
l_list_free (l_list_t * list, l_free_cb_t free_data_cb)
{
    l_list_clear (list, free_data_cb);
    ho_free (list);
}


/* 
 * ***************************************************************************
 * Name:            l_list_clear
 * Access:          public
 *
 * Description:     Removes all entries from the list. free_data_cb is a
 *                  callback that can be used to free the memory used by data.
 * ***************************************************************************
 */
void
l_list_clear (l_list_t * list, l_free_cb_t free_data_cb)
{
    assert (list);

    while (list->first) {
        void *data = l_list_remove (list, list->first->data);
        if (free_data_cb && data)
            free_data_cb (data);
    }

    assert (list->length == 0);
    assert (list->first == NULL);
    assert (list->last == NULL);
    assert (list->current == NULL);
}


/* 
 * ***************************************************************************
 * Name:            l_list_sort
 * Access:          public
 *
 * Description:     Sorts the list. swap_cb is a callback function, that is
 *                  used to determin if two entries must be swapped.
 * ***************************************************************************
 */
void
l_list_sort (l_list_t * list, l_swap_cb_t swap_cb)
{
    assert (list);
    assert (swap_cb);

    l_item_t *p1 = list->first;

    while (p1) {
        l_item_t *p2 = p1;
        while (p2) {
            if (swap_cb (p1->data, p2->data)) {
                void *tmp = p1->data;
                p1->data = p2->data;
                p2->data = tmp;
            }
            p2 = p2->next;
        }
        p1 = p1->next;
    }
}


/* 
 * ***************************************************************************
 * Name:            l_list_move_down
 * Access:          public
 *
 * Description:     Moves the entry identified by data one position towards
 *                  the end of the list.
 * ***************************************************************************
 */
void
l_list_move_down (l_list_t * list, void *data)
{
    assert (list);
    assert (data);

    l_item_t *p1 = list->first;

    while (p1) {
        if (p1->data == data) {
            p1->data = p1->next->data;
            p1->next->data = data;
            break;
        }
        p1 = p1->next;
    }
}


/* 
 * ***************************************************************************
 * Name:            l_list_move_up
 * Access:          public
 *
 * Description:     Moves the entry identified by data one position towards
 *                  the front of the list.
 * ***************************************************************************
 */
void
l_list_move_up (l_list_t * list, void *data)
{
    assert (list);
    assert (data);

    l_item_t *p1 = list->first;

    while (p1) {
        if (p1->next->data == data) {
            p1->next->data = p1->data;
            p1->data = data;
            break;
        }
        p1 = p1->next;
    }
}


/* 
 * ***************************************************************************
 * Name:            l_list_concat
 * Access:          public
 *
 * Description:     Adds list l2 to the end of list l1 and frees l2.
 * ***************************************************************************
 */
void
l_list_concat (l_list_t * l1, l_list_t * l2)
{
    assert (l1);
    assert (l2);

    if (l1->first) {
        assert (l1->last);
        if (l2->first) {
            assert (l2->last);
            l1->last->next = l2->first;
            l2->first->prev = l1->last;
            l1->last = l2->last;
        }
    } else {
        l1->first = l2->first;
        l1->last = l2->last;
    }

    l1->current = NULL;
    l1->length += l2->length;

    ho_free (l2);
}


/* 
 * ***************************************************************************
 * Name:            l_list_insert
 * Access:          public
 *
 * Description:     Inserts a new entry with data at the given position.
 * ***************************************************************************
 */
void *
l_list_insert (l_list_t * list, int position, void *data)
{
    assert (list);
    assert (data);
    assert (position >= 0);

    l_item_t *item = ho_new (l_item_t);
    item->data = data;

    if (!list->first) {
        item->prev = NULL;
        item->next = NULL;
        list->first = item;
        list->last = item;
    } else if (position == 0) {
        item->prev = NULL;
        item->next = list->first;
        list->first->prev = item;
        list->first = item;
    } else if (position >= list->length) {
        item->prev = list->last;
        item->next = NULL;
        list->last->next = item;
        list->last = item;
    } else {
        int p = 0;
        l_item_t *cur = list->first;
        while (cur && p != position) {
            p += 1;
            cur = cur->next;
        }
        item->prev = cur->prev;
        item->next = cur;

        cur->prev->next = item;
        cur->prev = item;
    }

    list->length++;
    list->current = NULL;

#ifdef L_LIST_TEST
    l_list_test (list);
#endif

    return data;
}


/* 
 * ***************************************************************************
 * Name:            l_list_append
 * Access:          public
 *
 * Description:     Appends a new entry to the end of the list.
 * ***************************************************************************
 */
void *
l_list_append (l_list_t * list, void *data)
{
    assert (list);
    assert (data);

    return l_list_insert (list, list->length, data);
}


/* 
 * ***************************************************************************
 * Name:            l_list_prepend
 * Access:          public
 *
 * Description:     Prepends a new entry to the beginning of the list.
 * ***************************************************************************
 */
void *
l_list_prepend (l_list_t * list, void *data)
{
    assert (list);
    assert (data);

    return l_list_insert (list, 0, data);
}


/* 
 * ***************************************************************************
 * Name:            l_list_remove
 * Access:          public
 *
 * Description:     Removes the entry identified by data from the list.
 * ***************************************************************************
 */
void *
l_list_remove (l_list_t * list, void *data)
{
    assert (list);
    assert (data);

    if (list->first) {
        l_item_t *current = list->first;
        while (current) {
            if (current->data == data) {
                if (current->next)
                    current->next->prev = current->prev;
                if (current->prev)
                    current->prev->next = current->next;
                if (list->first == current)
                    list->first = current->next;
                if (list->last == current)
                    list->last = current->prev;

                ho_free (current);

                list->length--;
                list->current = NULL;
                break;
            }
            current = current->next;
        }
    }
#ifdef L_LIST_TEST
    l_list_test (list);
#endif

    return data;
}


/* 
 * ***************************************************************************
 * Name:            l_list_first
 * Access:          public
 *
 * Description:     Returns the first entry of the list.
 * ***************************************************************************
 */
void *
l_list_first (l_list_t * list)
{
    assert (list);

    if (list->first) {
        list->current = list->first;
        return list->first->data;
    }
    return NULL;
}


/* 
 * ***************************************************************************
 * Name:            l_list_last
 * Access:          public
 *
 * Description:     Returns the last entry of the list.
 * ***************************************************************************
 */
void *
l_list_last (l_list_t * list)
{
    assert (list);

    if (list->last) {
        list->current = list->last;
        return list->last->data;
    }
    return NULL;
}


/* 
 * ***************************************************************************
 * Name:            l_list_next
 * Access:          public
 *
 * Description:     Returns the next entry. The current entry is identified by
 *                  data.
 * ***************************************************************************
 */
void *
l_list_next (l_list_t * list, void *data)
{
    assert (list);
    assert (data);

    if (list->current && list->current->data == data) {
        list->current = list->current->next;
    } else {
        list->current = list->first;
        while (list->current) {
            if (list->current && list->current->data == data) {
                list->current = list->current->next;
                break;
            }
            list->current = list->current->next;
        }
    }
    if (list->current) {
        return list->current->data;
    }

    return NULL;
}


/* 
 * ***************************************************************************
 * Name:            l_list_prev
 * Access:          public
 *
 * Description:     Returns the previous entry. The current entry is
 *                  identified by data.
 * ***************************************************************************
 */
void *
l_list_prev (l_list_t * list, void *data)
{
    assert (list);
    assert (data);

    if (list->current && list->current->data == data) {
        list->current = list->current->prev;
    } else {
        list->current = list->last;
        while (list->current) {
            if (list->current && list->current->data == data) {
                list->current = list->current->prev;
                break;
            }
            list->current = list->current->prev;
        }
    }
    if (list->current) {
        return list->current->data;
    }

    return NULL;
}


/* 
 * ***************************************************************************
 * Name:            l_list_length
 * Access:          public
 *
 * Description:     Returns the number of entries in the list.
 * ***************************************************************************
 */
int
l_list_length (l_list_t * list)
{
    assert (list);

    return list->length;
}
