//                                               -*- C++ -*-
/**
 * @file  NumericalMathEvaluationImplementation.hxx
 * @brief Abstract top-level class for all numerical math function implementations
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * \author $LastChangedBy: lebrun $
 * \date   $LastChangedDate: 2012-04-17 11:31:33 +0200 (Tue, 17 Apr 2012) $
 */

#ifndef OPENTURNS_NUMERICALMATHEVALUATIONIMPLEMENTATION_HXX
#define OPENTURNS_NUMERICALMATHEVALUATIONIMPLEMENTATION_HXX

#include "PersistentObject.hxx"
#include "NumericalPoint.hxx"
#include "NumericalPointWithDescription.hxx"
#include "NumericalSample.hxx"
#include "TimeSeries.hxx"
#include "HistoryStrategy.hxx"
#include "Description.hxx"
#include "Indices.hxx"
#include "Matrix.hxx"
#include "Collection.hxx"
#include "Pointer.hxx"
#include "StorageManager.hxx"
#include "Cache.hxx"

BEGIN_NAMESPACE_OPENTURNS

/**
 * @class NumericalMathEvaluationImplementation
 *
 * This class offers an abstract interface for the implementation
 * of an real numerical mathematical function into the platform.
 */
class NumericalMathEvaluationImplementation
  : public PersistentObject
{
  CLASSNAME;
public:

  typedef Pointer<NumericalMathEvaluationImplementation>         Implementation;
  typedef PersistentCollection<NumericalScalar>                  CacheKeyType;
  typedef PersistentCollection<NumericalScalar>                  CacheValueType;
  typedef Cache<CacheKeyType, CacheValueType>                    CacheType;
  typedef Pointer<CacheType>                                     CacheImplementation;

  /** Default constructor */
  NumericalMathEvaluationImplementation();

  /** Virtual constructor */
  virtual NumericalMathEvaluationImplementation * clone() const;

  /** Comparison operator */
  Bool operator ==(const NumericalMathEvaluationImplementation & other) const;

  /** String converter */
  virtual String __repr__() const;
  virtual String __str__(const String & offset = "") const;


  /** Description Accessor, i.e. the names of the input and output parameters */
  virtual void setDescription(const Description & description);
  Description getDescription() const;

  /** Input description Accessor, i.e. the names of the input parameters */
  Description getInputDescription() const;

  /** Output description Accessor, i.e. the names of the Output parameters */
  Description getOutputDescription() const;

  /** Enable or disable the input/output history */
  void enableHistory() const;
  void disableHistory() const;
  Bool isHistoryEnabled() const;
  void resetHistory() const;
  HistoryStrategy getInputHistory() const;
  HistoryStrategy getOutputHistory() const;

  /* Here is the interface that all derived class must implement */

  /** Enable or disable the internal cache */
  void enableCache() const;
  void disableCache() const;
  Bool isCacheEnabled() const;
  UnsignedLong getCacheHits() const;
  void addCacheContent(const NumericalSample & inSample, const NumericalSample & outSample);
  NumericalSample getCacheInput() const;

  /** Test for actual implementation */
  virtual Bool isActualImplementation() const;

  /** Operator () */
  virtual NumericalPoint operator() (const NumericalPoint & inP) const;

  /** Operator () on a sample, not pure virtual because a generic implementation is given */
  virtual NumericalSample operator() (const NumericalSample & inSample) const;

  /** Operator () on a time series, not pure virtual because a generic implementation is given */
  virtual TimeSeries operator() (const TimeSeries & inTimeSeries) const;

  /** Accessor for input point dimension */
  virtual UnsignedLong getInputDimension() const;

  /** Accessor for output point dimension */
  virtual UnsignedLong getOutputDimension() const;

  /** Get the i-th marginal evaluation */
  virtual NumericalMathEvaluationImplementation * getMarginal(const UnsignedLong i) const;

  /** Get the evaluation corresponding to indices components */
  virtual NumericalMathEvaluationImplementation * getMarginal(const Indices & indices) const;

  /** Gradient according to the marginal parameters */
  virtual Matrix parametersGradient(const NumericalPoint & inP) const;

  /** Parameters value and description accessor */
  virtual NumericalPointWithDescription getParameters() const;
  virtual void setParameters(const NumericalPointWithDescription & parameters);

  /** Get the number of calls to operator() */
  UnsignedLong getCallsNumber() const;

  /** Method save() stores the object through the StorageManager */
  void save(Advocate & adv) const;

  /** Method load() reloads the object from the StorageManager */
  void load(Advocate & adv);


protected:

  /** Number of calls since the construction */
  mutable UnsignedLong callsNumber_;

  /** A cache to store already computed points */
  mutable CacheImplementation p_cache_;

  /** An history mechanism that allows to remember all the input/output associations including duplicate calls */
  mutable HistoryStrategy inputStrategy_;

  mutable HistoryStrategy outputStrategy_;

  /** Flag to activate or deactivate the history mechanism */
  mutable Bool isHistoryEnabled_;

private:

  /** The description of all the components */
  Description description_;

  /** The value and description of all the parameters */
  NumericalPointWithDescription parameters_;

}; /* class NumericalMathEvaluationImplementation */


END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_NUMERICALMATHEVALUATIONIMPLEMENTATION_HXX */
