class oci::swiftstore(
  $machine_hostname          = undef,
  $machine_ip                = undef,
  $network_ipaddr            = undef,
  $network_cidr              = undef,
  $block_devices             = undef,
  $statsd_hostname           = undef,
  $pass_swift_hashpathsuffix = undef,
  $pass_swift_hashpathprefix = undef,
  $zoneid                    = undef,
  $use_ssl                   = true,
){

  package { 'rsyslog':
    ensure => present,
  }
  service { 'rsyslog':
    ensure  => running,
    enable  => true,
    require => Package['rsyslog'],
  }

  file { '/var/log/swift':
    ensure => directory,
    mode   => '0750',
  }

  file { '/etc/rsyslog.d/10-swift.conf':
    ensure  => present,
    source  => "puppet:///modules/${module_name}/rsyslog-swift.conf",
    require => [Package['rsyslog'], File['/var/log/swift']],
    notify  => Service['rsyslog'],
  }

  File<| title == '/var/log/swift' |> {
    owner => 'swift',
    group => 'adm'
  }

  class { '::memcached':
    listen_ip => '127.0.0.1',
    udp_port  => 0,
  }

  class { 'swift':
    swift_hash_path_suffix => $pass_swift_hashpathsuffix,
    swift_hash_path_prefix => $pass_swift_hashpathprefix,
  }

  class { '::swift::storage':
    storage_local_net_ip => $machine_ip,
  }
  include swift::storage::container
  include swift::storage::account
  include swift::storage::object

  if $statsd_hostname == ''{
    $statsd_enabled = false
  } else {
    $statsd_enabled = true
  }

  swift::storage::server { '6002':
    type                 => 'account',
    devices              => '/srv/node',
    config_file_path     => 'account-server.conf',
    storage_local_net_ip => "${machine_ip}",
    pipeline             => ['healthcheck', 'recon', 'account-server'],
    statsd_enabled           => $statsd_enabled,
    log_statsd_host          => $statsd_hostname,
    log_statsd_metric_prefix => $machine_hostname,
  }

  swift::storage::server { '6001':
    type                 => 'container',
    devices              => '/srv/node',
    config_file_path     => 'container-server.conf',
    storage_local_net_ip => "${machine_ip}",
    pipeline             => ['healthcheck', 'recon', 'container-server'],
    statsd_enabled           => $statsd_enabled,
    log_statsd_host          => $statsd_hostname,
    log_statsd_metric_prefix => $machine_hostname,
  }

  swift::storage::server { '6200':
    type                   => 'object',
    devices                => '/srv/node',
    config_file_path       => 'object-server.conf',
    storage_local_net_ip   => "${machine_ip}",
    pipeline               => ['healthcheck', 'recon', 'object-server'],
    servers_per_port       => 1,
    replicator_concurrency => 10,
    statsd_enabled           => $statsd_enabled,
    log_statsd_host          => $statsd_hostname,
    log_statsd_metric_prefix => $machine_hostname,
  }

  $block_devices.each |Integer $index, String $value| {
    swift::storage::disk { "${value}":
       mount_type  => 'uuid',
      require => Class['swift'],
    }->
    exec { "fix-unix-right-of-${value}":
      path    => "/bin",
      command => "chown swift:swift /srv/node/${value}",
      unless  => "cat /proc/mounts | grep -E ^/dev/${value}",
    }
  }

  $rings = [
    'account',
    'object',
    'container']
  swift::storage::filter::recon { $rings: }
  swift::storage::filter::healthcheck { $rings: }

  Swift::Ringsync<<||>>
}
