/*
 * Copyright (c) 1998 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the Computer Systems
 *       Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * ---------------------------
 *
 * Filename: supp-srmv2.cc
 *   -- Author: Suchitra Raman <suchi@cs.berkeley.edu>
 *
 * @(#) $Header: /usr/mash/src/repository/srmv2/srmv2/supp-srmv2.cc,v 1.18 2000/12/10 02:37:24 lim Exp $
 * 
 */


#include "supp-srmv2.h"
#include "srmv2-api.h"

void
SRMv2_Request::timeout() 
{
	queue_->send(this);
}

void
SRMv2_Request::send_request() 
{
  session_->send_rreq(this, SRMv2_RREQ);
  backoff();
}


void
SRMv2_Request::send_nsrequest() 
{
  session_->send_rreq(this, SRMv2_NSREQUEST);
  cancel();
}

void
SRMv2_Request::send_nsreply() 
{
  //	session_->send_nsreply(this);
	backoff();
}

void
SRMv2_Request::send_reply() 
{
	u_int32_t ss = left.adu, sbytes = left.byte;
	u_int32_t es = right.adu, ebytes = right.byte;
	u_int32_t last;
	unsigned char *data;
	unsigned int len;
	srm_free_proc free_proc;
	srm_read_adu_proc read_adu_proc;
	srm_adu_info info = { 0, {0, 0} };

	last = ((ebytes==0) ? es - 1 : es);
	read_adu_proc = source_->session()->callbacks()->read_adu_proc;
	srm_trace(srmSRM, ("S REP %d %d.%d--%d.%d (Q 0x%p)\n", cid_, left.adu, 
			   left.byte, right.adu, right.byte, this));

	int save = session_->delay_until_full_packet(1);
	for (unsigned int i = ss; i <= last; i++) {
		data = NULL;
		len = 0;
		free_proc = NULL;

		/* FIXME: what does one do if read_adu_proc is NULL */
		if (read_adu_proc) {
		  (*read_adu_proc)(source_, cid_, i, &data, &len, &free_proc,
				   &info);
		  
		  
		  /* Calculate left and right boundaries. */
		  u_int32_t rtx_sbytes = 0, rtx_ebytes = len - 1;
		  
		  if (i == ss) rtx_sbytes = sbytes;
		  if (i == es) {
		    rtx_ebytes = MIN(ebytes,len) - 1;
		  }

		  srm_trace(srmSRM, ("S (****RTX****) ADU %d %d %d--%d\n",
				     cid_, i, rtx_sbytes, rtx_ebytes));
		  session_->send_adu(source_, (unsigned char*)data,
				     (unsigned) i, rtx_sbytes,
				     rtx_ebytes, len - 1, cid_, SRMv2_REXMIT,
				     uaddr_, ((rtx_sbytes==0 &&
					       rtx_ebytes==len-1) ?0:1),&info);
		  if (free_proc) 
		    (*free_proc)(data);
		  
		}
	}
	session_->delay_until_full_packet(save);
}

void
SRMv2_Request::backoff()
{
	cancel();
	/* Request scope expands with each trial. */
	ttl_ = MIN(2*ttl_, session_->ttl());
	if (backoff_level_ < 10)
		backoff_level_ ++;
	period_ = session_->backoff(source_, backoff_level_);
	msched((int) period_);
}

void 
SRMv2_ReqQueue::process(SRMv2_Request *prev, Bool flag, u_short *ttl) 
{
	SRMv2_Request *p = 0;
	switch(flag) {
	case SRMv2_CANCEL:
		if (prev == 0) {
			if (list_ != 0) {
				p = list_;
				list_ = p->next;
			}
		} else {
			p = prev->next;
			prev->next = p->next;
		} 
		if (p != 0) {
			if (ttl != 0) *ttl = p->ttl_/2;
			delete p;
		}
		break;

	case SRMv2_BACKOFF:
		p = (prev == 0) ? list_ : prev->next;
		if (p != 0) p->backoff();
		break;

	default: 
		srm_trace(srmSRM,("Unknown flag %d\n", flag));
	}	
}

void
SRMv2_ReqQueue::schedule(SRMv2_Request *range)
{
	SRMv2_Session *s  = range->session();
	SRMv2_Source *src = range->source();	

	float period = s->backoff(src, 1);
	range->sched_timeout(period, s, src);
	range->queue(this);
	range->display();
	insert(range);
}

void 
SRMv2_ReqQueue::insert(SRMv2_Request *range, SRMv2_Request *before) 
{
	if (!range) return;
	if (!range->valid()) {
		delete range; /* Invalid range. */
		return;
	}
	if (before) {
		range->next = before->next; 
		before->next = range;
	} else {
		range->next = list_;
		list_ = range;
	}
}

/*
 * Since the incoming range is guaranteed not to exist in the queue already, 
 * simply compare the left edges of the list and insert at the right position. 
 */
void 
SRMv2_ReqQueue::insert(SRMv2_Request *range) 
{
	SRMv2_Request** r;
	SRMv2_Request*  p=0;
	SRMv2_Edge e = range->left;
	for (r = &list_; *r != 0; p = *r, r = &(*r)->next) {
	  if (e < (*r)->left)
	    break;
	}
	/* If the exact same request exists, return. */
	if (p != 0 && *p == *range) {
	  delete range;
	  return;
	}
	range->next = *r;
	*r = range;
}

/*
 * (1) Check queue and backoff matching ranges.
 * (2) Update the set of intersecting intervals. 
 * (3) Return the intersection set.
 *
 * If the flag cancel is set, the intersecting intervals 
 * are canceled. If not, they are simply backed off. 
 */

void 
SRMv2_ReqQueue::intersection(SRMv2_Request *r, Bool flag, u_short *ttl) 
{
	SRMv2_Request *cr = list_;
	SRMv2_Request *pr = 0, *next = 0;
	unsigned int cid = r->cid_;
	SRMv2_Edge left = r->left;
	SRMv2_Edge right = r->right;
	SRMv2_Session *session;
	SRMv2_Source *source;
	double period; /* ms */

	while (cr) {
		next = cr->next;
		srm_trace(srmSRM, ("Q INT: cr (%d.%d--%d.%d), "
				   "recv (%d.%d--%d.%d) ptrs(%p, %p)",
				   cr->left.adu, cr->left.byte, cr->right.adu,
				   cr->right.byte, left.adu, left.byte,
				   right.adu, right.byte, pr, cr));
		session = cr->session_;
		source  = cr->source_;
		period  = cr->period_;

		/* cr  : |---|        or         |---|
		 * recv:       |---|       |---| 
		 */
		if (cr->right <= left || right <= cr->left) {
			pr = cr;
			cr = next;
			continue;
		}
		/* cr  : |-------| or |-------| or |-------| or |------|
		 * recv:   |--|       |--|              |--|    |------|
		 */
		if (cr->left <= left && right <= cr->right) {
			SRMv2_Request* r1 = new SRMv2_Request(cid, cr->left,
							      left);
			SRMv2_Request* r2 = new SRMv2_Request(cid, right,
							      cr->right);
			r1->queue(this); r2->queue(this);
			process(pr, flag, ttl);
			r1->sched_timeout(period, session, source);
			r2->sched_timeout(period, session, source);
			insert(r1, pr);
			if (pr) pr = pr->next;
			insert(r2, pr);
			if (pr)
				cr = pr->next;
			else 
				cr = next;
			continue;
		}
		
		/* cr  :   |--|    or |--|      or      |--|
		 * recv: |-------|    |-------|    |-------|
		 */
		if (left <= cr->left && cr->right <= right) {
			process(pr, flag, ttl);
			if (pr)
				cr = pr->next;
			else 
				cr = next;
			continue;
		}
		
		/* 
		 * cr  : |-------|          
		 * recv:      |-------|
		 */
		if (cr->left < left && left < cr->right && cr->right < right) {
			SRMv2_Request* r1 = new SRMv2_Request(cid, cr->left, left);
			r1->queue(this);
			cr->left = right;
			process(pr, flag, ttl);
			insert(r1, pr);
			r1->sched_timeout(period, session, source);
			pr = cr;
			cr = next;
			continue;
		}
		/* 
		 * cr  :      |-------|
		 * recv: |-------|          
		 */
		if (left < cr->left && cr->left < right && right < cr->right) {
			SRMv2_Request* r1 = new SRMv2_Request(cid, right, cr->right);
			r1->queue(this);
			cr->right = right;
			process(pr, flag, ttl);
			insert(r1, pr);
			r1->sched_timeout(period, session, source);
			pr = cr;
			cr = next;
			continue;
		}
	}
}


void 
SRMv2_ReqQueue::append(SRMv2_Request *range) 
{
	append(range, list_);
}

void 
SRMv2_ReqQueue::append(SRMv2_Request *range, SRMv2_Request *head) 
{
	SRMv2_Request *p, *b = 0;
	if (!range->valid()) return;
	/* Traverse the list to its end */
	for (p = head; p != 0; p = p->next)
		b = p;
	if (b) {
		b->next = range;
	} else head = range;
}
void 
SRMv2_ReqQueue::append_cpy(SRMv2_Request *range, SRMv2_Request *head) 
{
	if (!range->valid()) return;
	SRMv2_Request cp = *range;
	append(&cp, head);
}








