# ui-notebook.tcl --
#
#       Defines widgets that server as a substitute for Tk tab widget
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


# Known "Issues"
#
# can't do a tab-config
# buttonRelease works even if cursor leaves tab
# doesn't do sideways tabs, or multilayer tabs
# Can't change tabs, etc once they are set up

#provide notebook

namespace eval NotebookWidget {

# The valid widget commands
set commands {configure cget index addTab addFrame addCommand tabPress}

# The valid widget configuration options
set options {
    -height -width -tabheight -tabwidth -tabpadx -bd -bg -fg -variable -font
}

# Defaults for the options listed above
set defaults(-height) 200
set defaults(-width) 400
set defaults(-tabheight) 30
set defaults(-tabwidth) 0
set defaults(-font) {Helvetica -12 bold}
set defaults(-bd) 2
set defaults(-bg) gray87
set defaults(-fg) black
set defaults(-tabpadx) 5
set defaults(-variable) {}

# This command is exported outside this module
proc notebook {n args} {
    eval newWidget $n $args
}

# Creates a new notebook widget
#
proc newWidget {n args} {
    variable options
    variable defaults

    # Initialize widget options to defaults
    foreach o $options {
	vset $n $o $defaults($o)
    }

    eval configure $n $args
    getMemberVars $n

    # Calculate light and dark bg colors
    calcBG $n

    # Create canvas that contains notebook
    set pad 2
    incr height [expr 2*$pad]
    incr width [expr 2*$pad]
    canvas $n -height $height -width $width -bg $bg -highlightthickness 0
    bind $n <Configure> "NotebookWidget::resizeContentsFrame $n"

    # Create the frame which holds the notebook's "contents"
    vset $n pad $pad
    createContentsFrame $n frame frame_ids
    vset $n frame $frame
    vset $n frame_ids $frame_ids

    # Initialize some values for the notebook
    vset $n numtabs 0
    vset $n activeTab -1
    vset $n tabOffset [expr $pad+$bd]
    vset $n callback 0
    # Export the canvas's name as the notebook's widget procedure
    if {[info commands _$n] != ""} {
	rename _$n ""
    }
    rename $n _$n
    proc $n {command args} "eval widgetCommand \$command $n \$args"
    namespace export $n
    namespace eval [namespace parent] "namespace import NotebookWidget::$n"

    return $n
}

# Called when the user invokes a command on a notebook widget
# Checks to make sure command is valid, then dispatches command.
#
proc widgetCommand {command n args} {
    variable commands

    if {[lsearch  $commands  $command] < 0} {
	error "bad option \"$command\": must be one of $commands"
    }
    eval $command $n $args
}

#### WIDGET COMMANDS  ###############

# Works like the standard Tk widget command config,
# except that it returns list of {-switch default value} lists.
#
proc configure {n args} {
    variable $n
    variable options
    variable defaults

    set len [llength $args]
    if {$len == 0} {
	foreach switch $options {
	    lappend results [list $switch $defaults($switch) [cget $n $switch]]
	}
	return $results
    }
    if {$len == 1} {
	return [eval cget $n $args]
    }
    if {$len > 1} {
	for {set i 0} {$i < $len} {incr i 2} {
	    cset $n [lindex $args $i] [lindex $args [expr $i+1]]
	}
    }
}

# Used to get a particular widget configuration option.
# Checks for valididy of option, wrapper around vget
#
proc cget {n switch} {
    variable options

    if {[lsearch  $options $switch] < 0} {
	error "unknown option \"$switch\""
    } else {
	return [vget $n $switch]
    }
}


# Creates a new tab with name "label".  It should be associate
# with a frame later
#
proc addTab {n label} {
    set t [vget $n numtabs]

    createTab $n $label win ids

    bind $win <ButtonRelease-1> "$n tabPress $t"

    vset $n tab${t}_win $win
    vset $n tab${t}_ids $ids

    activateTab $n $t
    deactivateTab $n $t
    vset $n numtabs [expr $t+1]
}

# Associates a frame with a particular tab
#
proc addFrame {n frame tab args} {
    set total [vget $n numtabs]

    if {$tab < 0 || $tab >= $total} {
	error "bad tab \"$tab\": want range \[0 [expr $total-1]\]"
    }
    vset $n frame${tab} $frame
    vset $n frame${tab}_args $args

    if {[vget $n activeTab] == $tab} {
	tabPress $n $tab
    }
}

# Associates a command with a particular tab
#
proc addCommand {n tab command} {
    set total [vget $n numtabs]

    if {$tab < 0 || $tab >= $total} {
	error "bad tab \"$tab\": want range \[0 [expr $total-1]\]"
    }
    vset $n command${tab} $command
}

# Shows the contents for the current tab.  Bound to a button
# press on a tab or can be called explicitly
#
proc tabPress {n newTab} {
    if [vget $n callback] {
	return
    }

    set total [vget $n numtabs]
    set t [vget $n activeTab]
    set contents [vget $n frame]

    if {$newTab < 0 || $newTab >= $total} {
	error "bad tab \"$newTab\": want range \[0 [expr $total-1]\]"
    }

    catch {deactivateTab $n $t}
    activateTab $n $newTab
    vset $n activeTab $newTab

    if {[catch {vget $n frame${t}} frame] == 0} {
	pack forget $frame
    }
    if {[catch {vget $n frame${newTab}} frame] == 0} {
	set packArgs [vget $n frame${newTab}_args]
	eval pack $frame -in $contents $packArgs
    }
    if {[catch {vget $n command${newTab}} command] == 0} {
	uplevel #0 $command
    }

    set traceVar [vget $n -variable]
    if {$traceVar != {}} {
	vset $n callback 1
	global $traceVar
	set $traceVar $newTab
	vset $n callback 0
    }
}

# Returns the currently active tab index
#
proc index {n} {
    variable $n
    return [vget $n activeTab]
}

#### LOCAL PROCEDURES ###############

# given a background color, calculate the light and dark versions
# for shadows on tabs
proc calcBG {n} {
    #set rgb [winfo rgb $n [cget $n -bg]]
    set rgb [winfo rgb . [cget $n -bg]]
    set r [lindex $rgb 0]
    set g [lindex $rgb 1]
    set b [lindex $rgb 2]

    set dr [expr $r*3/5]
    set dg [expr $g*3/5]
    set db [expr $b*3/5]

    set l1 [expr $r*7/5]
    if {$l1 > 65535} {
	set lr 65535
    } else {
	set l2 [expr (65535+$r)/2]
	if {$l1 > $l2} { set lr $l1 } else { set lr $l2 }
    }

    set l1 [expr $g*7/5]
    if {$l1 > 65535} {
	set lg 65535
    } else {
	set l2 [expr (65535+$g)/2]
	if {$l1 > $l2} { set lg $l1 } else { set lg $l2 }
    }

    set l1 [expr $b*7/5]
    if {$l1 > 65535} {
	set lb 65535
    } else {
	set l2 [expr (65535+$b)/2]
	if {$l1 > $l2} { set lb $l1 } else { set lb $l2 }
    }

    vset $n bgd [format "#%04x%04x%04x" $dr $dg $db]
    vset $n bgl [format "#%04x%04x%04x" $lr $lg $lb]
}

# Used to set a particular widget configuration option.
# Checks for valididy of option, wrapper around vset
#
proc cset {n switch value} {
    variable options

    if {[lsearch  $options $switch] < 0} {
	error "unknown option \"$switch\""
    }

    # Code to be evaluated before certain options change
    switch -- $switch {
	-variable {
	    global [vget $n -variable]
	    trace vdelete [vget $n -variable] w "NotebookWidget::callback $n"
	}
    }

    vset $n $switch $value

    # Code to be evaluated after certain options change
    switch -- $switch {
	-variable {
	    global $value
	    trace variable $value w "NotebookWidget::callback $n"
	}
	-bg {
	    calcBG $n
	}
    }
}

# Sets a member variable in a notebook widget
proc vset {n switch value} {
    variable $n
    set ${n}($switch) $value
}

# Gets a member variable in a notebook widget
proc vget {n switch} {
    variable $n
    return [set ${n}($switch)]
}

proc callback {n var nothing op} {
    upvar $var index
    tabPress $n $index
}

# Cool utility function that fetches all of the notebook's member
# variables into the local scope, removing dashes from "-switches"
#
proc getMemberVars {n} {
    variable $n
    foreach o [array names $n] {
	if {[string index $o 0] == "-"} {
	    upvar [string range $o 1 end] opt
	} else {
	    upvar $o opt
	}
	set opt [vget $n $o]
    }
}

set tnum 0
set cnum 0

# Creates an tab and adds it to the
# notebook's canvas
#
proc createTab {n text win_ptr ids_ptr} {
    variable tnum
    upvar $win_ptr win $ids_ptr ids
    getMemberVars $n

    set win [label $n.t[incr tnum] -text $text -bg $bg -fg $fg -padx $tabpadx -font $font]

    set x $tabOffset
    set y $pad
    set w [expr $tabwidth ? $tabwidth : [winfo reqwidth $win]]
    set h $tabheight

    # label window
    lappend ids [_$n create window [expr $x+$bd] [expr $y+$bd] \
	    -height [expr $h-2*$bd] -width [expr $w-2*$bd] \
	    -window $win -anchor nw]
    # left strip
    lappend ids [_$n create rectangle $x [expr $y+$bd] [expr $x+$bd] \
	    [expr $y+$h-$bd] -fill $bgl -outline {}]
    # left arc
    lappend ids [_$n create arc $x $y [expr $x+2*$bd] [expr $y+2*$bd] \
	    -extent 90 -start 90 -fill $bgl -outline {}]
    # top strip
    lappend ids [_$n create rectangle [expr $x+$bd] $y \
	    [expr $x+$w-$bd] [expr $y+$bd] -fill $bgl -outline {}]
    # right arc
    lappend ids [_$n create arc [expr $x+$w-2*$bd] $y \
	    [expr $x+$w] [expr $y+2*$bd] -extent 90 -start 0 \
	    -fill $bgd -outline {}]
    # right strip
    lappend ids [_$n create rectangle [expr $x+$w-$bd] \
	    [expr $y+$bd] [expr $x+$w] [expr $y+$h-$bd] \
	    -fill $bgd -outline {}]
    # bottom strip
    lappend ids [_$n create rectangle [expr $x+$bd] [expr $y+$h-$bd] \
	    [expr $x+$w-$bd] [expr $y+$h] -fill $bg -outline {}]

    vset $n tabOffset [expr $tabOffset+$w+1]
}

# Changes look of tab when it is activated
#
proc activateTab {n tab} {
    variable $n

    set ids [vget $n tab${tab}_ids]
    set bd [vget $n -bd]

    # Resize each part of tab in turn
    # window
    set id [lindex $ids 0]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]-$bd] [expr [lindex $coords 1]-$bd]
    # left strip
    set id [lindex $ids 1]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]-$bd] [expr [lindex $coords 1]-$bd] \
	    [expr [lindex $coords 2]-$bd] [lindex $coords 3]
    _$n raise $id
    # left arc
    set id [lindex $ids 2]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]-$bd] [expr [lindex $coords 1]-$bd] \
	    [expr [lindex $coords 2]-$bd] [expr [lindex $coords 3]-$bd]
    # top strip
    set id [lindex $ids 3]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]-$bd] [expr [lindex $coords 1]-$bd] \
	    [expr [lindex $coords 2]+$bd] [expr [lindex $coords 3]-$bd]
    # right arc
    set id [lindex $ids 4]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]+$bd] [expr [lindex $coords 1]-$bd] \
	    [expr [lindex $coords 2]+$bd] [expr [lindex $coords 3]-$bd]
    # right strip
    set id [lindex $ids 5]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]+$bd] [expr [lindex $coords 1]-$bd] \
	    [expr [lindex $coords 2]+$bd] [lindex $coords 3]
    _$n raise $id
    # bottom strip
    set id [lindex $ids 6]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]-$bd] [lindex $coords 1] \
	    [expr [lindex $coords 2]+$bd] [lindex $coords 3]

    _$n raise $id
}


# Changes look of tab back to normal
#
proc deactivateTab {n tab} {
    variable $n

    set ids [vget $n tab${tab}_ids]
    set bd [vget $n -bd]

    # Resize each part of tab in turn
    # window
    set id [lindex $ids 0]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]+$bd] [expr [lindex $coords 1]+$bd]
    # left strip
    set id [lindex $ids 1]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]+$bd] [expr [lindex $coords 1]+$bd] \
	    [expr [lindex $coords 2]+$bd] [lindex $coords 3]
    # left arc
    set id [lindex $ids 2]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]+$bd] [expr [lindex $coords 1]+$bd] \
	    [expr [lindex $coords 2]+$bd] [expr [lindex $coords 3]+$bd]
    # top strip
    set id [lindex $ids 3]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]+$bd] [expr [lindex $coords 1]+$bd] \
	    [expr [lindex $coords 2]-$bd] [expr [lindex $coords 3]+$bd]
    # right arc
    set id [lindex $ids 4]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]-$bd] [expr [lindex $coords 1]+$bd] \
	    [expr [lindex $coords 2]-$bd] [expr [lindex $coords 3]+$bd]
    # right strip
    set id [lindex $ids 5]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]-$bd] [expr [lindex $coords 1]+$bd] \
	    [expr [lindex $coords 2]-$bd] [lindex $coords 3]
    _$n raise $id
    # bottom strip
    set id [lindex $ids 6]
    set coords [_$n coords $id]
    _$n coords $id [expr [lindex $coords 0]+$bd] [lindex $coords 1] \
	    [expr [lindex $coords 2]-$bd] [lindex $coords 3]
    _$n lower $id
}

# Creates an empty frame (used for the contents) and adds it to the
# notebook's canvas
#
proc createContentsFrame {n win_ptr ids_ptr} {
    variable cnum
    upvar $win_ptr win $ids_ptr ids
    getMemberVars $n

    set win [frame $n.c[incr cnum] -bg $bg]
    pack propagate $win 0
    grid propagate $win 0

    set x $pad
    set y [expr $tabheight-$bd+$pad]
    set height [expr $height-$tabheight+$bd-$pad]

    # create frame
    lappend ids [$n create window [expr $x+$bd] [expr $y+$bd] \
	    -height [expr $height - 2*$bd] -width [expr $width - 2*$bd] \
	    -window $win -anchor nw]
    # left strip
    lappend ids [$n create rectangle $x $y [expr $x+$bd] [expr $y+$height] -fill $bgl -outline {}]
    # top strip
    lappend ids [$n create rectangle $x $y [expr $x+$width] [expr $y+$bd] -fill $bgl -outline {}]
    # right strip
    lappend ids [$n create rectangle [expr $x+$width-$bd] [expr $y+$bd] [expr $x+$width] [expr $y+$height] -fill $bgd -outline {}]
    # bottom strip
    lappend ids [$n create rectangle [expr $x+$width] [expr $y+$height] [expr $x+$bd] [expr $y+$height-$bd] -fill $bgd -outline {}]
}

proc resizeContentsFrame {n} {
    variable $n
    getMemberVars $n

    # Get new height and width of canvas
    set height [expr [winfo height $n] - 2]
    set width [expr [winfo width $n] - 2]

    set win [vget $n frame]
    set ids [vget $n frame_ids]

    set x $pad
    set y [expr $tabheight-$bd+$pad]
    set height [expr $height-$tabheight+$bd-$pad]

    # frame id
    set id [lindex $ids 0]
    _$n coords $id [expr $x+$bd] [expr $y+$bd]
    _$n itemconfigure $id -height [expr $height - 2*$bd] -width [expr $width - 2*$bd]
    # left strip id
    set id [lindex $ids 1]
    _$n coords $id $x $y [expr $x+$bd] [expr $y+$height]
    # top strip id
    set id [lindex $ids 2]
    _$n coords $id $x $y [expr $x+$width] [expr $y+$bd]
    # right strip id
    set id [lindex $ids 3]
    _$n coords $id [expr $x+$width-$bd] [expr $y+$bd] [expr $x+$width] [expr $y+$height]
    # bottom strip id
    set id [lindex $ids 4]
    _$n coords $id [expr $x+$width] [expr $y+$height] [expr $x+$bd] [expr $y+$height-$bd]
}



namespace export notebook

}

namespace import NotebookWidget::notebook
