# application-snap_sds.tcl --
#
#       The SDS server implementation.  SDS services register their attributes
#       with the server and others query it later.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import Timer/Periodic Application SRMv2_Session SRMv2_Source CServiceTableEntry

#
# Code for SdsApplication
#
Class SdsApplication -superclass Application

#
# SdsApplication instproc init { vargs }
#
# Input: vargs - list of strings - the command line arguments
#

SdsApplication instproc init { argv } {
    $self next "snap_sds"

    # deal the command line arguments
    set options [$self options]
    $self InitArguments $options
    $self InitResources $options

    # parse the arguments
    set argv [$options parse_args $argv]

    # now deal with the snap session stuff
    $self instvar m_snapSession

    set inetAddr [$self get_option optServiceAddress]
    set iSPort [$self get_option optServicePort]
    set iRPort [$self get_option optServicePort]
    set iTTL [$self get_option optServiceTTL]

    set m_snapSession [new CSDSSession $inetAddr $iSPort $iRPort $iTTL]
}

SdsApplication instproc InitArguments { options } {
    $options register_option -sa optServiceAddress
    $options register_option -sp optServicePort
    $options register_option -st optServiceTTL
}

SdsApplication instproc InitResources { options } {
    $options add_default optServiceAddress "224.4.6.8"
    $options add_default optServicePort "12344"
    $options add_default optServiceTTL "16"
}

#
# Code for CSDSSession
#
# this object pretty much does the bulk of the work in the application
# the purpose of this object is to keep a simple table with containers ids
# the containers will be created when a service in the network periodically
# announces that it has a service to offer.  The service will have a path
# from the root.  This path will serve as the path of the containers.
#
Class CSDSSession -superclass {SRMv2_Session Timer/Periodic}

#
# CSDSSession constructor
#
# input: inetAddr - a string with the internet address
#        iSPort - sending port
#        iRPort - receiving port
#

CSDSSession instproc init {inetAddr iSPort iRPort iTTL} {
    $self next $inetAddr $iSPort $iRPort $iTTL

    # initialize this array
    $self instvar m_aServiceTable
    set m_aServiceTable("") ""

    # create the one and only source object
    $self instvar m_SDSSource
    $self instvar m_cidQueryRequest

    set m_SDSSource [new SRMv2_Source $self ]

    # set the source info
    $m_SDSSource app_info { "Service Discover Service Source" }

    # create some default containers
    set m_cidQueryRequest [$m_SDSSource calloc 0 "Query Response"]

    # Ok now let's set the timer to go off periodically
    # when the timer goes off it will call timeout
    $self start
}

#
# CSDSSession instproc srm_recv { src cid seqno data }
#
# this is a call back function that is overloaded from SRMv2_Session object
#
# input: same as the srm_rec of SRMv2_Session object
#
CSDSSession instproc srm_recv { src cid seqno data } {
    $self instvar m_aServiceTable

    # ok so far there are only two types of messages that should
    # arrive, query requests and update messages
    if {[lindex $data 0] == "UPDATE:"} {

	set index [concat [$src source_id] $cid]

	# if there's an entry already then create it, else
	# just update the time on the one that's already existing
	if {![info exists m_aServiceTable($index)]} {
	    set tableEntry [new CServiceTableEntry $src $cid \
		    [lindex $data 1] [lindex $data 2]]
	    set m_aServiceTable($index) $tableEntry
	} else {
	    set tableEntry $m_aServiceTable($index)

	    # update the with new info
	    $tableEntry ContactInfo [lindex $data 1]
	    $tableEntry ServiceProperty [lindex $data 2]
	}

	$tableEntry UpdateTime
    }

    # ok if it's not an UPDATE, then it can be a QUERY
    # for now just return everything
    # FIXME this is really basic and need to be improved
    if {[lindex $data 0] == "QUERY_REQUEST:"} {
	set lSendData "QUERY_RESPONSE:"

	foreach {index tableEntry} [array get m_aServiceTable] {
	    if {$tableEntry != {}} {
		# collect all the info into the lSendData list
		lappend lSendData [$tableEntry ContactInfo]
		lappend lSendData [$tableEntry ServiceProperty]
	    }
	}

	# send the data out to the querier
	$self instvar m_SDSSource m_cidQueryRequest
	$m_SDSSource send $m_cidQueryRequest $lSendData
    }
}

#
# CSDSSession instproc srm_read_adu { src cid seqno } {
#
# don't repair anything
#
# input: same as the srm_rec of SRMv2_Session object
#
CSDSSession instproc srm_read_adu { src cid seqno } {
    # no recovery mechanism
}

#
# CSDSSession public srm_should_recover { src cid sseq eseq } {
#
# don't repair any losses
#
# input: same as the srm_rec of SRMv2_Session object
#
CSDSSession instproc srm_should_recover { src cid sseq eseq } {
    return 0
}


#
# CSDSSession instproc timeout {} {
#
# time out functions.  debug output to print out the table entries
# periodically.  Also decrement the time entry and if it reaches zero
# then remove it
#
# input: same as the srm_rec of SRMv2_Session object
#
CSDSSession instproc timeout {} {
    $self instvar m_aServiceTable

    foreach {index tableEntry} [array get m_aServiceTable] {
	if {$tableEntry != {}} {
	    # decrement the time for table entry
	    $tableEntry Time [expr [$tableEntry Time] - 1]
	    if {[$tableEntry Time] <= 0} {
		unset m_aServiceTable($index)
		delete $tableEntry 
	    }
	}
    }

    # this is debug outputs
#    foreach {index tableEntry} [array get m_aServiceTable] {
#	if {$tableEntry != {}} {
#	    puts -nonewline "src=[$tableEntry Source];"
#	    puts -nonewline "cid=[$tableEntry Cid];"
#	    puts -nonewline "time=[$tableEntry Time];"
#	    puts -nonewline "contact=[$tableEntry ContactInfo];"
#	    puts "prop=[$tableEntry ServiceProperty];"
#	}
#    }
}
