# mash-server.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/applications/pathfinder/mash-server.tcl,v 1.7 2002/02/03 04:22:06 lim Exp $


import HTTP_Server


#
# The MASH_Server class is a basic HTTP server that is geared towards
# simplifying the Webifying of MASH tools. The server can contain any
# number of HTTP_Agent objects each of which handle certain client
# requests. These agents are added through the add_agent method. Refer
# to the abstract HTTP_Agent class for specifics on implementing agents.
#
Class HTTP_Server/MASH_Server -superclass HTTP_Server


HTTP_Server/MASH_Server public init { } {

	$self next
	$self instvar html_dir_

	# Initialize the html directory.
	#set html_dir_ ~/mash/tcl/applications/mash_server/html/
	#set o [$self options]
	#$o load_preferences "mserver"
	set html_dir_ [$self get_option html_dir]
}


#
# The add_agent method takes an HTTP_Agent object and adds it to the
# current list of agents managed by the MASH_Server.
#
HTTP_Server/MASH_Server public add_agent { agent } {
	$self instvar agents_
	lappend agents_ $agent
}


#
# When the MASH_Server receives a client HTTP request, the handle_request
# method is called. In turn, the server looks to each of its agents and
# has each process the request until one of the agents handles the
# request. The data returned by the agent is then returned to the client
# in the form of an HTTP response.
#
HTTP_Server/MASH_Server public handle_request { socket hdr_var data } {
	$self instvar agents_ html_dir_
	upvar $hdr_var headers

	set u [$self extract_key $headers(url)]
	set url [lindex $u 0]
	set key [lindex $u 1]

	if { $url == "/" } {
		set url /index.html
	}

	set data ""
	set isRawData 0
	set page { }
	set type "text/html"

	# Find out the source of this request.
	set source [$self find_source $socket]

	# create a reply array
	set reply(headers) {}
	set reply(data) {}
	set reply(status) -1

	foreach a $agents_ {
		# A page is returned if the url was recognized by the
		# agent.  In this case, break out of the for loop.
		set r [$a handle_request $url $key $source reply]
		puts "[$a info class] returned $r"
		if $r {
			puts "headers: $reply(headers)\nstatus $reply(status)"
			break
		}
	}

	if { $reply(status) == -1 } {
		# none of the agents managed to handle this URL
		# return the error page

		# assume the last agent in the list was the static agent
		if ![$a handle_request /notfound.html "" $source reply] {
			set reply(headers) [list content-type text/html]
			set reply(data) "<HTML><HEAD><TITLE>404 Not found\
					</TITLE></HEAD>\
					\n<BODY><h1>404 Not found</h1></BODY>\
					\n</HTML>"
		}
		set reply(status) 404
	}

	$self send_reply $socket reply
}


#
# The get_data method extracts the data portion of the page returned
# by the agent.
#
HTTP_Server/MASH_Server private get_data { page } {
	return [lindex $page 0]
}


#
# The get_status method extracts the status portion of the page
# returned by the agent.
#
HTTP_Server/MASH_Server private get_status { page } {
	return [lindex $page 1]
}


#
# The get_type method extracts the status portion of the page
# returned by the agent.
#
HTTP_Server/MASH_Server private get_type { page } {
	return [lindex $page 2]
}


#
# The extract_key method extracts the key from the magic urls.  The
# standard form of a magic URL is [description]^[unique key].
#
HTTP_Server/MASH_Server private extract_key { url } {
	set offset [string last ^ $url]
	if { $offset < 0 } { return [list $url {}] }
	set key [string range $url [expr $offset + 1] end]
	set url [string range $url 0 [expr $offset-1]]
	return [list $url [string trimleft [string tolower $key] -:]]
}


#
# The find_source method uses the socket to determine the source
# of a HTTP request.
#
HTTP_Server/MASH_Server private find_source { socket } {

	# Find the hostname of the client.
	set chan [$socket channel]
	set host [lindex [fconfigure $chan -peername] 1]
	mtrace trcNet "-> Client hostname: $host"

	return $host
}

# -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=


proc write_to_file { filename string } {
	set fileid [open $filename "WRONLY CREAT TRUNC"]
	puts $fileid $string
	close $fileid
}


proc file_dump { filename } {
	set fileid [open $filename "r"]
	set data [read $fileid]
	close $fileid
	return $data
}


proc prepend_to_file { filename string } {

	# Read the original data in the file
	set orig [read_file $filename]

	# Truncate all the data in the file
	set fileid [open $filename "WRONLY CREAT TRUNC"]
	close $fileid

	# Prepend string to the file
	set fileid [open $filename "WRONLY CREAT APPEND"]
	puts $fileid $string
	puts $fileid $orig
	close $fileid
}


proc read_file { filename } {
	set exists [file exists $filename]
	set data ""
	if { $exists == 1 } {
		set fileid [open $filename "r"]
		set data [read $fileid]
		close $fileid
	}
	return $data
}


proc get_key { program } {
	return [string tolower [string trimleft [$program unique_key] -:]]
}


proc edit_html { str } {

	# Replace "<" and ">" so that they are not specially treated if
	# there is an email address within them.
	#    regsub -all -- (<)(\[^>\]*)(@)(\[^>\]*)(>) $str {\&lt\2\3\4\&gt} str
	regsub -all -- < $str {\&lt;} str
	regsub -all -- > $str {\&gt;} str

	return $str
}

