/*
 * ns-srmv2.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <tcl.h>

#include "tclcl.h"
#include "ntp-time.h"
#include "srmv2.h"
#include "ns-srmv2.h"
#include "source-srmv2.h"
#include "random.h"

static class SRMv2_NameSpaceClass : public TclClass {
public:
	SRMv2_NameSpaceClass() : TclClass("SRMv2_NameSpace") { }
	TclObject* create(int, const char*const*) {
		return new SRMv2_NameSpace(0);
	}
} srmv2_ns_class;


SRMv2_NameSpace::SRMv2_NameSpace(NameMap *map, u_int32_t rootID)
{
	/* Actually, this should be random, but let's worry about that later */
	cid_ = 0;
	root_ = new NS_Node(cid_++);
	namemap_ = map;
}

int
SRMv2_NameSpace::command(int argc, const char*const* argv)
{
	return (TclObject::command(argc, argv));
}

NS_Node*
SRMv2_NameSpace::calloc(unsigned int parent, int *created, unsigned int cid)
{
	NS_Node *node  = namemap_->getnode(namemap_->getname(parent));
	if (!cid) cid = cid_++;
	if (node)
		return node->calloc(cid, created);
	return 0;
}
/*
 * Recursively compute the signature of nodes in the
 * tree, starting from 'node'. Since the 128-bit MD-5
 * hash function is too long, we XOR the individual
 * random 32-bit quantities to get a 32-bit quantity.
 * Since MD-5 guarantees that the 128-bits generated
 * are random, so is their XOR.
 */
void
SRMv2_NameSpace::compute_signature()
{
	root_->compute_signature();
}


NS_Node::NS_Node(unsigned int oid, int max_degree, int max_dbufs) :
	degree_(0), max_degree_(max_degree), ndbufs_(0), max_dbufs_(max_dbufs),
	cid_(oid), child_(0)
{
	if (max_degree > 0)
		child_ = new NS_Node*[max_degree];
	if (max_dbufs > 0)
		dbufs_ = new adubuf[max_dbufs];
}

NS_Node::~NS_Node()
{
	for (int i = 0; i < degree_; i ++)
		delete [] child_[i];
	delete [] child_;
}

/*
 * writef() generates a repair request.
 */
void
NS_Node::writef(pktbuf *pb, SRMv2_Source* src, SRMv2_Session* session,
		u_int32_t seqno, Bool eoa)
{
	/* FIXME else, need to grow the array */
	if (seqno >= (unsigned) max_dbufs_) {
		printf("# slots in node exceeded\n");
		abort();
	}
	SRMv2_Edge left_edge(seqno, pb->layer);

	/* Compute the next byte past the last one we received. */
	SRMv2_Edge right_edge;
	if (eoa)
		right_edge.set_edge(seqno + 1, 0);
	else
		right_edge.set_edge(seqno, pb->layer + pb->len);

	/*
	 * We do not explicitly query the application through the
	 * session object to before requesting a repair. We need a
	 * way to garbage collect unassembled fragments.
	 */
	dbufs_[seqno].insert(pb);
	requests_.cancel(cid_, left_edge, right_edge);
	replies_.cancel(cid_, left_edge, right_edge);

	/*
	 * Check for missing pieces only if there is receiver interest.
	 * If this returns 1, a repair request is scheduled for the
	 * entire range. Both start and end are included in the range.
	 */
	check_holes(src, session, left_edge, eoa);

	/* Update the right edge if necessary. */
	if (expected_ < right_edge)
		expected_.set_edge(right_edge.adu, right_edge.byte);
}

/* check_holes --
 * Use left to detect a gap. If one is found, schedule a request.
 */
int
NS_Node::check_holes(SRMv2_Source* src, SRMv2_Session* session,
		     SRMv2_Edge& left, Bool eoa)
{
	int gap = 0;
	if (expected_ < left) {
		gap = 1;
		unsigned int last;
		if (left.byte == 0)
			last = left.adu - 1;
		else last = left.adu;

		if (session->recover(src->namemap()->getname(cid_), expected_.adu, last)) {
			SRMv2_Range *range = new SRMv2_Range(cid_, expected_, left);
			requests_.insert_range(range);
			float period = session->backoff(src, 1);
			range->sched_timeout(period, session, src);
		}
	}
	return gap;
}

void
NS_Node::set_edge(unsigned int a, unsigned int b) {
	expected_.set_edge(a, b);
}

NS_Node*
NS_Node::is_child(unsigned int id)
{
	NS_Node *child;

        for (int i = 0; i < degree_; i++) {
		child = child_[i];
		if (id == child->getcid())
			return child;
	}
	return NULL;
}

NS_Node*
NS_Node::child(unsigned int id)
{
	if (id < (unsigned) degree_) {
		return child_[id];
	} else return 0;
}

NS_Node*
NS_Node::calloc(unsigned int cid, int *created)
{
	NS_Node *child = is_child(cid);
	*created = 0;

	if (!child) {
		*created = 1;
		child = new NS_Node(cid);
		if (degree_ < max_degree_)
			child_[degree_++] = child;
	}
	return child;
}

adubuf*
NS_Node::reassemble(int seqno)
{
	if (dbufs_[seqno].concatenate() > 0)
		return &dbufs_[seqno];
	else return 0;
}

void
NS_Node::regadu(unsigned int seqno, unsigned int ebytes)
{
	if ((unsigned) dbufs_[seqno].len < ebytes + 1)
		dbufs_[seqno].len = ebytes + 1;
	/*
	 * Since this is only meaningful on the sender's side,
	 * we need not schedule a RREQ. We must push ndbufs_
	 * to the right, and never to the left.
	 */
	if ((unsigned) ndbufs_ < seqno)
		ndbufs_ = seqno;
	expected_.set_edge(seqno + 1, 0);
}

adubuf*
NS_Node::get_nodeinfo()
{
	int i;
	adubuf* buffer = new adubuf;
	buffer->data = (unsigned char *) new srmv2_announcehdr[degree_ + 1];
	srmv2_announcehdr* ahdr = (srmv2_announcehdr*) buffer->data;
	/* First fill in data about this node : ndbufs_ and ebytes */
	ahdr[0].cid    = cid_;
	ahdr[0].sign   = ndbufs_;
	ahdr[0].ebytes = expected_.byte;
	for (i=0; i<degree_; i++) {
		ahdr[i+1].cid    = child_[i]->getcid();
		ahdr[i+1].sign   = child_[i]->signature()->adu;
		ahdr[i+1].ebytes = child_[i]->signature()->byte;
	}
	buffer->len = (degree_ + 1) * sizeof(srmv2_announcehdr);
	return buffer;
}

Bool
NS_Node::match_signature(SRMv2_Edge *sign)
{
	return (*sign == signature_);
}

void
NS_Node::display(int indent)
{
	int i;
	for (i = 0; i < indent; i++)
		printf("   ");
	printf("%4d | %d %d\n", cid_, signature_.adu, signature_.byte);
	for (i = 0; i < degree_; i++)
		child_[i]->display(indent+1);
}

adubuf::adubuf() : len(0), unused(0),
	seqno(SRMv2_ERROR), data(0),
	dp(0), fraglist(0), lastseen(0)
{ }

adubuf::adubuf(unsigned int length) : len(length), fraglist(0)
{
	data = new unsigned char[length];
	dp = data;
}

adubuf::~adubuf()
{
	if (data) delete [] data;
}

void
adubuf::copy(unsigned char *buffer, int buflen)
{
	len = buflen;
	data = new unsigned char [len+1];
	memcpy(data, buffer, buflen);
	dp = data;
}

/*
 * Caller must ascertain that there is enough space.
 * The only way to do this now is to allocate enough
 * at the beginning.
 */

void
adubuf::append(unsigned *buffer, int buflen)
{
	memcpy(data + unused, buffer, buflen);
	unused += buflen;
	dp += buflen;
}


void
adubuf::insert(pktbuf *pb)
{
	pktbuf *prev = 0;
	pktbuf *curr;
	int i;

	if (pb->layer + pb->len > len) {
		len = pb->layer + pb->len;
	}

	/*
	 * Do nothing if the current ADU was already passed
	 * up to the application.
	 */
	if (fraglist == 0 && lastseen == 1) {
		pb->release();
		return;
	}

        /*
         * Find a segment which begins after this one does.
         */
	for (curr = fraglist; curr != 0; prev = curr, curr = curr->next)
		if (curr->layer > pb->layer)
			break;

	/*
         * If there is a preceding segment, it may provide some of
         * our data already.  If so, drop the data from the incoming
         * segment.  If it provides all of our data, drop us.
         */
	if (prev != 0) {
		i = prev->layer + prev->len - pb->layer;
		if (i > 0) {
			if (i >= pb->len)
				pb->release();
			else {
				pb->layer += i;
				pb->dp  += i;
				pb->len -= i;
				prev->next = pb;
				pb->next = curr;
				prev = pb;
			}
		} else {
			prev->next = pb;
			pb->next = curr;
			prev = pb;
		}
	} else {
		pb->next = curr;
		fraglist = pb;
	}

	/*
         * While we overlap succeeding segments trim them or,
         * if they are completely covered, dequeue them.
         */
	while (curr != 0 && pb->layer + pb->len > curr->layer) {
		i = (pb->layer + pb->len) - curr->layer;
		if (i < curr->len) {
			curr->len -= i;
			curr->layer += i;
			curr->dp += i;
			break;
		}
		prev->next = curr->next;
		curr->release();
		curr = prev->next;
	}
}

/*
 * Retransmissions should have the "more" bit set to 0
 * This will trigger the following test.
 */
int
adubuf::contiguous()
{
	pktbuf *p;
	unsigned int bytes = 0;
	for (p = fraglist; p != 0; p = p->next) {
		if ((unsigned) p->layer != bytes)
			return (-1);
		bytes += p->len;
	}
	return (bytes);
}

/*
 * Concatenate fragments.
 */
int
adubuf::concatenate()
{
	pktbuf *prev = 0;
	pktbuf *curr = fraglist;

	/*
	 * This returns the length of the ADU.
	 */
	int adulen = contiguous();
	if (adulen > 0) {
		len = adulen;
		data = new unsigned char[adulen+1];
		unsigned char *dp = data;
		while (curr) {
			memcpy(dp, curr->dp, curr->len);
			dp += curr->len;
			prev = curr;
			curr = curr->next;
			if (prev != 0)
				prev->release();
		}
		fraglist = 0;
	}
	return adulen;
}

/*
 * The signature function is computed as
 * s(n) = md5_digest(ndbufs_, s(n1), ..., s(nk)), for internal nodes.
 *              = ndbufs_, for leaf nodes.
 * Since MD-5 only takes strings, we feed it with the concatenation
 * of signatures of nodes n1, n2, ..., nk in hex, prepended by the
 * ndbufs_.
 */

void
NS_Node::compute_signature() {
	int sig_words[4];
	int i = 0;
	/* Leaf node */
	if (degree_ == 0) {
		if (expected_.byte == 0 && expected_.adu > 0) {
			signature_.adu = expected_.adu - 1;
			signature_.byte = dbufs_[expected_.adu - 1].len;
//			printf("Length = %d\n", signature_.byte);
		} else {
			signature_.adu = expected_.adu;
			signature_.byte = expected_.byte;
		}
		return;
	}
//	printf("Internal node...\n");
	/* Internal node */
	char* inpstr = new char[degree_*10 + 10];
	sprintf(inpstr, "%x ", expected_.adu - 1);
	for (i = 0; i < degree_; i ++) {
		child_[i]->compute_signature();
		sprintf(inpstr, "%x ", child_[i]->signature()->adu);
	}
	md5_digest(inpstr, (unsigned char *) &sig_words[0]);
	signature_.adu = sig_words[0];
	for (i = 1; i < 4; i++)
		signature_.adu ^= sig_words[i];
	delete [] inpstr;
	return;
}

