/*
 * manager-srmv2.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <tcl.h>

#include "tclcl.h"
#include "srmv2.h"
#include "ns-srmv2.h"
#include "source-srmv2.h"
#include "manager-srmv2.h"

static class SRMv2_ManagerClass : public TclClass {
public:
	SRMv2_ManagerClass() : TclClass("SRMv2_Manager") { }
	TclObject* create(int , const char*const* ) {
		return new SRMv2_Manager;
	}
} srmv2_manager_class;


int
SRMv2_Manager::command(int argc, const char*const* argv)
{

	if (strcmp(argv[1], "session") == 0) {
		session_ = (SRMv2_Session*)TclObject::lookup(argv[2]);
		return (TCL_OK);
	}
	return (TclObject::command(argc, argv));
}

void
SRMv2_Manager::handle_data(SRMv2_Source *src, pktbuf *pb, unsigned int cid,
			 int seqno, unsigned char type, u_int32_t offset, int more)
{
	Tcl& tcl = Tcl::instance();
	Tcl_Obj *objv[5];
	adubuf *ab = 0;

	NameMap *map = src->namemap();
	NS_Node *node = map->getnode(map->getname(cid));

	/* If this is a namespace packet, (from container #1),
	   create the node */
	if (cid == SRMv2_MAPPINGS)
		map->copen((char *)pb->dp, seqno);

	/* Reassemble ADU and pass up to the application */
	if (node) {
		node->writef(pb, src, session_, seqno, !more);
//		node->requests()->display();
		if (!more) node->set_lastseen(seqno);
		if (node->lastseen(seqno)) {
			ab = node->reassemble(seqno);
			/* Pass up to the application via tcl */
			if (ab) {
				char *label = map->getname(cid);
				char *otcl_name = (char *)name();
//				printf("Received %s [%d B]\n\n", label, ab->len);
				objv[0] = Tcl_NewStringObj(otcl_name, -1);
				objv[1] = Tcl_NewStringObj("recv", -1);
				objv[2] = Tcl_NewStringObj(label, -1);
				objv[3] = Tcl_NewIntObj(seqno);
				objv[4] = Tcl_NewStringObj((char *)ab->data, ab->len);
				tcl.evalObjs(5, objv);
				delete ab->data;
			}
		}
	}
}

void
SRMv2_Manager::handle_rreq(SRMv2_Source *src, u_int32_t ss, unsigned int sbytes,
			   u_int32_t es, unsigned int ebytes,
			   unsigned int cid)
{
	unsigned int i;
	adubuf *adu;
	NameMap *map = src->namemap();
	NS_Node *nd  = map->getnode(map->getname(cid));
	/* Normal request processing */
	SRMv2_Edge left(ss, sbytes);
	SRMv2_Edge right(es, ebytes);
	if (nd) {
		nd->requests()->backoff(cid, left, right);
		/* Special name space repair request. */
		if (ss == SRMv2_ERROR &&  es == SRMv2_ERROR) {
			adu = nd->get_nodeinfo();
			srmv2_announcehdr *ahdr = (srmv2_announcehdr *) adu->data;
			int count = adu->len/sizeof(srmv2_announcehdr);
			session_->send_announce(src, ahdr, count, SRMv2_NSREPAIR);
			return;
		}
		/*
		 * Read ADU to make sure that we are capable of handling this request.
		 * We register a request_to_send() with session_. The actual transmission
		 * occurs when the session object invokes the application again.
		 */
		for (i = ss; i <= es; i ++) {
			adu = reada(src, cid, i);
			if (adu != 0) {
				/* Send only 'sbytes' to 'ebytes' of this ADU */
				if (i == ss && i == es) {
					if (ebytes - sbytes > 0)
						prepare_to_send(src, cid, (unsigned) i, sbytes, ebytes - 1);
					delete adu;
					continue;
				}

				/* Send 'sbytes' to 'EoA' */
				if (i == ss) {
					if (adu->len - sbytes + 1 > 0)
						prepare_to_send(src, cid, i, sbytes, adu->len - 1);
					delete adu;
					continue;
				}

				/* Send 0 to 'ebytes - 1' */
				if (i == es) {
					if (ebytes > 0)
						prepare_to_send(src, cid, i, 0, ebytes - 1);
					delete adu;
					continue;
				}

				/* Send all of this ADU */
				if (adu->len > 0)
					prepare_to_send(src, cid, i, 0, adu->len - 1);
				delete adu;
				continue;
			}
		}
	}
}

void
SRMv2_Manager::prepare_to_send(SRMv2_Source *src, unsigned int cid, unsigned int seqno,
			       unsigned int ss, unsigned int es)
{
	NameMap *map = src->namemap();
	char* aduname   = map->getname(cid);
	Tcl& tcl = Tcl::instance();
	tcl.evalf("%s enqueue %s %d %d %d", name(), aduname, seqno, ss, es);
	return;
}

void
SRMv2_Manager::handle_announce(SRMv2_Source *src, unsigned int cid, unsigned int signature, unsigned int ebytes)
{
	NameMap *map = src->namemap();
	NS_Node *node  = map->getnode(map->getname(cid));
	SRMv2_Edge sign(signature, ebytes);

	/*
	 * If we have no node allocated for this CID, we ignore it
	 * until the mapping arrives. The mapping automatically
	 * allocates a node.
	 */

	if (node && !node->match_signature(&sign)) {
		/* Mismatch */
		if (node->degree() != 0) { /* Internal node */
			session_->send_rreq(src, SRMv2_ERROR, SRMv2_ERROR,
					    SRMv2_ERROR, SRMv2_ERROR, cid);
		} else { /* Leaf */
			SRMv2_Edge left_edge(signature, ebytes);
			SRMv2_Edge right_edge(signature, ebytes);

			/*
			 * For a leaf node, the 'signature' is the last ADU
			 * seqno. in that container.
			 */
			node->check_holes(src, session_, left_edge, 1);
			right_edge.advance(node->lastseen(right_edge.adu));
			node->set_edge(right_edge.adu, right_edge.byte);
		}
	}
}

void
SRMv2_Manager::handle_nsrepair(SRMv2_Source *src, srmv2_announcehdr *hdrs, int count)
{

	unsigned int cid, sign, ebytes;
	unsigned int pid, plast, pebytes;

	NameMap *map = src->namemap();
	pid = ntohl(hdrs[0].cid);
	plast = ntohl(hdrs[0].sign);
	pebytes = ntohl(hdrs[0].ebytes);

	NS_Node *parent = map->getnode(map->getname(pid));
	parent->compute_signature();
	SRMv2_Edge *e = new SRMv2_Edge(SRMv2_ERROR, SRMv2_ERROR);
	parent->requests()->cancel(pid, *e, *e);
	parent->replies()->cancel(pid, *e, *e);
	for (int i = 1; i < count; i++) {
		cid = ntohl(hdrs[i].cid);
		sign = ntohl(hdrs[i].sign);
		ebytes = ntohl(hdrs[i].ebytes);
		if (!parent->is_child(cid)) {
			int created;
			parent->calloc(cid, &created);
				parent->compute_signature();
		}
		handle_announce(src, cid, sign, ebytes);
	}
}

/* This must also pass the source ID */
adubuf*
SRMv2_Manager::reada(SRMv2_Source *src, unsigned int cid, unsigned int seqno)
{
	Tcl& tcl = Tcl::instance();
	adubuf* ab = 0;
	char *data = 0;
	char *adu_name = 0;

	NameMap *map = src->namemap();
	if (map)
		adu_name = map->getname(cid);

	if (adu_name) {
		if (strcmp(adu_name, "/map") == 0) {
			data = map->getname(seqno);
		} else {
			tcl.evalf("%s reada %s %d", name(), adu_name, seqno);
			data = tcl.result();
		}
		if (data > 0 && strlen(data) > 0) {
			int len = strlen(data);
			ab = new adubuf;
			ab->copy((unsigned char*) data, len);
		}
	}
	return ab;
}

